/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: CreateWarpVectorSurface.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <iostream>
// VTK Common
#include "vtkMath.h"
#include "vtkPolygon.h"
#include "vtkCellData.h"
#include "vtkGenericCell.h"
#include "vtkDataArray.h"
#include "vtkFloatArray.h"
#include "vtkLookupTable.h"
#include "vtkPointData.h"
#include "vtkDataSet.h"
#include "vtkPolyData.h"
#include "vtkDataSetCollection.h"
// VTK Filtering
#include "vtkDataSetToDataSetFilter.h"
#include "vtkCastToConcrete.h"
// VTK Graphics
#include "vtkProgrammableAttributeDataFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkWarpVector.h"
#include "vtkPolyDataNormals.h"


// ----------------------------------------------------------------------------
static void
ExecutePolyNormalDot (void* aPtr)
{
  vtkProgrammableAttributeDataFilter* filter =
    reinterpret_cast<vtkProgrammableAttributeDataFilter*>(aPtr);

  vtkDataSet* d0;
  vtkDataSet* d1;
  vtkDataSet* d2;

  // Input to 'polyNormalDot' (inflationVector->GetPolyDataOutput()).
  if ((d0 = filter->GetInput()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): failed to get input!");
    return;
    }

  std::cerr << "  0: " << d0->GetSource()->GetClassName()<< std::endl;

  // Input to 'inflationVector' (pointVectors->GetPolyDataOutput()).
  if ((d1 = vtkDataSet::SafeDownCast(*(d0->GetSource()->GetInputs()))) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): failed to get secondary input!");
    return;
    }

  std::cerr << "  1: " << d1->GetSource()->GetClassName()<< std::endl;

  // Input to 'pointVectors' (inputNormals->GetOutput()).
  if ((d2 = vtkDataSet::SafeDownCast(*(d1->GetSource()->GetInputs()))) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): failed to get tertiary input!");
    return;
    }

  std::cerr << "  2: " << d2->GetSource()->GetClassName()<< std::endl;

  vtkDataSet* inflatedDataSet = d0;
  vtkDataSet* originalDataSet = d2;
  vtkIdType   numCells;
  vtkIdType   numPoints;

  if ((numCells = inflatedDataSet->GetNumberOfCells()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): inflated input has no cells!");
    return;
    }
  if (originalDataSet->GetNumberOfCells() < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): original input has no cells!");
    return;
    }
  if ((numPoints = inflatedDataSet->GetNumberOfPoints()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): inflated input has no points!");
    return;
    }
  if (originalDataSet->GetNumberOfPoints() < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePolyNormalDot(" << aPtr
                           << "): original input has no points!");
    return;
    }

  //originalDataSet->Update();

  std::cerr << "ExecutePolyNormalDot(" << aPtr << "): "
            << numCells << " cells in inflated DataSet, "
            << originalDataSet->GetNumberOfCells()
            << " cells in original DataSet." << std::endl;

  vtkDataSet*    outputDataSet  = filter->GetOutput();
  vtkCellData*   outputCellData = outputDataSet->GetCellData();
  vtkDataArray*  outputScalars;
  vtkFloatArray* floatArray;

  if ( ((outputScalars = outputCellData->GetScalars()) == NULL) ||
       (outputScalars->GetNumberOfTuples() != numCells) ||
       (outputScalars->GetNumberOfComponents() != 1) ||
       (outputScalars->GetDataType() != VTK_FLOAT) )
    {
    floatArray = vtkFloatArray::New();
    floatArray->SetNumberOfValues(numCells);
    floatArray->SetName("Scalars");
    outputScalars = floatArray;
    
    outputCellData->SetScalars(outputScalars);
    outputCellData->CopyScalarsOn();
    outputCellData->SetActiveScalars("Scalars");
    outputScalars->Delete();
    }
  else
    {
    floatArray = vtkFloatArray::SafeDownCast(outputScalars);
    }

  //unsigned long outputScalarsMTime = outputScalars->GetMTime();

  //std::cerr << "ExecutePolyNormalDot(" << aPtr << ") output scalars MTime: "
  //          << outputScalarsMTime << std::endl;

  //if ( (originalDataSet->GetMTime() > outputScalarsMTime) ||
  //    (inflatedDataSet->GetMTime() > outputScalarsMTime) )
  //  {
  vtkGenericCell* originalCell = vtkGenericCell::New();
  vtkGenericCell* inflatedCell = vtkGenericCell::New();
  float n1[3], n2[3], p1[3], p2[3], dot;

  for (vtkIdType cellId=0; cellId<numCells; cellId++)
    {
    originalDataSet->GetCell(cellId, originalCell);
    inflatedDataSet->GetCell(cellId, inflatedCell);
    // Compute the unit normal for these point Ids.
    vtkPolygon::ComputeNormal(originalCell->GetPoints(), n1);
    vtkPolygon::ComputeNormal(inflatedCell->GetPoints(), n2);
    // Compute the scalar product of the unit normals.
    dot = vtkMath::Dot(n1, n2);
    floatArray->SetValue(cellId, dot);
    }
  //}

  for (vtkIdType pointId=0; pointId<numPoints; pointId++)
    {
    originalDataSet->GetPoint(pointId, p1);
    inflatedDataSet->GetPoint(pointId, p2);
  

    }

  vtkLookupTable* lookupTable;

  if ((lookupTable = outputScalars->GetLookupTable()) == NULL)
    {
    float range[2];
    
    lookupTable = vtkLookupTable::New();
    outputScalars->GetRange(range);
    lookupTable->SetValueRange(range);
    outputScalars->SetLookupTable(lookupTable);
    lookupTable->Delete();
    
    std::cerr << "\nScalar product range: [ " << range[0]
              << " .. " << range[1] << " ]\n" << std::endl;
    }
}

// ----------------------------------------------------------------------------
static void
ExecutePointVectors (void* aPtr)
{
  vtkProgrammableAttributeDataFilter* filter =
    reinterpret_cast<vtkProgrammableAttributeDataFilter*>(aPtr);

  vtkDataSet* inDataSet  = filter->GetInput();
  vtkDataSet* outDataSet = filter->GetOutput();
  vtkIdType   numPoints;

  if ((inDataSet == NULL) || ((numPoints = inDataSet->GetNumberOfPoints()) < 1))
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors(" << aPtr << "):"
                           << " no input!");
    return;
    }

  std::cerr << "\nInput PolyData ( " << inDataSet << " ): " << numPoints
            << " points.\n" << std::endl;

  vtkPointData* inPointData;

  if ((inPointData = inDataSet->GetPointData()) != NULL)
    {
    vtkDataArray* inVectors;
    vtkDataArray* inNormals;

    //std::cerr << "\nInput PointData: " << std::endl;
    //inPointData->Print(std::cerr);

    if ((inNormals = inPointData->GetNormals()) == NULL)
      {
      vtkGenericWarningMacro(<< "ExecutePointVectors(" << aPtr
                             << "): Failed to get input normals!");
      return;
      }

    vtkPointData*  outPointData;

    if ((outPointData = outDataSet->GetPointData()) != NULL)
      {
      vtkDataArray*  outVectors;
  
      if ((inVectors = inPointData->GetVectors()) == NULL)
        {
        // 
        // We use the ProgrammableAttributeDataFilter here to simply to 
        // deep copy existing normals or vectors as the output's vectors. 
        // 
        if ((outVectors = outPointData->GetVectors()) == NULL)
          {
          outVectors = vtkFloatArray::New();
          outVectors->DeepCopy(inNormals);
          outVectors->SetName("Vectors");

          outPointData->SetVectors(outVectors);
          outPointData->CopyVectorsOn();
          outPointData->SetActiveVectors("Vectors");
          outVectors->Delete();
          }
        else
          {
          outVectors->DeepCopy(inNormals);
          outVectors->SetName("Vectors");
          }
        }
      else
        {
        //
        // We have input point data vectors.
        //
        vtkDataArray*  outScalars;
        vtkFloatArray* floatArray;

        if ( ((outScalars = outPointData->GetScalars()) == NULL) ||
             (outScalars->GetNumberOfTuples() != numPoints) ||
             (outScalars->GetNumberOfComponents() != 1) ||
             (outScalars->GetDataType() != VTK_FLOAT) )
          {
          floatArray = vtkFloatArray::New();
          floatArray->SetNumberOfValues(numPoints);
          floatArray->SetName("Scalars");
          outScalars = floatArray;

          outPointData->SetScalars(outScalars);
          outPointData->CopyScalarsOn();
          outPointData->SetActiveScalars("Scalars");
          outScalars->Delete();
          }
        else
          {
          floatArray = vtkFloatArray::SafeDownCast(outScalars);
          }

        // Be sure that the output point data has a usable vector array.
        if ( ((outVectors = outPointData->GetVectors()) == NULL) ||
             (outVectors->GetNumberOfTuples() != numPoints) ||
             (outVectors->GetNumberOfComponents() != 3) ||
             (outVectors->GetDataType() != VTK_FLOAT) )
          {
          outVectors = vtkFloatArray::New();
          outVectors->SetNumberOfComponents(3);
          outVectors->SetNumberOfTuples(numPoints);
          outVectors->SetName("Vectors");

          outPointData->SetVectors(outVectors);
          outPointData->CopyVectorsOn();
          outPointData->SetActiveVectors("Vectors");
          outVectors->Delete();
          }

        unsigned long outVectorsMTime = outVectors->GetMTime();

        if ( (inNormals->GetMTime() > outVectorsMTime) ||
             (inVectors->GetMTime() > outVectorsMTime) ||
             (outScalars->GetMTime() < outVectorsMTime) )
          {
          float vec1[3], vec2[3], dot;
          float mag1, mag2;
          float cosAngle = cos(double(vtkMath::Pi()) / 2.0); // 90 degrees

          for (vtkIdType i=0; i<numPoints; i++)
            {
            inNormals->GetTuple(i,vec1);
            inVectors->GetTuple(i,vec2);

            mag1 = vtkMath::Normalize(vec1);
            mag2 = vtkMath::Normalize(vec2);
            dot  = vtkMath::Dot(vec1,vec2);

            floatArray->SetValue(i,dot);

            for(int j=0; j<3; j++) vec2[j] *= -1.f;

            outVectors->SetTuple(i,vec2);
            }
          }

        vtkLookupTable* lookupTable;

        if ((lookupTable = outScalars->GetLookupTable()) == NULL)
          {
          float range[2];

          lookupTable = vtkLookupTable::New();
          outScalars->GetRange(range);
          lookupTable->SetValueRange(range);
          outScalars->SetLookupTable(lookupTable);
          lookupTable->Delete();
          }
        }

      //std::cerr << "\nOutput PointData: " << std::endl;
      //outPointData->Print(std::cerr);
      }
    else
      {
      vtkGenericWarningMacro(<< "ExecutePointVectors(" << aPtr
                             << "): Failed to get output point data.");
      }

    }
  else
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors(" << aPtr
                           << "): Failed to get input point data.");
    }
}


// ----------------------------------------------------------------------------
static void
ExecutePointVectors2 (void* aPtr)
{
  vtkProgrammableAttributeDataFilter* filter =
    reinterpret_cast<vtkProgrammableAttributeDataFilter*>(aPtr);

  vtkDataSet*    inDataSet;
  vtkPointSet*   inPointSet;
  vtkPoints*     inPoints;
  vtkPointData*  inPointData;
  vtkDataArray*  inNormals;
  vtkIdType      numPoints;
  vtkIdType      numCells;

  if ((inDataSet = filter->GetInput()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input DataSet!");
    return;
    }
  if ((inPointSet = vtkPointSet::SafeDownCast(inDataSet)) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to cast input to PointSet!");
    return;
    }
  if ((inPoints = inPointSet->GetPoints()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input Points!");
    return;
    }
  if ((numPoints = inPointSet->GetNumberOfPoints()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): no input Points!");
    return;
    }
  if ((numCells = inPointSet->GetNumberOfCells()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): no input Cells!");
    return;
    }
  if ((inPointData = inPointSet->GetPointData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input PointData!");
    return;
    }
  if ((inNormals = inPointData->GetNormals()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input Normals!");
    return;
    }

  vtkDataArray* inScalars = inPointData->GetScalars();

  std::cerr << "\nInput Points ( " << inPoints << " ): " << numPoints
            << " points.\n" << std::endl;

  vtkDataSet*    outDataSet;
  vtkPointSet*   outPointSet;
  vtkPoints*     outPoints;
  vtkPointData*  outPointData;
  vtkCellData*   outCellData;

  if ((outDataSet = filter->GetOutput()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output DataSet!");
    return;
    }
  else if (outDataSet == inDataSet)
    {
    vtkPolyData* polyData = vtkPolyData::New();
    polyData->DeepCopy(inDataSet);
    filter->SetOutput(polyData);
    polyData->Delete();
    }
  if ((outPointSet = vtkPointSet::SafeDownCast(outDataSet)) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to cast output to PointSet!");
    return;
    }
  if ((outPoints = outPointSet->GetPoints()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output Points!");
    return;
    }
  if ((outPointData = outPointSet->GetPointData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output PointData.");
    return;
    }
  if ((outCellData = outPointSet->GetCellData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output CellData.");
    return;
    }

  vtkFloatArray* outVectors;

  if ( (outVectors = vtkFloatArray::SafeDownCast(outPointData->GetVectors()))
       == NULL )
    {
    outVectors = vtkFloatArray::New();
    outVectors->SetNumberOfComponents(3);
    outVectors->SetNumberOfTuples(numPoints);
    outVectors->Initialize();
    outVectors->SetName("Vectors");

    outPointData->SetVectors(outVectors);
    outPointData->CopyVectorsOn();
    outPointData->SetActiveVectors("Vectors");
    outVectors->Delete();
    }

  std::cerr << "\nOutput Points ( " << outPoints << " ): " << numPoints
            << " points.\n" << std::endl;

  unsigned long outPointsMTime = outPoints->GetMTime();

  if ( (inPoints->GetMTime() > outPointsMTime) ||
       (inNormals->GetMTime() > outPointsMTime) )
    {
    float n[3], p[3], v[3];

    if (inScalars != NULL)
      {
      if (inScalars->GetMTime() < outPointsMTime)
        {
        float s;
      
        for (vtkIdType i=0; i<numPoints; i++)
          {
          inPoints->GetPoint(i,p);
          inNormals->GetTuple(i,n);
          inScalars->GetTuple(i,&s);
          for (int j=0; j<3; j++) v[j] = n[j] * s;
          outPoints->SetPoint(i, p[0]+v[0], p[1]+v[1], p[2]+v[2]);
          outVectors->SetTuple(i, v);
          }
        }
      }
    else
      {
      for (vtkIdType i=0; i<numPoints; i++)
        {
        inPoints->GetPoint(i,p);
        inNormals->GetTuple(i,n);
        for (int j=0; j<3; j++) v[j] = n[j];
        outPoints->SetPoint(i, p[0]+v[0], p[1]+v[1], p[2]+v[2]);
        outVectors->SetTuple(i, v);
        }
      }

    vtkPolyData* outPolyData;

    if ((outPolyData = vtkPolyData::SafeDownCast(outPointSet)) == NULL)
      {
      vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                             << "): Failed to cast output to PolyData.");
      return;
      }

    outPolyData->BuildCells();
    //outPolyData->BuildLinks();

    //
    // We have input point data vectors.
    //
    vtkDataArray*  outScalars;
    vtkFloatArray* floatArray = NULL;

    if (inScalars != NULL)
      {

      if ( ((outScalars = outPointData->GetScalars()) == NULL) ||
           (outScalars->GetNumberOfTuples() != numPoints) ||
           (outScalars->GetNumberOfComponents() != 1) ||
           (outScalars->GetDataType() != VTK_FLOAT) )
        {
        floatArray = vtkFloatArray::New();
        floatArray->SetNumberOfValues(numPoints);
        floatArray->SetName("Scalars");
        outScalars = floatArray;

        outPointData->SetScalars(outScalars);
        outPointData->CopyScalarsOn();
        outPointData->SetActiveScalars("Scalars");
        outScalars->Delete();
        }
      else
        {
        floatArray = vtkFloatArray::SafeDownCast(outScalars);
        }

      floatArray->DeepCopy(inScalars);
      }

    vtkGenericCell* inCell = vtkGenericCell::New();
    vtkGenericCell* outCell = vtkGenericCell::New();
    vtkIdList*      inPointIds;
    vtkIdList*      outPointIds;

    float n1[3], n2[3], dot, vv0[3], vv1[3], vv2[3], vv[3];
    float ip[3][3], op[3][3], ie[3][3], oe[3][3], im[3], om[3], ea[3];
    vtkIdType outIds[3];

    for (vtkIdType cellId=0; cellId<numCells; cellId++)
      {
      inPointSet->GetCell(cellId, inCell);
      outPolyData->GetCell(cellId, outCell);

      // Compute the unit normal for these point Ids.
      vtkPolygon::ComputeNormal(inCell->GetPoints(), n1);
      vtkPolygon::ComputeNormal(outCell->GetPoints(), n2);

      // Compute the scalar product of the unit normals.
      if ((dot = vtkMath::Dot(n1, n2)) < 0)
        {
        inPointIds  = inCell->GetPointIds();
        outPointIds = outCell->GetPointIds();

        if ( (inPointIds->GetNumberOfIds() != 3) ||
             (outPointIds->GetNumberOfIds() != 3) )
          {
          vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                                 << "): Found illegal non-triangle Cell.");
          break;
          }

        // Vertices.
        for (int j=0; j<3; j++)
          {
          inPointSet->GetPoint(inPointIds->GetId(j) , ip[j]);
          outPointSet->GetPoint((outIds[j] = outPointIds->GetId(j)) , op[j]);
          }

        // Edge 0 (vertex 0 -> 1).
        ie[0][0] = ip[0][0] - ip[1][0]; oe[0][0] = op[0][0] - op[1][0];
        ie[0][1] = ip[0][1] - ip[1][1]; oe[0][1] = op[0][1] - op[1][1];
        ie[0][2] = ip[0][2] - ip[1][2]; oe[0][2] = op[0][2] - op[1][2];
        // Edge 1 (vertex 1 -> 2).
        ie[1][0] = ip[1][0] - ip[2][0]; oe[1][0] = op[1][0] - op[2][0];
        ie[1][1] = ip[1][1] - ip[2][1]; oe[1][1] = op[1][1] - op[2][1];
        ie[1][2] = ip[1][2] - ip[2][2]; oe[1][2] = op[1][2] - op[2][2];
        // Edge 2 (vertex 2 -> 0).
        ie[2][0] = ip[2][0] - ip[0][0]; oe[2][0] = op[2][0] - op[0][0];
        ie[2][1] = ip[2][1] - ip[0][1]; oe[2][1] = op[2][1] - op[0][1];
        ie[2][2] = ip[2][2] - ip[0][2]; oe[2][2] = op[2][2] - op[0][2];

        for (int j=0; j<3; j++)
          {
          im[j] = vtkMath::Normalize(ie[j]);
          om[j] = vtkMath::Normalize(oe[j]);
          // Cosine of angle between original and inflated edges.
          ea[j] = vtkMath::Dot(ie[j], oe[j]);
          }

        int count = 0; // count reversed edges.
        for (int j=0; j<3; j++) if (ea[j] < 0) count++;

        float value;

        if      (count == 1)
          {
          if      (ea[0] < 0)
            {
            outVectors->GetTuple(outIds[0], vv0);
            outVectors->GetTuple(outIds[1], vv1);
            for (int j=0; j<3; j++)
              {
              vv[j] = (vv0[j] + vv1[j]) / 2.f;
              v[j] = (op[0][j] + op[1][j]) / 2.f;
              }
            outPoints->SetPoint(outIds[0], v);
            outPoints->SetPoint(outIds[1], v);
            if (floatArray != NULL)
              {
              value = ( floatArray->GetValue(outIds[0]) +
                        floatArray->GetValue(outIds[1]) ) / 2.f;
              floatArray->SetValue(outIds[0], value);
              floatArray->SetValue(outIds[1], value);
              }
            outVectors->SetTuple(outIds[0], vv);
            outVectors->SetTuple(outIds[1], vv);
            outPolyData->DeleteCell(cellId);
            }
          else if (ea[1] < 0)
            {
            outVectors->GetTuple(outIds[1], vv1);
            outVectors->GetTuple(outIds[2], vv2);
            for (int j=0; j<3; j++)
              {
              vv[j] = (vv1[j] + vv2[j]) / 2.f;
              v[j] = (op[1][j] + op[2][j]) / 2.f;
              }
            outPoints->SetPoint(outIds[1], v);
            outPoints->SetPoint(outIds[2], v);
            if (floatArray != NULL)
              {
              value = ( floatArray->GetValue(outIds[1]) +
                        floatArray->GetValue(outIds[2]) ) / 2.f;
              floatArray->SetValue(outIds[1], value);
              floatArray->SetValue(outIds[2], value);
              }
            outVectors->SetTuple(outIds[1], vv);
            outVectors->SetTuple(outIds[2], vv);
            outPolyData->DeleteCell(cellId);
            }
          else if (ea[2] < 0)
            {
            outVectors->GetTuple(outIds[2], vv2);
            outVectors->GetTuple(outIds[0], vv0);
            for (int j=0; j<3; j++)
              {
              vv[j] = (vv2[j] + vv0[j]) / 2.f;
              v[j] = (op[2][j] + op[0][j]) / 2.f;
              }
            outPoints->SetPoint(outIds[2], v);
            outPoints->SetPoint(outIds[0], v);
            if (floatArray != NULL)
              {
              value = ( floatArray->GetValue(outIds[2]) +
                        floatArray->GetValue(outIds[0]) ) / 2.f;
              floatArray->SetValue(outIds[2], value);
              floatArray->SetValue(outIds[0], value);
              }
            outVectors->SetTuple(outIds[2], vv);
            outVectors->SetTuple(outIds[0], vv);
            outPolyData->DeleteCell(cellId);
            }
          }
        else if (count >= 2)
          {
          outVectors->GetTuple(outIds[0], vv0);
          outVectors->GetTuple(outIds[1], vv1);
          outVectors->GetTuple(outIds[2], vv2);
          for (int j=0; j<3; j++)
            {
            vv[j] = (vv0[j] + vv1[j] + vv2[j]) / 3.f;
            v[j] = (op[0][j] + op[1][j] + op[2][j]) / 3.f;
            }
          outPoints->SetPoint(outIds[0], v);
          outPoints->SetPoint(outIds[1], v);
          outPoints->SetPoint(outIds[2], v);
          if (floatArray != NULL)
            {
            value = ( floatArray->GetValue(outIds[0]) +
                      floatArray->GetValue(outIds[1]) +
                      floatArray->GetValue(outIds[2]) ) / 3.f;
            floatArray->SetValue(outIds[0], value);
            floatArray->SetValue(outIds[1], value);
            floatArray->SetValue(outIds[2], value);
            }
          outVectors->SetTuple(outIds[0], vv);
          outVectors->SetTuple(outIds[1], vv);
          outVectors->SetTuple(outIds[2], vv);
          outPolyData->DeleteCell(cellId);
          }

        }

      }

    inCell->Delete();
    outCell->Delete();

    vtkLookupTable* lookupTable;
    
    if ((lookupTable = outScalars->GetLookupTable()) == NULL)
      {
      float range[2];
      
      lookupTable = vtkLookupTable::New();
      outScalars->GetRange(range);
      lookupTable->SetValueRange(range);
      outScalars->SetLookupTable(lookupTable);
      lookupTable->Delete();
      }
    }
}

// ----------------------------------------------------------------------------
static void
ExecutePointVectors3 (void* aPtr)
{
  vtkProgrammableAttributeDataFilter* filter =
    reinterpret_cast<vtkProgrammableAttributeDataFilter*>(aPtr);

  vtkDataSet*    inDataSet;
  vtkPointSet*   inPointSet;
  vtkPoints*     inPoints;
  vtkPointData*  inPointData;
  vtkDataArray*  inNormals;
  vtkIdType      numPoints;
  vtkIdType      numCells;

  if ((inDataSet = filter->GetInput()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input DataSet!");
    return;
    }
  if ((inPointSet = vtkPointSet::SafeDownCast(inDataSet)) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to cast input to PointSet!");
    return;
    }
  if ((inPoints = inPointSet->GetPoints()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input Points!");
    return;
    }
  if ((numPoints = inPointSet->GetNumberOfPoints()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): no input Points!");
    return;
    }
  if ((numCells = inPointSet->GetNumberOfCells()) < 1)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): no input Cells!");
    return;
    }
  if ((inPointData = inPointSet->GetPointData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input PointData!");
    return;
    }
  if ((inNormals = inPointData->GetNormals()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get input Normals!");
    return;
    }

  vtkDataArray* inScalars = inPointData->GetScalars();

  std::cerr << "\nInput Points ( " << inPoints << " ): " << numPoints
            << " points.\n" << std::endl;

  vtkDataSet*    outDataSet;
  vtkPointSet*   outPointSet;
  vtkPoints*     outPoints;
  vtkPointData*  outPointData;
  vtkCellData*   outCellData;

  if ((outDataSet = filter->GetOutput()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output DataSet!");
    return;
    }
  else if (outDataSet == inDataSet)
    {
    vtkPolyData* polyData = vtkPolyData::New();
    polyData->DeepCopy(inDataSet);
    filter->SetOutput(polyData);
    polyData->Delete();
    }
  if ((outPointSet = vtkPointSet::SafeDownCast(outDataSet)) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to cast output to PointSet!");
    return;
    }
  if ((outPoints = outPointSet->GetPoints()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output Points!");
    return;
    }
  if ((outPointData = outPointSet->GetPointData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output PointData.");
    return;
    }
  if ((outCellData = outPointSet->GetCellData()) == NULL)
    {
    vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                           << "): Failed to get output CellData.");
    return;
    }

  vtkFloatArray* outVectors;

  if ( (outVectors = vtkFloatArray::SafeDownCast(outPointData->GetVectors()))
       == NULL )
    {
    outVectors = vtkFloatArray::New();
    outVectors->SetNumberOfComponents(3);
    outVectors->SetNumberOfTuples(numPoints);
    outVectors->Initialize();
    outVectors->SetName("Vectors");

    outPointData->SetVectors(outVectors);
    outPointData->CopyVectorsOn();
    outPointData->SetActiveVectors("Vectors");
    outVectors->Delete();
    }

  std::cerr << "\nOutput Points ( " << outPoints << " ): " << numPoints
            << " points.\n" << std::endl;

  unsigned long outPointsMTime = outPoints->GetMTime();

  if ( (inPoints->GetMTime() > outPointsMTime) ||
       (inNormals->GetMTime() > outPointsMTime) )
    {
    float n[3], p[3], v[3];

    if (inScalars != NULL)
      {
      if (inScalars->GetMTime() < outPointsMTime)
        {
        float s;
      
        for (vtkIdType i=0; i<numPoints; i++)
          {
          inPoints->GetPoint(i,p);
          inNormals->GetTuple(i,n);
          inScalars->GetTuple(i,&s);
          for (int j=0; j<3; j++) v[j] = -n[j] * s;
          outPoints->SetPoint(i, p[0]+v[0], p[1]+v[1], p[2]+v[2]);
          outVectors->SetTuple(i, v);
          }
        }
      }
    else
      {
      for (vtkIdType i=0; i<numPoints; i++)
        {
        inPoints->GetPoint(i,p);
        inNormals->GetTuple(i,n);
        for (int j=0; j<3; j++) v[j] = -n[j];
        outPoints->SetPoint(i, p[0]+v[0], p[1]+v[1], p[2]+v[2]);
        outVectors->SetTuple(i, v);
        }
      }

    vtkPolyData* outPolyData;

    if ((outPolyData = vtkPolyData::SafeDownCast(outPointSet)) == NULL)
      {
      vtkGenericWarningMacro(<< "ExecutePointVectors2(" << aPtr
                             << "): Failed to cast output to PolyData.");
      return;
      }

    outPolyData->BuildCells();
    //outPolyData->BuildLinks();
    }
}

// ----------------------------------------------------------------------------
vtkDataSetToDataSetFilter*
CreateInflatedSurface (vtkPolyData* aPolyData, float D)
{
  vtkCastToConcrete* castToConcrete = vtkCastToConcrete::New();
    {
    vtkProgrammableAttributeDataFilter* polyNormalDot =
      vtkProgrammableAttributeDataFilter::New();
      {
      vtkWarpVector* inflationVector = vtkWarpVector::New();
        {
        vtkProgrammableAttributeDataFilter* pointVectors =
          vtkProgrammableAttributeDataFilter::New();
          {
          // 
          // We use the ProgrammableAttributeDataFilter here to simply to 
          // deep copy existing normals or vectors as the output's vectors. 
          // 
          vtkPolyDataNormals* inputNormals = vtkPolyDataNormals::New();
            {
            vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
              {
              cleanPolyData->SetInput(aPolyData);
              cleanPolyData->ConvertStripsToPolysOn();
              cleanPolyData->PointMergingOn();
              }
            inputNormals->SetInput(cleanPolyData->GetOutput());
            inputNormals->SplittingOff();
            inputNormals->ConsistencyOn();
            inputNormals->ComputePointNormalsOn();
            inputNormals->ComputeCellNormalsOff();
            inputNormals->FlipNormalsOff();
            inputNormals->NonManifoldTraversalOff();
            inputNormals->ReleaseDataFlagOff();
            cleanPolyData->Delete();
            }
          pointVectors->SetInput(inputNormals->GetOutput());
          pointVectors->SetExecuteMethod( ExecutePointVectors,
                                          (void *) pointVectors );
          inputNormals->Delete();
          }
        inflationVector->SetInput(pointVectors->GetPolyDataOutput());
        inflationVector->SetScaleFactor(D);
        pointVectors->Delete();
        }
      polyNormalDot->SetInput(inflationVector->GetPolyDataOutput());
      polyNormalDot->SetExecuteMethod( ExecutePolyNormalDot,
                                       (void *) polyNormalDot );
      inflationVector->Delete();
      }
    castToConcrete->SetInput(polyNormalDot->GetPolyDataOutput());
    polyNormalDot->Delete();
    }
  
  return castToConcrete;
}

vtkDataSetToDataSetFilter*
CreateInflatedSurface2 (vtkPolyData* aPolyData, float D)
{
  vtkCastToConcrete* castToConcrete = vtkCastToConcrete::New();
    {
    vtkProgrammableAttributeDataFilter* inflaterFilter =
      vtkProgrammableAttributeDataFilter::New();
      {
      vtkPolyDataNormals* inputNormals = vtkPolyDataNormals::New();
        {
        vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
          {
          cleanPolyData->SetInput(aPolyData);
          cleanPolyData->ConvertStripsToPolysOn();
          cleanPolyData->PointMergingOn();
          }
        inputNormals->SetInput(cleanPolyData->GetOutput());
        inputNormals->SplittingOff();
        inputNormals->ConsistencyOn();
        inputNormals->ComputePointNormalsOn();
        inputNormals->ComputeCellNormalsOff();
        inputNormals->FlipNormalsOff();
        inputNormals->NonManifoldTraversalOff();
        inputNormals->ReleaseDataFlagOff();
        cleanPolyData->Delete();

        inputNormals->Update();
        vtkPolyData*  pd   = inputNormals->GetOutput();
        vtkIdType     numPoints = pd->GetNumberOfPoints();
        vtkPointData* attr = pd->GetPointData();
        vtkDataArray* scalars = attr->GetScalars();

        if (D != 1)
          {
          vtkFloatArray* fa;

          if (scalars == NULL)
            {
            fa = vtkFloatArray::New();
            fa->SetNumberOfValues(numPoints);
            fa->SetName("Scalars");
            scalars = fa;

            attr->SetScalars(scalars);
            attr->CopyScalarsOn();
            attr->SetActiveScalars("Scalars");
            scalars->Delete();
            }
          else
            {
            fa = static_cast<vtkFloatArray*>(scalars);
            }

          for (vtkIdType j=0; j<numPoints; j++)
            fa->SetValue(j, D);
          }
        else if (scalars != NULL)
          {
          attr->SetScalars(NULL);
          }
        }
      inflaterFilter->SetInput(inputNormals->GetOutput());
      inflaterFilter->SetExecuteMethod( ExecutePointVectors2,
                                        (void *) inflaterFilter );
      inputNormals->Delete();
      }
    castToConcrete->SetInput(inflaterFilter->GetPolyDataOutput());
    inflaterFilter->Delete();
    }
  
  return castToConcrete;
}

// ----------------------------------------------------------------------------
vtkDataSetToDataSetFilter*
CreateDeflatedSurface (vtkPolyData* aPolyData, float D)
{
  vtkCastToConcrete* castToConcrete = vtkCastToConcrete::New();
    {
    vtkWarpVector* deflationVector = vtkWarpVector::New();
      {
      // 
      // We use the ProgrammableAttributeDataFilter to compute the cosine
      // of the angle between the two vector fields (i.e. the dot product 
      // normalized by the product of the vector lengths).
      // 
      vtkProgrammableAttributeDataFilter* pointVectors =
        vtkProgrammableAttributeDataFilter::New();
        {
        vtkPolyDataNormals* warpedNormals = vtkPolyDataNormals::New();
          {
          warpedNormals->SetInput(aPolyData);
          warpedNormals->SplittingOff();
          warpedNormals->ConsistencyOn();
          warpedNormals->ComputePointNormalsOn();
          warpedNormals->ComputeCellNormalsOff();
          warpedNormals->FlipNormalsOff();
          warpedNormals->NonManifoldTraversalOff();
          }
        pointVectors->SetInput(warpedNormals->GetOutput());
        pointVectors->SetExecuteMethod( ExecutePointVectors,
                                        (void *) pointVectors );
        warpedNormals->Delete();
        }
      deflationVector->SetInput(pointVectors->GetPolyDataOutput());
      deflationVector->SetScaleFactor(D);
      pointVectors->Delete();
      }
    castToConcrete->SetInput(deflationVector->GetPolyDataOutput());
    deflationVector->Delete();
    }

  return castToConcrete;
}

vtkDataSetToDataSetFilter*
CreateDeflatedSurface2 (vtkPolyData* aPolyData, float D)
{
  vtkCastToConcrete* castToConcrete = vtkCastToConcrete::New();
    {
    vtkProgrammableAttributeDataFilter* deflaterFilter =
      vtkProgrammableAttributeDataFilter::New();
      {
      vtkPolyDataNormals* normalsFilter = vtkPolyDataNormals::New();
        {
        normalsFilter->SetInput(aPolyData);
        normalsFilter->SplittingOff();
        normalsFilter->ConsistencyOn();
        normalsFilter->ComputePointNormalsOn();
        normalsFilter->ComputeCellNormalsOff();
        normalsFilter->FlipNormalsOff();
        normalsFilter->NonManifoldTraversalOff();
        }
      deflaterFilter->SetInput(normalsFilter->GetOutput());
      deflaterFilter->SetExecuteMethod( ExecutePointVectors3,
                                        (void *) deflaterFilter );
      normalsFilter->Delete();
      }
    castToConcrete->SetInput(deflaterFilter->GetPolyDataOutput());
    deflaterFilter->Delete();
    }

  return castToConcrete;
}

/* 
 * End of: $Id: CreateWarpVectorSurface.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $.
 * 
 */
