/* 
 * $Id: SurfaceWarp.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $
 * 
 */
#include <iostream>
// SurfaceWarp
#include "SurfaceWarp.H"
// SVV
#include "svvController.h"

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);

static StringList inputArgs;

static void PrintUsage(ostream&, const char*);
static int  ParseArgs(int argc, char** argv, int& i);


int
main (int argc, char* argv[], char* envp[])
{
  // Note that this will create a vtkMPIController if MPI
  // is configured, vtkThreadedController otherwise.
  vtkMultiProcessController* multiProcessController;

  multiProcessController = vtkMultiProcessController::New();
  multiProcessController->Initialize(&argc, &argv);

  // When using MPI, the number of processes is determined
  // by the external program which launches this application.
  // However, when using threads, we need to set it ourselves.
  if (multiProcessController->IsA("vtkThreadedController"))
    {
    // Set the number of processes to 4 for this example.
    multiProcessController->SetNumberOfProcesses(4);
    // Forces controller to use deep copies instead of shallow. This is 
    // needed when asynch. processing occurs (i.e. pipeline parallelism).
    multiProcessController->ForceDeepCopyOn();
    }

  if (multiProcessController->GetNumberOfProcesses() != 4)
    {
    std::cerr << "SurfaceWarp requires four processes." << std::endl;
    multiProcessController->Finalize();
    multiProcessController->Delete();
    return 1;
    }

  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if (!(ret = svvController::Args(&argc, &argv, &envp, i, ParseArgs)))
    {
    PrintUsage(std::cerr,argv[0]);
    std::exit(0);
    }
  else
    {
    if (svv.GetDebug()) std::cerr << "Args() returned "<< ret << std::endl;
    }

  multiProcessController->SetMultipleMethod(0, Warper0, &inputArgs);
  multiProcessController->SetMultipleMethod(1, Warper1, &inputArgs);
  multiProcessController->SetMultipleMethod(2, Warper2, &inputArgs);
  multiProcessController->SetMultipleMethod(3, Warper3, &inputArgs);
  multiProcessController->MultipleMethodExecute();

  multiProcessController->Finalize();
  multiProcessController->Delete();
  
  return 0;
}

// ----------------------------------------------------------------------------
static int
ParseArgs (int argc, char** argv, int& i)
{
  // pointer to the current flag
  const char* flag = argv[i];
  char**      args;

  bool valid = false;

  // determines how many arguments follow the current flag
  int nargs = 0;
  {
  while ((i+nargs+1) < argc && argv[i+nargs+1][0]!='-') nargs++;
    
  if (svv.GetDebug()) std::cerr << " '" << &flag[0] << "' + ( ";

  if (nargs)
    {
    args = argv + std::ptrdiff_t(i+1);
    if (svv.GetDebug())
      {
      for (int n=0; n<nargs; n++) std::cerr<< "'"<< args[n]<< "' ";
      }
    }
  else
    {
    args = 0;
    if (svv.GetDebug()) std::cerr << "none ";
    }

  if (svv.GetDebug()) std::cerr << ")" << std::endl;
  }

  // first check for flags that do not take arguments
  if (args==0)
    {

    // display help and exit
    if (flag[1]=='h' && !flag[2])
      { PrintUsage(std::cerr,(svv.argv())[0]); std::exit(0); }

    // be verbose
    else if(flag[1]=='v' && !flag[2])
      { svv.DebugOn(); inputArgs.push_back(flag); valid = true; }

    // be quiet
    else if(flag[1]=='q' && !flag[2])
      { svv.DebugOff(); inputArgs.push_back(flag); valid = true; }

    }
  else
    {

    // specify input pathnames
    if (flag[1]=='i' && !flag[2])
      {
      inputArgs.push_back(flag);
      for (int n=0; n<nargs; n++)
        {
	inputArgs.push_back(args[n]);
        }
      valid = true;
      } // '-i'

    // specify composite geometry output pathname
    else if (flag[1]=='o' && !flag[2])
      {
      inputArgs.push_back(flag);

      std::string            name = args[0];
      const std::string      sfx  = ".vtk";
      std::string::size_type pos  = name.rfind(sfx);

      if (pos == std::string::npos)
        {
	name += sfx;
        }
      else if (pos != (name.length() - sfx.length()))
        {
	name.replace(pos, (name.length() - pos), sfx);
        }

      inputArgs.push_back(name);

      valid = true;
      } // '-o'

    // specify dmax as fraction of bounding box length.
    else if(flag[1]=='D' && !flag[2])
      {
      inputArgs.push_back(flag);
      inputArgs.push_back(args[0]);
      valid = true;
      } // '-D'
    }
  
  if (valid)
    {
    i += (nargs+1);
    return (nargs+1);
    }
  else
    {
    return 0;
    }
}

// ----------------------------------------------------------------------------
static void
PrintUsage (std::ostream& os, const char* progname)
{
  os << "Usage: " << progname << " -i <input> -o <output>"
     << " [-D <scale fraction>]\n"
     << std::endl;
}

/* 
 * End of: $Id: SurfaceWarp.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $.
 * 
 */
