/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: VoxelToContour.cxx,v 1.1.1.1 2006/12/19 22:59:41 christianh Exp $
 * 
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
// IOStreams
#include <iostream>
// STL
#include <string>
// FLTK
#include <FL/Fl.H>
// VTK Common
#include "vtkImageData.h"
#include "vtkPolyData.h"
#include "vtkLookupTable.h"
#include "vtkScalarsToColors.h"
// VTK Filtering
#include "vtkPiecewiseFunction.h"
#include "vtkColorTransferFunction.h"
// VTK Rendering
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkCamera.h"
#include "vtkPolyDataMapper.h"
#include "vtkCellPicker.h"
#include "vtkTextProperty.h"
#include "vtkProperty.h"
#include "vtkActor.h"
// VTK IO
#include "vtkPolyDataReader.h"
#include "vtkXMLPolyDataReader.h"
// VTK Hybrid
#include "vtkImagePlaneWidget.h"
// SVV
#include "VoxelToContourUI.h"


// ----------------------------------------------------------------------------
static const char*      gInputName = NULL;
static const char*      gReferenceName = NULL;

// ----------------------------------------------------------------------------
static float            gReferenceBounds[6] = { VTK_LARGE_FLOAT,
                                                -VTK_LARGE_FLOAT,
                                                VTK_LARGE_FLOAT,
                                                -VTK_LARGE_FLOAT,
                                                VTK_LARGE_FLOAT,
                                                -VTK_LARGE_FLOAT };
static float            gReferenceCenter[3] = { 0, 0, 0 };
static int              gReferenceWholeExtent[6] = { 0, -1, 0, -1, 0, -1 };

static float            gOutputBounds[6] = { VTK_LARGE_FLOAT,
                                             -VTK_LARGE_FLOAT,
                                             VTK_LARGE_FLOAT,
                                             -VTK_LARGE_FLOAT,
                                             VTK_LARGE_FLOAT,
                                             -VTK_LARGE_FLOAT };
static float            gOutputCenter[3] = { 0, 0, 0 };
static int              gOutputWholeExtent[6] = { 0, -1, 0, -1, 0, -1 };

static float            gInputScalarRange[2] = { 0, 0 };

// ----------------------------------------------------------------------------
// Create a transfer function mapping scalar value to opacity
static vtkPiecewiseFunction*            gOpacityTF = NULL;
// Create a transfer function mapping scalar value to color (color)
static vtkColorTransferFunction*        gColorTF = NULL;

// ----------------------------------------------------------------------------
static vtkActor*                        gReferenceActor = NULL;
static vtkImagePlaneWidget**            gImagePlaneWidgets = NULL;

// ----------------------------------------------------------------------------
static void
print_usage(std::ostream& aTarget, int& aArgc, char**& aArgv)
{
  const char* ptr = strrchr(aArgv[0],'/');
  aTarget << "Usage: " << (ptr ? ptr+1 : aArgv[0]) << std::endl
          << Fl::help << std::endl;
}

// ----------------------------------------------------------------------------
static int
parse_args(int argc, char** argv, int& i)
{
  if(argv[i][1] == 'r' && !argv[i][2] && i+1 < argc)
    {
    gReferenceName = argv[i+1];
    i += 2;
    return 2;
    }
  else if(argv[i][1] == 'h' && !argv[i][2])
    {
    print_usage(std::cerr, argc, argv);
    i += 1;
    return 1;
    }

  return 0;
}

// ----------------------------------------------------------------------------
int
main(int argc, char* argv[])
{
  int i = 1;
  if (!Fl::args(argc,argv,i,parse_args) || i < argc-1)
    {
    print_usage(std::cerr, argc, argv);
    return 1;
    }

  // create the interface
  VoxelToContourUI* GUI = new VoxelToContourUI;

  vtkRenderWindowInteractor* interactor = GUI->GetView()->GetInteractor();
  vtkRenderWindow* renderWindow = GUI->GetView()->GetRenderWindow();
  vtkRenderer* renderer = GUI->GetView()->GetDefaultRenderer();
  vtkCamera* camera = GUI->GetView()->GetDefaultCamera();

  renderWindow->SetDesiredUpdateRate(4.0);
  renderWindow->LineSmoothingOff();
  interactor->SetDesiredUpdateRate(4.0);
  
  const double inputSpacing[3] = { 1.5, 1.5, 3.0 };
  const double resliceRAS[9] = { 1,0,0, 0,1,0, 0,0,1 };
  const double resliceOrigin[3] = { 0, 0, 0 };
  const double isoValue = 0;

  GUI->SetInputSpacingValues(inputSpacing);
  GUI->SetResliceAxesDirectionCosinesValues(resliceRAS);
  GUI->SetResliceAxesOriginValues(resliceOrigin);
  GUI->SetImageThresholdValue(isoValue);

  // show() the main Fl_Window
  GUI->Show();

  // this is the standard way of "starting" a fltk application
  int fl_ret = Fl::run();

  // Clean up
  gImagePlaneWidgets[0]->Delete();
  gImagePlaneWidgets[1]->Delete();
  gImagePlaneWidgets[2]->Delete();
  delete [] gImagePlaneWidgets;
  delete GUI;

  return fl_ret;
}

// ----------------------------------------------------------------------------
static vtkActor*
CreateReferenceActor(const char* const aFileName, VoxelToContourUI* aGUI)
{
  if (gReferenceActor)
    {
    aGUI->GetView()->GetDefaultRenderer()->RemoveProp(gReferenceActor);
    gReferenceActor = NULL;
    }

  if (!aFileName || !*aFileName)
    {
    std::cerr << "Input filename is empty!" << std::endl;
    return NULL;
    }

  std::string fileName(aFileName);
  std::string::size_type idx = fileName.rfind('.');
  std::string suffix(idx==std::string::npos ? "" : fileName.substr(idx+1));

  if (suffix == "vtk")
    {
    gReferenceActor = vtkActor::New();
      {
      vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
        {
        vtkPolyDataReader* reader = vtkPolyDataReader::New();
          {
          reader->SetFileName(aFileName);
          reader->Update();
          reader->GetOutput()->GetBounds(gReferenceBounds);
          reader->GetOutput()->GetCenter(gReferenceCenter);
          reader->GetOutput()->GetWholeExtent(gReferenceWholeExtent);
          }
        mapper->SetInput(reader->GetOutput());
        mapper->ScalarVisibilityOff();
        reader->Delete();
        }
      gReferenceActor->SetMapper(mapper);
      mapper->Delete();
      }
    }
  else if (suffix == "vtp" || suffix == "xml")
    {
    gReferenceActor = vtkActor::New();
      {
      vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
        {
        vtkXMLPolyDataReader* reader = vtkXMLPolyDataReader::New();
          {
          reader->SetFileName(aFileName);
          reader->Update();
          reader->GetOutput()->GetBounds(gReferenceBounds);
          reader->GetOutput()->GetCenter(gReferenceCenter);
          reader->GetOutput()->GetWholeExtent(gReferenceWholeExtent);
          }
        mapper->SetInput(reader->GetOutput());
        mapper->ScalarVisibilityOff();
        reader->Delete();
        }
      gReferenceActor->SetMapper(mapper);
      gReferenceActor->GetProperty()->SetColor(1.f, 1.f, 0.9f);
      gReferenceActor->GetProperty()->SetSpecular(0.5f);
      gReferenceActor->GetProperty()->SetSpecularColor(0.5f, 0.7f, 0.9f);
      gReferenceActor->GetProperty()->SetSpecularPower(24.f);
      mapper->Delete();
      }
    }

  return gReferenceActor;
}

// ----------------------------------------------------------------------------
static void
BuildTransferFunctions(float min, float max)
{
  float span = max - min;

  // Create transfer mapping scalar value to opacity
  if (gOpacityTF) gOpacityTF->RemoveAllPoints();
  else gOpacityTF = vtkPiecewiseFunction::New();

  gOpacityTF->AddSegment(0,                0.0, 0,                0.0);
  gOpacityTF->AddSegment(1,                1.0, min + span * 0.2, 1.0);
  gOpacityTF->AddSegment(min + span * 0.2, 0.0, min + span * 0.8, 0.0);
  gOpacityTF->AddSegment(min + span * 0.8, 1.0, min + span * 1.0, 1.0);

  // Create transfer mapping scalar value to color
  if (gColorTF) gColorTF->RemoveAllPoints();
  else gColorTF = vtkColorTransferFunction::New();

  gColorTF->AddRGBPoint(1,                1.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.1, 0.0, 0.0, 1.0);
  gColorTF->AddRGBPoint(min + span * 0.2, 0.0, 1.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.5, 0.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.8, 1.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.9, 0.0, 0.0, 1.0);
  gColorTF->AddRGBPoint(min + span * 1.0, 0.0, 1.0, 0.0);
}

// ----------------------------------------------------------------------------
void
CheckImagePlaneWidgets(vtkRenderWindowInteractor* aInteractor,
                       vtkImageData* aImageData)
{
  int dimensions[3];

  aImageData->Update();
  aImageData->GetDimensions(dimensions);
  aImageData->GetScalarRange(gInputScalarRange);
  
  std::cerr << "scalar range: [ " << gInputScalarRange[0] << " .. "
            << gInputScalarRange[1] << " ]" << std::endl;

  BuildTransferFunctions(gInputScalarRange[0],gInputScalarRange[1]);

  // --------------------------------------------------------------------------
  if (!gImagePlaneWidgets)
    {
    gImagePlaneWidgets = new vtkImagePlaneWidget* [3];

    // Define the internal picker. The set of three orthogonal planes can 
    // share the same picker so that picking is performed correctly.
    vtkCellPicker* cellPicker = vtkCellPicker::New();
      {
      cellPicker->SetTolerance(0.005);
      }

    // Define the shared property for the display text.
    vtkTextProperty* textProperty = vtkTextProperty::New();
      {
      textProperty->SetColor(1,1,0.9);
      textProperty->ShadowOn();
      textProperty->AntiAliasingOn();
      }

    gImagePlaneWidgets[0] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[0]->SetInteractor(aInteractor);
      gImagePlaneWidgets[0]->SetInput(aImageData);
      gImagePlaneWidgets[0]->SetPicker(cellPicker);
      gImagePlaneWidgets[0]->SetTextProperty(textProperty);
      gImagePlaneWidgets[0]->DisplayTextOn();
      gImagePlaneWidgets[0]->SetKeyPressActivationValue('x');
      gImagePlaneWidgets[0]->GetPlaneProperty()->SetColor(1,0,0);
      gImagePlaneWidgets[0]->
        SetCursorProperty(gImagePlaneWidgets[0]->GetPlaneProperty());
      gImagePlaneWidgets[0]->TextureInterpolateOff();
      gImagePlaneWidgets[0]->SetPlaneOrientationToXAxes();
      gImagePlaneWidgets[0]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
      gImagePlaneWidgets[0]->On();
      }
    gImagePlaneWidgets[1] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[1]->SetInteractor(aInteractor);
      gImagePlaneWidgets[1]->SetInput(aImageData);
      gImagePlaneWidgets[1]->SetPicker(cellPicker);
      gImagePlaneWidgets[1]->SetTextProperty(textProperty);
      gImagePlaneWidgets[1]->DisplayTextOn();
      gImagePlaneWidgets[1]->SetKeyPressActivationValue('y');
      gImagePlaneWidgets[1]->GetPlaneProperty()->SetColor(1,1,0);
      gImagePlaneWidgets[1]->
        SetCursorProperty(gImagePlaneWidgets[1]->GetPlaneProperty());
      gImagePlaneWidgets[1]->TextureInterpolateOff();
      gImagePlaneWidgets[1]->SetPlaneOrientationToYAxes();
      gImagePlaneWidgets[1]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
      gImagePlaneWidgets[1]->On();
      }
    gImagePlaneWidgets[2] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[2]->SetInteractor(aInteractor);
      gImagePlaneWidgets[2]->SetInput(aImageData);
      gImagePlaneWidgets[2]->SetPicker(cellPicker);
      gImagePlaneWidgets[2]->SetTextProperty(textProperty);
      gImagePlaneWidgets[2]->DisplayTextOn();
      gImagePlaneWidgets[2]->SetKeyPressActivationValue('z');
      gImagePlaneWidgets[2]->GetPlaneProperty()->SetColor(0,0,1);
      gImagePlaneWidgets[2]->
        SetCursorProperty(gImagePlaneWidgets[2]->GetPlaneProperty());
      gImagePlaneWidgets[2]->TextureInterpolateOff();
      gImagePlaneWidgets[2]->SetPlaneOrientationToZAxes();
      gImagePlaneWidgets[2]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
      gImagePlaneWidgets[2]->On();
      }

    // Set the internal lut to that of the first plane. In this way, the set 
    // of three orthogonal planes can share the same lut so that 
    // window-levelling is performed uniformly among planes.
    vtkLookupTable* lut = gImagePlaneWidgets[0]->GetLookupTable();
      {
      vtkIdType nValues = lut->GetNumberOfTableValues();
      float rgba[4];
      float value = gInputScalarRange[0];
      float incr = (gInputScalarRange[1]-gInputScalarRange[0])/float(nValues);
    
      for(vtkIdType n=0; n<nValues; n++)
        {
        gColorTF->GetColor(value,rgba);
        rgba[3] = gOpacityTF->GetValue(value);
        lut->SetTableValue(n,rgba);
        value += incr;
        }
      }

    gImagePlaneWidgets[1]->SetLookupTable(lut);
    gImagePlaneWidgets[2]->SetLookupTable(lut);
    }
  else
    {
    gImagePlaneWidgets[0]->SetInteractor(aInteractor);
    gImagePlaneWidgets[0]->SetInput(aImageData);
    gImagePlaneWidgets[0]->SetPlaneOrientationToXAxes();
    gImagePlaneWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
    gImagePlaneWidgets[1]->SetInteractor(aInteractor);
    gImagePlaneWidgets[1]->SetInput(aImageData);
    gImagePlaneWidgets[1]->SetPlaneOrientationToYAxes();
    gImagePlaneWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
    gImagePlaneWidgets[2]->SetInteractor(aInteractor);
    gImagePlaneWidgets[2]->SetInput(aImageData);
    gImagePlaneWidgets[2]->SetPlaneOrientationToZAxes();
    gImagePlaneWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
    }
}

/* 
 * End of: $Id: VoxelToContour.cxx,v 1.1.1.1 2006/12/19 22:59:41 christianh Exp $.
 * 
 */
