/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: VoxelToContourImpl.cxx,v 1.1.1.1 2006/12/19 22:59:42 christianh Exp $
 * 
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
// FLTK
#include <FL/Fl.H>
#include <FL/fl_ask.H> /* fl_message() */
// VTK Common
#include "vtkMatrix4x4.h"
#include "vtkMatrixToLinearTransform.h"
#include "vtkImageData.h"
// VTK Graphics
#include "vtkContourFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkTriangleFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataNormals.h"
#include "vtkCurvatures.h"
#include "vtkMassProperties.h"
// VTK Imaging
#include "vtkImageThreshold.h"
#include "vtkImageResample.h"
#include "vtkImageDilateErode3D.h"
// VTK Rendering
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkActor.h"
#include "vtkActorCollection.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
// VTK IO
#include "vtkImageReader.h"
// SVV
#include "VoxelToContourUI.h"

// ----------------------------------------------------------------------------
int
VoxelToContourUI::Initialize(void)
{
  if (!(this->mImageReader = this->CreateImageReader()))
    {
    Fl::error("Initialize() failed to create ImageReader.\n");
    return 0;
    }
  if (!(this->mImageThreshold = this->CreateImageThreshold()))
    {
    Fl::error("Initialize() failed to create ImageThreshold.\n");
    return 0;
    }
  if (!(this->mImageResample = this->CreateImageResample()))
    {
    Fl::error("Initialize() failed to create ImageResample.\n");
    return 0;
    }
  if (!(this->mResliceMatrix = this->CreateResliceMatrix()))
    {
    Fl::error("Initialize() failed to create ResliceMatrix.\n");
    return 0;
    }
  if (!(this->mResliceTransform = this->CreateResliceTransform()))
    {
    Fl::error("Initialize() failed to create ResliceMatrix.\n");
    return 0;
    }

  this->mResliceScale = 1.0;

  if (!(this->mImagePreThreshold = this->CreateImagePreThreshold()))
    {
    Fl::error("Initialize() failed to create ImagePreThreshold.\n");
    return 0;
    }
  if (!(this->mImageDilateErode3D = this->CreateImageDilateErode3D()))
    {
    Fl::error("Initialize() failed to create ImageDilateErode3D.\n");
    return 0;
    }
  if (!(this->mContourFilter = this->CreateContourFilter()))
    {
    Fl::error("Initialize() failed to create ContourFilter.\n");
    return 0;
    }
  if (!(this->mCleanPolyData = this->CreateCleanPolyData()))
    {
    Fl::error("Initialize() failed to create CleanPolyData.\n");
    return 0;
    }
  if (!(this->mTriangleFilter = this->CreateTriangleFilter()))
    {
    Fl::error("Initialize() failed to create TriangleFilter.\n");
    return 0;
    }
  if (!(this->mWSincPolyDataFilter = this->CreateWSincPolyDataFilter()))
    {
    Fl::error("Initialize() failed to create WSincPolyDataFilter.\n");
    return 0;
    }
  if (!(this->mPolyDataNormals = this->CreatePolyDataNormals()))
    {
    Fl::error("Initialize() failed to create PolyDataNormals.\n");
    return 0;
    }
  if (!(this->mCurvatures = this->CreateCurvatures()))
    {
    Fl::error("Initialize() failed to create Curvatures.\n");
    return 0;
    }
  if (!(this->mMassProperties = this->CreateMassProperties()))
    {
    Fl::error("Initialize() failed to create MassProperties.\n");
    return 0;
    }
  if (!(this->mContourMapper = this->CreateContourMapper()))
    {
    Fl::error("Initialize() failed to create ContourMapper.\n");
    return 0;
    }
  if (!(this->mContourActor = this->CreateContourActor()))
    {
    Fl::error("Initialize() failed to create ContourActor.\n");
    return 0;
    }

  this->mImageThreshold->SetInput(this->mImageReader->GetOutput());
  this->mImageResample->SetInput(this->mImageThreshold->GetOutput());
  this->mResliceTransform->SetInput(this->mResliceMatrix);
  this->mImageResample->SetResliceTransform(this->mResliceTransform);
  this->mImageDilateErode3D->SetInput(this->mImageResample->GetOutput());
  this->mContourFilter->SetInput(this->mImageDilateErode3D->GetOutput());
  this->mCleanPolyData->SetInput(this->mContourFilter->GetOutput());
  this->mTriangleFilter->SetInput(this->mCleanPolyData->GetOutput());
  this->mWSincPolyDataFilter->SetInput(this->mTriangleFilter->GetOutput());
  this->mPolyDataNormals->SetInput(this->mWSincPolyDataFilter->GetOutput());
  this->mCurvatures->SetInput(this->mPolyDataNormals->GetOutput());
  this->mMassProperties->SetInput(this->mCurvatures->GetOutput());
  this->mContourMapper->SetInput(this->mCurvatures->GetOutput());
  this->mContourActor->SetMapper(this->mContourMapper);

  return 1;
}

int
VoxelToContourUI::Finalize(void)
{
  if (this->mImageReader)
    this->mImageReader->Delete();

  if (this->mImageThreshold)
    this->mImageThreshold->Delete();

  if (this->mImageResample)
    this->mImageResample->Delete();

  if (this->mResliceMatrix)
    this->mResliceMatrix->Delete();

  if (this->mResliceTransform)
    this->mResliceTransform->Delete();

  if (this->mImagePreThreshold)
    this->mImagePreThreshold->Delete();

  if (this->mImageDilateErode3D)
    this->mImageDilateErode3D->Delete();

  if (this->mContourFilter)
    this->mContourFilter->Delete();

  if (this->mCleanPolyData)
    this->mCleanPolyData->Delete();

  if (this->mTriangleFilter)
    this->mTriangleFilter->Delete();

  if (this->mWSincPolyDataFilter)
    this->mWSincPolyDataFilter->Delete();

  if (this->mPolyDataNormals)
    this->mPolyDataNormals->Delete();

  if (this->mCurvatures)
    this->mCurvatures->Delete();

  if (this->mMassProperties)
    this->mMassProperties->Delete();

  if (this->mContourMapper)
    this->mContourMapper->Delete();

  if (this->mContourActor)
    {
    this->mViewWindow->RemoveProp(this->mContourActor);
    this->mContourActor->Delete();
    }


  return 1;
}

// ----------------------------------------------------------------------------
vtkImageReader*
VoxelToContourUI::GetImageReader(void)
{
  return this->mImageReader;
}

vtkImageThreshold*
VoxelToContourUI::GetImageThreshold(void)
{
  return this->mImageThreshold;
}

vtkImageResample*
VoxelToContourUI::GetImageResample(void)
{
  return this->mImageResample;
}

vtkMatrix4x4*
VoxelToContourUI::GetResliceMatrix(void)
{
  return this->mResliceMatrix;
}

vtkMatrixToLinearTransform*
VoxelToContourUI::GetResliceTransform(void)
{
  return this->mResliceTransform;
}

vtkImageThreshold*
VoxelToContourUI::GetImagePreThreshold(void)
{
  return this->mImagePreThreshold;
}

vtkImageDilateErode3D*
VoxelToContourUI::GetImageDilateErode3D(void)
{
  return this->mImageDilateErode3D;
}

vtkContourFilter*
VoxelToContourUI::GetContourFilter(void)
{
  return this->mContourFilter;
}

vtkCleanPolyData*
VoxelToContourUI::GetCleanPolyData(void)
{
  return this->mCleanPolyData;
}

vtkTriangleFilter*
VoxelToContourUI::GetTriangleFilter(void)
{
  return this->mTriangleFilter;
}

vtkWindowedSincPolyDataFilter*
VoxelToContourUI::GetWSincPolyDataFilter(void)
{
  return this->mWSincPolyDataFilter;
}

vtkPolyDataNormals*
VoxelToContourUI::GetPolyDataNormals(void)
{
  return this->mPolyDataNormals;
}

vtkCurvatures*
VoxelToContourUI::GetCurvatures(void)
{
  return this->mCurvatures;
}

vtkMassProperties*
VoxelToContourUI::GetMassProperties(void)
{
  return this->mMassProperties;
}

vtkPolyDataMapper*
VoxelToContourUI::GetContourMapper(void)
{
  return this->mContourMapper;
}

vtkActor*
VoxelToContourUI::GetContourActor(void)
{
  return this->mContourActor;
}

// ----------------------------------------------------------------------------
Fl_VTK_Window*
VoxelToContourUI::GetView(void)
{
  return this->mViewWindow;
}

void
VoxelToContourUI::UpdateView(void)
{
  this->mViewWindow->Update();

  if (!this->mViewWindow->shown())
    {
    this->mViewWindow->show();
    }
}

// ----------------------------------------------------------------------------
vtkImageSource*
VoxelToContourUI::GetPrevImageSource(vtkImageToImageFilter* aRef)
{
  if      (aRef == this->mImageDilateErode3D)
    {
    if      (this->mImageDilateErode3DToggle->value())
      if      (this->mPreThresholdImageToggle->value())
        return this->GetImagePreThreshold();
    else if (this->mImageResampleToggle->value())
      return this->GetImageResample();
    }
  else if (aRef == this->mImagePreThreshold)
    {
    if      (this->mImageResampleToggle->value())
      return this->GetImageResample();
    }
  else if (aRef == this->mImageThreshold)
    {
    return this->GetImageReader();
    }

  return this->GetImageThreshold();
}

vtkImageSource*
VoxelToContourUI::GetPrevImageSource(vtkContourFilter* aRef)
{
  if      (this->mImageDilateErode3DToggle->value())
    return this->GetImageDilateErode3D();
  else if (this->mImageResampleToggle->value())
    return this->GetImageResample();
  else
    return this->GetImageThreshold();
}

vtkImageToImageFilter*
VoxelToContourUI::GetNextImageToImageFilter(vtkImageSource* aRef)
{
  if      (aRef == this->mImageReader)
    {
    return this->GetImageThreshold();
    }
  else if (aRef == this->mImageThreshold)
    {
    if      (this->mImageResampleToggle->value())
      return this->GetImageResample();
    else if (this->mImageDilateErode3DToggle->value())
      if      (this->mPreThresholdImageToggle->value())
        return this->GetImagePreThreshold();
      else
        return this->GetImageDilateErode3D();
    }
  else if (aRef == this->mImageResample)
    {
    if      (this->mImageDilateErode3DToggle->value())
      if      (this->mPreThresholdImageToggle->value())
        return this->GetImagePreThreshold();
      else
        return this->GetImageDilateErode3D();
    }
  else if (aRef == this->mImagePreThreshold)
    {
    if      (this->mImageDilateErode3DToggle->value())
      return this->GetImageDilateErode3D();
    }

  return NULL; // i.e. ContourFilter
}

// ----------------------------------------------------------------------------
vtkPolyDataSource*
VoxelToContourUI::GetPrevPolyDataSource(vtkPolyDataToPolyDataFilter* aRef)
{
  if      (aRef == this->mTriangleFilter)
    {
    if      (this->mCleanPolyDataToggle->value())
      return this->GetCleanPolyData();
    }
  else if (aRef == this->mWSincPolyDataFilter)
    {
    if      (this->mTriangleFilterToggle->value())
      return this->GetTriangleFilter();
    else if (this->mCleanPolyDataToggle->value())
      return this->GetCleanPolyData();
    }
  else if (aRef == this->mPolyDataNormals)
    {
    if      (this->mWSincPolyDataFilterToggle->value())
      return this->GetWSincPolyDataFilter();
    else if (this->mTriangleFilterToggle->value())
      return this->GetTriangleFilter();
    else if (this->mCleanPolyDataToggle->value())
      return this->GetCleanPolyData();
    }
  else if (aRef == this->mCurvatures)
    {
    if      (this->mPolyDataNormalsToggle->value())
      return this->GetPolyDataNormals();
    else if (this->mWSincPolyDataFilterToggle->value())
      return this->GetWSincPolyDataFilter();
    else if (this->mTriangleFilterToggle->value())
      return this->GetTriangleFilter();
    else if (this->mCleanPolyDataToggle->value())
      return this->GetCleanPolyData();
    }

  return this->GetContourFilter();
}

vtkPolyDataSource*
VoxelToContourUI::GetPrevPolyDataSource(vtkMassProperties* aRef)
{
  if      (this->mCurvaturesToggle->value())
    return this->GetCurvatures();
  else if (this->mPolyDataNormalsToggle->value())
    return this->GetPolyDataNormals();
  else if (this->mWSincPolyDataFilterToggle->value())
    return this->GetWSincPolyDataFilter();
  else if (this->mTriangleFilterToggle->value())
    return this->GetTriangleFilter();
  else if (this->mCleanPolyDataToggle->value())
    return this->GetCleanPolyData();

  return this->GetContourFilter();
}

vtkPolyDataSource*
VoxelToContourUI::GetPrevPolyDataSource(vtkPolyDataMapper* aRef)
{
  if      (this->mCurvaturesToggle->value())
    return this->GetCurvatures();
  else if (this->mPolyDataNormalsToggle->value())
    return this->GetPolyDataNormals();
  else if (this->mWSincPolyDataFilterToggle->value())
    return this->GetWSincPolyDataFilter();
  else if (this->mTriangleFilterToggle->value())
    return this->GetTriangleFilter();
  else if (this->mCleanPolyDataToggle->value())
    return this->GetCleanPolyData();

  return this->GetContourFilter();
}

// ----------------------------------------------------------------------------
vtkPolyDataToPolyDataFilter*
VoxelToContourUI::GetNextPolyDataToPolyDataFilter(vtkPolyDataSource* aRef)
{
  if      (aRef == this->mContourFilter)
    {
    if      (this->mCleanPolyDataToggle->value())
      return this->GetCleanPolyData();
    else if (this->mTriangleFilterToggle->value())
      return this->GetTriangleFilter();
    else if (this->mWSincPolyDataFilterToggle->value())
      return this->GetWSincPolyDataFilter();
    else if (this->mPolyDataNormalsToggle->value())
      return this->GetPolyDataNormals();
    else if (this->mCurvaturesToggle->value())
      return this->GetCurvatures();
    }
  else if (aRef == this->mCleanPolyData)
    {
    if      (this->mTriangleFilterToggle->value())
      return this->GetTriangleFilter();
    else if (this->mWSincPolyDataFilterToggle->value())
      return this->GetWSincPolyDataFilter();
    else if (this->mPolyDataNormalsToggle->value())
      return this->GetPolyDataNormals();
    else if (this->mCurvaturesToggle->value())
      return this->GetCurvatures();
    }
  else if (aRef == this->mTriangleFilter)
    {
    if      (this->mWSincPolyDataFilterToggle->value())
      return this->GetWSincPolyDataFilter();
    else if (this->mPolyDataNormalsToggle->value())
      return this->GetPolyDataNormals();
    else if (this->mCurvaturesToggle->value())
      return this->GetCurvatures();
    }
  else if (aRef == this->mWSincPolyDataFilter)
    {
    if      (this->mPolyDataNormalsToggle->value())
      return this->GetPolyDataNormals();
    else if (this->mCurvaturesToggle->value())
      return this->GetCurvatures();
    }
  else if (aRef == this->mPolyDataNormals)
    {
    if      (this->mCurvaturesToggle->value())
      return this->GetCurvatures();
    }

  return NULL; // mMassProperties
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::UpdateImageReader(void)
{
  vtkImageReader* imageReader;

  if (!(imageReader = this->GetImageReader()))
    {
    Fl::error("Update failed to get ImageReader.\n");
    return;
    }

  if (!imageReader->GetFileName() && !imageReader->GetFilePattern())
    {
    Fl::error("Update failed to get ImageReader FileName or FilePattern.\n");
    return;
    }

  float spacing[3], origin[3];
  int extent[6], voi[6];

  this->GetInputSpacingValues(spacing);
  imageReader->SetDataSpacing(spacing);

  imageReader->Update();
  imageReader->GetDataExtent(extent);

  origin[0] = -((float(1 + extent[1] - extent[0]) * spacing[0]) / 2.f);
  origin[1] = -((float(1 + extent[3] - extent[2]) * spacing[1]) / 2.f);
  origin[2] = -((float(1 + extent[5] - extent[4]) * spacing[2]) / 2.f);

  imageReader->SetDataOrigin(origin);

  this->GetInputDataVOIValues(voi);
  imageReader->SetDataVOI(voi);

#if 0
  float range[2];

  imageReader->Update();
  imageReader->GetOutput()->GetScalarRange(range);
#endif /* 0 */

  this->SetInputExtentValues(extent);
}

void
VoxelToContourUI::UpdateImageThreshold(void)
{
  vtkImageThreshold* imageThreshold;

  if (!(imageThreshold = this->GetImageThreshold()))
    {
    Fl::error("Update failed to get ImageThreshold.\n");
    return;
    }

  double value;

  this->GetImageThresholdValue(value);

  if (value == 0.f) imageThreshold->ThresholdByUpper(1.f);
  else              imageThreshold->ThresholdBetween(value, value);

  if (this->mImageThresholdOutputType->value() == 1)
    imageThreshold->SetOutputScalarTypeToFloat();
  else
    imageThreshold->SetOutputScalarTypeToUnsignedChar();
}

void
VoxelToContourUI::UpdateImageResample(void)
{
  if (!this->mImageResampleToggle->value())
    {
    return;
    }

  vtkImageResample* imageResample;

  if (!(imageResample = this->GetImageResample()))
    {
    Fl::error("Update failed to get ImageResample.\n");
    return;
    }

  double resliceAxesDirectionCosines[9];
  double resliceAxesOrigin[9];
  double resliceSpacing[3];

  this->GetResliceAxesDirectionCosinesValues(resliceAxesDirectionCosines);
  this->GetResliceAxesOriginValues(resliceAxesOrigin);
  this->GetResliceSpacingValues(resliceSpacing);

  imageResample->SetResliceAxesDirectionCosines(resliceAxesDirectionCosines);
  imageResample->SetResliceAxesOrigin(resliceAxesOrigin);
  imageResample->SetAxisOutputSpacing(0,resliceSpacing[0]);
  imageResample->SetAxisOutputSpacing(1,resliceSpacing[1]);
  imageResample->SetAxisOutputSpacing(2,resliceSpacing[2]);

  // // SHOULD I REALLY DO THIS???
  // imageResample->SetOutputOrigin( 0.0, 0.0, 0.0 );
  imageResample->UpdateWholeExtent();

  vtkMatrix4x4* resliceMatrix;
  double m[16] = { 1,0,0,0, 0,1,0,0, 0,0,1,0, 0,0,0,1 };

  if (resliceMatrix = this->GetResliceMatrix())
    {

    for (int i=0; i<4; i++)
      for (int j=0; j<4; j++)
        m[i*4+j] = resliceMatrix->GetElement(i,j);
    }

  this->SetResliceMatrixValues(m);

  float spacing[3], origin[3], bounds[6];
  int extent[6];

  imageResample->Update();
  imageResample->GetOutput()->GetSpacing(spacing);
  imageResample->GetOutput()->GetOrigin(origin);
  imageResample->GetOutput()->GetExtent(extent);
  imageResample->GetOutput()->GetBounds(bounds);
  
  this->SetResliceImageSpacingValues(spacing);
  this->SetResliceImageOriginValues(origin);
  this->SetResliceImageExtentValues(extent);
  this->SetResliceImageBoundsValues(bounds);
}

void
VoxelToContourUI::UpdateImageDilateErode3D(void)
{
  if (!this->mImageDilateErode3DToggle->value())
    {
    return;
    }

  if (this->mPreThresholdImageToggle->value())
    {
    vtkImageThreshold* imagePreThreshold;

    if (!(imagePreThreshold = this->GetImagePreThreshold()))
      {
      Fl::error("Update failed to get ImagePreThreshold.\n");
      return;
      }

    if (this->mImageThresholdOutputType->value() == 1)
      {
      imagePreThreshold->ThresholdBetween(0.5, 1.5);
      //imagePreThreshold->ThresholdByUpper(1);
      imagePreThreshold->SetOutputScalarTypeToFloat();
      }
    else
      {
      imagePreThreshold->ThresholdByUpper(1);
      imagePreThreshold->SetOutputScalarTypeToUnsignedChar();
      }
    }

  vtkImageDilateErode3D* imageDilateErode3D;

  if (!(imageDilateErode3D = this->GetImageDilateErode3D()))
    {
    Fl::error("Update failed to get ImageDilateErode3D.\n");
    return;
    }

  int kernel[3];

  this->GetDilateErode3DKernelValues(kernel);
  imageDilateErode3D->SetKernelSize(kernel[0], kernel[1], kernel[2]);
  //imageDilateErode3D->SetDilateValue(1);
  //imageDilateErode3D->SetErodeValue(1);
}

void
VoxelToContourUI::UpdateContourFilter(void)
{
  vtkContourFilter* contourFilter;

  if (!(contourFilter = this->GetContourFilter()))
    {
    Fl::error("Update failed to get ContourFilter.\n");
    return;
    }

  contourFilter->SetValue(0, float(this->mContourFilterValue->value()));
  contourFilter->
    SetComputeNormals(int(this->mContourFilterComputeNormals->value()));
  contourFilter->
    SetComputeGradients(int(this->mContourFilterComputeGradients->value()));
  contourFilter->
    SetComputeScalars(int(this->mContourFilterComputeScalars->value()));
}

void
VoxelToContourUI::UpdateCleanPolyData(void)
{
  if (!this->mCleanPolyDataToggle->value())
    {
    return;
    }

  vtkCleanPolyData* cleanPolyData;

  if (!(cleanPolyData = this->GetCleanPolyData()))
    {
    Fl::error("Update failed to get CleanPolyData.\n");
    return;
    }
}

void
VoxelToContourUI::UpdateTriangleFilter(void)
{
  if (!this->mTriangleFilterToggle->value())
    {
    return;
    }

  vtkTriangleFilter* triangleFilter;

  if (!(triangleFilter = this->GetTriangleFilter()))
    {
    Fl::error("Update failed to get TriangleFilter.\n");
    return;
    }
}

void
VoxelToContourUI::UpdateWSincPolyDataFilter(void)
{
  if (!this->mWSincPolyDataFilterToggle->value())
    {
    return;
    }

  vtkWindowedSincPolyDataFilter* wsincPolyDataFilter;

  if (!(wsincPolyDataFilter = this->GetWSincPolyDataFilter()))
    {
    Fl::error("Update failed to get WSincPolyDataFilter.\n");
    return;
    }

  wsincPolyDataFilter->
    SetNumberOfIterations(int(this->
                              mWSincPolyDataFilterNumberOfIterations->value()));
  wsincPolyDataFilter->
    SetPassBand(float(this->mWSincPolyDataFilterPassBand->value()));
  wsincPolyDataFilter->
    SetFeatureAngle(float(this->mWSincPolyDataFilterFeatureAngle->value()));
  wsincPolyDataFilter->
    SetEdgeAngle(float(this->mWSincPolyDataFilterEdgeAngle->value()));
  wsincPolyDataFilter->
    SetFeatureEdgeSmoothing(int(this->
                                mWSincPolyDataFilterFeatureEdgeSmoothing->
                                value()));
  wsincPolyDataFilter->
    SetBoundarySmoothing(int(this->
                             mWSincPolyDataFilterBoundarySmoothing->value()));
  wsincPolyDataFilter->
    SetNonManifoldSmoothing(int(this->
                                mWSincPolyDataFilterNonManifoldSmoothing->
                                value()));
}

void
VoxelToContourUI::UpdatePolyDataNormals(void)
{
  if (!this->mPolyDataNormalsToggle->value())
    {
    return;
    }

  vtkPolyDataNormals* polyDataNormals;

  if (!(polyDataNormals = this->GetPolyDataNormals()))
    {
    Fl::error("Update failed to get PolyDataNormals.\n");
    return;
    }

  polyDataNormals->
    SetFeatureAngle(float(this->mPolyDataNormalsFeatureAngle->value()));
  polyDataNormals->
    SetSplitting(int(this->mPolyDataNormalsSplitting->value()));
  polyDataNormals->
    SetConsistency(int(this->mPolyDataNormalsConsistency->value()));
  polyDataNormals->
    SetNonManifoldTraversal(int(this->
                                mPolyDataNormalsNonManifoldTraversal->value()));
}

void
VoxelToContourUI::UpdateCurvatures(void)
{
  if (!this->mCurvaturesToggle->value())
    {
    return;
    }

  vtkCurvatures* curvatures;

  if (!(curvatures = this->GetCurvatures()))
    {
    Fl::error("Update failed to get Curvatures.\n");
    return;
    }

  if (this->mCurvaturesType->value() == 1)
    curvatures->SetCurvatureTypeToMean();
  else
    curvatures->SetCurvatureTypeToGaussian();

  curvatures->SetInvertMeanCurvature(int(this->mCurvaturesInvertMeanCurvature->
                                         value()));
}

void
VoxelToContourUI::UpdateMassProperties(void)
{
  vtkMassProperties* massProperties;

  if (!(massProperties = this->GetMassProperties()))
    {
    Fl::error("Update failed to get MassProperties.\n");
    return;
    }

  double volume, area, vx, vy, vz, kx, ky, kz, nsi;

  massProperties->Update();

  this->mMassPropertiesVolume->value(massProperties->GetVolume());
  this->mMassPropertiesSurfaceArea->value(massProperties->GetSurfaceArea());
  this->mMassPropertiesNormalizedShapeIndex->
    value(massProperties->GetNormalizedShapeIndex());
  this->mMassPropertiesVolumeX->value(massProperties->GetVolumeX());
  this->mMassPropertiesVolumeY->value(massProperties->GetVolumeY());
  this->mMassPropertiesVolumeZ->value(massProperties->GetVolumeZ());
  this->mMassPropertiesKX->value(massProperties->GetKx());
  this->mMassPropertiesKY->value(massProperties->GetKy());
  this->mMassPropertiesKZ->value(massProperties->GetKz());

  this->mMainWindow->redraw();
}

void
VoxelToContourUI::UpdateContourMapper(void)
{
  vtkPolyDataMapper* contourMapper;

  if (!(contourMapper = this->GetContourMapper()))
    {
    Fl::error("Update failed to get ContourMapper.\n");
    return;
    }

  vtkPolyDataSource* polyDataSource;

  if (!(polyDataSource = this->GetPrevPolyDataSource(contourMapper)))
    {
    Fl::error("Update failed to get PolyData source.\n");
    return;
    }

  contourMapper->SetInput(polyDataSource->GetOutput());

#if 0
  contourMapper->Update();

  vtkPolyData* polyData;

  if (!(polyData = contourMapper->GetInput()))
    {
    Fl::error("Update failed to get ContourMapper input PolyData.\n");
    return;
    }

  float bounds[6], center[3], length, range[2];

  polyData->GetBounds(bounds);
  polyData->GetCenter(center);
  length = polyData->GetLength();
  polyData->GetScalarRange(range);
#endif /* 0 */
}

void
VoxelToContourUI::UpdateContourActor(void)
{
  vtkActor* contourActor;

  if (!(contourActor = this->GetContourActor()))
    {
    Fl::error("Update failed to get ContourActor.\n");
    return;
    }

  contourActor->SetMapper(this->GetContourMapper());

  vtkRenderer* renderer;

  if (!(renderer = this->GetView()->GetDefaultRenderer()))
    {
    Fl::error("Update failed to get DefaultRenderer.\n");
    return;
    }

  vtkActorCollection* actorCollection;

  if (!(actorCollection = renderer->GetActors()))
    {
    Fl::error("Update failed to get Actors from DefaultRenderer.\n");
    return;
    }

  if (!actorCollection->IsItemPresent(contourActor))
    {
    renderer->AddProp(contourActor);
    renderer->ResetCamera();
    this->GetView()->GetRenderWindow()->Render();
    }
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetInputDimensions(const int a[3])
{
  int extent[6], voi[6];

  extent[0] = 0; extent[1] = a[0] - 1;
  extent[2] = 0; extent[3] = a[1] - 1;
  extent[4] = 0; extent[5] = a[2] - 1;

  vtkImageReader* imageReader;

  if (imageReader = this->GetImageReader())
    {
    float spacing[3], origin[3];

    imageReader->SetDataExtent(extent);
    imageReader->SetFileDimensionality(a[2]>1 ? 3 : 2);

    this->GetInputSpacingValues(spacing);
    imageReader->SetDataSpacing(spacing);

    origin[0] = -((float(a[0]) * spacing[0]) / 2.f);
    origin[1] = -((float(a[1]) * spacing[1]) / 2.f);
    origin[2] = -((float(a[2]) * spacing[2]) / 2.f);
    
    imageReader->SetDataOrigin(origin);
    }

  this->SetInputExtentValues(extent);
  this->SetInputDataVOIValues(extent);
}

void
VoxelToContourUI::GetInputDimensions(int a[3])
{
  vtkImageReader* imageReader;

  if (!(imageReader = this->GetImageReader()))
    {
    return;
    }

  int extent[6];

  imageReader->Update();
  imageReader->GetDataExtent(extent);

  a[0] = 1 + extent[1] - extent[0];
  a[1] = 1 + extent[3] - extent[2];
  a[2] = 1 + extent[5] - extent[4];

  this->SetInputExtentValues(extent);
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetInputSpacingValues(const double a[3])
{
  this->mInputSpacing0->value(a[0]);
  this->mInputSpacing1->value(a[1]);
  this->mInputSpacing2->value(a[2]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetInputSpacingValues(float a[3])
{
  a[0] = float(this->mInputSpacing0->value());
  a[1] = float(this->mInputSpacing1->value());
  a[2] = float(this->mInputSpacing2->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetInputDataVOIValues(const int a[6])
{
  this->mInputVOIX0->value(a[0]);
  this->mInputVOIX1->value(a[1]);
  this->mInputVOIY0->value(a[2]);
  this->mInputVOIY1->value(a[3]);
  this->mInputVOIZ0->value(a[4]);
  this->mInputVOIZ1->value(a[5]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetInputDataVOIValues(int a[6])
{
  a[0] = int(this->mInputVOIX0->value());
  a[1] = int(this->mInputVOIX1->value());
  a[2] = int(this->mInputVOIY0->value());
  a[3] = int(this->mInputVOIY1->value());
  a[4] = int(this->mInputVOIZ0->value());
  a[5] = int(this->mInputVOIZ1->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetInputExtentValues(const int a[6])
{
  this->mInputExtentX0->value(a[0]);
  this->mInputExtentX1->value(a[1]);
  this->mInputExtentY0->value(a[2]);
  this->mInputExtentY1->value(a[3]);
  this->mInputExtentZ0->value(a[4]);
  this->mInputExtentZ1->value(a[5]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetImageThresholdValue(const double a)
{
  this->mImageThresholdValue->value(a);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetImageThresholdValue(double& a)
{
  a = double(this->mImageThresholdValue->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceAxesDirectionCosinesValues(const double a[9])
{
  this->mResliceRASX0->value(a[0]);
  this->mResliceRASX1->value(a[1]);
  this->mResliceRASX2->value(a[2]);
  this->mResliceRASY0->value(a[3]);
  this->mResliceRASY1->value(a[4]);
  this->mResliceRASY2->value(a[5]);
  this->mResliceRASZ0->value(a[6]);
  this->mResliceRASZ1->value(a[7]);
  this->mResliceRASZ2->value(a[8]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetResliceAxesDirectionCosinesValues(double a[9])
{
  a[0] = double(this->mResliceRASX0->value());
  a[1] = double(this->mResliceRASX1->value());
  a[2] = double(this->mResliceRASX2->value());
  a[3] = double(this->mResliceRASY0->value());
  a[4] = double(this->mResliceRASY1->value());
  a[5] = double(this->mResliceRASY2->value());
  a[6] = double(this->mResliceRASZ0->value());
  a[7] = double(this->mResliceRASZ1->value());
  a[8] = double(this->mResliceRASZ2->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceAxesOriginValues(const double a[3])
{
  this->mResliceOriginX->value(a[0]);
  this->mResliceOriginY->value(a[1]);
  this->mResliceOriginZ->value(a[2]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetResliceAxesOriginValues(double a[3])
{
  a[0] = double(this->mResliceOriginX->value());
  a[1] = double(this->mResliceOriginY->value());
  a[2] = double(this->mResliceOriginZ->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceSpacingValues(const double a[3])
{
  this->mResliceSpacing0->value(a[0]);
  this->mResliceSpacing1->value(a[1]);
  this->mResliceSpacing2->value(a[2]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetResliceSpacingValues(double a[3])
{
  a[0] = double(this->mResliceSpacing0->value());
  a[1] = double(this->mResliceSpacing1->value());
  a[2] = double(this->mResliceSpacing2->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetDilateErode3DKernelValues(const int a[3])
{
  this->mImageDilateErode3DKernelX->value(a[0]);
  this->mImageDilateErode3DKernelY->value(a[1]);
  this->mImageDilateErode3DKernelZ->value(a[2]);
  this->mMainWindow->redraw();
}

void
VoxelToContourUI::GetDilateErode3DKernelValues(int a[3])
{
  a[0] = int(this->mImageDilateErode3DKernelX->value());
  a[1] = int(this->mImageDilateErode3DKernelY->value());
  a[2] = int(this->mImageDilateErode3DKernelZ->value());
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceMatrixValues(const double a[16])
{
  this->mResliceMatrix00->value(a[0]);
  this->mResliceMatrix01->value(a[1]);
  this->mResliceMatrix02->value(a[2]);
  this->mResliceMatrix03->value(a[3]);
  this->mResliceMatrix10->value(a[4]);
  this->mResliceMatrix11->value(a[5]);
  this->mResliceMatrix12->value(a[6]);
  this->mResliceMatrix13->value(a[7]);
  this->mResliceMatrix20->value(a[8]);
  this->mResliceMatrix21->value(a[9]);
  this->mResliceMatrix22->value(a[10]);
  this->mResliceMatrix23->value(a[11]);
  this->mResliceMatrix30->value(a[12]);
  this->mResliceMatrix31->value(a[13]);
  this->mResliceMatrix32->value(a[14]);
  this->mResliceMatrix33->value(a[15]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetOutputScalingValues(const double a[3])
{
  this->mOutputScaling0->value(a[0]);
  this->mOutputScaling1->value(a[1]);
  this->mOutputScaling2->value(a[2]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceImageSpacingValues(const float a[3])
{
  this->mResliceImageSpacingX->value(a[0]);
  this->mResliceImageSpacingY->value(a[1]);
  this->mResliceImageSpacingZ->value(a[2]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceImageOriginValues(const float a[3])
{
  this->mResliceImageOriginX->value(a[0]);
  this->mResliceImageOriginY->value(a[1]);
  this->mResliceImageOriginZ->value(a[2]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceImageExtentValues(const int a[6])
{
  this->mResliceImageExtentX0->value(a[0]);
  this->mResliceImageExtentX1->value(a[1]);
  this->mResliceImageExtentY0->value(a[2]);
  this->mResliceImageExtentY1->value(a[3]);
  this->mResliceImageExtentZ0->value(a[4]);
  this->mResliceImageExtentZ1->value(a[5]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetResliceImageBoundsValues(const float a[6])
{
  this->mResliceImageBoundsX0->value(a[0]);
  this->mResliceImageBoundsX1->value(a[1]);
  this->mResliceImageBoundsY0->value(a[2]);
  this->mResliceImageBoundsY1->value(a[3]);
  this->mResliceImageBoundsZ0->value(a[4]);
  this->mResliceImageBoundsZ1->value(a[5]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetPolyDataScalarRangeValues(const float a[2])
{
  this->mPolyDataScalarRange0->value(a[0]);
  this->mPolyDataScalarRange1->value(a[1]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetPolyDataCenterValues(const float a[3])
{
  this->mPolyDataCenterX->value(a[0]);
  this->mPolyDataCenterY->value(a[1]);
  this->mPolyDataCenterZ->value(a[2]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
void
VoxelToContourUI::SetPolyDataBoundsValues(const float a[6])
{
  this->mPolyDataBoundsX0->value(a[0]);
  this->mPolyDataBoundsX1->value(a[1]);
  this->mPolyDataBoundsY0->value(a[2]);
  this->mPolyDataBoundsY1->value(a[3]);
  this->mPolyDataBoundsZ0->value(a[4]);
  this->mPolyDataBoundsZ1->value(a[5]);
  this->mMainWindow->redraw();
}

// ----------------------------------------------------------------------------
vtkImageReader*
VoxelToContourUI::CreateImageReader(void)
{
  vtkImageReader* imageReader = vtkImageReader::New();
  imageReader->SetHeaderSize(0);
  imageReader->SetDataByteOrderToBigEndian();
  imageReader->SetNumberOfScalarComponents(1);
  imageReader->FileLowerLeftOff();
  return imageReader;
}

// ----------------------------------------------------------------------------
vtkImageThreshold*
VoxelToContourUI::CreateImageThreshold(void)
{
  vtkImageThreshold* imageThreshold = vtkImageThreshold::New();
  imageThreshold->SetOutputScalarTypeToUnsignedChar();
  imageThreshold->SetInValue(1);
  imageThreshold->SetOutValue(0);
  imageThreshold->ReplaceInOn();
  imageThreshold->ReplaceOutOn();
  return imageThreshold;
}

// ----------------------------------------------------------------------------
vtkImageResample*
VoxelToContourUI::CreateImageResample(void)
{
  vtkImageResample* imageResample = vtkImageResample::New();
  imageResample->AutoCropOutputOn();
  imageResample->SetDimensionality(3);
  return imageResample;
}

// ----------------------------------------------------------------------------
vtkMatrix4x4*
VoxelToContourUI::CreateResliceMatrix(void)
{
  vtkMatrix4x4* resliceMatrix = vtkMatrix4x4::New();
  return resliceMatrix;
}

// ----------------------------------------------------------------------------
vtkMatrixToLinearTransform*
VoxelToContourUI::CreateResliceTransform(void)
{
  vtkMatrixToLinearTransform* resliceTransform =
    vtkMatrixToLinearTransform::New();
  return resliceTransform;
}

// ----------------------------------------------------------------------------
vtkImageThreshold*
VoxelToContourUI::CreateImagePreThreshold(void)
{
  vtkImageThreshold* imageThreshold = vtkImageThreshold::New();
  imageThreshold->SetOutputScalarTypeToUnsignedChar();
  imageThreshold->SetInValue(1);
  imageThreshold->SetOutValue(0);
  imageThreshold->ReplaceInOn();
  imageThreshold->ReplaceOutOn();
  return imageThreshold;
}

// ----------------------------------------------------------------------------
vtkImageDilateErode3D*
VoxelToContourUI::CreateImageDilateErode3D(void)
{
  vtkImageDilateErode3D* imageDilateErode3D = vtkImageDilateErode3D::New();
  imageDilateErode3D->SetKernelSize(3,3,3);
  imageDilateErode3D->SetDilateValue(1);
  imageDilateErode3D->SetErodeValue(0);
  return imageDilateErode3D;
}

// ----------------------------------------------------------------------------
vtkContourFilter*
VoxelToContourUI::CreateContourFilter(void)
{
  vtkContourFilter* contourFilter = vtkContourFilter::New();
  contourFilter->SetValue(0, 0.5);
  contourFilter->ComputeNormalsOff();
  contourFilter->ComputeGradientsOff();
  contourFilter->ComputeScalarsOff();
  return contourFilter;
}

// ----------------------------------------------------------------------------
vtkCleanPolyData*
VoxelToContourUI::CreateCleanPolyData(void)
{
  vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
  cleanPolyData->SetTolerance(0.0);
  cleanPolyData->ConvertLinesToPointsOn();
  cleanPolyData->ConvertPolysToLinesOn();
  cleanPolyData->ConvertStripsToPolysOn();
  cleanPolyData->PointMergingOn();
  return cleanPolyData;
}

// ----------------------------------------------------------------------------
vtkTriangleFilter*
VoxelToContourUI::CreateTriangleFilter(void)
{
  vtkTriangleFilter* triangleFilter = vtkTriangleFilter::New();
  triangleFilter->PassLinesOff();
  triangleFilter->PassVertsOff();
  return triangleFilter;
}

// ----------------------------------------------------------------------------
vtkWindowedSincPolyDataFilter*
VoxelToContourUI::CreateWSincPolyDataFilter(void)
{
  vtkWindowedSincPolyDataFilter* wsincPolyDataFilter =
    vtkWindowedSincPolyDataFilter::New();

  return wsincPolyDataFilter;
}

// ----------------------------------------------------------------------------
vtkPolyDataNormals*
VoxelToContourUI::CreatePolyDataNormals(void)
{
  vtkPolyDataNormals* polyDataNormals = vtkPolyDataNormals::New();
  polyDataNormals->SetFeatureAngle(30.f); /* default */
  polyDataNormals->SplittingOff();
  polyDataNormals->ConsistencyOn();
  polyDataNormals->NonManifoldTraversalOff();
  return polyDataNormals;
}

// ----------------------------------------------------------------------------
vtkCurvatures*
VoxelToContourUI::CreateCurvatures(void)
{
  vtkCurvatures* curvatures = vtkCurvatures::New();
  curvatures->SetCurvatureTypeToGaussian();
  curvatures->InvertMeanCurvatureOff();
  return curvatures;
}

// ----------------------------------------------------------------------------
vtkMassProperties*
VoxelToContourUI::CreateMassProperties(void)
{
  vtkMassProperties* massProperties = vtkMassProperties::New();
  return massProperties;
}

// ----------------------------------------------------------------------------
vtkPolyDataMapper*
VoxelToContourUI::CreateContourMapper(void)
{
  vtkPolyDataMapper* contourMapper = vtkPolyDataMapper::New();
  //vtkLookupTable* lut = vtkLookupTable::New();
  //  {
  //  lut->SetNumberOfColors(256);
  //  lut->SetHueRange(0.15, 1.0);
  //  lut->SetSaturationRange(1.0, 1.0);
  //  lut->SetValueRange(1.0, 1.0);
  //  lut->SetAlphaRange(1.0, 1.0);
  //  lut->SetRange(-20, 20);
  //  }
  //contourMapper->SetLookupTable(lut);
  //lut->Delete();
  contourMapper->SetUseLookupTableScalarRange(1);
  contourMapper->ImmediateModeRenderingOn();
  return contourMapper;
}

// ----------------------------------------------------------------------------
vtkActor*
VoxelToContourUI::CreateContourActor(void)
{
  vtkActor* contourActor = vtkActor::New();
  contourActor->GetProperty()->SetSpecular(0.4);
  contourActor->GetProperty()->SetSpecularPower(24);
  contourActor->GetProperty()->SetRepresentationToWireframe();
  return contourActor;
}

/* 
 * End of: $Id: VoxelToContourImpl.cxx,v 1.1.1.1 2006/12/19 22:59:42 christianh Exp $.
 * 
 */
