/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvContour.cxx,v 1.1.1.1 2006/12/19 22:59:41 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
// IOStreams
#include <iostream>
#include <fstream>
// STL
#include <string>
#include <fstream>
#include <vector>
// FLTK
#include <FL/Fl_File_Chooser.H> /* for fl_file_chooser() */
// VTK Common
#include "vtkMath.h"
#include "vtkMatrix4x4.h"
#include "vtkMatrixToLinearTransform.h"
#include "vtkTransform.h"
#include "vtkImageData.h"
#include "vtkStructuredPoints.h"
#include "vtkPolyData.h"
// VTK Graphics
#include "vtkPolyDataNormals.h"
#include "vtkCleanPolyData.h"
#include "vtkContourFilter.h"
#include "vtkTriangleFilter.h"
#include "vtkCurvatures.h"
// VTK Imaging
#include "vtkImageThreshold.h"
#include "vtkImageResample.h"
#include "vtkImageToStructuredPoints.h"
// VTK Rendering
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkCamera.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkActor.h"
#include "vtkLODActor.h"
// VTK IO
#include "vtkImageReader.h"
#include "vtkDataSetWriter.h"
#include "vtkXMLDataSetWriter.h"
#include "vtkZLibDataCompressor.h"
#include "vtkPolyDataReader.h"
#include "vtkXMLPolyDataReader.h"

#include "vtkImagePlaneWidget.h"
#include "vtkCellPicker.h"
#include "vtkTextProperty.h"

#include "vtkLookupTable.h"
#include "vtkScalarsToColors.h"
#include "vtkPiecewiseFunction.h"
#include "vtkColorTransferFunction.h"
//
#include "vtkRenderingContext.h"
// SVV
#include "svvContourUI.h"
#include "Fl_VTK_View.H"


VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);


// ----------------------------------------------------------------------------
typedef std::vector<vtkImagePlaneWidget*> ImagePlaneWidgetVec;

// ----------------------------------------------------------------------------
static const char* gInputName     = NULL;
static const char* gReferenceName = NULL;

// ----------------------------------------------------------------------------
static float gReferenceBounds[6]      = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                                          VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                                          VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

static float gReferenceCenter[3]      = { 0, 0, 0 };

static int   gReferenceWholeExtent[6] = { 0, -1, 0, -1, 0, -1 };

static float gOutputBounds[6]         = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                                          VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                                          VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

static float gOutputCenter[3]         = { 0, 0, 0 };

static int   gOutputWholeExtent[6]    = { 0, -1, 0, -1, 0, -1 };

static float gInputScalarRange[2]     = { 0, 0 };

// ----------------------------------------------------------------------------
static vtkRenderingContext*             gRenderingContext        = NULL;

// ----------------------------------------------------------------------------
static vtkPolyData*                     gContourData             = NULL;

// ----------------------------------------------------------------------------
static vtkImageReader*                  gImageReader             = NULL;
static vtkImageResample*                gImageResample           = NULL;
static vtkImageToStructuredPoints*      gImageToStructuredPoints = NULL;

static vtkImageThreshold*               gInputThreshold          = NULL;
static vtkCleanPolyData*                gCleaner                 = NULL;
static vtkTriangleFilter*               gTriangleFilter          = NULL;
static vtkPolyDataNormals*              gNormals                 = NULL;

// ----------------------------------------------------------------------------
// Create a transfer function mapping scalar value to opacity
static vtkPiecewiseFunction*            gOpacityTF               = NULL;
// Create a transfer function mapping scalar value to color (grey)
static vtkPiecewiseFunction*            gGrayTF                  = NULL;
// Create a transfer function mapping scalar value to color (color)
static vtkColorTransferFunction*        gColorTF                 = NULL;
// Create a transfer function mapping magnitude of gradient to opacity
static vtkPiecewiseFunction*            gGradientTF              = NULL;

// ----------------------------------------------------------------------------
static vtkActor*                        gReferenceActor          = NULL;
static vtkActor*                        gContourActor            = NULL;
static vtkImagePlaneWidget**            gImagePlaneWidgets       = NULL;

// ----------------------------------------------------------------------------
int
main (int argc, char* argv[])
{
  if (argc < 2)
    {
    std::cerr << "\nUsage: " << argv[0] << " <in.buchar>"
              << " [<reference.vtk>|<reference.vtp>]\n" << std::endl;
    return 0;
    }

  gInputName = argv[1];

  if (argc > 2) gReferenceName = argv[2];

  // create the interface
  svvContourUI* GUI = svvContourUI::New();

  if ((gRenderingContext = GUI->GetClientRenderingContext()) == NULL)
    {
    std::cerr << "Failed to get RenderingContext! ... exiting" << std::endl;
    return 1;
    }

  vtkRenderWindow* renderWindow = gRenderingContext->GetRenderWindow();

  gRenderingContext->SetDesiredUpdateRate(4.0);
  renderWindow->LineSmoothingOff();
  
  const double inputSpacing[3]  = { 1.5, 1.5, 3.0 };
  const double resliceRAS[9]    = { 1,0,0, 0,1,0, 0,0,1 };
  const double resliceOrigin[3] = { 0, 0, 0 };
  const double isoValue         = 0;

  GUI->SetInputSpacing(inputSpacing);
  GUI->SetResliceAxesDirectionCosines(resliceRAS);
  GUI->SetResliceAxesOrigin(resliceOrigin);
  GUI->SetContourIsoValue(isoValue);

  // show() the main Fl_Window
  GUI->Show(argc, argv);

  // this is the standard way of "starting" a fltk application
  int fl_ret = Fl::run();

  // Clean up
  if (gImageReader) gImageReader->Delete();
  if (gContourActor) gContourActor->Delete();
  gImagePlaneWidgets[0]->Delete();
  gImagePlaneWidgets[1]->Delete();
  gImagePlaneWidgets[2]->Delete();
  delete [] gImagePlaneWidgets;
  GUI->Delete();

  return fl_ret;
}

// ----------------------------------------------------------------------------
static vtkActor*
CreateReferenceActor (const char* const aFileName, svvContourUI* aGUI)
{
  if (gReferenceActor)
    {
    aGUI->GetView()->GetDefaultRenderer()->RemoveProp(gReferenceActor);
    gReferenceActor = NULL;
    }

  if (!aFileName || !*aFileName)
    {
    std::cerr << "Input filename is empty!" << std::endl;
    return NULL;
    }

  std::string fileName(aFileName);
  std::string::size_type idx = fileName.rfind('.');
  std::string suffix(idx==std::string::npos ? "" : fileName.substr(idx+1));

  if (suffix == "vtk")
    {
    gReferenceActor = vtkActor::New();
      {
      vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
        {
        vtkPolyDataReader* reader = vtkPolyDataReader::New();
          {
          reader->SetFileName(aFileName);
          reader->Update();
          reader->GetOutput()->GetBounds(gReferenceBounds);
          reader->GetOutput()->GetCenter(gReferenceCenter);
          reader->GetOutput()->GetWholeExtent(gReferenceWholeExtent);
          }
        mapper->SetInput(reader->GetOutput());
        mapper->ScalarVisibilityOff();
        reader->Delete();
        }
      gReferenceActor->SetMapper(mapper);
      mapper->Delete();
      }
    }
  else if (suffix == "vtp" || suffix == "xml")
    {
    gReferenceActor = vtkActor::New();
      {
      vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
        {
        vtkXMLPolyDataReader* reader = vtkXMLPolyDataReader::New();
          {
          reader->SetFileName(aFileName);
          reader->Update();
          reader->GetOutput()->GetBounds(gReferenceBounds);
          reader->GetOutput()->GetCenter(gReferenceCenter);
          reader->GetOutput()->GetWholeExtent(gReferenceWholeExtent);
          }
        mapper->SetInput(reader->GetOutput());
        mapper->ScalarVisibilityOff();
        reader->Delete();
        }
      gReferenceActor->SetMapper(mapper);
      gReferenceActor->GetProperty()->SetColor(1.f, 1.f, 0.9f);
      gReferenceActor->GetProperty()->SetSpecular(0.5f);
      gReferenceActor->GetProperty()->SetSpecularColor(0.5f, 0.7f, 0.9f);
      gReferenceActor->GetProperty()->SetSpecularPower(24.f);
      mapper->Delete();
      }
    }

  return gReferenceActor;
}

// ----------------------------------------------------------------------------
static bool
CheckHeaderInfo (const char* const aFileName)
{
  if (!aFileName || !*aFileName)
    {
    std::cerr << "Input filename is empty!" << std::endl;
    return false;
    }

  return true;
}

// ----------------------------------------------------------------------------
static vtkImageReader*
GetImageReader (svvContourUI* aGUI)
{
  if (!gImageReader) gImageReader = vtkImageReader::New();

  int inputDimensions[3] = { -1, -1, -1 };
  int inputExtent[6] = { 0, -1, 0, -1, 0, -1 };
  int inputScalarType = -1;

  if ( (!gImageReader->GetFileName() && gInputName) ||
       (std::string(gImageReader->GetFileName()) != std::string(gInputName)) )
    {
    gImageReader->SetFileName(gInputName);

    std::string fileName(gImageReader->GetFileName());
    std::string::size_type idx = fileName.rfind('.');
    std::string suffix(idx==std::string::npos ? "" : fileName.substr(idx+1));
    const std::string hdr(".hdr");
    std::string hdrName;

    if      (suffix == "buchar")
      {
      inputScalarType = VTK_UNSIGNED_CHAR;
      hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;
      }
    else if (suffix == "bushort")
      {
      inputScalarType = VTK_UNSIGNED_SHORT;
      hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;
      }
    else if (suffix == "bshort")
      {
      inputScalarType = VTK_SHORT;
      hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;
      }
    else if (suffix == "bfloat")
      {
      inputScalarType = VTK_FLOAT;
      hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;
      }
    else
      {
      std::cerr << "Failed to determine scalar type from suffix!" << std::endl;
      gImageReader->Delete();
      gImageReader = NULL;
      return NULL;
      }

    if (hdrName.empty())
      {
      std::cerr << "Failed to determine header name!" << std::endl;
      gImageReader->Delete();
      gImageReader = NULL;
      return NULL;
      }

    std::ifstream hdrStream;
    
    // Opens the file and positions the stream pointer at EOF ...
    hdrStream.open(hdrName.c_str(), std::ios::in | std::ios::ate);
  
    if (!hdrStream.is_open())
      {
      std::cerr << "Failed to open header '" << hdrName << "'" << std::endl;
      gImageReader->Delete();
      gImageReader = NULL;
      return NULL;
      }

    int xsize, ysize, zsize;

    // Rewind the stream ... and start reading the items.
    hdrStream.seekg(0, std::ios::beg);
    // Read the file dimensions from the header (rows, cols, slices, 0).
    hdrStream >> ysize >> xsize >> zsize;
    hdrStream.close();
    
    inputDimensions[0] = xsize;
    inputDimensions[1] = ysize;
    inputDimensions[2] = zsize;

    // update the user interface
    aGUI->SetInputDimensions(inputDimensions);
    
    // Derive the extent from the dimensions.
    inputExtent[0] = 0;
    inputExtent[1] = inputDimensions[0] - 1;
    inputExtent[2] = 0;
    inputExtent[3] = inputDimensions[1] - 1;
    inputExtent[4] = 0;
    inputExtent[5] = inputDimensions[2] - 1;

    // update the user interface
    aGUI->SetInputDataVOI(inputExtent);

    gImageReader->SetHeaderSize(0);
    gImageReader->SetDataByteOrderToBigEndian();
    gImageReader->SetDataScalarType(inputScalarType);
    gImageReader->SetNumberOfScalarComponents(1);
    gImageReader->SetFileDimensionality(inputDimensions[2]>1 ? 3 : 2);

    std::cerr << "Setting Input Extent to { " << inputExtent[0] << ", "
              << inputExtent[1] << ", " << inputExtent[2] << ", "
              << inputExtent[3] << ", " << inputExtent[4] << ", "
              << inputExtent[5]  << " }" << std::endl;

    gImageReader->SetDataExtent(inputExtent);
    //gImageReader->SetFileNameSliceOffset(1/*count from 1*/);
    gImageReader->FileLowerLeftOff();
    }

  double inputSpacing[3];

  aGUI->GetInputSpacing(inputSpacing);

  std::cerr << "Setting Input Spacing to { " << inputSpacing[0] << ", "
            << inputSpacing[1] << ", " << inputSpacing[2] << " }" << std::endl;

  gImageReader->SetDataSpacing(inputSpacing[0],inputSpacing[1],inputSpacing[2]);

  float inputOrigin[3];

  gImageReader->Update();
  gImageReader->GetDataExtent(inputExtent);

  inputOrigin[0] = -( ( float(1 + inputExtent[1] - inputExtent[0]) *
                        inputSpacing[0] ) / 2.f );
  inputOrigin[1] = -( ( float(1 + inputExtent[3] - inputExtent[2]) *
                        inputSpacing[1] ) / 2.f );
  inputOrigin[2] = -( ( float(1 + inputExtent[5] - inputExtent[4]) *
                        inputSpacing[2] ) / 2.f );

  std::cerr << "Setting Input Origin to { " << inputOrigin[0] << ", "
            << inputOrigin[1] << ", " << inputOrigin[2] << " }" << std::endl;

  gImageReader->SetDataOrigin(inputOrigin);

  int dataVOI[6];

  aGUI->GetInputDataVOI(dataVOI);

  std::cerr << "Setting Input VOI to { " << dataVOI[0] << ", "
            << dataVOI[1] << ", " << dataVOI[2] << ", " << dataVOI[3]
            << ", " << dataVOI[4] << ", " << dataVOI[5]  << " }" << std::endl;

  gImageReader->SetDataVOI(dataVOI);

  return gImageReader;
}

// ----------------------------------------------------------------------------
static void
BuildTransferFunctions (float aMin, float aMax)
{
  float span = aMax - aMin;

  // Create transfer mapping scalar value to opacity
  if (gOpacityTF)
    gOpacityTF->RemoveAllPoints();
  else
    gOpacityTF = vtkPiecewiseFunction::New();

  gOpacityTF->AddSegment( 0,                 0.0,                 0, 0.0 );
  gOpacityTF->AddSegment( 1,                 1.0, aMin + span * 0.2, 1.0 );
  gOpacityTF->AddSegment( aMin + span * 0.2, 0.0, aMin + span * 0.8, 0.0 );
  gOpacityTF->AddSegment( aMin + span * 0.8, 1.0, aMin + span * 1.0, 1.0 );
  
  // Create transfer mapping scalar value to grayscale
  if (gGrayTF)
    gGrayTF->RemoveAllPoints();
  else
    gGrayTF = vtkPiecewiseFunction::New();

  gGrayTF->AddSegment(aMin, 0.0, aMax, 1.0);

  // Create transfer mapping scalar value to color
  if (gColorTF)
    gColorTF->RemoveAllPoints();
  else
    gColorTF = vtkColorTransferFunction::New();

  gColorTF->AddRGBPoint( 1,                 1.0, 0.0, 0.0 );
  gColorTF->AddRGBPoint( aMin + span * 0.1, 0.0, 0.0, 1.0 );
  gColorTF->AddRGBPoint( aMin + span * 0.2, 0.0, 1.0, 0.0 );
  gColorTF->AddRGBPoint( aMin + span * 0.5, 0.0, 0.0, 0.0 );
  gColorTF->AddRGBPoint( aMin + span * 0.8, 1.0, 0.0, 0.0 );
  gColorTF->AddRGBPoint( aMin + span * 0.9, 0.0, 0.0, 1.0 );
  gColorTF->AddRGBPoint( aMin + span * 1.0, 0.0, 1.0, 0.0 );

  // Create a transfer function mapping magnitude of gradient to opacity
  if (gGradientTF) gGradientTF->RemoveAllPoints();
  else gGradientTF = vtkPiecewiseFunction::New();

  gGradientTF->AddPoint( aMin + span * 0.0, 0.0 );
  gGradientTF->AddPoint( aMin + span * 0.5, 0.5 );
  gGradientTF->AddPoint( aMin + span * 1.0, 0.5 );
}

// ----------------------------------------------------------------------------
void
CheckImagePlaneWidgets (vtkRenderWindowInteractor* aInteractor,
                        vtkImageData*              aImageData)
{
  int dimensions[3];

  aImageData->Update();
  aImageData->GetDimensions(dimensions);
  aImageData->GetScalarRange(gInputScalarRange);
  
  std::cerr << "scalar range: [ " << gInputScalarRange[0] << " .. "
            << gInputScalarRange[1] << " ]" << std::endl;

  BuildTransferFunctions(gInputScalarRange[0],gInputScalarRange[1]);

  // --------------------------------------------------------------------------
  if (!gImagePlaneWidgets)
    {
    gImagePlaneWidgets = new vtkImagePlaneWidget* [3];

    // Define the internal picker. The set of three orthogonal planes can 
    // share the same picker so that picking is performed correctly.
    vtkCellPicker* cellPicker = vtkCellPicker::New();
      {
      cellPicker->SetTolerance(0.005);
      }

    // Define the shared property for the display text.
    vtkTextProperty* textProperty = vtkTextProperty::New();
      {
      textProperty->SetColor(1,1,0.9);
      textProperty->ShadowOn();
      textProperty->AntiAliasingOn();
      }

    gImagePlaneWidgets[0] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[0]->SetInteractor(aInteractor);
      gImagePlaneWidgets[0]->SetInput(aImageData);
      gImagePlaneWidgets[0]->SetPicker(cellPicker);
      gImagePlaneWidgets[0]->SetTextProperty(textProperty);
      gImagePlaneWidgets[0]->DisplayTextOn();
      gImagePlaneWidgets[0]->SetKeyPressActivationValue('x');
      gImagePlaneWidgets[0]->GetPlaneProperty()->SetColor(1,0,0);
      gImagePlaneWidgets[0]->
        SetCursorProperty(gImagePlaneWidgets[0]->GetPlaneProperty());
      gImagePlaneWidgets[0]->TextureInterpolateOff();
      gImagePlaneWidgets[0]->SetPlaneOrientationToXAxes();
      gImagePlaneWidgets[0]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
      gImagePlaneWidgets[0]->On();
      }
    gImagePlaneWidgets[1] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[1]->SetInteractor(aInteractor);
      gImagePlaneWidgets[1]->SetInput(aImageData);
      gImagePlaneWidgets[1]->SetPicker(cellPicker);
      gImagePlaneWidgets[1]->SetTextProperty(textProperty);
      gImagePlaneWidgets[1]->DisplayTextOn();
      gImagePlaneWidgets[1]->SetKeyPressActivationValue('y');
      gImagePlaneWidgets[1]->GetPlaneProperty()->SetColor(1,1,0);
      gImagePlaneWidgets[1]->
        SetCursorProperty(gImagePlaneWidgets[1]->GetPlaneProperty());
      gImagePlaneWidgets[1]->TextureInterpolateOff();
      gImagePlaneWidgets[1]->SetPlaneOrientationToYAxes();
      gImagePlaneWidgets[1]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
      gImagePlaneWidgets[1]->On();
      }
    gImagePlaneWidgets[2] = vtkImagePlaneWidget::New();
      {
      gImagePlaneWidgets[2]->SetInteractor(aInteractor);
      gImagePlaneWidgets[2]->SetInput(aImageData);
      gImagePlaneWidgets[2]->SetPicker(cellPicker);
      gImagePlaneWidgets[2]->SetTextProperty(textProperty);
      gImagePlaneWidgets[2]->DisplayTextOn();
      gImagePlaneWidgets[2]->SetKeyPressActivationValue('z');
      gImagePlaneWidgets[2]->GetPlaneProperty()->SetColor(0,0,1);
      gImagePlaneWidgets[2]->
        SetCursorProperty(gImagePlaneWidgets[2]->GetPlaneProperty());
      gImagePlaneWidgets[2]->TextureInterpolateOff();
      gImagePlaneWidgets[2]->SetPlaneOrientationToZAxes();
      gImagePlaneWidgets[2]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
      gImagePlaneWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
      gImagePlaneWidgets[2]->On();
      }

    // Set the internal lut to that of the first plane. In this way, the set 
    // of three orthogonal planes can share the same lut so that 
    // window-levelling is performed uniformly among planes.
    vtkLookupTable* lut = gImagePlaneWidgets[0]->GetLookupTable();
      {
      vtkIdType nValues = lut->GetNumberOfTableValues();
      float rgba[4];
      float value = gInputScalarRange[0];
      float incr = (gInputScalarRange[1]-gInputScalarRange[0])/float(nValues);
    
      for(vtkIdType n=0; n<nValues; n++)
        {
        gColorTF->GetColor(value,rgba);
        rgba[3] = gOpacityTF->GetValue(value);
        lut->SetTableValue(n,rgba);
        value += incr;
        }
      }

    gImagePlaneWidgets[1]->SetLookupTable(lut);
    gImagePlaneWidgets[2]->SetLookupTable(lut);
    }
  else
    {
    gImagePlaneWidgets[0]->SetInteractor(aInteractor);
    gImagePlaneWidgets[0]->SetInput(aImageData);
    gImagePlaneWidgets[0]->SetPlaneOrientationToXAxes();
    gImagePlaneWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
    gImagePlaneWidgets[1]->SetInteractor(aInteractor);
    gImagePlaneWidgets[1]->SetInput(aImageData);
    gImagePlaneWidgets[1]->SetPlaneOrientationToYAxes();
    gImagePlaneWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
    gImagePlaneWidgets[2]->SetInteractor(aInteractor);
    gImagePlaneWidgets[2]->SetInput(aImageData);
    gImagePlaneWidgets[2]->SetPlaneOrientationToZAxes();
    gImagePlaneWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
    }
}

// ----------------------------------------------------------------------------
vtkPolyData*
CreateContourData (vtkImageData* aImageData)
{
  vtkCurvatures* curvatures = vtkCurvatures::New();
    {
    gNormals = vtkPolyDataNormals::New();
      {
      gTriangleFilter = vtkTriangleFilter::New();
        {
        gCleaner = vtkCleanPolyData::New();
          {
          vtkContourFilter* contourFilter = vtkContourFilter::New();
            {
            contourFilter->SetInput(aImageData);
            contourFilter->SetValue(0, 1);
            contourFilter->ComputeNormalsOff();
            }
          gCleaner->SetInput(contourFilter->GetOutput());
          //gCleaner->SetTolerance(0.005);
          gCleaner->ConvertLinesToPointsOn();
          gCleaner->ConvertPolysToLinesOn();
          gCleaner->ConvertStripsToPolysOn();
          gCleaner->PointMergingOn();
          contourFilter->Delete();
          }
        gTriangleFilter->SetInput(gCleaner->GetOutput());
        gTriangleFilter->PassLinesOff();
        gTriangleFilter->PassVertsOff();
        gCleaner->Delete();
        }
      gNormals->SetInput(gTriangleFilter->GetOutput());
      gNormals->SplittingOff();
      gNormals->ConsistencyOn();
      gNormals->NonManifoldTraversalOff();
      gTriangleFilter->Delete();
      }
    curvatures->SetInput(gNormals->GetOutput());
    curvatures->SetCurvatureTypeToGaussian();
    gNormals->Delete();
    }

  return curvatures->GetOutput();
}

// ----------------------------------------------------------------------------
vtkActor*
CreateContourActor (vtkPolyData* aPolyData)
{
  vtkActor* actor = vtkActor::New();
    {
    vtkPolyDataMapper* polyDataMapper = vtkPolyDataMapper::New();
      {
#if 0
      vtkLookupTable* lut = vtkLookupTable::New();
        {
        lut->SetNumberOfColors(256);
        lut->SetHueRange(0.15, 1.0);
        lut->SetSaturationRange(1.0, 1.0);
        lut->SetValueRange(1.0, 1.0);
        lut->SetAlphaRange(1.0, 1.0);
        lut->SetRange(-20, 20);
        }
#endif /* 0 */
      polyDataMapper->SetInput(aPolyData);
      //polyDataMapper->SetLookupTable(lut);
      polyDataMapper->SetUseLookupTableScalarRange(1);
      polyDataMapper->ImmediateModeRenderingOn();
      //lut->Delete();
      }
    actor->SetMapper(polyDataMapper);
    actor->GetProperty()->SetSpecular(0.4);
    actor->GetProperty()->SetSpecularPower(24);
    actor->GetProperty()->SetRepresentationToWireframe();
    polyDataMapper->Delete();
    }
  return actor;
}

// ----------------------------------------------------------------------------
void
UpdateButtonCb (Fl_Return_Button* aButton, void* aPtr)
{
  if (!aButton)
    return;

  svvContourUI* GUI;

  if (!(GUI = reinterpret_cast<svvContourUI*>(aPtr)))
    return;

  // --------------------------------------------------------------------------
  if (gReferenceName && !gReferenceActor)
    {
    if (gReferenceActor = CreateReferenceActor(gReferenceName,GUI))
      {
      GUI->GetView()->GetDefaultRenderer()->AddProp(gReferenceActor);
      gReferenceActor->Delete();
      GUI->GetView()->GetDefaultRenderer()->ResetCamera();
      GUI->GetView()->GetRenderWindow()->Render();
      }
    }

  // --------------------------------------------------------------------------
  if (!(gImageReader = GetImageReader(GUI)))
    {
    std::cerr << "Failed to get image reader!" << std::endl;
    return;
    }

  if (!gImageToStructuredPoints)
    {
    gImageToStructuredPoints = vtkImageToStructuredPoints::New();
    gImageToStructuredPoints->SetInput(gImageReader->GetOutput());
    }
  else if (gImageToStructuredPoints->GetInput() != gImageReader->GetOutput())
    {
    gImageToStructuredPoints->SetInput(gImageReader->GetOutput());
    }

  if (!gImageResample)
    {
    gImageResample = vtkImageResample::New();
    gImageResample->SetInput(gImageToStructuredPoints->GetOutput());
    }
  else if (gImageResample->GetInput() != gImageToStructuredPoints->GetOutput())
    {
    gImageResample->SetInput(gImageToStructuredPoints->GetOutput());
    }

  gImageResample->AutoCropOutputOn();
  gImageResample->SetDimensionality(3);

  double resliceXYZ[9], resliceOrigin[3];

  GUI->GetResliceAxesDirectionCosines(resliceXYZ);
  GUI->GetResliceAxesOrigin(resliceOrigin);

  gImageResample->SetResliceAxesDirectionCosines(resliceXYZ);
  gImageResample->SetResliceAxesOrigin(resliceOrigin);

  gImageResample->SetOutputOrigin(0,0,0);

  // --------------------------------------------------------------------------
  gImageResample->UpdateWholeExtent();

  vtkHomogeneousTransform* resliceTransform;

  if ( resliceTransform =
       vtkHomogeneousTransform::SafeDownCast( gImageResample->
                                              GetResliceTransform() ) )
    {
    vtkMatrix4x4* resliceMatrix = resliceTransform->GetMatrix();
    double m[16];

    for (int i=0; i<4; i++)
      for (int j=0; j<4; j++)
        m[i*4+j] = resliceMatrix->GetElement(i,j);

    GUI->SetOutputMatrix(m);
    }

  float outputSpacing[3], outputOrigin[3], outputBounds[3];
  int outputExtent[6];

  gImageResample->GetOutput()->Update();
  gImageResample->GetOutput()->GetSpacing(outputSpacing);
  gImageResample->GetOutput()->GetOrigin(outputOrigin);
  gImageResample->GetOutput()->GetExtent(outputExtent);
  gImageResample->GetOutput()->GetBounds(outputBounds);
  
  GUI->SetOutputSpacing(outputSpacing);
  GUI->SetOutputOrigin(outputOrigin);
  GUI->SetOutputExtent(outputExtent);
  GUI->SetOutputBounds(outputBounds);

  // --------------------------------------------------------------------------
  CheckImagePlaneWidgets( GUI->GetView()->GetInteractor(),
                          gImageResample->GetOutput() );

  // --------------------------------------------------------------------------
  if (!gInputThreshold)
    {
    gInputThreshold = vtkImageThreshold::New();
    gInputThreshold->SetInput(gImageResample->GetOutput());
    gInputThreshold->SetOutputScalarTypeToUnsignedChar();
    gInputThreshold->SetInValue(1);
    gInputThreshold->SetOutValue(0);
    gInputThreshold->ReplaceInOn();
    gInputThreshold->ReplaceOutOn();
    gImageResample->Delete();
    }
  else if (gInputThreshold->GetInput() != gImageResample->GetOutput())
    {
    gInputThreshold->SetInput(gImageResample->GetOutput());
    gImageResample->Delete();
    }

  double isoValue;

  GUI->GetContourIsoValue(isoValue);

  if (isoValue == 0.f) gInputThreshold->ThresholdByUpper(1.f);
  else                 gInputThreshold->ThresholdBetween(isoValue, isoValue);

  // --------------------------------------------------------------------------
  if (!gContourData)
    {
    gContourData = CreateContourData(gInputThreshold->GetOutput());
    if (gContourActor)
      {
      GUI->GetView()->GetDefaultRenderer()->RemoveProp(gContourActor);
      gContourActor = NULL;
      }
    }

  if (gContourData)
    {
    gContourData->Update();
    gContourData->GetBounds(gOutputBounds);
    gContourData->GetCenter(gOutputCenter);
    gContourData->GetWholeExtent(gOutputWholeExtent);
    }
  
  // --------------------------------------------------------------------------
  if (!gContourActor)
    {
    gContourActor = CreateContourActor(gContourData);
    GUI->SetContour(gContourActor);
    gContourActor->Delete();
    GUI->GetView()->GetDefaultRenderer()->ResetCamera();
    GUI->GetView()->GetRenderWindow()->Render();
    }

  if (gContourActor)
    {
    gContourActor->GetMapper()->Update();
    if (gContourData)
      {
      gContourData->GetBounds(gOutputBounds);
      gContourData->GetCenter(gOutputCenter);
      gContourData->GetWholeExtent(gOutputWholeExtent);
      }

    std::cerr << "\n";

    if (gReferenceActor)
      {
      std::cerr << "Ref Bounds: { " << gReferenceBounds[0] << ", "
                << gReferenceBounds[1] << ", " << gReferenceBounds[2] << ", "
                << gReferenceBounds[3] << ", " << gReferenceBounds[4] << ", "
                << gReferenceBounds[5] << " }\n"
                << "    Center: { " << gReferenceCenter[0] << ", "
                << gReferenceCenter[1] << ", " << gReferenceCenter[2] << " }\n"
                << "    Extent: { " << gReferenceWholeExtent[0] << ", "
                << gReferenceWholeExtent[1] << ", " << gReferenceWholeExtent[2]
                << ", " << gReferenceWholeExtent[3] << ", "
                << gReferenceWholeExtent[4] << ", " << gReferenceWholeExtent[5]
                << " }\n";
      }

    std::cerr << "Out Bounds: { " << gOutputBounds[0] << ", "
              << gOutputBounds[1] << ", " << gOutputBounds[2] << ", "
              << gOutputBounds[3] << ", " << gOutputBounds[4] << ", "
              << gOutputBounds[5] << " }\n"
              << "    Center: { " << gOutputCenter[0] << ", "
              << gOutputCenter[1] << ", " << gOutputCenter[2] << " }\n"
              << "    Extent: { " << gOutputWholeExtent[0] << ", "
              << gOutputWholeExtent[1] << ", " << gOutputWholeExtent[2]
              << ", " << gOutputWholeExtent[3] << ", " << gOutputWholeExtent[4]
              << ", " << gOutputWholeExtent[5] << " }\n"
              << std::endl;
    }

  GUI->UpdateView();
}

// ----------------------------------------------------------------------------
void
ContourVisibilityToggleCb (Fl_Button* aButton, void* aPtr)
{
  svvContourUI* GUI;
  vtkProp3D*    contour;

  if (aButton == NULL)
    return;
  if ((GUI = reinterpret_cast<svvContourUI*>(aPtr)) == NULL)
    return;
  if ((contour = GUI->GetContour()) == NULL)
    return;

  Fl::check();
  contour->SetVisibility(aButton->value());
  GUI->UpdateView();
}

// ----------------------------------------------------------------------------
// Internal function to read in a line up to 'size' characters.
// Returns zero if there was an error, else number of char read.
static std::size_t
ReadLine (std::ifstream& aStream, char* aBuffer, std::size_t aMaxSize)
{
  aStream.getline(aBuffer, aMaxSize);

  std::size_t length = aStream.gcount();

  if (aStream.fail())
    {
    if (aStream.eof())
      {
      return 0;
      }
    if (length == (aMaxSize - 1))
      {
      // Read 'aMaxSize' chars; ignoring the rest of the line.
      aStream.clear();
      aStream.ignore(0xffffffff, '\n');
      }
    }

  return length;
}

// ----------------------------------------------------------------------------
static bool
OrientWithRegister (double aMatrix[16], svvContourUI* aGUI)
{
  int    toDimensions[3]                  = { 256, 256, 256 };
  double toSpacing[3]                     = { 1.0, 1.0, 1.0 };
  int    toRASGoodFlag                    = 1;
  double toResliceAxesDirectionCosines[9] = { -1,0,0, 0,0,-1, 0,1,0 };
  double toResliceAxesOrigin[3]           = { 0.0, 0.0, 0.0 };

  vtkMatrix4x4* r = vtkMatrix4x4::New();
  r->SetElement(0,0, aMatrix[0]);
  r->SetElement(0,1, aMatrix[1]);
  r->SetElement(0,2, aMatrix[2]);
  r->SetElement(0,3, aMatrix[3]);
  r->SetElement(1,0, aMatrix[4]);
  r->SetElement(1,1, aMatrix[5]);
  r->SetElement(1,2, aMatrix[6]);
  r->SetElement(1,3, aMatrix[7]);
  r->SetElement(2,0, aMatrix[8]);
  r->SetElement(2,1, aMatrix[9]);
  r->SetElement(2,2, aMatrix[10]);
  r->SetElement(2,3, aMatrix[11]);
  r->SetElement(3,0, aMatrix[12]);
  r->SetElement(3,1, aMatrix[13]);
  r->SetElement(3,2, aMatrix[14]);
  r->SetElement(3,3, aMatrix[15]);

  double determinant;

#if 1
  determinant = r->Determinant();

  if (determinant == 0.0)
    {
    std::cerr << "orient_with_register():"
              << " registration matrix has zero determinant" << std::endl;
    r->Print(std::cerr);
    return false;
    }
#endif /* 0 */

  vtkMatrix4x4* rinv = vtkMatrix4x4::New();
  rinv->DeepCopy(r);
  rinv->Invert();
  r->Delete();

  double m00 = toSpacing[0] * toResliceAxesDirectionCosines[0];
  double m01 = toSpacing[1] * toResliceAxesDirectionCosines[3];
  double m02 = toSpacing[2] * toResliceAxesDirectionCosines[6];
  double m10 = toSpacing[0] * toResliceAxesDirectionCosines[1];
  double m11 = toSpacing[1] * toResliceAxesDirectionCosines[4];
  double m12 = toSpacing[2] * toResliceAxesDirectionCosines[7];
  double m20 = toSpacing[0] * toResliceAxesDirectionCosines[2];
  double m21 = toSpacing[1] * toResliceAxesDirectionCosines[5];
  double m22 = toSpacing[2] * toResliceAxesDirectionCosines[8];

  double ci = (toDimensions[0] - 1.0) / 2.0;
  double cj = (toDimensions[1] - 1.0) / 2.0;
  double ck = (toDimensions[2] - 1.0) / 2.0;

  double m03 = toResliceAxesOrigin[0] - (m00 * ci + m01 * cj + m02 * ck);
  double m13 = toResliceAxesOrigin[1] - (m10 * ci + m11 * cj + m12 * ck);
  double m23 = toResliceAxesOrigin[2] - (m20 * ci + m21 * cj + m22 * ck);

  vtkMatrix4x4* sr = vtkMatrix4x4::New();

  sr->SetElement(0,0,m00);
  sr->SetElement(0,1,m01);
  sr->SetElement(0,2,m02);
  sr->SetElement(0,3,m03);
  
  sr->SetElement(1,0,m10);
  sr->SetElement(1,1,m11);
  sr->SetElement(1,2,m12);
  sr->SetElement(1,3,m13);
  
  sr->SetElement(2,0,m20);
  sr->SetElement(2,1,m21);
  sr->SetElement(2,2,m22);
  sr->SetElement(2,3,m23);

  sr->SetElement(3,0, 0.0);
  sr->SetElement(3,1, 0.0);
  sr->SetElement(3,2, 0.0);
  sr->SetElement(3,3, 1.0);

  vtkMatrix4x4* sa = vtkMatrix4x4::New();

  sa->SetElement(0,0, -toSpacing[0]);
  sa->SetElement(0,1,  0.0);
  sa->SetElement(0,2,  0.0);
  sa->SetElement(0,3,  toDimensions[0] * toSpacing[0] / 2.0);

  sa->SetElement(1,0,  0.0);
  sa->SetElement(1,1,  0.0);
  sa->SetElement(1,2,  toSpacing[2]);
  sa->SetElement(1,3, -toDimensions[2] * toSpacing[2] / 2.0);

  sa->SetElement(2,0,  0.0);
  sa->SetElement(2,1, -toSpacing[1]);
  sa->SetElement(2,2,  0.0);
  sa->SetElement(2,3,  toDimensions[1] * toSpacing[1] / 2.0);

  sa->SetElement(3,0,  0.0);
  sa->SetElement(3,1,  0.0);
  sa->SetElement(3,2,  0.0);
  sa->SetElement(3,3,  1.0);

#if 1
  determinant = sa->Determinant();

  if (determinant == 0.0)
    {
    std::cerr << "orient_with_register(): destination (ijk) ->"
              << " r space matrix has zero determinant" << std::endl;
    sa->Print(std::cerr);
    sr->Delete();
    sa->Delete();
    rinv->Delete();
    return false;
    }
#endif /* 0 */

  vtkMatrix4x4* sainv = vtkMatrix4x4::New();
  sainv->DeepCopy(sa);
  sainv->Invert();
  sa->Delete();

  int    fromDimensions[3];
  double fromSpacing[3];

  aGUI->GetInputDimensions(fromDimensions);
  aGUI->GetInputSpacing(fromSpacing);

  vtkMatrix4x4* fa = vtkMatrix4x4::New();

  fa->SetElement(0,0, -fromSpacing[0]);
  fa->SetElement(0,1,  0.0);
  fa->SetElement(0,2,  0.0);
  fa->SetElement(0,3,  fromDimensions[0] * fromSpacing[0] / 2.0);

  fa->SetElement(1,0,  0.0);
  fa->SetElement(1,1,  0.0);
  fa->SetElement(1,2,  fromSpacing[2]);
  fa->SetElement(1,3, -fromDimensions[2] * fromSpacing[2] / 2.0);

  fa->SetElement(2,0,  0.0);
  fa->SetElement(2,1, -fromSpacing[1]);
  fa->SetElement(2,2,  0.0);
  fa->SetElement(2,3,  fromDimensions[1] * fromSpacing[1] / 2.0);

  fa->SetElement(3,0,  0.0);
  fa->SetElement(3,1,  0.0);
  fa->SetElement(3,2,  0.0);
  fa->SetElement(3,3,  1.0);

  vtkMatrix4x4* r1 = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(rinv,fa,r1);

  rinv->Delete();
  fa->Delete();

  vtkMatrix4x4* r2 = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(sainv,r1,r2);

  r1->Delete();
  sainv->Delete();

  vtkMatrix4x4* fr = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(sr,r2,fr);

  sr->Delete();
  r2->Delete();

  double outSpacing[3];
  double outResliceAxesDirectionCosines[9];
  double outResliceAxesOrigin[3];

  double ras[3];
  double magnitude;
  
  ras[0] = fr->GetElement(0,0);
  ras[1] = fr->GetElement(1,0);
  ras[2] = fr->GetElement(2,0);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[0] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[1] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[2] = ras[2] / magnitude;
  outSpacing[0]                     = magnitude;
  
  ras[0] = fr->GetElement(0,1);
  ras[1] = fr->GetElement(1,1);
  ras[2] = fr->GetElement(2,1);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[3] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[4] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[5] = ras[2] / magnitude;
  outSpacing[1]                     = magnitude;
  
  ras[0] = fr->GetElement(0,2);
  ras[1] = fr->GetElement(1,2);
  ras[2] = fr->GetElement(2,2);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[6] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[7] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[8] = ras[2] / magnitude;
  outSpacing[2]                     = magnitude;
  
  double origin[4], c[4];
  
  origin[0] = (fromDimensions[0] - 1.0) / 2.0;
  origin[1] = (fromDimensions[1] - 1.0) / 2.0;
  origin[2] = (fromDimensions[2] - 1.0) / 2.0;
  origin[3] = 1.0;
  
  fr->MultiplyPoint(origin,c);
  
  outResliceAxesOrigin[0] = c[0];
  outResliceAxesOrigin[1] = c[1];
  outResliceAxesOrigin[2] = c[2];
  
  aGUI->SetResliceAxesDirectionCosines(outResliceAxesDirectionCosines);
  aGUI->SetResliceAxesOrigin(outResliceAxesOrigin);
  aGUI->SetResliceSpacing(outSpacing);
  
  fr->Delete();

  return true;
} /* end orient_with_register() */

// ----------------------------------------------------------------------------
void
LoadRegistrationButtonCb (Fl_Button* aButton, void* aPtr)
{
  svvContourUI* GUI;

  if (aButton == NULL)
    return;
  if ((GUI = reinterpret_cast<svvContourUI*>(aPtr)) == NULL)
    return;

  char* fileName = NULL;

  if ( !(fileName = fl_file_chooser(
           "Read Registration Data?",
           "Registration (register.{dat,txt})",
           0/*NULL ... defaults to last filename*/)) )
    {
    return;
    }

  struct stat fs;
  
  // Open the file containing the resolutions and rotation matrix.
  if (stat(fileName, &fs) != 0)
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return;
    }

  // Opens the file and positions the stream pointer at EOF ...
  std::ifstream ifs(fileName, ios::in|ios::ate);

  if (!ifs.is_open())
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return;
    }

  // The read position at EOF is the file size.
  std::streampos size = ifs.tellg();
  // Rewind the stream.
  ifs.seekg(0, ios::beg);

  char pathName[1024];
    
  if (!ReadLine(ifs, pathName, sizeof(pathName)))
    {
    return;
    }

  double ipr, tpr, scaling, m[16];

  ifs >> ipr >> tpr >> scaling
      >> m[0] >> m[1] >> m[2] >> m[3]
      >> m[4] >> m[5] >> m[6] >> m[7]
      >> m[8] >> m[9] >> m[10] >> m[11]
      >> m[12] >> m[13] >> m[14] >> m[15];

  if (ifs.fail())
    {
    return;
    }

  double inputSpacing[3];

  GUI->GetInputSpacing(inputSpacing);

  if ( (inputSpacing[0] != ipr) ||
       (inputSpacing[1] != ipr) ||
       (inputSpacing[2] != tpr) )
    {
    Fl::warning("Registration spacing does not match input spacing.");
    }

  if (!gImageResample) gImageResample = vtkImageResample::New();

  vtkMatrix4x4* matrix = vtkMatrix4x4::New();
  matrix->DeepCopy(m);
  matrix->Invert();

  vtkMatrixToLinearTransform* transform = vtkMatrixToLinearTransform::New();
  transform->SetInput(matrix);

  gImageResample->SetResliceTransform(transform);
  transform->Delete();

  if (!OrientWithRegister(m, GUI))
    {
    return;
    }

  double resliceAxesDirectionCosines[9];
  double resliceAxesOrigin[9];
  double resliceSpacing[3];

  GUI->GetResliceAxesDirectionCosines(resliceAxesDirectionCosines);
  GUI->GetResliceAxesOrigin(resliceAxesOrigin);
  GUI->GetResliceSpacing(resliceSpacing);

  gImageResample->SetResliceAxesDirectionCosines(resliceAxesDirectionCosines);
  gImageResample->SetResliceAxesOrigin(resliceAxesOrigin);

  gImageResample->SetAxisOutputSpacing(0,resliceSpacing[0]);
  gImageResample->SetAxisOutputSpacing(1,resliceSpacing[1]);
  gImageResample->SetAxisOutputSpacing(2,resliceSpacing[2]);
}

// ----------------------------------------------------------------------------
void
SaveContourButtonCb (Fl_Button* aButton, void* aPtr)
{
  svvContourUI* GUI;

  if (aButton == NULL)
    return;
  if ((GUI = reinterpret_cast<svvContourUI*>(aPtr)) == NULL)
    return;

  char* fileName = NULL;

  if ( fileName = fl_file_chooser(
         "Write Contour Polygonal Data?",
         "PolyData Files (*.{vtk,vtp})",
         0/*NULL ... defaults to last filename*/) )
    {
    std::string outName(fileName);
    std::string::size_type idx = outName.rfind('.');
    std::string suffix;

    if (idx==std::string::npos)
      {
      suffix = "vtp";
      outName.append(".vtp");
      }
    else
      {
      suffix = outName.substr(idx+1);
      if (suffix!="vtp" && suffix!="vtk")
        {
        suffix = "vtp";
        outName.replace(idx,outName.size(),".vtp");
        }
      }

    if (!outName.empty())
      {
      if (suffix == "vtp")
        {
        vtkXMLDataSetWriter* writer = vtkXMLDataSetWriter::New();
          {
          writer->SetInput(gNormals->GetOutput());
          writer->SetByteOrderToBigEndian();
          writer->SetFileName(outName.c_str());
          vtkZLibDataCompressor* compressor = vtkZLibDataCompressor::New();
          writer->SetCompressor(compressor);
          compressor->Delete();
          //writer->SetDataModeToAscii();
          //writer->SetDataModeToBinary();
          writer->SetDataModeToAppended();
          writer->EncodeAppendedDataOn();
          writer->Write();
          }
        writer->Delete();
        }
      else if (suffix == "vtk")
        {
        vtkDataSetWriter* writer = vtkDataSetWriter::New();
          {
          writer->SetInput(gNormals->GetOutput());
          writer->SetFileName(outName.c_str());
          writer->Write();
          }
        writer->Delete();          
        }
      }
    }
}

void
ExitButtonCb (Fl_Button* aButton, void* aPtr)
{
  std::exit(EXIT_SUCCESS);
}

/*
 * End of: $Id: svvContour.cxx,v 1.1.1.1 2006/12/19 22:59:41 christianh Exp $.
 *
 */
