/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Fl_Toggle_Node.H,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * Adapted from Flek (The Fast Light Environment Kit).
 * 
 */
#ifndef FL_TOGGLE_NODE_H_
#  define FL_TOGGLE_NODE_H_
// Flek-derived
#  include "Fl_Toggle_Node_Base.H"
#  include <FL/Fl.H>
// C++ forwarding ANSI C
#  include <cstring>

class Fl_Image;

/** \class   Fl_Toggle_Node
 *  \brief   Data storage and management object for Fl_Toggle_Tree.
 * 
 * \author  Sean McInerney (orig: James Dean Palmer, Flek library maintainer)
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:53 $
 * 
 * \sa
 * Fl_Toggle_Node Fl_Toggle_Node_Base
 */
class Fl_Toggle_Node : public Fl_Toggle_Node_Base
{
  friend class Fl_Toggle_Tree;

public:
  /** Constructor. */
  Fl_Toggle_Node (const char*   aLabel   = (char *) 0,
                  bool          aCanOpen = true,
                  Fl_Image*     aIcon    = (Fl_Image *) 0,
                  void*         aData    = (void *) 0)
    : Fl_Toggle_Node_Base(true),
      selected_(false),
      changed_(false),
      openable_(aCanOpen),
      label_(aLabel ? strcpy(new char [strlen(aLabel)+1], aLabel) : 0),
      icon_(aIcon),
      user_data_(aData)
    {}

  /*@{*/
  /** Get/Set the node label. */
  inline const char*    label (void) const
    { return this->label_; }
  inline void           label (const char* label);
  /*@}*/

  /*@{*/
  /** Get/Set the node icon. */
  inline Fl_Image*      icon (void) const
    { return this->icon_; }
  inline void           icon (Fl_Image* icon)
    { this->icon_ = icon; }
  /*@}*/

  /*@{*/
  /** Get/Set the node data. */
  inline void*          user_data (void) const
    { return this->user_data_; }
  inline void           user_data (void* a)
    { this->user_data_ = a; }
  /*@}*/

  /**  */
  inline bool           selected (void) const
    { return this->selected_; }

  /**  */
  inline bool           changed (void) const
    { return this->changed_; }

  /** Query ability to add subnodes to this node. */
  inline bool           openable (void) const
    { return this->openable_; }

  /** Get the parent of this element. */
  inline Fl_Toggle_Node*        parent_node (void) const;

  /** Get the previous sibling node on the branch. */
  inline Fl_Toggle_Node*        previous_sibling (void) const;

  /** Get the next sibling node on the branch. */
  inline Fl_Toggle_Node*        next_sibling (void) const;

  /** Get the first subnode (child) pointer. */
  inline Fl_Toggle_Node*        first_child (void) const;

  /** Get the first visible subnode (child) pointer. */
  inline Fl_Toggle_Node*        visible_child (void) const;

protected:
  bool          selected_;
  bool          changed_;
  bool          openable_;

  char*         label_;
  Fl_Image*     icon_;
  void*         user_data_;
};

inline Fl_Toggle_Node*
Fl_Toggle_Node::parent_node (void) const
{
  return
    static_cast<Fl_Toggle_Node*>(this->Fl_Toggle_Node_Base::parent_node());
}

inline Fl_Toggle_Node*
Fl_Toggle_Node::previous_sibling (void) const
{
  return
    static_cast<Fl_Toggle_Node*>(this->Fl_Toggle_Node_Base::previous_sibling());
}

inline Fl_Toggle_Node*
Fl_Toggle_Node::next_sibling (void) const
{
  return
    static_cast<Fl_Toggle_Node*>(this->Fl_Toggle_Node_Base::next_sibling());
}

inline Fl_Toggle_Node*
Fl_Toggle_Node::first_child (void) const
{
  return
    static_cast<Fl_Toggle_Node*>(this->Fl_Toggle_Node_Base::first_child());
}

inline Fl_Toggle_Node*
Fl_Toggle_Node::visible_child (void) const
{
  return
    static_cast<Fl_Toggle_Node*>(this->Fl_Toggle_Node_Base::visible_child());
}

inline void
Fl_Toggle_Node::label (const char* aLabel)
{
  if (this->label_ == NULL && aLabel == NULL)
    return;

  if (this->label_ != NULL && aLabel != NULL && strcmp(this->label_,aLabel)==0)
    return;

  if (this->label_ != NULL)
    {
    delete [] this->label_;
    this->label_ = (char *) 0;
    }

  if (aLabel != NULL)
    {
    try
      {
      this->label_ = strcpy(new char [strlen(aLabel) + 1], aLabel);
      }
    catch (...)
      {
      delete [] this->label_;
      this->label_ = (char *) 0;
      Fl::error("Fl_Toggle_Node::label failed to create label \"%s\".",aLabel);
      }
    }
}

#endif /* FL_TOGGLE_NODE_H_ */
/* 
 * End of: $Id: Fl_Toggle_Node.H,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $.
 * 
 */
