/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * "$Id: Fl_Toggle_Tree.H,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $"
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * Adapted from Flek (The Fast Light Environment Kit).
 * 
 */
#ifndef FL_TOGGLE_TREE_H_
#  define FL_TOGGLE_TREE_H_
// Flek-derived
#  include "Fl_Toggle_Tree_Base.H"
#  include "Fl_Toggle_Node.H"
// FLTK
#  include <FL/Fl_Image.H>
// IOStreams
#  include <iostream>
// STL
#  include <string>
#  include <map>


/** State of the Fl_Toggle_Tree widget. */
enum Fl_ToggleState
{
  FL_TOGGLE_NONE        = 0, /*!< none */
  FL_TOGGLE_SELECT      = 1, /*!< selected */
  FL_TOGGLE_RESELECT    = 2, /*!< reselected */
  FL_TOGGLE_SELECT_MASK = 3, /*!< selection mask */
  FL_TOGGLE_OPENED      = 4, /*!< open */
  FL_TOGGLE_CLOSED      = 8, /*!< closed */
  FL_TOGGLE_HIT         = 16 /*!< hit */
};

typedef std::map<std::string,Fl_Image*> ImageIconMap;

class Fl_Input;

/** \class   Fl_Toggle_Tree
 *  \brief   Tree widget with togglable branch visibility.
 * 
 * \author  Sean McInerney (orig: James Dean Palmer, Flek library maintainer)
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:53 $
 * 
 * \sa
 * Fl_Toggle_Node Fl_Toggle_Tree_Base
 */
class Fl_Toggle_Tree : public Fl_Toggle_Tree_Base
{
public:
  /** The constructor makes an empty Fl_Toggle_Tree. */
  Fl_Toggle_Tree (int x, int y, int w, int h);
  virtual ~Fl_Toggle_Tree();

  virtual int handle (int event);

  /** Returns the first (top) Fl_Toggle_Node_Base in the widget tree. */
  inline Fl_Toggle_Node*        first_node (void) const;

  /** Returns the current node (as set with the traversal functions). */
  inline Fl_Toggle_Node*        current_node (void) const;

  /** Return the selected node.
   *
   * If multiple nodes are selected, then \c selected_node() returns
   * \c null, and \c selection_next() should be checked.
   *
   * \note The \c current_node_ is not necessarily equal to the
   *       \c selected_node_. \c Fl_Toggle_Tree attempts to maintain the
   *       selected item even while the tree is being modified. Of
   *       course, if the node is unselected or deleted by some
   *       operation, \c selected_node() will return \c null.
   */
  inline Fl_Toggle_Node*        selected_node (void) const;

  /** Traverses forward and returns the next selected node. */
  Fl_Toggle_Node*               selection_next (void);

  /** Returns the i-th selected node. */
  Fl_Toggle_Node*               selection_node (int i);

  /** Returns the currently selected index. */
  int                           selection_count (void);

  /** Sets nodes between start and end as selected.
   *
   * \param start The first node to be selected.
   * \param end   The last node to be selected.
   * \param add   add == 0 - pick one only (unpick rest)
   *              add == 1 - add picked (never unpick)
   *              add  > 1 - toggle picked.
   */
  void                          select_range (Fl_Toggle_Node* start,
                                              Fl_Toggle_Node* end,
                                              int             add = 0);

  /** Unselects all selected items. */
  inline void                   select_none (void)
    { this->select_range((Fl_Toggle_Node *) 0, (Fl_Toggle_Node *) 0, 0); }

  /** Returns the state of the Fl_Toggle_Tree widget. */
  inline Fl_ToggleState         state (void) const
    { return this->state_; }

  /** Open \c node, revealing any sub items belonging to \c node. */
  void                          open (Fl_Toggle_Node* node);

  /** Close \c node, hiding any sub items belonging to \c node. */
  void                          close (Fl_Toggle_Node* node);

  /*@{*/
  /** Set/Get the selection text color for lines in the browser. */
  inline void                   selection_label_color (Fl_Color a)
    { this->selection_label_color_ = a; }
  inline Fl_Color               selection_label_color (void) const
    { return this->selection_label_color_; }
  /*@}*/

  /*@{*/
  /** Set/Get the alternating background color for lines in the browser. */
  inline void                   alternate_color (Fl_Color a)
    { this->alternate_color_ = a; }
  inline Fl_Color               alternate_color (void) const
    { return this->alternate_color_; }
  /*@}*/

  /*@{*/
  /** Set/Get the trim color that seperates lines of text in the browser. */
  inline void                   trim_color (Fl_Color a)
    { this->trim_color_ = a; }
  inline Fl_Color               trim_color (void) const
    { return this->trim_color_; }
  /*@}*/

  /*@{*/
  /** Set/Get if toggled items are indented or not. */
  inline void                   indent_toggles (bool a)
    { this->indent_toggles_ = a; }
  inline bool                   indent_toggles (void) const
    { return this->indent_toggles_; }
  /*@}*/

  /*@{*/
  /** Set/Get if lines are drawn between toggle nodes. */
  inline void                   line_visibility (bool a)
    { this->line_visibility_ = a; }
  inline bool                   line_visibility (void) const
    { return this->line_visibility_; }
  /*@}*/

  /*@{*/
  /** Set/Get the horizontal label offset in pixels. */
  inline void                   label_offset (int a)
    { this->label_offset_ = a; }
  inline int                    label_offset (void) const
    { return this->label_offset_; }
  /*@}*/

  /*@{*/
  /** Set/Get the horizontal icon offset in pixels. */
  inline void                   icon_offset (int a)
    { this->icon_offset_ = a; }
  inline int                    icon_offset (void) const
    { return this->icon_offset_; }
  /*@}*/

  /*@{*/
  /** Set/Get the icon displayed when a node is "open". */
  void                          opened_icon (Fl_Image*);
  inline Fl_Image*              opened_icon (void) const
    { return this->opened_icon_; }
  /*@}*/

  /*@{*/
  /** Set/Get the icon displayed when a node is "closed". */
  void                          closed_icon (Fl_Image*);
  inline Fl_Image*              closed_icon (void) const
    { return this->closed_icon_; }
  /*@}*/

  /** Find an icon in the static map by name. */
  static Fl_Image*      find_icon (std::string);

  /** Add a named icon to the static map. */
  static void           add_icon (std::string, Fl_Image*);

  /** Add a named icon (created as pixmap from data) to the static map. */
  static Fl_Image*      make_icon (std::string, const char* const *);

  /** Remove a named icon from the static map. */
  static void           remove_icon (std::string, bool = true);

  /*@{*/
  /** Set/Get the current column widths array.
   *
   * This array is zero-terminated and specifies the widths in pixels
   * of each column. The text is split at each \c column_delimiter and
   * each part is formatted into it's own column. After the last column
   * any remaining text is formatted into the space between the last
   * column and the right edge of the \c Fl_Toggle_Tree wdiget, even if
   * the text contains instances of \c column_delimiter. The default value
   * is a one-element array of just a zero, which makes there are no
   * columns.
   *
   * \note Make sure the last entry is zero.
   */
  inline void                   column_widths (const int* a)
    { this->column_widths_ = a; }
  inline const int*             column_widths (void) const
    { return this->column_widths_; }
  /*@}*/

  /*@{*/
  /** Set/Get the current column separator character.
   *
   * By default this is '\\t' (tab).
   *
   * \note This will only have an effect if you also set \c column_widths.
   */
  inline void                   column_delimiter (char a)
    { this->column_delimiter_ = a; }
  inline char                   column_delimiter (void) const
    { return this->column_delimiter_; }
  /*@}*/

  /*@{*/
  /** Set/Get the default text font for the lines in the browser. */
  inline void                   textfont (Fl_Font a)
    { this->textfont_ = a; }
  inline Fl_Font                textfont (void) const
    { return this->textfont_; }
  /*@}*/

  /*@{*/
  /** Set/Get the default text size for the lines in the browser. */
  inline void                   textsize (unsigned int a)
    { this->textsize_ = a; }
  inline unsigned int           textsize (void) const
    { return this->textsize_; }
  /*@}*/

  /*@{*/
  /** Set/Get the default text color for the lines in the browser.  */
  inline void                   textcolor (Fl_Color a)
    { this->textcolor_ = a; }
  inline Fl_Color               textcolor (void) const
    { return this->textcolor_; }
  /*@}*/

  /* Remove an item from the tree but also select the previous line
     if possible. */

  /** Removes \c node (and all of it's sub nodes) from the tree.
   * If successful remove returns 1, otherwise it returns 0. 
   */
  inline int                    remove (Fl_Toggle_Node* node);

  /**
   * Performs a find() with \c p as argument and removes the returned
   * node if it exists.
   */
  int                           remove (void* p);

  /**
   * Performs a find() with the string \c s as it's argument and remove()s 
   * the returned node if it exists. 
   */
  int                           remove (const char* s);

  /** Sets the current node pointer to \c a. */
  inline Fl_Toggle_Node*        traverse_start (Fl_Toggle_Node_Base* a);

  /** Sets the traversal pointer to the first node and then returns first.
   * The second form is provided for convenience. 
   */
  inline Fl_Toggle_Node*        traverse_start (void);

  /** Depth first traversal through the tree, updating current node pointer.
   *
   * \param visible Flag should be \c true if you want to restrict
   *                traversal to the visible tree (the nodes that are not
   *                closed).
   * \param depth   The variable is updated with the new node depth, if
   *                the old node depth is passed to \c traverse_forward.
   *
   * \return  If \c traverse_forward has reached the end of the tree, the
   *          \c null current node pointer is returned. Otherwise,
   *          \c traverse_forward returns the current node pointer set
   *          to the next node in the tree.
   */
  inline Fl_Toggle_Node*        traverse_forward (bool visible, int& depth);

  /** Equivalent to \c traverse_forward(false, temp). */
  inline Fl_Toggle_Node*        traverse_forward (void);

  /** Traverse back one. */
  inline Fl_Toggle_Node*        traverse_backward (void);

  /**
   * Returns the first node in the tree whose data pointer is equal to 
   * the pointer \c p, or returns \c null if no node matches \c p.
   */
  Fl_Toggle_Node*               find (void* p);

  /**
   * Returns the first node in the tree that matches the string \c s,
   * or returns \c null if no node matches \c s.
   */
  Fl_Toggle_Node*               find (const char* s);

  /** Inserts node as the next sub item after the current node.
   *
   * If the tree is empty, then node becomes the first node of the toggle tree.
   */
  inline Fl_Toggle_Node*        add_next_child (const char* label   = 0,
                                                bool        canOpen = true,
                                                Fl_Image*   icon    = 0,
                                                void*       data    = 0);

  /** Inserts node as the next item after the current node.
   *
   * If the tree is empty, then node becomes the first node of the toggle tree. 
   */
  inline Fl_Toggle_Node*        add_next_sibling (const char* label   = 0,
                                                  bool        canOpen = true,
                                                  Fl_Image*   icon    = 0,
                                                  void*       data    = 0);

  /** Equiv. to add_next_child(new Fl_Toggle_Node(label, true, pixmap, d)). */
  inline void                   add_next_child (Fl_Toggle_Node* a)
    { this->Fl_Toggle_Tree_Base::add_next_child(a); }

  /** Equiv. to add_next_sibling(new Fl_Toggle_Node(label,true,pixmap,d)). */
  inline void                   add_next_sibling (Fl_Toggle_Node* a)
    { this->Fl_Toggle_Tree_Base::add_next_sibling(a); }

  /** Sort the tree by label. */
  static int                    sort_by_label (Fl_Toggle_Node_Base* a,
                                               Fl_Toggle_Node_Base* b);

  /*@{*/
  /** If on, this causes editing to occur if an item is reselected. */
  inline void                   edit_on_reselect (bool a)
    { this->edit_on_reselect_ = a; }
  inline bool                   edit_on_reselect (void) const
    { return this->edit_on_reselect_; }
  /*@}*/

  /*@{*/
  /** Set the callback to invoke when editing. */
  void                          edit_callback (Fl_Callback*, void*);
  void                          edit_callback (Fl_Callback*);
  void                          edit_callback (Fl_Callback0*);
  void                          edit_callback (Fl_Callback1*, long = 0);
  /*@}*/

  /** The default edit callback. */
  static void                   edit_default_callback (Fl_Input* in, void* p);

  /** Stop editing an entry. */
  void                          end_edit (void);

  void Print (std::ostream& aTarget)
    {
      if (this->first_node_ != NULL)
        aTarget << "first_node_     (" << this->first_node_ << "): "
                << static_cast<Fl_Toggle_Node*>(this->first_node_)->label()
                << std::endl;
      else
        aTarget << "first_node_     (NULL)" << std::endl;

      if (this->top_node_ != NULL)
        aTarget << "top_node_       (" << this->top_node_ << "): "
                << static_cast<Fl_Toggle_Node*>(this->top_node_)->label()
                << std::endl;
      else
        aTarget << "top_node_       (NULL)" << std::endl;

      if (this->current_node_ != NULL)
        aTarget << "current_node_ (" << this->current_node_ << "): "
                << static_cast<Fl_Toggle_Node*>(this->current_node_)->label()
                << std::endl;
      else
        aTarget << "current_node_ (NULL)" << std::endl;

      if (this->selected_node_ != NULL)
        aTarget << "selected_node_   (" << this->selected_node_ << "): "
                << static_cast<Fl_Toggle_Node*>(this->selected_node_)->label()
                << std::endl;
      else
        aTarget << "selected_node_   (NULL)" << std::endl;

      aTarget << "\ntop_depth_:   " << this->top_depth_ << std::endl;
      aTarget << "top_yoffset_: " << this->top_yoffset_ << "\n" << std::endl;

      if (this->damaged_node_ != (Fl_Toggle_Node_Base *) 0)
        aTarget << "damaged_node_   (" << this->damaged_node_ << "): "
                << static_cast<Fl_Toggle_Node*>(this->damaged_node_)->label()
                << std::endl;
      else
        aTarget << "damaged_node_   (NULL)" << std::endl;

      if (this->selection_node_ != NULL)
        aTarget << "\nselection_node_ (" << this->selection_node_ << "): "
                << this->selection_node_->label()
                << std::endl;
      else
        aTarget << "\nselection_node_ (NULL)" << std::endl;
    }

protected:
  virtual int   node_height (Fl_Toggle_Node_Base* node);

  virtual void  edit (Fl_Toggle_Node* t, int cx, int cy);

  void          draw_label (char* str, int indent, int x, int y, int w, int h);
  virtual void  draw_node (int depth, int cy, Fl_Toggle_Node_Base* node);

  int                   icon_offset_;
  int                   label_offset_;

  bool                  indent_toggles_;

  bool                  edit_on_reselect_;
  bool                  line_visibility_;

  Fl_Color              selection_label_color_;
  Fl_Color              alternate_color_;
  Fl_Color              trim_color_;

  const int*            column_widths_;
  char                  column_delimiter_;

  Fl_Font               textfont_;
  unsigned int          textsize_;
  Fl_Color              textcolor_;

  Fl_ToggleState        state_;

  Fl_Image*             opened_icon_;
  Fl_Image*             closed_icon_;

  static ImageIconMap*  s_icon_map_;

  Fl_Toggle_Node*       selection_node_;
  int                   selection_index_;
  int                   selection_count_;

  Fl_Input*             edit_input_;

  static void   ClassInitialize (void);
  static void   ClassFinalize (void);

public:
  /** \internal
   * Nested initializer class.
   * The static initializer object is declared below.
   */
  class Init
  {
  public:
    Init (void);
    ~Init();
  private:
    static long TranslationUnits;
    friend class Fl_Toggle_Tree;
   };

  friend class Init;
};

// ----------------------------------------------------------------------------
inline Fl_Toggle_Node*
Fl_Toggle_Tree::first_node (void) const
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::first_node());
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::current_node (void) const
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::current_node());
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::selected_node (void) const
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::selected_node());
}

// ----------------------------------------------------------------------------
inline int
Fl_Toggle_Tree::remove (Fl_Toggle_Node* aNode)
{
  Fl_Toggle_Node* current = (Fl_Toggle_Node *) 0;

  if (aNode->selected_)
    {
    // Determine if any remaining node will inherit selection state.
    if      (aNode->parent_node_ != (Fl_Toggle_Node_Base *) 0)
      {
      current = static_cast<Fl_Toggle_Node*>(aNode->parent_node_);
      }
    else if (aNode->previous_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      current = static_cast<Fl_Toggle_Node*>(aNode->previous_sibling_);
      }
    }

  this->Fl_Toggle_Tree_Base::remove(static_cast<Fl_Toggle_Node_Base*>(aNode));

  if (current == (Fl_Toggle_Node *) 0)
    {
    current = static_cast<Fl_Toggle_Node*>(this->first_node_);
    }

  if (current != (Fl_Toggle_Node *) 0)
    {
    current->selected_ = true;
    }

  this->selected_node_ = current;

  this->redraw();

  return 1;
}

// ----------------------------------------------------------------------------
inline Fl_Toggle_Node*
Fl_Toggle_Tree::traverse_start (Fl_Toggle_Node_Base* a)
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::traverse_start(a));
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::traverse_start (void)
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::traverse_start());
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::traverse_forward (bool aVisible, int& aDepth)
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::traverse_forward(aVisible, aDepth));
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::traverse_forward (void)
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::traverse_forward());
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::traverse_backward (void)
{
  return static_cast<Fl_Toggle_Node*>
    (this->Fl_Toggle_Tree_Base::traverse_backward());
}

// ----------------------------------------------------------------------------
inline Fl_Toggle_Node*
Fl_Toggle_Tree::add_next_child (const char* aLabel,
                                bool        aCanOpen,
                                Fl_Image*   aImage,
                                void*       aData)
{
  Fl_Toggle_Node* node;

  try
    {
    node = new Fl_Toggle_Node(aLabel, aCanOpen, aImage, aData);
    this->Fl_Toggle_Tree_Base::add_next_child(node);
    }
  catch (...)
    {
    delete node;
    node = (Fl_Toggle_Node *) 0;
    Fl::error("Fl_Toggle_Tree::add_next_child failed to create node.");
    }

  return node;
}

inline Fl_Toggle_Node*
Fl_Toggle_Tree::add_next_sibling (const char* aLabel,
                                  bool        aCanOpen,
                                  Fl_Image*   aImage,
                                  void*       aData)
{
  Fl_Toggle_Node* node;

  try
    {
    node = new Fl_Toggle_Node(aLabel, aCanOpen, aImage, aData);
    this->Fl_Toggle_Tree_Base::add_next_sibling(node);
    }
  catch (...)
    {
    delete node;
    node = (Fl_Toggle_Node *) 0;
    Fl::error("Fl_Toggle_Tree::add_next_sibling failed to create node.");
    }

  return node;
}

#  ifndef FL_TOGGLE_TREE_INITIALIZED_
#    define FL_TOGGLE_TREE_INITIALIZED_
/** \internal
 * Global initializer object, to ensure construction of static objects.
 */
static Fl_Toggle_Tree::Init flToggleTreeInitInstance;
#  endif /* FL_TOGGLE_TREE_INITIALIZED_ */

#endif /* FL_TOGGLE_TREE_H_ */
/* 
 * End of: $Id: Fl_Toggle_Tree.H,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $.
 * 
 */
