/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * "$Id: Fl_Toggle_Tree_Base.H,v 1.1.1.1 2006/12/19 22:59:54 christianh Exp $"
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * Adapted from Flek (The Fast Light Environment Kit).
 * 
 */
#ifndef FL_TOGGLE_TREE_BASE_H_
#  define FL_TOGGLE_TREE_BASE_H_
// FLTK
#  include <FL/Fl_Widget.H>
// Flek-derived
#  include "Fl_Toggle_Node_Base.H"

#  define FL_DAMAGE_TREE 0x40

enum Fl_SortOrder
{
  FL_FORWARD_SORT = 0,
  FL_REVERSE_SORT = 1
};

extern int (*s_node_compare_)(Fl_Toggle_Node_Base*, Fl_Toggle_Node_Base*);

/** \class   Fl_Toggle_Tree_Base
 *  \brief   Base class for Fl_Toggle_Tree.
 * 
 * This class has been designed to maintain a doubly linked tree list, 
 * and defers data storage and management to subclasses of 
 * Fl_Toggle_Tree_Base and Fl_Toggle_Node_Base. This class also maintains 
 * whether it's branches are visible or not (i.e. "opened" or "closed"). 
 * 
 * \author  Sean McInerney (orig: James Dean Palmer, Flek library maintainer)
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:54 $
 * 
 * \sa
 * Fl_Toggle_Node_Base Fl_Toggle_Tree
 */
class Fl_Toggle_Tree_Base : public Fl_Widget 
{
public:
  /** The constructor makes an empty Fl_Toggle_Tree_Base. */
  Fl_Toggle_Tree_Base (int x, int y, int w, int h)
    : Fl_Widget(x, y, w, h),
      first_node_(0),
      top_node_(0),
      current_node_(0),
      selected_node_(0),
      top_depth_(0),
      top_yoffset_(0),
      damaged_node_(0)
    {}

  /** Returns the first (top) Fl_Toggle_Node_Base in the widget tree. */
  inline Fl_Toggle_Node_Base*   first_node (void) const
    { return this->first_node_; }

  /** Return the top most node in the tree. */
  inline Fl_Toggle_Node_Base*   top_node (void) const
    { return this->top_node_; }

  /** Returns the current node (as set with the traversal functions). */
  inline Fl_Toggle_Node_Base*   current_node (void) const
    { return this->current_node_; }

  /** Returns the selected node. */
  inline Fl_Toggle_Node_Base*   selected_node (void) const
    { return this->selected_node_; }

  /** Find a node in the tree */
  Fl_Toggle_Node_Base*          find (int fy, int& depth, int& ry);

  /** Sets \c current_node_ to \c aNode and returns \c current_node_. */
  inline Fl_Toggle_Node_Base*   traverse_start (Fl_Toggle_Node_Base* aNode)
    { return (this->current_node_ = aNode); }

  /** Sets \c current_node_ to \c first_node_ and returns \c current_node_. */
  inline Fl_Toggle_Node_Base*   traverse_start (void)
    { return (this->current_node_ = this->first_node_); }

  /**
   * If the current node is \c non-null and has a parent, the
   * parent becomes the the current node and is returned, otherwise
   * returns \c null.
   */
  inline Fl_Toggle_Node_Base*   traverse_up (void);

  /** Depth first traversal through the tree, updating the current node pointer.
   *
   * \param visible Flag should be \c true if you want to restrict
   *                traversal to the visible tree (the nodes that are
   *                not closed).
   * \param depth   The variable is updated with the new node depth, if
   *                the old node depth is passed to \c traverse_forward.
   *
   * \return  If \c traverse_forward has reached the end of the tree, the
   *          \c null current node pointer is returned. Otherwise,
   *          \c traverse_forward returns the current node pointer set
   *          to the next node in the tree.
   */
  inline Fl_Toggle_Node_Base*   traverse_forward (bool visible, int& depth);

  /** Equivalent to \c traverse_forward(false, temp). */
  inline Fl_Toggle_Node_Base*   traverse_forward (void);

  /** Traverse back one. */
  inline Fl_Toggle_Node_Base*   traverse_backward (void);

  /** Insert a node as the next item after current node. */
  void                          add_next_sibling (Fl_Toggle_Node_Base* node);

  /** Insert a node as the next child item after the current node. */
  void                          add_next_child (Fl_Toggle_Node_Base* node);

  /** Remove a node (and all of it's sub nodes) from the tree.
   * \return If successful returns 1, else returns 0. 
   */
  int                           remove (Fl_Toggle_Node_Base* node);

  /** Clear the tree. */
  inline int                    clear (void);

  /** Update the height of the tree. */
  void                          update_height (void);

  /** Set \c node as "open" and return new total tree height. */
  inline int                    open (Fl_Toggle_Node_Base* node);

  /** Set \c node as "closed" and return new total tree height. */
  inline int                    close (Fl_Toggle_Node_Base* node);

  /** Virtual method that draws the tree. */
  virtual void                  draw (void);

protected:
  Fl_Toggle_Node_Base*  first_node_;
  Fl_Toggle_Node_Base*  top_node_;
  Fl_Toggle_Node_Base*  current_node_;
  Fl_Toggle_Node_Base*  selected_node_;

  int                   top_depth_;
  int                   top_yoffset_;

  Fl_Toggle_Node_Base*  damaged_node_;

  virtual int           node_height (Fl_Toggle_Node_Base* node);

  /** \internal height must be |1 , so Fl_ToggleTree can do color-swapping
   * with &1 on y coordinate
   */
  inline int            height_ (Fl_Toggle_Node_Base* aNode)
    { return this->node_height(aNode) | 1; }

  inline int            total_height (Fl_Toggle_Node_Base* node);

  void                  update_top (void);

  virtual void          draw_node (int depth,int cy, Fl_Toggle_Node_Base* node);

  virtual int           handle (int aEvent)
    { return Fl_Widget::handle(aEvent); }

  Fl_Toggle_Node_Base*  sort_ ( Fl_Toggle_Node_Base* start,
                                int (*compar)( Fl_Toggle_Node_Base*,
                                               Fl_Toggle_Node_Base* ),
                                int down,
                                Fl_SortOrder order = FL_FORWARD_SORT );
  
public:
  static inline int     s_compare_ (void* a, void *b);
  static inline int     s_compare_reverse_ (void* a, void *b);

  inline Fl_Toggle_Node_Base*   sort (Fl_Toggle_Node_Base* start,
                                      int (*compar)(Fl_Toggle_Node_Base*,
                                                    Fl_Toggle_Node_Base*),
                                      Fl_SortOrder order = FL_FORWARD_SORT);

  inline Fl_Toggle_Node_Base*   sort_tree (Fl_Toggle_Node_Base* start,
                                           int (*compar)(Fl_Toggle_Node_Base*,
                                                         Fl_Toggle_Node_Base*),
                                           Fl_SortOrder order =FL_FORWARD_SORT);

  inline void                   sort (int (*compar)(Fl_Toggle_Node_Base*,
                                                    Fl_Toggle_Node_Base*),
                                      Fl_SortOrder order = FL_FORWARD_SORT);

  inline void                   sort_tree (int (*compar)(Fl_Toggle_Node_Base*,
                                                         Fl_Toggle_Node_Base*),
                                           Fl_SortOrder order =FL_FORWARD_SORT);
};

// ----------------------------------------------------------------------------
inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::traverse_up (void)
{
  if (this->current_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    this->current_node_ = this->current_node_->parent_node_;
    }

  return this->current_node_;
}

inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::traverse_forward (bool aVisibleOnly, int& aDepth)
{
  // If the current node is set and has a visible child ...
  if      ( aVisibleOnly &&
            this->current_node_                 != (Fl_Toggle_Node_Base *) 0 &&
            this->current_node_->visible_child_ != (Fl_Toggle_Node_Base *) 0 )
    {
    this->current_node_ = this->current_node_->visible_child_;
    aDepth++;
    }
  // If the current node is set and has a child ...
  else if ( !aVisibleOnly &&
            this->current_node_               != (Fl_Toggle_Node_Base *) 0 &&
            this->current_node_->first_child_ != (Fl_Toggle_Node_Base *) 0 )
    {
    this->current_node_ = this->current_node_->first_child_;
    aDepth++;
    }
  // If the current node is set and has a next sibling ...
  else if ( this->current_node_                != (Fl_Toggle_Node_Base *) 0 &&
            this->current_node_->next_sibling_ != (Fl_Toggle_Node_Base *) 0 )
    {
    this->current_node_ = current_node_->next_sibling_;
    }
  // Traverse upward in search of a next sibling ...
  else
    {
    while ( this->current_node_                != (Fl_Toggle_Node_Base *) 0 &&
            this->current_node_->next_sibling_ == (Fl_Toggle_Node_Base *) 0 )
      {
      this->current_node_ = this->current_node_->parent_node_;
      aDepth--;
      }
    if (this->current_node_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = this->current_node_->next_sibling_;
      }
    }

  return this->current_node_;
}

inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::traverse_forward (void)
{
  int d;
  return this->traverse_forward(false, d);
}

inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::traverse_backward (void)
{
  if ( this->current_node_                    != (Fl_Toggle_Node_Base *) 0 &&
       this->current_node_->previous_sibling_ != (Fl_Toggle_Node_Base *) 0 )
    {
    this->current_node_ = this->current_node_->previous_sibling_;

    while (this->current_node_->first_child_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = this->current_node_->first_child_;

      while (this->current_node_->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
        {
        this->current_node_ = this->current_node_->next_sibling_;

        if ( this->current_node_->next_sibling_ == (Fl_Toggle_Node_Base *) 0 &&
             this->current_node_->first_child_  != (Fl_Toggle_Node_Base *) 0 )
          {
          this->current_node_ = this->current_node_->first_child_;
          }
        }
      }
    }
  else
    {
    this->current_node_ = this->current_node_->parent_node_;
    }

  return this->current_node_;
}

// ----------------------------------------------------------------------------
inline int
Fl_Toggle_Tree_Base::clear (void)
{
  while (this->first_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    this->remove(this->first_node_);
    }

  return 1;
}

// ----------------------------------------------------------------------------
inline int
Fl_Toggle_Tree_Base::total_height (Fl_Toggle_Node_Base* aNode)
{
  int result = 0;
  int depth  = 1;

  while (aNode != (Fl_Toggle_Node_Base *) 0)
    {
    result += this->node_height(aNode);

    if      (aNode->visible_child_ != (Fl_Toggle_Node_Base *) 0)
      {
      aNode = aNode->visible_child_;
      depth++;
      }
    else if (aNode->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      aNode = aNode->next_sibling_;
      }
    else
      {
      while ( aNode                != (Fl_Toggle_Node_Base *) 0 &&
              aNode->next_sibling_ == (Fl_Toggle_Node_Base *) 0 )
        {
        aNode = aNode->parent_node_;
        depth--;

        if (depth <= 0)
          {
          aNode = (Fl_Toggle_Node_Base *) 0;
          }
        }

      if (aNode != (Fl_Toggle_Node_Base *) 0)
        {
        aNode = aNode->next_sibling_;
        }
      }
    }

  return result;
}

// ----------------------------------------------------------------------------
inline int
Fl_Toggle_Tree_Base::open (Fl_Toggle_Node_Base* aNode)
{
  int th = this->total_height((aNode->visible_child_ = aNode->first_child_));

  aNode->opened_ = true;

  return th;
}

inline int
Fl_Toggle_Tree_Base::close (Fl_Toggle_Node_Base* aNode)
{
  int th = this->total_height(aNode->visible_child_);

  aNode->opened_        = false;
  aNode->visible_child_ = (Fl_Toggle_Node_Base *) 0;

  return th;
}

// ----------------------------------------------------------------------------
inline int
Fl_Toggle_Tree_Base::s_compare_ (void* aPtrA, void* aPtrB)
{
  Fl_Toggle_Node_Base* nodeA =
    *(reinterpret_cast<Fl_Toggle_Node_Base**>(aPtrA));
  Fl_Toggle_Node_Base* nodeB =
    *(reinterpret_cast<Fl_Toggle_Node_Base**>(aPtrB));

  return s_node_compare_(nodeA, nodeB);
}

inline int
Fl_Toggle_Tree_Base::s_compare_reverse_ (void* aPtrA, void* aPtrB)
{
  Fl_Toggle_Node_Base* nodeA =
    *(reinterpret_cast<Fl_Toggle_Node_Base**>(aPtrA));
  Fl_Toggle_Node_Base* nodeB =
    *(reinterpret_cast<Fl_Toggle_Node_Base**>(aPtrB));

  return -(s_node_compare_(nodeA, nodeB));
}

// ----------------------------------------------------------------------------
inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::sort ( Fl_Toggle_Node_Base* aStartNode,
                            int (*aCompare)( Fl_Toggle_Node_Base*,
                                             Fl_Toggle_Node_Base* ),
                            Fl_SortOrder aOrder )
{
  return ( this->first_node_ != (Fl_Toggle_Node_Base *) 0 ?
           this->sort_(aStartNode, aCompare, 0, aOrder) :
           (Fl_Toggle_Node_Base *) 0 );
}

inline Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::sort_tree ( Fl_Toggle_Node_Base* aStartNode,
                                 int (*aCompare)( Fl_Toggle_Node_Base*,
                                                  Fl_Toggle_Node_Base* ),
                                 Fl_SortOrder aOrder )
{
  return ( this->first_node_ != (Fl_Toggle_Node_Base *) 0 ?
           this->sort_(aStartNode, aCompare, 1, aOrder) :
           (Fl_Toggle_Node_Base *) 0 );
}

inline void
Fl_Toggle_Tree_Base::sort ( int (*aCompare)( Fl_Toggle_Node_Base*,
                                             Fl_Toggle_Node_Base* ),
                            Fl_SortOrder aOrder )
{
  if (this->first_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    this->first_node_ = this->top_node_ =
      this->sort(this->first_node_, aCompare, aOrder);
    }
}

inline void
Fl_Toggle_Tree_Base::sort_tree ( int (*aCompare)( Fl_Toggle_Node_Base*,
                                                  Fl_Toggle_Node_Base* ),
                                 Fl_SortOrder aOrder )
{
  if (this->first_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    this->first_node_ = this->top_node_ =
      this->sort_tree(this->first_node_, aCompare, aOrder);
    }
}

#endif /* FL_TOGGLE_TREE_BASE_H_ */
/* 
 * End of: $Id: Fl_Toggle_Tree_Base.H,v 1.1.1.1 2006/12/19 22:59:54 christianh Exp $.
 * 
 */
