/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Fl_Toggle_Tree_Base.cxx,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * Adapted from Flek (The Fast Light Environment Kit).
 * 
 */
// Flek-derived
#include "Fl_Toggle_Tree_Base.H"
// C++ forwarding ANSI C
#include <cstdlib> /* qsort() */
// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <FL/fl_draw.H>

static int s_icon_size = 24;

// ----------------------------------------------------------------------------
/*                        w() - d*16
 *        cy            /            \
 *        |            /              \
 *        v           /                \
 * x()--->+----------+------------------+
 *      / |          |                  |
 *    16  | FL_BLACK |     FL_WHITE     |
 *      \ |          |                  |
 *        +----------+------------------+
 *         \        /
 *          \      /
 *            d*16
 */
void
Fl_Toggle_Tree_Base::draw_node (int aDepth, int aY, Fl_Toggle_Node_Base*)
{
  fl_color(FL_BLACK);
  fl_rectf( this->x(),                      aY,
            aDepth*s_icon_size,             s_icon_size);
  fl_color(FL_WHITE);
  fl_rectf( this->x() + aDepth*s_icon_size, aY,
            this->w() - aDepth*s_icon_size, s_icon_size );
}

// ----------------------------------------------------------------------------
Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::find (int aFindY, int& aDepth, int& aReturnY)
{
  Fl_Group* parent;
  int   begY, endY;

  if ((parent = this->parent()) != (Fl_Group *) 0)
    {
    begY = parent->y() + this->top_yoffset_;
    endY = parent->y() + parent->h();
    }
  else
    {
    begY = this->y() + this->top_yoffset_;
    endY = this->y() + this->h();
    }

  if (aFindY < begY)
    {
    return (Fl_Toggle_Node_Base *) 0;
    }

  Fl_Toggle_Node_Base* node = this->top_node_;

  aDepth = this->top_depth_;

  this->traverse_start(this->top_node_);

  while (begY < endY && node != (Fl_Toggle_Node_Base *) 0)
    {
    aReturnY  = begY;
    begY += this->node_height(node);

    if (begY > aFindY)
      {
      return node;
      }

    node = this->traverse_forward(true, aDepth);
    }

  return (Fl_Toggle_Node_Base *) 0;
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree_Base::update_height (void)
{
  this->resize( this->x(),
                this->y(),
                this->w(),
                this->total_height(this->first_node_) );
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree_Base::draw (void)
{
  this->update_top();

  Fl_Group* parent;
  int   begY, endY;

  if ((parent = this->parent()) != (Fl_Group *) 0)
    {
    begY = parent->y() + this->top_yoffset_;
    endY = parent->y() + parent->h();
    }
  else
    {
    begY = this->y() + this->top_yoffset_;
    endY = this->y() + this->h();
    }

  int   depth   = this->top_depth_;
  bool  drawing = false;

  if (this->damage() == FL_DAMAGE_ALL)
    {
    drawing = true;
    }

  if ( this->damage()      == FL_DAMAGE_CHILD &&
       this->damaged_node_ == (Fl_Toggle_Node_Base *) 0 )
    {
    drawing = true;
    }

  Fl_Toggle_Node_Base* node = this->top_node_;

  while (begY < endY && node != (Fl_Toggle_Node_Base *) 0)
    {
    if (this->damaged_node_ == node)
      {
      if (this->damage() == FL_DAMAGE_CHILD)
        {
        this->draw_node(depth, begY, node);
        this->damaged_node_ = (Fl_Toggle_Node_Base *) 0; // Added 20030815
        return;
        }
      drawing = true;
      }

    if (drawing)
      {
      this->draw_node(depth, begY, node);
      }

    begY += this->node_height(node);

    if      (node->visible_child_ != (Fl_Toggle_Node_Base *) 0)
      {
      node = node->visible_child_;
      depth++;
      }
    else if (node->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      node = node->next_sibling_;
      }
    else
      {
      while ( node                != (Fl_Toggle_Node_Base *) 0 &&
              node->next_sibling_ == (Fl_Toggle_Node_Base *) 0 )
        {
        node = node->parent_node_;
        depth--;
        }

      if (node != (Fl_Toggle_Node_Base *) 0)
        {
        node = node->next_sibling_;
        }
      }
    }

  fl_color(parent == (Fl_Group *) 0 ? this->color() : parent->color());
  fl_rectf(this->x(), begY, this->w(), endY - begY);
}

// ----------------------------------------------------------------------------
// The static comparison functions.
// ----------------------------------------------------------------------------

// function pointer
int (*s_node_compare_)(Fl_Toggle_Node_Base*, Fl_Toggle_Node_Base*) = 0;

// ----------------------------------------------------------------------------
Fl_Toggle_Node_Base*
Fl_Toggle_Tree_Base::sort_ (Fl_Toggle_Node_Base* aStartNode,
                            int (*aCompare)(Fl_Toggle_Node_Base*,
                                          Fl_Toggle_Node_Base*),
                            int aDownFlag,
                            Fl_SortOrder aOrder)
{
  int i;
  Fl_Toggle_Node_Base* node;

  i = 0;
  node = aStartNode;
    
  while (node != (Fl_Toggle_Node_Base *) 0)
    {
    node = node->next_sibling_;
    i++;
    }

  Fl_Toggle_Node_Base** nodeArray = (Fl_Toggle_Node_Base **) 0;

  try
    {
    nodeArray = new Fl_Toggle_Node_Base * [i];

    i = 0;
    node = aStartNode;

    while (node != (Fl_Toggle_Node_Base *) 0)
      {
      nodeArray[i] = node;
      node         = node->next_sibling_;
      i++;
      }

    s_node_compare_ = aCompare;

    if (aOrder == FL_REVERSE_SORT)
      {
      qsort( nodeArray, i, sizeof(Fl_Toggle_Node_Base*),
             (int (*)(const void*, const void*))
             Fl_Toggle_Tree_Base::s_compare_reverse_ );
      }
    else
      {
      qsort( nodeArray, i, sizeof(Fl_Toggle_Node_Base*),
             (int (*)(const void*, const void*))
             Fl_Toggle_Tree_Base::s_compare_ );
      }

    aStartNode = nodeArray[0];

    node                    = aStartNode;
    node->previous_sibling_ = (Fl_Toggle_Node_Base *) 0;

    int j = 1;

    while (j < i)
      {
      node->next_sibling_                    = nodeArray[j];
      node->next_sibling_->previous_sibling_ = node;
      node                                   = node->next_sibling_;
      j++;
      }

    node->next_sibling_ = (Fl_Toggle_Node_Base *) 0;

    if (aDownFlag)
      {
      node = aStartNode;

      while (node != (Fl_Toggle_Node_Base *) 0)
        {
        if (node->first_child_ != (Fl_Toggle_Node_Base *) 0)
          {
          node->first_child_ =
            this->sort_tree(node->first_child_, aCompare, aOrder);
          }

        if (node->visible_child_ != (Fl_Toggle_Node_Base *) 0)
          {
          node->visible_child_ = node->first_child_;
          }

        node = node->next_sibling_;
        }
      }

    delete [] nodeArray;
    }
  catch (...)
    {
    delete [] nodeArray;
    Fl::error("Fl_Toggle_Tree_Base::sort_ failed to create nodeArray.");
    }

  return aStartNode;
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree_Base::update_top (void)
{
  Fl_Toggle_Node_Base* node = this->first_node_;

  int   py      = (this->parent() == (Fl_Group *) 0 ? 0 : this->parent()->y());
  int   ly      = this->y();
  int   h       = 0;
  int   depth   = 0;

  while ( node != (Fl_Toggle_Node_Base *) 0 &&
          (ly + (h = this->node_height(node))) <= py )
    {
    ly += h;
    if      (node->visible_child_ != (Fl_Toggle_Node_Base *) 0)
      {
      node = node->visible_child_;
      depth++;
      }
    else if (node->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      node = node->next_sibling_;
      }
    else
      {
      while ( node                != (Fl_Toggle_Node_Base *) 0 &&
              node->next_sibling_ == (Fl_Toggle_Node_Base *) 0 )
        {
        node = node->parent_node_;
        depth--;
        }

      if (node != (Fl_Toggle_Node_Base *) 0)
        {
        node = node->next_sibling_;
        }
      }
    }

  this->top_node_    = node;
  this->top_depth_   = depth;
  this->top_yoffset_ = ly - py;
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree_Base::node_height (Fl_Toggle_Node_Base*)
{
  return 17;
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree_Base::add_next_sibling (Fl_Toggle_Node_Base* aNode)
{
  if (this->first_node_ == (Fl_Toggle_Node_Base *) 0)
    {
    this->first_node_   = aNode;
    this->current_node_ = aNode;
    }
  else
    {
    if (this->current_node_ == (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = this->first_node_;
      }

    aNode->next_sibling_ = this->current_node_->next_sibling_;

    if (this->current_node_->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_->next_sibling_->previous_sibling_ = aNode;
      }

    this->current_node_->next_sibling_ = aNode;
    aNode->previous_sibling_           = this->current_node_;
    aNode->parent_node_                     = this->current_node_->parent_node_;
    this->current_node_                = aNode;
    }

  this->update_height();
  if (this->parent() != (Fl_Group *) 0) this->parent()->damage(FL_DAMAGE_CHILD);
  this->redraw();
}

void
Fl_Toggle_Tree_Base::add_next_child (Fl_Toggle_Node_Base* aNode)
{
  if (this->first_node_ == (Fl_Toggle_Node_Base *) 0)
    {
    this->first_node_   = aNode;
    this->current_node_ = aNode;
    }
  else
    {
    if (this->current_node_ == (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = this->first_node_;
      }

    aNode->next_sibling_ = this->current_node_->first_child_;

    if (this->current_node_->first_child_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_->first_child_->previous_sibling_ = aNode;
      }

    aNode->previous_sibling_          = (Fl_Toggle_Node_Base *) 0;
    this->current_node_->first_child_ = aNode;

    if (this->current_node_->opened_)
      {
      this->current_node_->visible_child_ = aNode;
      }

    aNode->parent_node_      = this->current_node_;
    this->current_node_ = aNode;
    }

  this->update_height();
  if (this->parent() != (Fl_Group *) 0) this->parent()->damage(FL_DAMAGE_CHILD);
  this->redraw();
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree_Base::remove (Fl_Toggle_Node_Base* aNode)
{
  Fl_Toggle_Node_Base* tmpNode = aNode->first_child_;

  while (tmpNode != (Fl_Toggle_Node_Base *) 0)
    { // Remove all children
    this->remove(tmpNode);
    tmpNode = aNode->first_child_;
    }

  if (aNode->previous_sibling_ != (Fl_Toggle_Node_Base *) 0)
    {
    aNode->previous_sibling_->next_sibling_ = aNode->next_sibling_;

    if (aNode->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      aNode->next_sibling_->previous_sibling_ = aNode->previous_sibling_;
      }
    }
  else if (aNode->parent_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    Fl_Toggle_Node_Base* parentNode = aNode->parent_node_;

    parentNode->first_child_ = aNode->next_sibling_;

    if (parentNode->opened_ == true)
      {
      parentNode->visible_child_ = aNode->next_sibling_;
      }
    else
      {
      parentNode->visible_child_ = (Fl_Toggle_Node_Base *) 0;
      }

    if (aNode->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      aNode->next_sibling_->parent_node_          = parentNode;
      aNode->next_sibling_->previous_sibling_ = aNode->previous_sibling_;
      }
    }

  if (aNode == this->first_node_)
    {
    if (aNode->next_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->first_node_                  = aNode->next_sibling_;
      this->first_node_->parent_node_          = (Fl_Toggle_Node_Base *) 0;
      this->first_node_->previous_sibling_ = (Fl_Toggle_Node_Base *) 0;
      }
    else
      {
      this->first_node_ = (Fl_Toggle_Node_Base *) 0;
      this->top_node_   = (Fl_Toggle_Node_Base *) 0;
      }
    }

  if (aNode == this->selected_node_)
    {
    if      (aNode->parent_node_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->selected_node_ = aNode->parent_node_;
      }
    else if (aNode->previous_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->selected_node_ = aNode->previous_sibling_;
      }
    else
      {
      this->selected_node_ = (Fl_Toggle_Node_Base *) 0;
      }
    }

  if (aNode == this->current_node_)
    {
    if (aNode->parent_node_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = aNode->parent_node_;
      }
    else if (aNode->previous_sibling_ != (Fl_Toggle_Node_Base *) 0)
      {
      this->current_node_ = aNode->previous_sibling_;
      }
    else
      {
      this->current_node_ = (Fl_Toggle_Node_Base *) 0;
      }
    }

  if (this->damaged_node_ == aNode)
    {
    this->damaged_node_ = (Fl_Toggle_Node_Base *) 0;
    }

  this->update_height();
  if (this->parent() != (Fl_Group *) 0) this->parent()->damage(FL_DAMAGE_CHILD);
  this->redraw();

  delete aNode;
  aNode = (Fl_Toggle_Node_Base *) 0;

  return 1;
}

/* 
 * End of: $Id: Fl_Toggle_Tree_Base.cxx,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $.
 * 
 */
