/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *
 * $Id: LightUI.cxx,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "LightUI.H"
// C++ forwarding ANSI C
#include <cstdio>
#include <cstring>
// VTK Rendering
#include "vtkLight.h"
#include "vtkLightCollection.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
// vtkExtensions
#include "vtkRenderingContext.h"


VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);


// ----------------------------------------------------------------------------
Fl_Menu_Item LightUI::mLightTypeMenuItems[] =
{
  { "headlight",    0,  0, 0,  8, 0, 0, 12, 56 },
  { "camera light", 0,  0, 0,  8, 0, 0, 12, 56 },
  { "scene light",  0,  0, 0, 12, 0, 0, 12, 56 },
  { 0 }
};


// ----------------------------------------------------------------------------
void
LightUI::LightToggleCb (Fl_Button* aButton, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aButton == (Fl_Button *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  light->SetSwitch(int(aButton->value()));

  lightUI->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
LightUI::LightTypeCb (Fl_Menu_Button* aMenu, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aMenu == (Fl_Menu_Button *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  switch (aMenu->value())
    {
    case 0: light->SetLightTypeToHeadlight();   break;
    case 1: light->SetLightTypeToCameraLight(); break;
    case 2: light->SetLightTypeToSceneLight();  break;
    }

  lightUI->UpdateClientRenderingContext();
  lightUI->Receive();
}

// ----------------------------------------------------------------------------
void
LightUI::PositionalToggleCb (Fl_Button* aButton, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aButton == (Fl_Button *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  light->SetPositional(int(aButton->value()));

  lightUI->UpdateClientRenderingContext();
  lightUI->Receive();
}

// ----------------------------------------------------------------------------
void
LightUI::FollowCameraToggleCb (Fl_Button* aButton, void* aPtr)
{
  LightUI*             lightUI;
  vtkRenderer*         renderer;

  if (aButton == (Fl_Button *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((renderer = lightUI->GetCurrentClientRenderer()) == (vtkRenderer* ) 0)
    return;

  renderer->SetLightFollowCamera(int(aButton->value()));

  lightUI->UpdateClientRenderingContext();
  lightUI->Receive();
}

// ----------------------------------------------------------------------------
void
LightUI::IntensityCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  light->SetIntensity(float(aInput->value()));

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::RedCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float rgb[3];

  light->GetColor(rgb);
  rgb[0] = float(aInput->value());
  light->SetColor(rgb);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::GreenCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float rgb[3];

  light->GetColor(rgb);
  rgb[1] = float(aInput->value());
  light->SetColor(rgb);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::BlueCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float rgb[3];

  light->GetColor(rgb);
  rgb[2] = float(aInput->value());
  light->SetColor(rgb);

  lightUI->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
LightUI::FocalPointXCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float focus[3];

  light->GetFocalPoint(focus);
  focus[0] = float(aInput->value());
  light->SetFocalPoint(focus);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::FocalPointYCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float focus[3];

  light->GetFocalPoint(focus);
  focus[1] = float(aInput->value());
  light->SetFocalPoint(focus);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::FocalPointZCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float focus[3];

  light->GetFocalPoint(focus);
  focus[2] = float(aInput->value());
  light->SetFocalPoint(focus);

  lightUI->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
LightUI::PositionXCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float position[3];

  light->GetPosition(position);
  position[0] = float(aInput->value());
  light->SetPosition(position);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::PositionYCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float position[3];

  light->GetPosition(position);
  position[1] = float(aInput->value());
  light->SetPosition(position);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::PositionZCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float position[3];

  light->GetPosition(position);
  position[2] = float(aInput->value());
  light->SetPosition(position);

  lightUI->UpdateClientRenderingContext();
}


// ----------------------------------------------------------------------------
void
LightUI::ExponentCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  light->SetExponent(float(aInput->value()));

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::ConeAngleCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  light->SetConeAngle(float(aInput->value()));

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::ConstantAttenuationCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float attenuation[3];

  light->GetAttenuationValues(attenuation);
  attenuation[0] = float(aInput->value());
  light->SetAttenuationValues(attenuation);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::LinearAttenuationCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float attenuation[3];

  light->GetAttenuationValues(attenuation);
  attenuation[1] = float(aInput->value());
  light->SetAttenuationValues(attenuation);

  lightUI->UpdateClientRenderingContext();
}

void
LightUI::QuadraticAttenuationCb (Fl_Value_Input* aInput, void* aPtr)
{
  LightUI*  lightUI;
  vtkLight* light;

  if (aInput == (Fl_Value_Input *) 0)
    return;
  if ((lightUI = reinterpret_cast<LightUI*>(aPtr)) == (LightUI *) 0)
    return;
  if ((light = lightUI->GetLight()) == (vtkLight *) 0)
    return;

  float attenuation[3];

  light->GetAttenuationValues(attenuation);
  attenuation[2] = float(aInput->value());
  light->SetAttenuationValues(attenuation);

  lightUI->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
LightUI::Initialize (vtkRenderingContext* aContext, vtkLight* aLight)
{
  this->ClientRenderingContext = aContext;
  this->Light                  = aLight;

  vtkLightCollection* currentLights;

  if (currentLights = this->ClientRenderingContext->GetCurrentLights())
    {
    int i;

    if (i = currentLights->IsItemPresent(this->Light))
      {
      if (this->Name) delete [] this->Name;

      this->Name = new char[16];

      (void) snprintf(this->Name,16,"light%d",int(i-1));
      //cerr << "Initialize( " << aContext << ", " << aLight << " ) ..."
      //     << " setting name to \"" << this->Name << "\"." << endl;
      this->label(this->Name);
      this->mLightToggle.label(this->Name);
      }
    }
}

// ----------------------------------------------------------------------------
vtkRenderingContext*
LightUI::GetClientRenderingContext (void)
{
  return this->ClientRenderingContext;
}

vtkRenderer*
LightUI::GetCurrentClientRenderer (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentRenderer();
    }

  return (vtkRenderer *) 0;
}

vtkLightCollection*
LightUI::GetCurrentLights (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentLights();
    }

  return (vtkLightCollection *) 0;
}

vtkLight*
LightUI::GetLight (void)
{
  return this->Light;
}

// ----------------------------------------------------------------------------
void
LightUI::Send (vtkObject* vtkNotUsed(a))
{
  Fl::check();

  vtkRenderer* renderer;

  if ((renderer = this->GetCurrentClientRenderer()) != (vtkRenderer* ) 0)
    {
    renderer->SetLightFollowCamera(this->mFollowCameraToggle.value());
    }

  if (this->Light != (vtkLight *) 0)
    {
    float rgb[3], position[3], focus[3], attenuation[3];

    // Toggle on/off.
    this->Light->SetSwitch(this->mLightToggle.value());
    // Toggle directional vs. positional lighting.
    this->Light->SetPositional(this->mPositionalToggle.value());

    switch (this->mLightTypeMenu.value())
      {
      case 0:
        this->Light->SetLightType(VTK_LIGHT_TYPE_HEADLIGHT);
        break;
      case 1:
        this->Light->SetLightType(VTK_LIGHT_TYPE_CAMERA_LIGHT);
        break;
      case 2:
      default:
        this->Light->SetLightType(VTK_LIGHT_TYPE_SCENE_LIGHT);
        break;
      } // switch (type)

    // Set XYZ position.
    this->Light->SetPosition( this->mPositionX.value(),
                              this->mPositionY.value(),
                              this->mPositionZ.value() );
    // Set XYZ focal point.
    this->Light->SetFocalPoint( this->mFocalPointX.value(),
                                this->mFocalPointY.value(),
                                this->mFocalPointZ.value() );
    // Set RGB color and color intensity.
    this->Light->SetColor( this->mRed.value(),
                           this->mGreen.value(),
                           this->mBlue.value() );
    this->Light->SetIntensity(this->mIntensity.value());
    // Positional lighting parameters.
    this->Light->SetExponent(this->mExponent.value());
    this->Light->SetConeAngle(this->mConeAngle.value());
    // Set attenuation values for positional lighting.
    this->Light->SetAttenuationValues( this->mConstantAttenuation.value(),
                                       this->mLinearAttenuation.value(),
                                       this->mQuadraticAttenuation.value() );
    }
}

// ----------------------------------------------------------------------------
void
LightUI::Receive (vtkObject* vtkNotUsed(a))
{
  Fl::check();

  vtkRenderer* renderer;

  if ((renderer = this->GetCurrentClientRenderer()) != (vtkRenderer* ) 0)
    {
    this->mFollowCameraToggle.value(renderer->GetLightFollowCamera());
    }

  if (this->Light != (vtkLight *) 0)
    {
    float rgb[3], position[3], focus[3], attenuation[3];

    // Toggle on/off.
    this->mLightToggle.value(this->Light->GetSwitch());
    // Toggle directional vs. positional lighting.
    this->mPositionalToggle.value(this->Light->GetPositional());

    if (this->mPositionalToggle.value() == 0)
      {
      this->mPositionalGroup.deactivate();
      this->mPositionalToggle.label("directional");
      }
    else
      {
      this->mPositionalToggle.label("positional");
      this->mPositionalGroup.activate();
      }
    this->mPositionalToggle.redraw();

    switch (this->Light->GetLightType())
      {
      case VTK_LIGHT_TYPE_HEADLIGHT:
        this->mLightTypeMenu.value(0);
        break;
      case VTK_LIGHT_TYPE_CAMERA_LIGHT:
        this->mLightTypeMenu.value(1);
        break;
      case VTK_LIGHT_TYPE_SCENE_LIGHT:
      default:
        this->mLightTypeMenu.value(2);
        break;
      } // switch (type)

    // Label the menu button with the light type.
    this->mLightTypeMenu.label(this->mLightTypeMenu.mvalue()->label());
    this->mLightTypeMenu.redraw();

    // Set XYZ position.
    this->Light->GetPosition(position);
    this->mPositionX.value(double(position[0]));
    this->mPositionY.value(double(position[1]));
    this->mPositionZ.value(double(position[2]));
    // Set XYZ focal point.
    this->Light->GetFocalPoint(focus);
    this->mFocalPointX.value(double(focus[0]));
    this->mFocalPointY.value(double(focus[1]));
    this->mFocalPointZ.value(double(focus[2]));
    // Set RGB color and color intensity.
    this->Light->GetColor(rgb);
    this->mRed.value(double(rgb[0]));
    this->mGreen.value(double(rgb[1]));
    this->mBlue.value(double(rgb[2]));
    this->mIntensity.value(double(this->Light->GetIntensity()));
    // Positional lighting parameters.
    this->mExponent.value(double(this->Light->GetExponent()));
    this->mConeAngle.value(double(this->Light->GetConeAngle()));
    // Set attenuation values for positional lighting.
    this->Light->GetAttenuationValues(attenuation);
    this->mConstantAttenuation.value(double(attenuation[0]));
    this->mLinearAttenuation.value(double(attenuation[1]));
    this->mQuadraticAttenuation.value(double(attenuation[2]));
    }
}

// ----------------------------------------------------------------------------
void
LightUI::Default (vtkObject* aObject)
{
  this->mFollowCameraToggle.value(1);

  // Toggle on.
  this->mLightToggle.value(1);
  // Toggle directional vs. positional lighting.
  this->mPositionalToggle.value(0);
  this->mPositionalGroup.deactivate();
  this->mPositionalToggle.label("directional");
  this->mPositionalToggle.redraw();

  this->mLightTypeMenu.value(2);
  // Label the menu button with the light type.
  this->mLightTypeMenu.label(this->mLightTypeMenu.mvalue()->label());
  this->mLightTypeMenu.redraw();

  // Set XYZ position.
  this->mPositionX.value(0.0);
  this->mPositionY.value(0.0);
  this->mPositionZ.value(1.0);
  // Set XYZ focal point.
  this->mFocalPointX.value(0.0);
  this->mFocalPointY.value(0.0);
  this->mFocalPointZ.value(0.0);
  // Set RGB color and color intensity.
  this->mRed.value(1.0);
  this->mGreen.value(1.0);
  this->mBlue.value(1.0);
  this->mIntensity.value(1.0);
  // Positional lighting parameters.
  this->mExponent.value(1);
  this->mConeAngle.value(30);
  // Set attenuation values for positional lighting.
  this->mConstantAttenuation.value(1);
  this->mLinearAttenuation.value(0);
  this->mQuadraticAttenuation.value(0);

  this->Send(aObject);

  if (this->Light != (vtkLight *) 0)
    {
    this->Light->SetTransformMatrix(0);
    }

  this->Receive(aObject);
}

// ----------------------------------------------------------------------------
void
LightUI::UpdateClientRenderingContext (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    this->ClientRenderingContext->Update();
    this->Receive();
    }
}

// ----------------------------------------------------------------------------
// 324 x 160
// ----------------------------------------------------------------------------
LightUI::LightUI (int x, int y, int w, int h, const char* l)
  : Fl_Group(x, y, w, h, l),
    mLightToggle(            x+4,   y+4,   50, 20, l ),
    mLightTypeMenu(         x+58,   y+4,   96, 20, "scene light" ),
    mPositionalToggle(     x+156,   y+4,   72, 20, "directional" ),
    mFollowCameraToggle(   x+230,   y+4,   90, 20, "follow camera" ),

    mPositionGroup(        x+84,   y+30,  228, 24, "position" ),
    mPositionX(            x+100,  y+33,   60, 17, "x" ),
    mPositionY(            x+174,  y+33,   60, 17, "y" ),
    mPositionZ(            x+248,  y+33,   60, 17, "z" ),
    mPositionEnd(),

    mFocalPointGroup(      x+84,   y+56,  228, 24, "focal point" ),
    mFocalPointX(          x+100,  y+59,   60, 17, "x" ),
    mFocalPointY(          x+174,  y+59,   60, 17, "y" ),
    mFocalPointZ(          x+248,  y+59,   60, 17, "z" ),
    mFocalPointEnd(),

    mColorGroup(            x+84,  y+82,  236, 24, "color" ),
    mIntensity(             x+88,  y+85,   46, 17 ),
    mRed(                  x+156,  y+85,   46, 17, " r" ),
    mGreen(                x+214,  y+85,   46, 17, "g" ),
    mBlue(                 x+272,  y+85,   46, 17, "b" ),
    mColorEnd(),

    mPositionalGroup(        x+4,  y+108, 316, 48, "positional" ),
    mExponent(             x+154,  y+112,  46, 17, "exponent:" ),
    mConeAngle(            x+272,  y+112,  46, 17, "cone angle:" ),
    mAttenuationGroup(     x+118,  y+132, 188, 20, "attenuation = 1(" ),
    mConstantAttenuation(  x+118,  y+134,  28, 17, "+ d " ),
    mLinearAttenuation(    x+180,  y+134,  38, 17, "+ d " ),
    mQuadraticAttenuation( x+258,  y+134,  46, 17, ")" ),
    mAttenuationEnd(),
    mPositionalEnd(),

    mLightUIEnd(),
    ClientRenderingContext(0),
    Light(0),
    Name(l != (char *) 0 ? strcpy(new char [strlen(l)+1],l) : (char *) 0)
{
  this->box(FL_DOWN_BOX);
  this->labelcolor(FL_FOREGROUND_COLOR);
  this->color(FL_DARK3);
  this->align(FL_ALIGN_TOP_LEFT);
  this->labeltype(FL_NORMAL_LABEL);

  // On/Off Switch
  this->mLightToggle.type(1);
  this->mLightToggle.box(FL_THIN_UP_BOX);
  this->mLightToggle.down_box(FL_THIN_DOWN_BOX);
  this->mLightToggle.color(FL_DARK2);
  this->mLightToggle.align(FL_ALIGN_CLIP|FL_ALIGN_WRAP|FL_ALIGN_INSIDE);
  this->mLightToggle.labeltype(FL_EMBOSSED_LABEL);
  this->mLightToggle.labelfont(FL_COURIER);
  this->mLightToggle.labelsize(12);
  this->mLightToggle.labelcolor(FL_FOREGROUND_COLOR);
  this->mLightToggle.value(1);
  this->mLightToggle.callback((Fl_Callback *) LightUI::LightToggleCb, this);

  // Light Type
  this->mLightTypeMenu.down_box(FL_BORDER_FRAME);
  this->mLightTypeMenu.labelsize(12);
  this->mLightTypeMenu.labelcolor(FL_FOREGROUND_COLOR);
  this->mLightTypeMenu.textsize(12);
  this->mLightTypeMenu.callback((Fl_Callback *) LightUI::LightTypeCb, this);
  this->mLightTypeMenu.align(FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
  this->mLightTypeMenu.menu(LightUI::mLightTypeMenuItems);

  // Positional/Directional Lighting Toggle
  this->mPositionalToggle.type(1);
  this->mPositionalToggle.down_box(FL_DOWN_BOX);
  this->mPositionalToggle.labelsize(12);
  this->mPositionalToggle.labelcolor(FL_FOREGROUND_COLOR);
  this->mPositionalToggle.callback( (Fl_Callback *) LightUI::PositionalToggleCb,
                                    this );

  // Light Follow Camera Toggle
  this->mFollowCameraToggle.type(1);
  this->mFollowCameraToggle.down_box(FL_DOWN_BOX);
  this->mFollowCameraToggle.value(1);
  this->mFollowCameraToggle.labelsize(12);
  this->mFollowCameraToggle.labelcolor(FL_FOREGROUND_COLOR);
  this->mFollowCameraToggle.callback(
    (Fl_Callback *) LightUI::FollowCameraToggleCb, this );

  this->mPositionGroup.box(FL_BORDER_FRAME);
  this->mPositionGroup.color(FL_LIGHT3);
  this->mPositionGroup.labeltype(FL_SHADOW_LABEL);
  this->mPositionGroup.labelfont(FL_HELVETICA_BOLD);
  this->mPositionGroup.labelsize(14);
  this->mPositionGroup.labelcolor(FL_LIGHT2);
  this->mPositionGroup.align(FL_ALIGN_LEFT);
    {
    this->mPositionX.box(FL_FLAT_BOX);
    this->mPositionX.color(FL_DARK3);
    this->mPositionX.labelfont(FL_HELVETICA_BOLD);
    this->mPositionX.labelcolor(FL_LIGHT2);
    this->mPositionX.minimum(-10000);
    this->mPositionX.maximum(10000);
    this->mPositionX.step(0.01);
    this->mPositionX.value(0);
    this->mPositionX.textfont(FL_COURIER_BOLD);
    this->mPositionX.textsize(12);
    this->mPositionX.textcolor(FL_LIGHT3);
    this->mPositionX.callback((Fl_Callback *) LightUI::PositionXCb, this);
    this->mPositionX.when(FL_WHEN_RELEASE);

    this->mPositionY.box(FL_FLAT_BOX);
    this->mPositionY.color(FL_DARK3);
    this->mPositionY.labelfont(FL_HELVETICA_BOLD);
    this->mPositionY.labelcolor(FL_LIGHT2);
    this->mPositionY.minimum(-10000);
    this->mPositionY.maximum(10000);
    this->mPositionY.step(0.01);
    this->mPositionY.value(0);
    this->mPositionY.textfont(FL_COURIER_BOLD);
    this->mPositionY.textsize(12);
    this->mPositionY.textcolor(FL_LIGHT3);
    this->mPositionY.callback((Fl_Callback *) LightUI::PositionYCb, this);
    this->mPositionY.when(FL_WHEN_RELEASE);

    this->mPositionZ.box(FL_FLAT_BOX);
    this->mPositionZ.color(FL_DARK3);
    this->mPositionZ.labelfont(FL_HELVETICA_BOLD);
    this->mPositionZ.labelcolor(FL_LIGHT2);
    this->mPositionZ.minimum(-10000);
    this->mPositionZ.maximum(10000);
    this->mPositionZ.step(0.01);
    this->mPositionZ.value(1);
    this->mPositionZ.textfont(FL_COURIER_BOLD);
    this->mPositionZ.textsize(12);
    this->mPositionZ.textcolor(FL_LIGHT3);
    this->mPositionZ.callback((Fl_Callback *) LightUI::PositionZCb, this);
    this->mPositionZ.when(FL_WHEN_RELEASE);
    }

  this->mFocalPointGroup.box(FL_BORDER_FRAME);
  this->mFocalPointGroup.color(FL_LIGHT3);
  this->mFocalPointGroup.labeltype(FL_SHADOW_LABEL);
  this->mFocalPointGroup.labelfont(FL_HELVETICA_BOLD);
  this->mFocalPointGroup.labelsize(14);
  this->mFocalPointGroup.labelcolor(FL_LIGHT2);
  this->mFocalPointGroup.align(FL_ALIGN_LEFT);
    {
    this->mFocalPointX.box(FL_FLAT_BOX);
    this->mFocalPointX.color(FL_DARK3);
    this->mFocalPointX.labelfont(FL_HELVETICA_BOLD);
    this->mFocalPointX.labelcolor(FL_LIGHT2);
    this->mFocalPointX.minimum(-10000);
    this->mFocalPointX.maximum(10000);
    this->mFocalPointX.step(0.01);
    this->mFocalPointX.value(0);
    this->mFocalPointX.textfont(FL_COURIER_BOLD);
    this->mFocalPointX.textsize(12);
    this->mFocalPointX.textcolor(FL_LIGHT3);
    this->mFocalPointX.callback((Fl_Callback *) LightUI::FocalPointXCb, this);
    this->mFocalPointX.when(FL_WHEN_RELEASE);

    this->mFocalPointY.box(FL_FLAT_BOX);
    this->mFocalPointY.color(FL_DARK3);
    this->mFocalPointY.labelfont(FL_HELVETICA_BOLD);
    this->mFocalPointY.labelcolor(FL_LIGHT2);
    this->mFocalPointY.minimum(-10000);
    this->mFocalPointY.maximum(10000);
    this->mFocalPointY.step(0.01);
    this->mFocalPointY.value(0);
    this->mFocalPointY.textfont(FL_COURIER_BOLD);
    this->mFocalPointY.textsize(12);
    this->mFocalPointY.textcolor(FL_LIGHT3);
    this->mFocalPointY.callback((Fl_Callback *) LightUI::FocalPointYCb, this);
    this->mFocalPointY.when(FL_WHEN_RELEASE);

    this->mFocalPointZ.box(FL_FLAT_BOX);
    this->mFocalPointZ.color(FL_DARK3);
    this->mFocalPointZ.labelfont(FL_HELVETICA_BOLD);
    this->mFocalPointZ.labelcolor(FL_LIGHT2);
    this->mFocalPointZ.minimum(-10000);
    this->mFocalPointZ.maximum(10000);
    this->mFocalPointZ.step(0.01);
    this->mFocalPointZ.value(0);
    this->mFocalPointZ.textfont(FL_COURIER_BOLD);
    this->mFocalPointZ.textsize(12);
    this->mFocalPointZ.textcolor(FL_LIGHT3);
    this->mFocalPointZ.callback((Fl_Callback *) LightUI::FocalPointZCb, this);
    this->mFocalPointZ.when(FL_WHEN_RELEASE);
    }

  this->mColorGroup.box(FL_BORDER_FRAME);
  this->mColorGroup.color(FL_LIGHT3);
  this->mColorGroup.labeltype(FL_SHADOW_LABEL);
  this->mColorGroup.labelfont(FL_HELVETICA_BOLD);
  this->mColorGroup.labelsize(14);
  this->mColorGroup.labelcolor(FL_LIGHT2);
  this->mColorGroup.align(FL_ALIGN_LEFT);
    {
    this->mIntensity.box(FL_FLAT_BOX);
    this->mIntensity.color(FL_DARK3);
    this->mIntensity.labelfont(FL_HELVETICA_BOLD);
    this->mIntensity.labelcolor(FL_LIGHT2);
    this->mIntensity.minimum(0);
    this->mIntensity.maximum(1);
    this->mIntensity.step(0.001);
    this->mIntensity.value(1);
    this->mIntensity.textfont(FL_COURIER_BOLD);
    this->mIntensity.textsize(12);
    this->mIntensity.textcolor(FL_LIGHT3);
    this->mIntensity.callback((Fl_Callback *) LightUI::IntensityCb, this);
    this->mIntensity.when(FL_WHEN_RELEASE);
    
    this->mRed.box(FL_FLAT_BOX);
    this->mRed.color(FL_DARK3);
    this->mRed.labelfont(FL_HELVETICA_BOLD);
    this->mRed.labelcolor(FL_LIGHT2);
    this->mRed.minimum(0);
    this->mRed.maximum(1);
    this->mRed.step(0.001);
    this->mRed.value(1);
    this->mRed.textfont(FL_COURIER_BOLD);
    this->mRed.textsize(12);
    this->mRed.textcolor(FL_RED);
    this->mRed.callback((Fl_Callback *) LightUI::RedCb, this);
    this->mRed.when(FL_WHEN_RELEASE);
    
    this->mGreen.box(FL_FLAT_BOX);
    this->mGreen.color(FL_DARK3);
    this->mGreen.labelfont(FL_HELVETICA_BOLD);
    this->mGreen.labelcolor(FL_LIGHT2);
    this->mGreen.minimum(0);
    this->mGreen.maximum(1);
    this->mGreen.step(0.001);
    this->mGreen.value(1);
    this->mGreen.textfont(FL_COURIER_BOLD);
    this->mGreen.textsize(12);
    this->mGreen.textcolor(FL_GREEN);
    this->mGreen.callback((Fl_Callback *) LightUI::GreenCb, this);
    this->mGreen.when(FL_WHEN_RELEASE);
    
    this->mBlue.box(FL_FLAT_BOX);
    this->mBlue.color(FL_DARK3);
    this->mBlue.labelfont(FL_HELVETICA_BOLD);
    this->mBlue.labelcolor(FL_LIGHT2);
    this->mBlue.minimum(0);
    this->mBlue.maximum(1);
    this->mBlue.step(0.001);
    this->mBlue.value(1);
    this->mBlue.textfont(FL_COURIER_BOLD);
    this->mBlue.textsize(12);
    this->mBlue.textcolor(FL_BLUE);
    this->mBlue.callback((Fl_Callback *) LightUI::BlueCb, this);
    this->mBlue.when(FL_WHEN_RELEASE);
    }

  this->mPositionalGroup.box(FL_BORDER_FRAME);
  this->mPositionalGroup.color(FL_LIGHT3);
  this->mPositionalGroup.labeltype(FL_SHADOW_LABEL);
  this->mPositionalGroup.labelfont(FL_HELVETICA_BOLD);
  this->mPositionalGroup.labelsize(14);
  this->mPositionalGroup.labelcolor(FL_LIGHT2);
  this->mPositionalGroup.align(FL_ALIGN_TOP_LEFT|FL_ALIGN_INSIDE);
  this->mPositionalGroup.deactivate();
    {
    this->mExponent.box(FL_FLAT_BOX);
    this->mExponent.color(FL_DARK3);
    this->mExponent.labelfont(FL_HELVETICA_BOLD);
    this->mExponent.labelsize(12);
    this->mExponent.labelcolor(FL_LIGHT2);
    this->mExponent.minimum(0);
    this->mExponent.maximum(128);
    this->mExponent.step(0.1);
    this->mExponent.value(1);
    this->mExponent.textfont(FL_COURIER_BOLD);
    this->mExponent.textsize(12);
    this->mExponent.textcolor(FL_LIGHT3);
    this->mExponent.callback((Fl_Callback *) LightUI::ExponentCb, this);
    this->mExponent.when(FL_WHEN_RELEASE);
    
    this->mConeAngle.box(FL_FLAT_BOX);
    this->mConeAngle.color(FL_DARK3);
    this->mConeAngle.labelfont(FL_HELVETICA_BOLD);
    this->mConeAngle.labelsize(12);
    this->mConeAngle.labelcolor(FL_LIGHT2);
    this->mConeAngle.minimum(0);
    this->mConeAngle.maximum(180);
    this->mConeAngle.step(0.1);
    this->mConeAngle.value(30);
    this->mConeAngle.textfont(FL_COURIER_BOLD);
    this->mConeAngle.textsize(12);
    this->mConeAngle.textcolor(FL_LIGHT3);
    this->mConeAngle.callback((Fl_Callback *) LightUI::ConeAngleCb, this);
    this->mConeAngle.when(FL_WHEN_RELEASE);
    
    this->mAttenuationGroup.labelfont(FL_HELVETICA_BOLD);
    this->mAttenuationGroup.labelsize(12);
    this->mAttenuationGroup.labelcolor(FL_LIGHT2);
    this->mAttenuationGroup.align(FL_ALIGN_LEFT);
      {
      this->mConstantAttenuation.box(FL_FLAT_BOX);
      this->mConstantAttenuation.color(FL_DARK3);
      this->mConstantAttenuation.labelfont(FL_HELVETICA_BOLD);
      this->mConstantAttenuation.labelsize(12);
      this->mConstantAttenuation.labelcolor(FL_LIGHT2);
      this->mConstantAttenuation.minimum(0);
      this->mConstantAttenuation.maximum(100);
      this->mConstantAttenuation.step(1);
      this->mConstantAttenuation.value(1);
      this->mConstantAttenuation.textfont(FL_COURIER_BOLD);
      this->mConstantAttenuation.textsize(12);
      this->mConstantAttenuation.textcolor(FL_LIGHT3);
      this->mConstantAttenuation.align(FL_ALIGN_RIGHT);
      this->mConstantAttenuation.callback(
        (Fl_Callback *) LightUI::ConstantAttenuationCb, this );
      this->mConstantAttenuation.when(FL_WHEN_RELEASE);
      
      this->mLinearAttenuation.box(FL_FLAT_BOX);
      this->mLinearAttenuation.color(FL_DARK3);
      this->mLinearAttenuation.labelfont(FL_HELVETICA_BOLD);
      this->mLinearAttenuation.labelsize(12);
      this->mLinearAttenuation.labelcolor(FL_LIGHT2);
      this->mLinearAttenuation.minimum(0);
      this->mLinearAttenuation.maximum(1);
      this->mLinearAttenuation.step(0.001);
      this->mLinearAttenuation.value(0);
      this->mLinearAttenuation.textfont(FL_COURIER_BOLD);
      this->mLinearAttenuation.textsize(12);
      this->mLinearAttenuation.textcolor(FL_LIGHT3);
      this->mLinearAttenuation.align(FL_ALIGN_RIGHT);
      this->mLinearAttenuation.callback(
        (Fl_Callback *) LightUI::LinearAttenuationCb, this );
      this->mLinearAttenuation.when(FL_WHEN_RELEASE);
      
      this->mQuadraticAttenuation.box(FL_FLAT_BOX);
      this->mQuadraticAttenuation.color(FL_DARK3);
      this->mQuadraticAttenuation.labelfont(FL_HELVETICA_BOLD);
      this->mQuadraticAttenuation.labelsize(12);
      this->mQuadraticAttenuation.labelcolor(FL_LIGHT2);
      this->mQuadraticAttenuation.minimum(0);
      this->mQuadraticAttenuation.maximum(1);
      this->mQuadraticAttenuation.step(0.0001);
      this->mQuadraticAttenuation.value(0);
      this->mQuadraticAttenuation.textfont(FL_COURIER_BOLD);
      this->mQuadraticAttenuation.textsize(12);
      this->mQuadraticAttenuation.textcolor(FL_LIGHT3);
      this->mQuadraticAttenuation.align(FL_ALIGN_RIGHT);
      this->mQuadraticAttenuation.callback(
        (Fl_Callback *) LightUI::QuadraticAttenuationCb, this );
      this->mQuadraticAttenuation.when(FL_WHEN_RELEASE);
      }
    }
}

/* 
 * End of: $Id: LightUI.cxx,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $.
 * 
 */
