#include <FL/Fl_Window.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Scroll.H>
#include <FL/Fl.H>

#include "Fl_Toggle_Tree.H"

#include <cstdio>
#include <cstdlib>

Fl_Pixmap* groupIcon;
Fl_Pixmap* propIcon;

void
TreeCb(Fl_Widget* widget, void*)
{
  Fl_Toggle_Tree* tree = static_cast<Fl_Toggle_Tree*>(widget);

  switch (tree->state())
    {
    case FL_TOGGLE_NONE:        printf("state: FL_TOGGLE_NONE\n"); break;
    case FL_TOGGLE_SELECT:      printf("state: FL_TOGGLE_SELECT\n"); break;
    case FL_TOGGLE_RESELECT:    printf("state: FL_TOGGLE_RESELECT\n"); break;
    case FL_TOGGLE_SELECT_MASK: printf("state: FL_TOGGLE_SELECT_MASK\n"); break;
    case FL_TOGGLE_OPENED:      printf("state: FL_TOGGLE_OPENED\n"); break;
    case FL_TOGGLE_CLOSED:      printf("state: FL_TOGGLE_CLOSED\n"); break;
    case FL_TOGGLE_HIT:         printf("state: FL_TOGGLE_HIT\n"); break;
    }

  if(tree->selected())
    {
    printf("current = '%s'\n", tree->selected()->label());
    }
  else
    {
    int i;
    // if selected()==0 then we have a multiple select
    printf("MULTIPLE:\n");
    Fl_Toggle_Node* s;
    while((s = tree->selection()))
      {
      printf(" selected=%s\n", s->label());
      }

    int n = tree->selection_count();
    printf("%d selections:\n", n);
    for(i=0; i<n; i++)
      {
      s = tree->selection(i);
      printf(" %d %s\n", i, s->label());
      }
    for(i=(n-1); i>=0; i--)
      {
      s = tree->selection(i);
      printf(" %d %s\n", i, s->label());
      }
    }
}

void
RemoveCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  if(tree->selected())
    {
    tree->remove(tree->selected());
    }
  else
    {
    Fl_Toggle_Node* s;
    while((s = tree->selection()))
      {
      tree->remove(s);
      }
    }
}

void
DrawLinesCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  tree->draw_lines(!tree->draw_lines());
  tree->redraw();
}

void
MacColorsCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  if(tree->alternate_color() == tree->color())
    {
    tree->alternate_color(FL_LIGHT2);
    tree->trim_color(FL_LIGHT1);
    }
  else
    {
    tree->alternate_color(tree->color());
    tree->trim_color(tree->color());
    }
  tree->redraw();
}

void
AddGroupCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  Fl_Toggle_Node* sel = tree->selected();
  if(sel == 0)
    {
    tree->add_next_sibling("Untitled", 1, groupIcon);
    return;
    }

  tree->traverse_start(sel);
  if (sel->is_open() && sel->can_open())
    {
    tree->add_next_child("Untitled", 1, groupIcon);
    }
  else
    {
    tree->add_next_sibling("Untitled", 1, groupIcon);
    }
}

void
AddPropCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  Fl_Toggle_Node* sel = tree->selected();
  if(sel == 0)
    {
    tree->add_next_sibling("Untitled", 0, propIcon);
    return;
    }

  tree->traverse_start(sel);
  if(sel->is_open() && sel->can_open())
    {
    tree->add_next_child("Untitled", 0, propIcon);
    }
  else
    {
    tree->add_next_sibling("Untitled", 0, propIcon);
    }
}

int
RandomCb(Fl_Toggle_Node_Base*, Fl_Toggle_Node_Base*)
{
  return (random()&1)*2 - 1; // -1 or 1
}

void
SortCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  tree->sort_tree(Fl_Toggle_Tree::sort_by_label);

  tree->redraw();
}

void
SortReverseCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  tree->sort_tree(Fl_Toggle_Tree::sort_by_label, REVERSE_SORT);

  tree->redraw();
}

void
RandomSortCb(Fl_Widget*, void* ptr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(ptr);

  tree->sort_tree(RandomCb);

  tree->redraw();
}


// Add some nodes with icons -- some open, some closed.
void
make_test_tree(Fl_Toggle_Tree& toggleTree);


int
main(int argc, char* argv[])
{
  Fl_Window mainWindow(240, 304, "Tree Example");
  Fl_Button removeButton(10, 200, 100, 22, "Remove");
  Fl_Button addPropButton(10, 224, 100, 22, "Add Paper");
  Fl_Button addGroupButton(10, 248, 100, 22, "Add Folder");
  Fl_Button drawLinesButton(10, 272, 100, 22, "Draw Lines");
  Fl_Button sortButton(130, 200, 100, 22, "Sort");
  Fl_Button sortRevrseButton(130, 224, 100, 22, "Reverse Sort");
  Fl_Button sortRandomButton(130, 248, 100, 22, "Randomize");
  Fl_Button macColorsButton(130, 272, 100, 22, "Mac Colors");
  Fl_Scroll scroll(10, 10, 220, 180);
  scroll.type(Fl_Scroll::VERTICAL_ALWAYS);

  // When constructing an Fl_Toggle_Tree, the height doesn't matter.

  Fl_Toggle_Tree toggleTree(12, 12, 220-scroll.scrollbar.w(), 10);

  scroll.end();

  // Add callbacks to the widgets
  removeButton.callback(RemoveCb, &toggleTree);
  addPropButton.callback(AddPropCb, &toggleTree);
  addGroupButton.callback(AddGroupCb, &toggleTree);
  sortButton.callback(SortCb, &toggleTree);
  sortRandomButton.callback(RandomSortCb, &toggleTree);
  sortRevrseButton.callback(SortReverseCb, &toggleTree);
  drawLinesButton.callback(DrawLinesCb, &toggleTree);
  macColorsButton.callback(MacColorsCb, &toggleTree);
  toggleTree.callback(TreeCb);

  int w[3] = {150, 200, 0};
  toggleTree.column_widths(w);

  // Make the icons
  groupIcon = new Fl_Pixmap(tt_group_xpm);
  propIcon  = new Fl_Pixmap(tt_prop_xpm);

  // Add some nodes with icons -- some open, some closed.
  make_test_tree(toggleTree);

  toggleTree.update_height();

  mainWindow.show();

  return Fl::run();
}

void
make_test_tree(Fl_Toggle_Tree& toggleTree)
{
  Fl_Toggle_Node* node;

  // Add some nodes with icons -- some open, some closed.
  // Fl_Toggle_Node::Fl_Toggle_Node( LABEL , CAN_OPEN (default=1) , ICON )

  toggleTree.add_next_sibling("aaa\t123", 1, groupIcon);

  toggleTree.add_next_child("bbb TWO\t456", 1, groupIcon);
  toggleTree.traverse_up();

  node = toggleTree.add_next_sibling("bbb", 1, groupIcon);
  toggleTree.close(node);

  toggleTree.add_next_child("ccc\t789", 1, groupIcon);
  toggleTree.add_next_child("ddd\t012", 0, propIcon);
  toggleTree.traverse_up();
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("eee", 1, groupIcon);
  toggleTree.add_next_child("fff", 0, propIcon);
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("ggg", 1, groupIcon);
  node = toggleTree.add_next_child("hhh", 1, propIcon);
  toggleTree.close(node);
  toggleTree.add_next_child("iii", 1, propIcon);
  toggleTree.traverse_up();
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("jjj", 1, groupIcon);
  toggleTree.add_next_child("kkk", 0, propIcon);
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("lll", 0, propIcon);
  node = toggleTree.add_next_sibling("mmm", 1, groupIcon);
  toggleTree.close(node);
  toggleTree.add_next_child("nnn", 1, groupIcon);
  toggleTree.add_next_child("ooo", 0, propIcon);
  toggleTree.traverse_up();
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("ppp", 1, groupIcon);
  toggleTree.add_next_child("qqq", 0, propIcon);
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("rrr", 1, groupIcon);
  toggleTree.add_next_child("sss", 1, groupIcon);
  toggleTree.add_next_child("ttt", 1, groupIcon);
  toggleTree.traverse_up();
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("uuu", 1, groupIcon);
  toggleTree.add_next_child("vvv", 0, propIcon);
  toggleTree.traverse_up();

  toggleTree.add_next_sibling("www", 0, propIcon);
  toggleTree.add_next_sibling("xxx", 0, propIcon);
  toggleTree.add_next_sibling("yyy", 0, propIcon);
  toggleTree.add_next_child("zzz", 0, propIcon);

  // Examples of removing items (successfully, and unsuccessfully)
  // by label name:

  if(toggleTree.remove("xxx"))
    {
    printf("Successfully deleted \"xxx\"\n");
    }
  else
    {
    printf("Could not delete \"xxx\"\n");
    }

  if(toggleTree.remove("nonexistant"))
    {
    printf("Successfully deleted \"nonexistant\"\n");
    }
  else
    {
    printf("Could not delete \"nonexistant\"\n");
    }
}
