/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMessagesRedirection.cxx,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvMessagesRedirection.h"
#include "svvMessagesUI.h"
// VTK Common
#include "vtkObjectFactory.h"

namespace vtkstd
{
/** \class   svvMessagesStreamBuf
 *  \brief   Ties the iostream to the widget.
 * 
 * svvMessagesStreamBuf is derived derived from \c streambuf.
 * Needed to tie the iostream to the widget.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:44 $
 * 
 * \sa streambuf
 */
class svvMessagesStreamBuf : public streambuf
{
  // could this be determined via CMake?
#if !defined(__GNUC__) || __GNUC__ < 3
  typedef char          char_type;
  typedef int           int_type;
  typedef streampos     pos_type;
  typedef streamoff     off_type;
#endif /* __GNUC__ */

private:
  svvMessagesBrowser* browser_;
  char_type*          buffer_[1024];

  /** Appends the streambuf buffer to the text Widget.
   *
   * This is done by first retrieving a string from the put area
   * (the character sequence in the put area is not necessarily
   * nul-terminated), then appending this string to the widget and
   * finally resetting the pointers to the put area.
   *
   * The number of characters currently found in the put area is is
   * calculated as the difference between the next put position
   * (returned by \c streambuf::pptr()) and the beginning of the put
   * area (\c streambuf::pbase()).
   *
   * The details how to append the string to the Widget's text are
   * obvious. To reset the pointers into the put area (mainly the
   * pointer to the put position) the values returned by \c pbase()
   * and \c epptr() are used.
   */
  void	put_buffer (void)
    {
      if (this->pbase() != this->pptr())
        {
        char*      buffer;
        streamsize size = (this->pptr() - this->pbase());
        
        try
          {
          buffer = new char [size + 1];
          }
        catch (...)
          {
          delete [] buffer;
          return;
          }

        strncpy(buffer, this->pbase(), size);
        buffer[size] = '\0';

        this->browser_->PutSn( static_cast<const char *>(buffer),
                               static_cast<int>(size) );

        this->setp(this->pbase(), this->epptr());
        
        delete [] buffer;
        }
    }

  void	put_char (int_type c) { this->browser_->PutC(static_cast<int>(c)); }

protected:
  /** \name Base class overrides */
  /*@{*/
  /** Called whenever the put area of a streambuf is full.
   *
   * The argument to is the character which didn't fit into the
   * put area or EOF, if there is no such character.
   *
   * Implemented by calling \c put_buffer() which writes the current
   * contents of the put area to the widget and then putting the
   * single missing character. The argument to overflow is put into
   * the put area if the \c svvMessagesStreamBuf is operating buffered:
   * \c streambuf::sputc() puts a single character into the put area.
   * Otherwise the character is directly put into the Widget with
   * \c put_char(). To check whether the output is buffered or
   * unbuffered, the length of the put area is used:
   * \c streambuf::pbase() points to the start of the put area and
   * \c streambuf::epptr() points to the end of the put area.
   *
   * \param c The char which caused the overflow.
   */
  int_type      overflow (int_type c = static_cast<int_type>(EOF))
    {
      this->put_buffer();
      
      if (c != static_cast<int_type>(EOF))
        {
        if (this->pbase() == this->epptr()) 
          this->put_char(c);            // buffered Output to widget
        else
          this->sputc(char_type(c));    // unbuffered Output
        }

      return 0;
    }

  int_type      underflow (void) { return 0; }

  /**
   * Called to bring the internal representation of a streambuf into
   * synchronization with its external representation (remember that a
   * \c streambuf represents a sequence of characters).
   *
   * Since only writing to a \c svvMessagesStreamBuf is supported this means
   * that the internal buffer has to be put into the widget. This is
   * done by calling \c put_buffer().
   *
   * \return 0
   */
  int           sync (void) { this->put_buffer(); return 0; }

  /** Multiple character insertion.
   *
   * \param s A buffer read area.
   * \param n Maximum number of characters to write.
   *
   * \return The number of characters written.
   */
  streamsize    sputn (const char_type* s, streamsize n)
    {
      this->browser_->PutSn(static_cast<const char *>(s), static_cast<int>(n));
      return 0;
    }
  /*@}*/

public:
  svvMessagesStreamBuf (svvMessagesBrowser*  browser = NULL)
    : streambuf(),
      browser_(browser)
    {
      this->setp( (char_type *) this->buffer_,
                  (char_type *) this->buffer_ + sizeof(this->buffer_) );
      this->setg(0, 0, 0);
    }

  virtual ~svvMessagesStreamBuf() { this->sync(); }
};

/** \class   svvMessagesOutputStream
 *  \brief   Output-to-widget stream.
 * 
 * svvMessagesOutputStream is derived from \c ostream. Output-to-widget stream.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:44 $
 * 
 * \sa ostream svvMessagesStreamBuf
 */
class svvMessagesOutputStream : public ostream, public svvMessagesStreamBuf
{
public:
  svvMessagesOutputStream (svvMessagesBrowser* browser)
    : ostream(this),
      svvMessagesStreamBuf(browser)
    {}
};
} // namespace vtkstd

// ----------------------------------------------------------------------------
//     s v v M e s s a g e s R e d i r e c t i o n
// ----------------------------------------------------------------------------

vtkCxxRevisionMacro(svvMessagesRedirection, "$Revision: 1.1.1.1 $");
// Needed when we don't use the vtkStandardNewMacro.
vtkInstantiatorNewMacro(svvMessagesRedirection);

// All are default initialized and explicitly initialized in ClassInitialize().
vtkOutputWindow*                        svvMessagesRedirection::Default;
vtkstd::streambuf*                      svvMessagesRedirection::OutBuffer;
vtkstd::streambuf*                      svvMessagesRedirection::ErrBuffer;
vtkstd::streambuf*                      svvMessagesRedirection::LogBuffer;
vtkstd::svvMessagesOutputStream*        svvMessagesRedirection::OutStream;
vtkstd::svvMessagesOutputStream*        svvMessagesRedirection::ErrStream;
vtkstd::svvMessagesOutputStream*        svvMessagesRedirection::LogStream;

long    svvMessagesRedirection::Init::TranslationUnits;

// ----------------------------------------------------------------------------
svvMessagesRedirection::Init::Init (void)
{ if (TranslationUnits++ == 0) svvMessagesRedirection::ClassInitialize(); }

svvMessagesRedirection::Init::~Init()
{ if (--TranslationUnits == 0) svvMessagesRedirection::ClassFinalize(); }

void
svvMessagesRedirection::ClassInitialize (void)
{
  svvMessagesRedirection::Default   = NULL;

  svvMessagesRedirection::OutBuffer = NULL;
  svvMessagesRedirection::ErrBuffer = NULL;
  svvMessagesRedirection::LogBuffer = NULL;

  svvMessagesRedirection::OutStream = NULL;
  svvMessagesRedirection::ErrStream = NULL;
  svvMessagesRedirection::LogStream = NULL;
}

void
svvMessagesRedirection::ClassFinalize (void)
{
  if (svvMessagesRedirection::Default != NULL)
    {
    svvMessagesRedirection::Default->Delete();
    svvMessagesRedirection::Default = NULL;
    }

  if (svvMessagesRedirection::OutBuffer != NULL)
    {
    cout.rdbuf(svvMessagesRedirection::OutBuffer);
    svvMessagesRedirection::OutBuffer = NULL;
    }
  if (svvMessagesRedirection::ErrBuffer != NULL)
    {
    cerr.rdbuf(svvMessagesRedirection::ErrBuffer);
    svvMessagesRedirection::ErrBuffer = NULL;
    }
  if (svvMessagesRedirection::LogBuffer != NULL)
    {
    vtkstd::clog.rdbuf(svvMessagesRedirection::LogBuffer);
    svvMessagesRedirection::LogBuffer = NULL;
    }

  if (svvMessagesRedirection::OutStream != NULL)
    {
    delete svvMessagesRedirection::OutStream;
    svvMessagesRedirection::OutStream = NULL;
    }
  if (svvMessagesRedirection::ErrStream != NULL)
    {
    delete svvMessagesRedirection::ErrStream;
    svvMessagesRedirection::ErrStream = NULL;
    }
  if (svvMessagesRedirection::LogStream != NULL)
    {
    delete svvMessagesRedirection::LogStream;
    svvMessagesRedirection::LogStream = NULL;
    }
}

// ----------------------------------------------------------------------------
void
svvMessagesRedirection::RedirectOutStream (void)
{
  if (svvMessagesRedirection::OutBuffer == NULL)
    {
    svvMessagesRedirection::OutBuffer = cout.rdbuf();
    if (svvMessagesRedirection::OutStream == NULL)
      {
      svvMessagesRedirection::OutStream =
        new vtkstd::svvMessagesOutputStream(&(SvvMsg.browser()));
      }
    cout.rdbuf(svvMessagesRedirection::OutStream->rdbuf());
    }
}

void
svvMessagesRedirection::RestoreOutStream (void)
{
  if (svvMessagesRedirection::OutBuffer != NULL)
    {
    cout.rdbuf(svvMessagesRedirection::OutBuffer);
    svvMessagesRedirection::OutBuffer = NULL;
    }
}

void
svvMessagesRedirection::RedirectErrStream (void)
{
  if (svvMessagesRedirection::ErrBuffer == NULL)
    {
    svvMessagesRedirection::ErrBuffer = cerr.rdbuf();
    if (svvMessagesRedirection::ErrStream == NULL)
      {
      svvMessagesRedirection::ErrStream =
        new vtkstd::svvMessagesOutputStream(&(SvvMsg.browser()));
      }
    cerr.rdbuf(svvMessagesRedirection::ErrStream->rdbuf());
    }
}

void
svvMessagesRedirection::RestoreErrStream (void)
{
  if (svvMessagesRedirection::ErrBuffer != NULL)
    {
    cerr.rdbuf(svvMessagesRedirection::ErrBuffer);
    svvMessagesRedirection::ErrBuffer = NULL;
    }
}

void
svvMessagesRedirection::RedirectLogStream (void)
{
  if (svvMessagesRedirection::LogBuffer == NULL)
    {
    svvMessagesRedirection::LogBuffer = vtkstd::clog.rdbuf();
    if (svvMessagesRedirection::LogStream == NULL)
      {
      svvMessagesRedirection::LogStream =
        new vtkstd::svvMessagesOutputStream(&(SvvMsg.browser()));
      }
    vtkstd::clog.rdbuf(svvMessagesRedirection::LogStream->rdbuf());
    }
}

void
svvMessagesRedirection::RestoreLogStream (void)
{
  if (svvMessagesRedirection::LogBuffer != NULL)
    {
    vtkstd::clog.rdbuf(svvMessagesRedirection::LogBuffer);
    svvMessagesRedirection::LogBuffer = NULL;
    }
}

// ----------------------------------------------------------------------------
svvMessagesRedirection::svvMessagesRedirection (void)
{
  vtkObject* ret = vtkObjectFactory::CreateInstance("svvMessagesRedirection");
  if (ret) ret->Delete();
}

// ----------------------------------------------------------------------------
void
svvMessagesRedirection::RedirectVtkStreams (void)
{
  if (svvMessagesRedirection::GetInstance()->IsA("svvMessagesRedirection"))
    {
    return;
    }

  if ( svvMessagesRedirection::Default != NULL &&
       svvMessagesRedirection::Default != svvMessagesRedirection::GetInstance())
    {
    svvMessagesRedirection::Default->Delete();
    svvMessagesRedirection::Default = NULL;
    }

  if (svvMessagesRedirection::Default == NULL)
    {
    svvMessagesRedirection::Default = svvMessagesRedirection::GetInstance();
    svvMessagesRedirection::Default->Register(NULL);
    }

  svvMessagesRedirection::SetInstance(new svvMessagesRedirection);
}

void
svvMessagesRedirection::RestoreVtkStreams (void)
{
  if (svvMessagesRedirection::GetInstance()->IsA("svvMessagesRedirection"))
    {
    svvMessagesRedirection::SetInstance(svvMessagesRedirection::Default);
    }
}

// ----------------------------------------------------------------------------
void
svvMessagesRedirection::DisplayText (const char* aMessage)
{
  if(aMessage == NULL || *aMessage == '\0') { return; }
  svvMessage(<< aMessage);
}

void
svvMessagesRedirection::DisplayErrorText (const char* aMessage)
{
  if(aMessage == NULL || *aMessage == '\0') { return; }
  svvError(<< aMessage);
}

void
svvMessagesRedirection::DisplayWarningText (const char* aMessage)
{
  if(aMessage == NULL || *aMessage == '\0') { return; }
  svvWarning(<< aMessage);
}

void
svvMessagesRedirection::DisplayGenericWarningText (const char* aMessage)
{
  if(aMessage == NULL || *aMessage == '\0') { return; }
  svvWarning(<< aMessage);
}

void
svvMessagesRedirection::DisplayDebugText (const char* aMessage)
{
  if(aMessage == NULL || *aMessage == '\0') { return; }
  svvDebug(<< aMessage);
}

/*
 * End of: $Id: svvMessagesRedirection.cxx,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $.
 *
 */
