/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMessagesUI.cxx,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvMessagesUI.h"
#include "svvMacros.h"  // SVV_ASTRCPY()
// STL
#include <new>
#include <iostream>
#include <fstream>
// C++ forwarding ANSI C
#include <cstdlib>      // exit(3C)
#include <cstdarg>      // va_list, va_start, va_end
#include <cstring>      // strncpy(3C), strncat(3C), strlen(3C), strstr(3C)
#include <cerrno>       // errno(3C)
// FLTK
#include <FL/filename.H>        // fl_filename_isdir()
#include <FL/Fl_File_Chooser.H> // fl_file_chooser()
#include <FL/fl_ask.H>          // fl_ask()
// UNIX
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <sys/resource.h>       // getrusage(3C)


// ----------------------------------------------------------------------------
void
Debug (void)
{
  ;
}

void
Exit (int aStatus)
{
#ifdef SVV_DEBUG
  if (aStatus != 0)
    {
    abort();
    }
#endif /* SVV_DEBUG */

  std::exit(aStatus);
}

void
GetResources (long& aSeconds, long& aMicro, long& aMemory)
{
  static struct rusage r;

  getrusage(RUSAGE_SELF, &r);

  aSeconds = long(r.ru_utime.tv_sec);
  aMicro   = long(r.ru_utime.tv_usec);
  aMemory  = long(r.ru_maxrss);
}

void
PrintResources (char* fmt, long s, long us, long mem)
{
  svvMessage(<< "Resources:  " << fmt << "cpu " << s << "." << us << " s / mem " << mem << " kb");
}

double
Cpu (void)
{
  long s, us, mem;
  GetResources(s, us, mem);
  return double(s) + double(us) / 1.e6;
}


// ----------------------------------------------------------------------------
//     s v v M e s s a g e s B r o w s e r
// ----------------------------------------------------------------------------
svvMessagesBrowser::svvMessagesBrowser (int x, int y, int w, int h)
  : Fl_Browser(x, y, w, h)
{
  memset(this->CBuffer,'\0',1024);
  memset(this->SnBuffer,'\0',1024);
}

void
svvMessagesBrowser::PutSn (const char* str, int len)
{
  bool           newlines = false;
  std::size_t lastline = 0L;
  int            oldend   = 0;

  //if (!this->active_r())
  //  return;

  this->SnBuffer[0] = '\0';  // 'clear' string buffer

  if (str == NULL || len == 0)
    return;

  lastline = this->size();

  const char* line = this->text(lastline);

  if (line != NULL) // Line contains text    
    {
    strcpy(this->SnBuffer,line);
    oldend = strlen(this->SnBuffer);
    }

  strncat(this->SnBuffer, str, len);

  char* substrbeg       = this->SnBuffer; 
  int   substrlen       = 0;
  // search only new part for newlines
  char* substrend       = this->SnBuffer + oldend;
  int   length          = 0 + oldend;

  while ((*substrend != '\0') && (length < oldend+len))
    {
    if (*substrend == '\n') 
      {
      *substrend = '\0';                // damages string

      this->text(lastline,substrbeg);   // add to current line
      this->add("");                    // and add new line for the next writes!
      this->bottomline(lastline);       // follow output, newest stays visible

      substrbeg = substrend;            // start new substring
      substrlen = 0; 
      newlines  = true;
      }

    substrend++;
    substrlen++;
    length++;
    }

  if (!newlines)        // this should be the normal case
    {
    this->text(lastline,substrbeg);
    this->bottomline(lastline); // follow output, newest output stays visible
   }

  return;
}

void
svvMessagesBrowser::PutC (int chr)
{
  char app[2];
  
  this->CBuffer[0] = '\0';  // 'clear' string buffer

  app[0] = char(chr);
  app[1] = '\0';

  int         lastline = this->size();
  const char* line     = this->text(lastline);

  if (line != 0)
    {
    strcpy(this->CBuffer,line);  
    }

  strcat(this->CBuffer, app);

  this->text(lastline, this->CBuffer);      
  this->bottomline(lastline); // follow output, newest output stays visible

  return;
}

// ----------------------------------------------------------------------------
//     s v v M e s s a g e s U I
// ----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// Begin class singleton internals
///////////////////////////////////////////////////////////////////////////////

/** \internal
 * this is for fake initialization
 */
template <typename T>
union _aligned_buffer_t
{
  char buf[sizeof(T)];
  struct { double a; double b; } padding;
};

// ----------------------------------------------------------------------------
char    svvMessagesUI::ErrorFileName[1024];
long    svvMessagesUI::Init::TranslationUnits;

// Singleton static space.
static _aligned_buffer_t<svvMessagesUI> fake;
// Singleton instance reference.
svvMessagesUI& SvvMsg = *(reinterpret_cast<svvMessagesUI*>(&fake));

// ----------------------------------------------------------------------------
static void
svv_msg_error (const char* aFormat, ...)
{
  static char buf[1024] = { '\0' };
  va_list args;

  va_start(args, aFormat);
  vsnprintf(buf, sizeof buf, aFormat, args);
  va_end(args);

  svvError(<< buf);
}

static void
svv_msg_fatal (const char* aFormat, ...)
{
  static char buf[1024] = { '\0' };
  va_list args;

  va_start(args, aFormat);
  vsnprintf(buf, sizeof buf, aFormat, args);
  va_end(args);

  svvFatal(<< buf);
}

static void
svv_msg_warning (const char* aFormat, ...)
{
  static char buf[1024] = { '\0' };
  va_list args;

  va_start(args, aFormat);
  vsnprintf(buf, sizeof buf, aFormat, args);
  va_end(args);

  svvWarning(<< buf);
}

// ----------------------------------------------------------------------------
svvMessagesUI::Init::Init (void)
{ if (TranslationUnits++ == 0) svvMessagesUI::ClassInitialize(); }

svvMessagesUI::Init::~Init()
{ if (--TranslationUnits == 0) svvMessagesUI::ClassFinalize(); }

void
svvMessagesUI::ClassInitialize (void)
{
  SVV_ASTRCPY(svvMessagesUI::ErrorFileName, "/var/tmp/svv-error-dump.txt");

  svvMessagesUI* ptr = new((void *) &fake) svvMessagesUI;
  ptr->hide();

  // subsume these FLTK facilities.
  Fl::warning = ::svv_msg_warning;
  Fl::error   = ::svv_msg_error;
  Fl::fatal   = ::svv_msg_fatal;
}

void
svvMessagesUI::ClassFinalize (void)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(&SvvMsg);
  ptr->hide();

  // EXPLICIT DESTRUCTOR CALL AHEAD ... think thrice!!!
  ptr->~svvMessagesUI();
}

///////////////////////////////////////////////////////////////////////////////
// End of class singleton internals
///////////////////////////////////////////////////////////////////////////////


// ----------------------------------------------------------------------------
void
svvMessagesUI::HelpCb (Fl_Button*, void* a)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(a);

  fl_message("Sorry, the Message Browser Help is not yet helpful.");
}

void
svvMessagesUI::LevelCb (Fl_Menu_* aMenu, void* a)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(a);

  ptr->threshold(long(aMenu->mvalue()->user_data()));
}

void
svvMessagesUI::SaveCb (Fl_Button*, void* a)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(a);

  char* name;

  if ((name = fl_file_chooser("Save messages", NULL, NULL)) != NULL)
    {
    if (fl_filename_isdir(name)==0 && strcmp(fl_filename_name(name)," ")!=0)
      ptr->save(name);
    else
      svvWarning(<< "SaveCb() Failed : '" << name << "' is a directory.");
    }
}

void
svvMessagesUI::ClearCb (Fl_Button*, void* a)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(a);

  ptr->clear();
}

void
svvMessagesUI::BrowserCb (Fl_Browser*, void*)
{
}

void
svvMessagesUI::CloseCb (Fl_Button*, void* a)
{
  svvMessagesUI* ptr = reinterpret_cast<svvMessagesUI*>(a);

  ptr->hide();
}

// ----------------------------------------------------------------------------
static const int WW = 540;      // window width
static const int WH = 300;      // window height
static const int WB = 4;        // window border
static const int BW = 42;       // button width
static const int BH = 20;       // button height

Fl_Menu_Item
svvMessagesUI::LevelMenuItems[] =
{
  {"DEBUG", 0, 0, (void *) DEBUG, 0, 0, FL_COURIER_BOLD, 14, FL_GREEN},
  {"INFO", 0, 0, (void *) INFO, 0, 0, FL_COURIER_BOLD, 14, FL_FOREGROUND_COLOR},
  {"WARNING", 0,  0, (void *) WARNING, 0, 0, FL_COURIER_BOLD, 14, FL_YELLOW},
  {"ERROR", 0, 0, (void *) ERROR, 0, 0, FL_COURIER_BOLD, 14, FL_RED},
  {"FATAL", 0, 0, (void *) FATAL, 0, 0, FL_COURIER_BOLD, 14, FL_RED},
  {0}
};

svvMessagesUI::svvMessagesUI (void)
  : Fl_Window    (WW,  WH, "SV\262: Messages"),
    mBrowser     (WB,  WB, WW - WB*2, WH - WB * 3 - BH),
    mHelpButton  (WB,        WH - WB - BH, BH,                    BH, "?"),
    mStatusBox   (BH + WB*2, WH - WB - BH, WW - BH - WB*7 - BW*5, BH),
    mLevelButton (WW - WB*4 - BW*5, WH - WB - BH, BW*2, BH, "severity"),
    mSaveButton  (WW - WB*3 - BW*3, WH - WB - BH, BW,   BH, "save"),
    mClearButton (WW - WB*2 - BW*2, WH - WB - BH, BW,   BH, "clear"),
    mCloseButton (WW - WB*1 - BW*1, WH - WB - BH, BW,   BH, "close"),
    mLevelPopup  (WB,  WB, WW - WB*2, WH - WB * 3 - BH),
    mEnd(),
    threshold_(INFO),
    next_save_start_(1)
{
  this->mBrowser.type(FL_MULTI_BROWSER);
  this->mBrowser.textfont(FL_COURIER);
  this->mBrowser.textsize(12);
  this->mBrowser.callback((Fl_Callback *) svvMessagesUI::BrowserCb, this);

  this->mHelpButton.box(FL_UP_BOX);
  this->mHelpButton.down_box(FL_DOWN_BOX);
  this->mHelpButton.labelfont(FL_HELVETICA_BOLD);
  this->mHelpButton.labelcolor(FL_FOREGROUND_COLOR);
  this->mHelpButton.align(FL_ALIGN_CENTER | FL_ALIGN_INSIDE);
  this->mHelpButton.callback((Fl_Callback *) svvMessagesUI::HelpCb, this);

  this->mStatusBox.box(FL_EMBOSSED_BOX);
  this->mStatusBox.labelcolor(FL_FOREGROUND_COLOR);
  this->mStatusBox.labelfont(FL_HELVETICA);
  this->mStatusBox.labelsize(12);
  this->mStatusBox.align(FL_ALIGN_LEFT | FL_ALIGN_INSIDE | FL_ALIGN_CLIP);

  this->mLevelButton.down_box(FL_BORDER_BOX);
  this->mLevelButton.labelcolor(FL_FOREGROUND_COLOR);
  this->mLevelButton.align(FL_ALIGN_INSIDE | FL_ALIGN_LEFT);
  this->mLevelButton.textfont(FL_COURIER_BOLD);
  this->mLevelButton.textcolor(FL_FOREGROUND_COLOR);
  this->mLevelButton.callback((Fl_Callback *) svvMessagesUI::LevelCb, this);
  this->mLevelButton.menu(svvMessagesUI::LevelMenuItems);
  this->mLevelButton.value(2/*INFO*/);

  this->mSaveButton.box(FL_UP_BOX);
  this->mSaveButton.down_box(FL_DOWN_BOX);
  this->mSaveButton.labelcolor(FL_FOREGROUND_COLOR);
  this->mSaveButton.align(FL_ALIGN_CENTER | FL_ALIGN_INSIDE);
  this->mSaveButton.callback((Fl_Callback *) svvMessagesUI::SaveCb, this);

  this->mClearButton.box(FL_UP_BOX);
  this->mClearButton.down_box(FL_DOWN_BOX);
  this->mClearButton.labelcolor(FL_FOREGROUND_COLOR);
  this->mClearButton.align(FL_ALIGN_CENTER | FL_ALIGN_INSIDE);
  this->mClearButton.callback((Fl_Callback *) svvMessagesUI::ClearCb, this);

  this->mCloseButton.box(FL_UP_BOX);
  this->mCloseButton.down_box(FL_DOWN_BOX);
  this->mCloseButton.labelcolor(FL_FOREGROUND_COLOR);
  this->mCloseButton.align(FL_ALIGN_CENTER | FL_ALIGN_INSIDE);
  this->mCloseButton.callback((Fl_Callback *) svvMessagesUI::CloseCb, this);

  this->mLevelPopup.type(Fl_Menu_Button::POPUP3);
  this->mLevelPopup.box(FL_UP_BOX);
  this->mLevelPopup.down_box(FL_BORDER_BOX);
  this->mLevelPopup.labeltype(FL_NO_LABEL);
  this->mLevelPopup.textfont(FL_COURIER_BOLD);
  this->mLevelPopup.textcolor(FL_FOREGROUND_COLOR);
  this->mLevelPopup.callback((Fl_Callback *) svvMessagesUI::LevelCb, this);
  this->mLevelPopup.menu(svvMessagesUI::LevelMenuItems);
  this->mLevelPopup.value(2/*INFO*/);

  this->user_data((void *) this);
  this->set_non_modal();
  this->resizable(this->mBrowser);

  SVV_ASTRCPY(this->filename_, "/tmp/svv-messages.txt");
  SVV_ASTRCPY(this->status_, "ready");
  this->mStatusBox.label(this->status_);
}

// ----------------------------------------------------------------------------
void
svvMessagesUI::threshold (uchar a)
{
  this->threshold_ = a;
}

uchar
svvMessagesUI::threshold (void) const
{
  return this->threshold_;
}

// ----------------------------------------------------------------------------
void
svvMessagesUI::log (uchar aLevel, const std::string& aMessage)
{
  Fl::lock();

  bool abort = false;
  bool log   = true;

  if (aMessage.empty())
    return;

  std::string::size_type pos, idx = 0;
  std::string msg;

  try
    {

    if      (aLevel & LOGFILE)
      {
      log = true;
      }
    else if (aLevel & STATUS)
      {
      if (aLevel == STATUS) log = false;

      msg.assign(aMessage);
      
      while ((pos = msg.find('\n')) != std::string::npos)
        msg.replace(pos,1,1,' ');
      if ((pos = msg.find_last_not_of(' ')) != std::string::npos)
        msg.erase(pos + 1);

      SVV_ASTRCPY(this->status_, msg.c_str());

      this->mStatusBox.label(this->status_);
      this->mStatusBox.redraw();

      msg.clear();
      }
  
    if (this->threshold_ < aLevel)
      {
      Fl::unlock();
      return;
      }

    if      (aLevel & FATAL)
      {
      abort = true;
      msg += "@C88@b@m@.";
      msg += "Fatal   : ";
      }
    else if (aLevel & DIRECT)
      {
      // ...
      }
    else if (aLevel & ERROR)
      {
      msg += "@C88@b@.";
      msg += "Error   : ";
      }
    else if (aLevel & WARNING)
      {
      msg += "@C95@b@.";
      msg += "Warning : ";
      }
    else if (aLevel & INFO)
      {
      msg += "@.";
      msg += "Info    : ";
      }
    else if (aLevel & STATUS)
      {
      msg += "@.";
      msg += "Status  : ";
      }
    else if (aLevel & DEBUG)
      {
      msg += "@C63@f@.";
      msg += "Debug   : ";
      }
    else if (aLevel & LOGFILE)
      {
      // ...
      }
    else
      {
      Fl::unlock();
      return;
      }
    }
  catch (...)
    {
    std::cerr << "\nCaught exception in message log()!\n" << std::endl;
    this->mBrowser.add(aMessage.c_str(), 0);
    this->mBrowser.add( "@C72@b@m@.Fatal   :"
                        " Caught exception in message log()!", 0 );
    aLevel = FATAL;
    abort  = true;
    }

  if (abort || (aLevel & FATAL))
    {
    Debug();

    this->save(svvMessagesUI::ErrorFileName);

    fl_alert( "A fatal error has occurred, which will force SV to exit\n"
              "All messages have been saved in the error log file:\n'%s'.",
              svvMessagesUI::ErrorFileName );

    Fl::unlock();

    Exit(-1);
    }
  else
    {
    // Append the message;
    msg += aMessage;

    if (aLevel != DEBUG && aLevel != STATUS)
      {
      // This is pretty costly, but permits to keep the app
      // responsive. The downside is that it can cause race
      // conditions: everytime we output a Msg, a pending callback can
      // be executed! We fix this by encapsulating critical sections
      // (mai3d(), CreateFile(), etc.) with 'CTX.threads_lock', but
      // this is far from perfect...
      Fl::check();
      }

    if (aLevel & DIRECT)
      {
      while ((pos = msg.find('\n', idx)) != std::string::npos)
        {
        this->mBrowser.add(msg.substr(idx, pos - idx).c_str(), 0);
        idx = pos + 1;
        }
      }
    else
      {
      while ((pos = msg.find('\n')) != std::string::npos)
        msg.replace(pos,1,1,' ');
      if ((pos = msg.find_last_not_of(' ')) != std::string::npos)
        msg.erase(pos + 1);
      
      this->mBrowser.add(msg.c_str(), 0);
      }
  
    this->mBrowser.bottomline(this->mBrowser.size());
    }

  Fl::unlock();
}

// ----------------------------------------------------------------------------
void
svvMessagesUI::save (void)
{
  if (this->filename_[0] == '\0')
    {
    this->log(svvMessagesUI::ERROR, "Empty file name specified to save().");
    return;
    }

  std::ofstream fileStream;

  // Opens file (maybe creates) and positions stream pointer at EOF for append.
#ifdef _WIN32
  fileStream.open(this->filename_, ios::binary|std::ios::out|std::ios::app);
#else
  fileStream.open(this->filename_, std::ios::out|std::ios::app);
#endif

  if (fileStream.is_open())
    {
    const char* p;
    const char* p2;

    for (int i=this->next_save_start_; i <= this->mBrowser.size(); i++)
      if ((p = this->mBrowser.text(i)) != NULL)
        fileStream << (p[0]=='@' ? ((p2=strstr(p,"@.")) ? &p2[2] : &p[3]) : p)
                   << std::endl;

    this->next_save_start_ = this->mBrowser.size() + 1;

    fileStream.close();
    }
}

void
svvMessagesUI::save (char aFileName[1024])
{
  SVV_ASTRCPY(this->filename_, aFileName);

  this->save();
}

void
svvMessagesUI::save (const std::string& aFileName)
{
  if (aFileName.size() == 0)
    {
    this->log(svvMessagesUI::ERROR, "Empty file name specified to save().");
    return;
    }

  SVV_ASTRCPY(this->filename_, aFileName.c_str());

  this->save();
}

// ----------------------------------------------------------------------------
void
svvMessagesUI::clear (void)
{
  this->mBrowser.clear();
  this->next_save_start_ = 1;
}

/*
 * End of: $Id: svvMessagesUI.cxx,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $.
 *
 */
