/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkAbstractCameraUI.cxx,v 1.1.1.1 2006/12/19 22:59:45 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "vtkCameraUI.h"
// C++ forwarding ANSI C
#include <cmath>
// VTK Common
#include "vtkMath.h"
// VTK Rendering
#include "vtkCamera.h"
// VTK Common
#include "vtkDebugLeaks.h"
#include "vtkObjectFactory.h"

// ----------------------------------------------------------------------------
//     v t k A b s t r a c t C a m e r a U I
// ----------------------------------------------------------------------------
vtkCxxRevisionMacro (vtkAbstractCameraUI, "$Revision: 1.1.1.1 $");
vtkInstantiatorNewMacro (vtkAbstractCameraUI);

// ----------------------------------------------------------------------------
vtkAbstractCameraUI*
vtkAbstractCameraUI::New (void)
{
  vtkObject* result = vtkObjectFactory::CreateInstance("vtkAbstractCameraUI");

  if (result != NULL)
    {
    return static_cast<vtkAbstractCameraUI*>(result);
    }

  vtkDebugLeaks::DestructClass("vtkAbstractCameraUI");

  return vtkCameraUI::New();
}

// ----------------------------------------------------------------------------
void
vtkAbstractCameraUI::Default (vtkObject* a)
{
  double tmp[3] = { 0.0, 0.0, 0.0 };

  this->SetWindowCenter(tmp);
  this->SetFocalPoint(tmp);
  tmp[2] = 1.0;
  this->SetPosition(tmp);
  tmp[0] = 0.01;
  tmp[1] = 1000.01;
  this->SetClippingRange(tmp);
  tmp[0] = 0.0;
  tmp[1] = 1.0;
  tmp[2] = 0.0;
  this->SetViewUp(tmp);
  this->SetViewAngle(30.0);
  tmp[0] = 0.0;
  tmp[1] = 0.0;
  tmp[2] = 1.0;
  this->SetViewShear(tmp);
  this->SetFocalDisk(1.0);
  this->SetEyeAngle(2.0);
  this->SetParallelProjection(0);
  this->SetUseHorizontalViewAngle(0);
  this->SetParallelScale(1.0);

  this->Send(a);
  this->ResetCurrentClientCamera();
  this->Receive(a);
}

void
vtkAbstractCameraUI::Receive (vtkObject* a)
{
  this->Check();

  vtkCamera* camera;
  double tmp[3];
  int i;

  if ((camera = this->GetCurrentClientCamera()) != (vtkCamera *) 0)
    {
    camera->GetWindowCenter(tmp);
    this->SetWindowCenter(tmp);

    camera->GetFocalPoint(tmp);
    this->SetFocalPoint(tmp);

    camera->GetPosition(tmp);
    this->SetPosition(tmp);

    camera->GetClippingRange(tmp);
    this->SetClippingRange(tmp);

    camera->GetViewUp(tmp);
    this->SetViewUp(tmp);

    tmp[0] = camera->GetViewAngle();
    this->SetViewAngle(tmp[0]);

    camera->GetViewShear(tmp);
    this->SetViewShear(tmp);

    tmp[0] = camera->GetFocalDisk();
    this->SetFocalDisk(tmp[0]);

    tmp[0] = camera->GetEyeAngle();
    this->SetEyeAngle(tmp[0]);

    i = camera->GetParallelProjection();
    this->SetParallelProjection(i);

    i = camera->GetUseHorizontalViewAngle();
    this->GetUseHorizontalViewAngle(i);

    tmp[0] = camera->GetParallelScale();
    this->SetParallelScale(tmp[0]);
    }
  else
    {
    this->Default(a);
    }

  this->GetViewShear(tmp);

  double alpha = vtkstd::atan2(tmp[1], tmp[0]);
  double beta  = vtkstd::atan2(vtkstd::cos(alpha), tmp[0]);

  tmp[0] = alpha * vtkMath::DoubleRadiansToDegrees();
  tmp[1] = beta * vtkMath::DoubleRadiansToDegrees();

  this->SetObliqueAngles(tmp);
}

void
vtkAbstractCameraUI::Send (vtkObject* vtkNotUsed(a))
{
  this->Check();

  vtkCamera* camera;
  double tmp[3];
  int i;

  if ((camera = this->GetCurrentClientCamera()) != (vtkCamera *) 0)
    {
    this->GetWindowCenter(tmp);
    camera->SetWindowCenter(tmp[0], tmp[1]);

    this->GetFocalPoint(tmp);
    camera->SetFocalPoint(tmp);

    this->GetPosition(tmp);
    camera->SetPosition(tmp);

    this->GetClippingRange(tmp);
    camera->SetClippingRange(tmp);

    this->GetViewUp(tmp);
    camera->SetViewUp(tmp);

    this->GetViewAngle(tmp[0]);
    camera->SetViewAngle(tmp[0]);

    this->GetViewShear(tmp);
    camera->SetViewShear(tmp);

    this->GetFocalDisk(tmp[0]);
    camera->SetFocalDisk(tmp[0]);

    this->GetEyeAngle(tmp[0]);
    camera->SetEyeAngle(tmp[0]);

    this->GetParallelProjection(i);
    camera->SetParallelProjection(i);

    this->GetUseHorizontalViewAngle(i);
    camera->SetUseHorizontalViewAngle(i);

    this->GetParallelScale(tmp[0]);
    camera->SetParallelScale(tmp[0]);
    }
}

// ----------------------------------------------------------------------------
void
vtkAbstractCameraUI::PrintSelf (ostream& aTarget, vtkIndent aIndent)
{
  this->Superclass::PrintSelf(aTarget, aIndent);

  aTarget << aIndent << "LastDolly:     " << LastDolly << endl;
  aTarget << aIndent << "LastRoll:      " << LastRoll << endl;
  aTarget << aIndent << "LastElevation: " << LastElevation << endl;
  aTarget << aIndent << "LastPitch:     " << LastPitch << endl;
  aTarget << aIndent << "LastAzimuth:   " << LastAzimuth << endl;
  aTarget << aIndent << "LastYaw:       " << LastYaw << endl;
  aTarget << aIndent << "LastZoom:      " << LastZoom << endl;
}

/*
 * End of: $Id: vtkAbstractCameraUI.cxx,v 1.1.1.1 2006/12/19 22:59:45 christianh Exp $.
 *
 */
