/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkAbstractUI.cxx,v 1.1.1.1 2006/12/19 22:59:46 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "vtkAbstractUI.h"
// VTK Rendering
#include "vtkProp.h"
#include "vtkAssembly.h"
#include "vtkCamera.h"
#include "vtkLightCollection.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
// vtkExtensions
#include "vtkRenderingContext.h"
// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Window.H>

VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);

// ----------------------------------------------------------------------------
//      v t k A b s t r a c t U I
// ----------------------------------------------------------------------------
vtkCxxRevisionMacro (vtkAbstractUI, "$Revision: 1.1.1.1 $");

vtkCxxSetObjectMacro(vtkAbstractUI,ClientRenderingContext,vtkRenderingContext);

// ----------------------------------------------------------------------------
void
vtkAbstractUI::SetFltkWindow (Fl_Window* aWindow)
{
  if      (this->FltkWindow != (Fl_Window *) 0)
    {
    vtkErrorMacro(<< "The FltkWindow is expected to be set ONCE,"
                  << " and ONLY ONCE, within the subclass constructor.");
    }
  else if (aWindow == (Fl_Window *) 0)
    {
    vtkErrorMacro(<< "The FltkWindow is expected to be set ONCE,"
                  << " and ONLY ONCE, within the subclass constructor,"
                  << " but the Fl_Window pointer is NULL.");
    }
  else
    {
    this->FltkWindow = aWindow;
    this->FltkWindow->user_data((void*)(this));
    }
}

Fl_Window*
vtkAbstractUI::GetFltkWindow (void) const
{
  return this->FltkWindow;
}

// ----------------------------------------------------------------------------
vtkAbstractUI::vtkAbstractUI (void)
  : ClientRenderingContext(0),
    FltkWindow(0),
    Name(0),
    Label(0),
    FltkWindowLabel(0)
{
}

vtkAbstractUI::~vtkAbstractUI()
{
  this->Hide();

  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    this->ClientRenderingContext->UnRegister(this);

  if (this->FltkWindow != (Fl_Window *) 0)
    delete this->FltkWindow;

  if (this->Name != (char *) 0)
    delete [] this->Name;

  if (this->Label != (char *) 0)
    delete [] this->Label;

  if (this->FltkWindowLabel != (char *) 0)
    delete [] this->FltkWindowLabel;
}

// ----------------------------------------------------------------------------
void
vtkAbstractUI::SetName (const char* aName)
{
  if ((this->Name == (char *) 0) && (aName == (char *) 0))
    return;

  if (this->Name != (char *) 0)
    {
    if ((aName != (char *) 0) && (strcmp(this->Name, aName) == 0))
      return;

    delete [] this->Name;
    this->Name = (char *) 0;
    }

  vtkDebugMacro(<< this->GetClassName() << " (" << this
                << "): setting Name to \"" << aName << "\"");

  if (aName != (char *) 0)
    {
    this->Name = strcpy(new char[strlen(aName)+1], aName);
    }

  this->UpdateFltkWindowLabel();
}

void
vtkAbstractUI::SetLabel (const char* aLabel)
{
  if ((this->Label == (char *) 0) && (aLabel == (char *) 0))
    return;

  if (this->Label != (char *) 0)
    {
    if ((aLabel != (char *) 0) && (strcmp(this->Label, aLabel) == 0))
      return;

    delete [] this->Label;
    this->Label = (char *) 0;
    }

  vtkDebugMacro(<< this->GetClassName() << " (" << this
                << "): setting Label to \"" << aLabel << "\"");

  if (aLabel != (char *) 0)
    {
    this->Label = strcpy(new char[strlen(aLabel)+1], aLabel);
    }

  this->UpdateFltkWindowLabel();
}

void
vtkAbstractUI::UpdateFltkWindowLabel (void)
{
  char* tmp = (char *) 0;

  if      (this->Name != (char *) 0)
    {
    if (this->Label != (char *) 0)
      {
      tmp = new char [strlen(this->Label) + strlen(this->Name) + 4];
      strcpy(tmp, this->Label);
      strcat(tmp, " (");
      }
    else
      {
      tmp = new char [strlen(this->Name) + 3];
      strcpy(tmp, "(");
      }
    strcat(tmp, this->Name);
    strcat(tmp, ")");
    }
  else if (this->Label != (char *) 0)
    {
    tmp = new char [strlen(this->Label) + 1];
    strcpy(tmp, this->Label);
    }

  if (this->FltkWindowLabel != (char *) 0)
    {
    delete [] this->FltkWindowLabel;
    }

  vtkDebugMacro(<< this->GetClassName() << " (" << this
                << "): setting FltkWindowLabel to \"" << tmp << "\"");

  this->FltkWindowLabel = tmp;

  Fl_Window* w;

  if ((w = this->GetFltkWindow()) != (Fl_Window *) 0)
    {
    w->label(this->FltkWindowLabel);
    }
}

// ----------------------------------------------------------------------------
void
vtkAbstractUI::UpdateClientRenderingContext (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    this->ClientRenderingContext->Update();
    this->Receive();
    }
}

// ----------------------------------------------------------------------------
vtkRenderWindowInteractor*
vtkAbstractUI::GetClientInteractor (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetInteractor();
    }

  return (vtkRenderWindowInteractor *) 0;
}

vtkRenderWindow*
vtkAbstractUI::GetClientRenderWindow (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetRenderWindow();
    }

  return (vtkRenderWindow *) 0;
}

vtkRenderer*
vtkAbstractUI::GetCurrentClientRenderer (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentRenderer();
    }

  return (vtkRenderer *) 0;
}

vtkLightCollection*
vtkAbstractUI::GetCurrentClientLights (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentLights();
    }

  return (vtkLightCollection *) 0;
}

vtkCamera*
vtkAbstractUI::GetCurrentClientCamera (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentCamera();
    }

  return (vtkCamera *) 0;
}

vtkProp*
vtkAbstractUI::GetCurrentClientProp (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentProp();
    }

  return (vtkProp *) 0;
}

vtkAssembly*
vtkAbstractUI::GetCurrentClientAssembly (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    return this->ClientRenderingContext->GetCurrentAssembly();
    }

  return (vtkAssembly *) 0;
}

// ----------------------------------------------------------------------------
void
vtkAbstractUI::ResetCurrentClientCamera (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    this->ClientRenderingContext->ResetCurrentCamera();
    this->Receive();
    }
}

void
vtkAbstractUI::ResetCurrentClientCameraClippingRange (void)
{
  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    Fl::check();
    this->ClientRenderingContext->ResetCurrentCameraClippingRange();
    this->Receive();
    }
}

// ----------------------------------------------------------------------------
void
vtkAbstractUI::Show (int aArgc, char* aArgv[])
{
  if (this->FltkWindow != (Fl_Window *) 0)
    {
    this->Receive();
    this->FltkWindow->show(aArgc, aArgv);
    }
}

void
vtkAbstractUI::Show (void)
{
  if (this->FltkWindow != (Fl_Window *) 0)
    {
    this->Receive();
    this->FltkWindow->show();
    }
}

int
vtkAbstractUI::Shown (void)
{
  if (this->FltkWindow != (Fl_Window *) 0)
    {
    return this->FltkWindow->shown();
    }

  return 0;
}

void
vtkAbstractUI::Hide (void)
{
  if (this->FltkWindow != (Fl_Window *) 0)
    {
    this->FltkWindow->hide();
    }
}

void
vtkAbstractUI::Show (int aArgc, char* aArgv[], int aFlag)
{
  this->Show(aArgc, aArgv);
  if (aFlag) while (!this->Shown()) Fl::wait();
}

void
vtkAbstractUI::Show (int aFlag)
{
  this->Show();
  if (aFlag) while (!this->Shown()) Fl::wait();
}

void
vtkAbstractUI::Hide (int aFlag)
{
   this->Hide();
   if (aFlag) while (this->Shown()) Fl::wait();
}

// ----------------------------------------------------------------------------
void
vtkAbstractUI::PrintSelf (ostream& aTarget, vtkIndent aIndent)
{
  this->Superclass::PrintSelf(aTarget,aIndent);

  aTarget << aIndent << "ClientRenderingContext: "
          << this->ClientRenderingContext << endl;

  if (this->ClientRenderingContext != (vtkRenderingContext *) 0)
    {
    this->ClientRenderingContext->PrintSelf(aTarget, aIndent.GetNextIndent());
    }

  aTarget << aIndent << "FltkWindow:             "
          << this->FltkWindow << endl;

  aTarget << aIndent << "Name:                   "
          << (this->Name != (char *) 0 ? this->Name : "(NULL)") << endl;

  aTarget << aIndent << "Label:                  "
          << (this->Label != (char *) 0 ? this->Label : "(NULL)") << endl;

  aTarget << aIndent << "FltkWindowLabel:        "
          << ( this->FltkWindowLabel != (char *) 0 ? this->FltkWindowLabel :
               "(NULL)" ) << endl;
}

/*
 * End of: $Id: vtkAbstractUI.cxx,v 1.1.1.1 2006/12/19 22:59:46 christianh Exp $.
 *
 */
