/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkCameraCb.cxx,v 1.1.1.1 2006/12/19 22:59:47 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "vtkAbstractCameraUI.h"
// VTK Rendering
#include "vtkCamera.h"
// FLTK
#include <FL/fl_ask.H>  // fl_message()
#include <FL/Fl_Roller.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Value_Input.H>
#include <FL/Fl_Check_Button.H>


// ----------------------------------------------------------------------------
void
VtkCameraRollCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Roll(gui->GetLastRoll() - aRoller->value()/*reversed*/);
  gui->SetLastRoll();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraRollDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Roll(15.0/*reversed*/);
  gui->SetLastRoll(gui->GetLastRoll() - 15.0);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraRollIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Roll(-15.0/*reversed*/);
  gui->SetLastRoll(gui->GetLastRoll() + 15.0);
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraDollyCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double diff = (aRoller->value() - gui->GetLastDolly());

  if (diff != 0)
    {
    // dolly moves the camera towards the focus
    double d = camera->GetDistance() / (diff<0 ? -25.0 : 25.0);
    double direction[3], position[3];

    camera->GetDirectionOfProjection(direction);
    camera->GetPosition(position);
    position[0] += (d * direction[0]);
    position[1] += (d * direction[1]);
    position[2] += (d * direction[2]);
    camera->SetPosition(position);

    gui->ResetCurrentClientCameraClippingRange();
    gui->SetLastDolly();
    gui->UpdateClientRenderingContext();
    }
}

void
VtkCameraDollyDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double d = camera->GetDistance() / -10.0;
  double direction[3], position[3];

  camera->GetDirectionOfProjection(direction);
  camera->GetPosition(position);
  position[0] += (d * direction[0]);
  position[1] += (d * direction[1]);
  position[2] += (d * direction[2]);
  camera->SetPosition(position);

  gui->ResetCurrentClientCameraClippingRange();
  gui->SetLastDolly(gui->GetLastDolly() - 1.0);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraDollyIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double d = camera->GetDistance() / 10.0;
  double direction[3], position[3];

  camera->GetDirectionOfProjection(direction);
  camera->GetPosition(position);
  position[0] += (d * direction[0]);
  position[1] += (d * direction[1]);
  position[2] += (d * direction[2]);
  camera->SetPosition(position);

  gui->ResetCurrentClientCameraClippingRange();
  gui->SetLastDolly(gui->GetLastDolly() + 1.0);
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraZoomCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Zoom(1.0 + aRoller->value() - gui->GetLastZoom());
  gui->SetLastZoom();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraZoomDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Zoom(0.75);
  gui->SetLastZoom(gui->GetLastZoom() - 0.25);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraZoomIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Zoom(1.0/0.75);
  gui->SetLastZoom(gui->GetLastZoom() + 1.0/3.0);
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraElevationCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Elevation(aRoller->value() - gui->GetLastElevation());
  gui->SetLastElevation();
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraElevationDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Elevation(+15.0);
  gui->SetLastElevation(gui->GetLastElevation() + 15.0);
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraElevationIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Elevation(-15.0);
  gui->SetLastElevation(gui->GetLastElevation() - 15.0);
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraPitchCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Pitch(aRoller->value() - gui->GetLastPitch());
  gui->SetLastPitch();
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraPitchDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Pitch(+15.0);
  gui->SetLastPitch(gui->GetLastPitch() + 15.0);
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraPitchIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Pitch(-15.0);
  gui->SetLastPitch(gui->GetLastPitch() - 15.0);
  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraOrthogonalizeViewUpButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->OrthogonalizeViewUp();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraAzimuthCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Azimuth(gui->GetLastAzimuth() - aRoller->value()/*reversed*/);
  gui->SetLastAzimuth();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraAzimuthDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Azimuth(15.0/*reversed*/);
  gui->SetLastAzimuth(gui->GetLastAzimuth() - 15.0);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraAzimuthIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Azimuth(-15.0/*reversed*/);
  gui->SetLastAzimuth(gui->GetLastAzimuth() + 15.0);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraYawCb (Fl_Roller* aRoller, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aRoller == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Yaw(aRoller->value() - gui->GetLastYaw());
  gui->SetLastYaw();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraYawDecrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Yaw(-15.0);
  gui->SetLastYaw(gui->GetLastYaw() + 15.0);
  gui->UpdateClientRenderingContext();
}

void
VtkCameraYawIncrCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->Yaw(15.0);
  gui->SetLastYaw(gui->GetLastYaw() - 15.0);
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraParallelScaleCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->SetParallelScale(aInput->value());
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraSetObliqueAnglesButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[2];

  gui->GetObliqueAngles(tmp);
  camera->SetObliqueAngles(tmp[0], tmp[1]);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraAlphaAngleCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  // nuthin' ...
}

void
VtkCameraBetaAngleCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  // nuthin' ...
}

// ----------------------------------------------------------------------------
void
VtkCameraWindowCenterXCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double x, y;

  camera->GetWindowCenter(x,y);
  x = aInput->value();
  camera->SetWindowCenter(x,y);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraWindowCenterYCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double x, y;

  camera->GetWindowCenter(x,y);
  y = aInput->value();
  camera->SetWindowCenter(x,y);

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraFocalPointXCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetFocalPoint(tmp);
  tmp[0] = aInput->value();
  camera->SetFocalPoint(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraFocalPointYCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetFocalPoint(tmp);
  tmp[1] = aInput->value();
  camera->SetFocalPoint(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraFocalPointZCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetFocalPoint(tmp);
  tmp[2] = aInput->value();
  camera->SetFocalPoint(tmp);

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraPositionXCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetPosition(tmp);
  tmp[0] = aInput->value();
  camera->SetPosition(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraPositionYCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetPosition(tmp);
  tmp[1] = aInput->value();
  camera->SetPosition(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraPositionZCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetPosition(tmp);
  tmp[2] = aInput->value();
  camera->SetPosition(tmp);

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraClippingRangeLowCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[2];

  camera->GetClippingRange(tmp);
  tmp[0] = aInput->value();
  camera->SetClippingRange(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraClippingRangeHighCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[2];

  camera->GetClippingRange(tmp);
  tmp[1] = aInput->value();
  camera->SetClippingRange(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraClippingRangeResetButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  gui->ResetCurrentClientCameraClippingRange();
  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraViewUpXCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewUp(tmp);
  tmp[0] = aInput->value();
  camera->SetViewUp(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraViewUpYCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewUp(tmp);
  tmp[1] = aInput->value();
  camera->SetViewUp(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraViewUpZCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewUp(tmp);
  tmp[2] = aInput->value();
  camera->SetViewUp(tmp);

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraViewAngleCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->SetViewAngle(aInput->value());

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraViewShearDXDZCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewShear(tmp);
  tmp[0] = aInput->value();
  camera->SetViewShear(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraViewShearDYDZCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewShear(tmp);
  tmp[1] = aInput->value();
  camera->SetViewShear(tmp);

  gui->UpdateClientRenderingContext();
}

void
VtkCameraViewShearCenterCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  double tmp[3];

  camera->GetViewShear(tmp);
  tmp[2] = aInput->value();
  camera->SetViewShear(tmp);

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraFocalDiskCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->SetFocalDisk(aInput->value());

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraEyeAngleCb (Fl_Value_Input* aInput, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aInput == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->SetEyeAngle(aInput->value());

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraParallelProjectionCb (Fl_Check_Button* aButton, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aButton == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  gui->SetParallelProjection(aButton->value());
  camera->SetParallelProjection(aButton->value());

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraUseHorizontalViewAngleCb (Fl_Check_Button* aButton, void* aPtr)
{
  vtkAbstractCameraUI*  gui;
  vtkCamera*            camera;

  if (aButton == NULL)
    return;
  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;
  if ((camera = gui->GetCurrentClientCamera()) == NULL)
    return;

  camera->SetUseHorizontalViewAngle(aButton->value());

  gui->UpdateClientRenderingContext();
}

// ----------------------------------------------------------------------------
void
VtkCameraHelpButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  fl_message("Sorry, the Viewport Camera Help is not yet helpful.");
}

void
VtkCameraApplyButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  gui->Send();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraDefaultButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  gui->Default();
  gui->UpdateClientRenderingContext();
}

void
VtkCameraUpdateButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  gui->UpdateClientRenderingContext();
  gui->Receive();
}

void
VtkCameraCloseButtonCb (Fl_Button*, void* aPtr)
{
  vtkAbstractCameraUI*  gui;

  if ((gui = reinterpret_cast<vtkAbstractCameraUI*>(aPtr)) == NULL)
    return;

  gui->Hide();
}

/*
 * End of: $Id: vtkCameraCb.cxx,v 1.1.1.1 2006/12/19 22:59:47 christianh Exp $.
 *
 */
