/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkTreeCb.cxx,v 1.1.1.1 2006/12/19 22:59:51 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvConfiguration.h"
// C system headers.
#include <sys/stat.h>  /* stat() */
#include <unistd.h>  /* access() */
#include <cerrno>
// STL
#include <sstream>
// IOStreams
#include <iostream>
// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Progress.H>
#include <FL/Fl_Output.H>
#include <FL/filename.H> /* for fl_filename_name(), etc. */
#include <FL/fl_ask.H> /* for fl_alert() */
#include <FL/Fl_Help_View.H>
#include <FL/Fl_Pixmap.H>
#include <FL/Fl_PNG_Image.H>
// VTK Common
#include "vtkCallbackCommand.h"
#include "vtkPolyData.h"
// vtkProp subtypes
#include "vtkActor2D.h"
#include "vtkImageActor.h"
#include "vtkAssembly.h"
#include "vtkActor.h"
#include "vtkLODProp3D.h"
#include "vtkVolume.h"
#include "vtkPropAssembly.h"
// 
#include "vtkProp3DCollection.h"
#include "vtkActorCollection.h"
// 
#include "vtkPolyDataSource.h"
#include "vtkDepthSortPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkRenderer.h"
#include "vtkImporter.h"
#include "vtkExporter.h"
#include "vtkInteractorObserver.h"
#include "vtkInteractorStyle.h"
#include "vtkRenderWindowInteractor.h"
// vtkExtensions
#include "vtkInputSource.h"
#include "vtkInputSourceCollection.h"
#include "vtkRenderingContext.h"
#include "vtkUpstream.h"
#include "vtkXMLString.h"
// pixmaps
#if 0
#  include "icons/rendererIcon.xpm"
#  include "icons/groupIcon.xpm"
#  include "icons/surfaceIcon.xpm"
#  include "icons/volumeIcon.xpm"
#else
#  include "newicons/treeIcon24.xpm"
#  include "newicons/groupIcon24.xpm"
#  include "newicons/surfaceIcon24.xpm"
#  include "newicons/volumeIcon24.xpm"
#endif /* 0 */
#include "icons/unknownIcon.xpm"
// SVV
#include "svvController.h"
#include "svv_file_chooser.h"
#include "Fl_Toggle_Tree.H"
#include "svvMessagesUI.h"
#include "SvvMenuItemType.h"
#include "Fl_VTK_View.H"
#include "vtkAboutUI.h"
#include "vtkLightingUI.h"
#include "vtkViewportUI.h"
#include "vtkCameraUI.h"
#include "vtkSurfacePropertiesUI.h"
#include "vtkNodeBrowserUI.h"
#include "vtkAbstractToolBarUI.h"

#ifdef SVV_SUPPORT_MRML
// vtkExtensions
#  include "vtkFileType.h"
// vtkMrml
#  include "vtkMrml.h"
#  include "vtkMrmlRegistry.h"
#  include "vtkMrmlImporter.h"
#  include "vtkMrmlExporter.h"
#  include "vtkMrmlNode.h"
#  include "vtkMrmlTree.h"
#endif /* SVV_SUPPORT_MRML */

VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);
VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSourceCollection);
VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);

#ifdef SVV_SUPPORT_MRML
VTK_EXTENSIONS_NAMESPACE_USING(vtkFileType);
VTK_MRML_NAMESPACE_USING(vtkMrml);
VTK_MRML_NAMESPACE_USING(vtkMrmlRegistry);
VTK_MRML_NAMESPACE_USING(vtkMrmlImporter);
VTK_MRML_NAMESPACE_USING(vtkMrmlExporter);
VTK_MRML_NAMESPACE_USING(vtkMrmlNode);
VTK_MRML_NAMESPACE_USING(vtkMrmlTree);
#endif /* SVV_SUPPORT_MRML */

SVV_NAMESPACE_USING(svv);

#if 0
static Fl_Image* gRendererIcon =
Fl_Toggle_Tree::make_icon("renderer", rendererIcon_xpm);
static Fl_Image* gGroupIcon =
Fl_Toggle_Tree::make_icon("group", groupIcon_xpm);
static Fl_Image* gSurfaceIcon =
Fl_Toggle_Tree::make_icon("surface", surfaceIcon_xpm);
static Fl_Image* gVolumeIcon =
Fl_Toggle_Tree::make_icon("volume", volumeIcon_xpm);
#else
static Fl_Image* gRendererIcon =
Fl_Toggle_Tree::make_icon("renderer", treeIcon24_xpm);
static Fl_Image* gGroupIcon =
Fl_Toggle_Tree::make_icon("group", groupIcon24_xpm);
static Fl_Image* gSurfaceIcon =
Fl_Toggle_Tree::make_icon("surface", surfaceIcon24_xpm);
static Fl_Image* gVolumeIcon =
Fl_Toggle_Tree::make_icon("volume", volumeIcon24_xpm);
#endif /* 0 */
static Fl_Image* gUnknownIcon =
Fl_Toggle_Tree::make_icon("unknown", unknownIcon_xpm);


// ----------------------------------------------------------------------------
// Some static file / filename utilities
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
static inline vtkstd::string
get_basename (const vtkstd::string& aFullName, bool aStripSuffix = true)
{
  vtkstd::string::size_type beg;

  if ((beg = aFullName.find_last_of('/')) == vtkstd::string::npos)
    beg = 0;
  else
    beg++;

  vtkstd::string            baseName = aFullName.substr(beg);
  vtkstd::string::size_type end      = baseName.find_last_of('.');

  if (end == vtkstd::string::npos || !aStripSuffix) end = baseName.length();

  return baseName.substr(0, end);
}

// ----------------------------------------------------------------------------
static inline bool
can_read_file (const char* const& aFileName)
{
  if (aFileName == NULL || aFileName[0] == '\0')
    {
    svvDebug(<< "Cannot read file: zero length filename.");
    return false;
    }

  struct stat sb;

  if (stat(aFileName, &sb))
    {
    svvDebug(<< "Cannot read file: '" << aFileName
             << "' " << strerror(errno));
    return false; /*failure*/
    }
  else if (S_ISDIR(sb.st_mode))
    {
    svvDebug(<< "Cannot read file: '" << aFileName
             << "' " << "is a directory.");
    return false; /*failure*/
    }
  else if (access(aFileName,R_OK))
    {
    svvDebug(<< "Cannot read file: '" << aFileName
             << "' " << strerror(errno));
    return false; /*failure*/
    }

  return true;
}

static inline bool
can_write_file (const char* const& aFileName)
{
  if (aFileName == NULL || aFileName[0] == '\0')
    {
    svvDebug(<< "Cannot write file: zero length filename.");
    return false;
    }

  struct stat sb;

  if (access(aFileName,F_OK))
    { // does not yet exist, so test directory
    const vtkstd::string dirsep("/");
    const vtkstd::string fileName(aFileName);
    vtkstd::string dirName;
    vtkstd::string::size_type end;

    if ((end = fileName.find_last_of(dirsep)) == vtkstd::string::npos)
      {
      dirName = ".";
      }
    else
      {
      dirName = fileName.substr(0, end);
      }

    if (stat(dirName.c_str(), &sb))
      {
      svvDebug(<< "Cannot write to directory: '" << dirName
               << "' " << strerror(errno));
      return false; /*failure*/
      }
    else if (access(dirName.c_str(),W_OK))
      {
      svvDebug(<< "Cannot write to directory: '" << dirName
               << "' " << strerror(errno));
      return false; /*failure*/
      }
    }
  else if (stat(aFileName, &sb))
    {
    svvDebug(<< "Cannot write to file: '" << aFileName
             << "' " << strerror(errno));
    return false; /*failure*/
    }
  else if (S_ISDIR(sb.st_mode))
    {
    svvDebug(<< "Cannot write to file: '" << aFileName
             << "' " << "is a directory.");
    return false; /*failure*/
    }
  else if (access(aFileName,W_OK))
    {
    svvDebug(<< "Cannot write to file: '" << aFileName
             << "' " << strerror(errno));
    return false; /*failure*/
    }

  return true;
}

// ----------------------------------------------------------------------------
// Static functions to get a VTK object from a toggle tree node.
// ----------------------------------------------------------------------------
static inline vtkObject*
sgGetNodeObject (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return reinterpret_cast<vtkObject*>(aNode->user_data());
}

static inline vtkProp*
sgGetNodeProp (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkProp::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkActor2D*
sgGetNodeActor2D (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkActor2D::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkImageActor*
sgGetNodeImageActor (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkImageActor::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkProp3D*
sgGetNodeProp3D (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkProp3D::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkActor*
sgGetNodeActor (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkActor::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkAssembly*
sgGetNodeAssembly (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkAssembly::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkLODProp3D*
sgGetNodeLODProp3D (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkLODProp3D::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkVolume*
sgGetNodeVolume (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkVolume::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkPropAssembly*
sgGetNodePropAssembly (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkPropAssembly::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

static inline vtkRenderer*
sgGetNodeRenderer (Fl_Toggle_Tree* aTree, Fl_Toggle_Node* aNode)
{
  return vtkRenderer::SafeDownCast(sgGetNodeObject(aTree, aNode));
}

// ----------------------------------------------------------------------------
static inline Fl_Toggle_Node*
AddRendererNode (vtkAbstractToolBarUI*  aGUI,
                 vtkRenderer*           aRenderer,
                 const char*            aName = NULL)
{
  if (aGUI == NULL)
    return NULL;
  if (aRenderer == NULL)
    return NULL;

  Fl_Toggle_Tree* toggleTree;
  Fl_VTK_View*    view;

  if ((toggleTree = aGUI->GetNodeBrowser()) == NULL)
    return NULL;
  if ((view = aGUI->GetView()) == NULL)
    return NULL;

  void* data = reinterpret_cast<void*>(aRenderer);

  Fl_Toggle_Node* rendererNode;

  if ((rendererNode = toggleTree->find(data)) == NULL)
    {
    if (aName == NULL) aName = "Renderer";

    int       canOpen = 1;
    Fl_Image* pIcon   = gRendererIcon;

    // Add Renderer to the RenderWindow.
    view->AddRenderer(aRenderer);
    // Only add Renderers to the top level.
    toggleTree->select_none();
    // Now add the group node to the tree interface.
    rendererNode = toggleTree->add_next_sibling(aName, canOpen, pIcon, data);
    }

  toggleTree->traverse_start(rendererNode);
  toggleTree->select_range(rendererNode, rendererNode, 0); // ???

  return rendererNode;
}

// ----------------------------------------------------------------------------
static inline Fl_Toggle_Node*
AddPropNode (vtkAbstractToolBarUI*      aGUI,
             vtkProp*                   aProp,
             const char*                aName = NULL)
{
  if (aGUI == NULL)
    return NULL;
  if (aProp == NULL)
    return NULL;

  Fl_Toggle_Tree* toggleTree;
  Fl_VTK_View*    view;

  if ((toggleTree = aGUI->GetNodeBrowser()) == NULL)
    return NULL;
  if ((view = aGUI->GetView()) == NULL)
    return NULL;

  Fl_Toggle_Node* selectedNode;

  if ((selectedNode = toggleTree->selected_node()) == NULL)
    {
    if ((selectedNode = toggleTree->first_node()) == NULL)
      {
      //
      // The tree is empty ...
      //
      vtkRenderer* renderer;

      if ((renderer = view->GetCurrentRenderer()) != NULL)
        {
        if ( (selectedNode = AddRendererNode(aGUI, renderer, "Renderer"))
             == NULL )
          {
          Fl::error("AddPropNode() Failed add current Renderer!");
          }
        }
      else
        {
        Fl::error("AddPropNode() Failed to get current Renderer!");
        }
      }
    else
      {
      toggleTree->select_range(selectedNode, selectedNode, 0); // ???
      }
    }

  toggleTree->traverse_start(selectedNode);

  int       canOpen = 0;
  Fl_Image* pIcon   = NULL;
  void*     data    = NULL;

  if (aName != NULL) aName = fl_filename_name(aName);

  if      (aProp->IsA("vtkActor"))
    {
    if (aName == NULL) aName = "untitled surface";
    canOpen = 0;
    pIcon   = gSurfaceIcon;
    data    = reinterpret_cast<void*>(aProp);
    }
  else if (aProp->IsA("vtkAssembly"))
    {
    if (aName == NULL) aName = "untitled transform group";
    canOpen = 1;
    pIcon   = gGroupIcon;
    data    = reinterpret_cast<void*>(aProp);
    }
  else if (aProp->IsA("vtkPropAssembly"))
    {
    if (aName == NULL) aName = "untitled group";
    canOpen = 1;
    pIcon   = gGroupIcon;
    data    = reinterpret_cast<void*>(aProp);
    }
  else if (aProp->IsA("vtkVolume"))
    {
    if (aName == NULL) aName = "untitled volume";
    canOpen = 0;
    pIcon   = gVolumeIcon;
    data    = reinterpret_cast<void*>(aProp);
    }
      
  Fl_Toggle_Node* newNode = NULL;
  vtkRenderer*    renderer;
  vtkAssembly*    assembly;

  if (selectedNode->opened() && selectedNode->openable())
    {
    // Add Prop to Assembly if there is one, otherwise to the Renderer.
    if      ((assembly = sgGetNodeAssembly(toggleTree, selectedNode)) != NULL)
      {
      view->SetCurrentAssembly(assembly);
      view->AddProp(aProp);
      // Now add the subgroup node to the tree interface.
      newNode = toggleTree->add_next_child(aName, canOpen, pIcon, data);
      }
    else if ((renderer = sgGetNodeRenderer(toggleTree,selectedNode)) != NULL)
      {
      view->SetCurrentAssembly(NULL);
      view->SetCurrentRenderer(renderer);
      view->AddProp(aProp);
      // Now add the subgroup node to the tree interface.
      newNode = toggleTree->add_next_child(aName, canOpen, pIcon, data);
      }
    else
      {
      Fl::error("AddPropNode() failed to get Renderer or Assembly!");
      }
    }
  else
    {
    Fl_Toggle_Node* parentNode;
      
    if ((parentNode = selectedNode->parent_node()) != NULL)
      {
      // Add Prop to Assembly if there is one, otherwise to the Renderer.
      if      ((assembly = sgGetNodeAssembly(toggleTree,parentNode)) != NULL)
        {
        view->SetCurrentAssembly(assembly);
        view->AddProp(aProp);
        // Now add as the next node in the tree interface.
        newNode = toggleTree->add_next_sibling(aName, canOpen, pIcon, data);
        }
      else if ((renderer = sgGetNodeRenderer(toggleTree,parentNode)) != NULL)
        {
        view->SetCurrentAssembly(NULL);
        view->SetCurrentRenderer(renderer);
        view->AddProp(aProp);
        // Now add as the next node in the tree interface.
        newNode = toggleTree->add_next_sibling(aName, canOpen, pIcon, data);
        }
      else
        {
        Fl::error("AddPropNode() failed to get Renderer or Assembly!");
        }
      }
    else
      {
      Fl::error("AddPropNode() failed to get parent node!");
      }
    }

  if (newNode != NULL)
    {
    vtkProp* newProp;
      
    if ((newProp = sgGetNodeProp(toggleTree, newNode)) != NULL)
      {
      vtkAssembly* newAssembly;
        
      // Make New Groups current.
      if ((newAssembly = sgGetNodeAssembly(toggleTree,newNode)) != NULL)
        {
        view->SetCurrentAssembly(newAssembly);
        }
      view->SetCurrentProp(newProp);
      }
    toggleTree->traverse_start(newNode);
    }

  return newNode;
}

// ----------------------------------------------------------------------------
static inline Fl_Toggle_Node*
AddSourceItemAndPropNode (vtkAbstractToolBarUI* aGUI, const char* aPathName)
{
  if (aGUI == NULL || aPathName == NULL)
    return NULL;

  Fl_Toggle_Node* node = NULL;
  vtkInputSource* inputSource;

  // make the "standard" surface
  if ((inputSource = svv.MakeSurfaceDataSource(aPathName)) != NULL)
    {
    char absPath[FL_PATH_MAX];
    Fl_Browser* browser;

    fl_filename_absolute(absPath, aPathName);

    svvInfo(<< "Loaded source: '" << absPath << "'");

    vtkstd::string path(absPath), name, ext, col;
    vtkstd::string::size_type pos1, pos2;

    if ((pos1 = path.find_last_of('/')) == vtkstd::string::npos)
      pos1 = 0;
    else
      pos1 += 1;

    if ((pos2 = path.find_last_of('.')) == vtkstd::string::npos || pos1>pos2)
      {
      name = path.substr(pos1);
      }
    else
      {
      name = path.substr(pos1, pos2 - pos1);
      ext = path.substr(pos2+1);
      }

    const vtkstd::string first("@B49@C0@b");
    const vtkstd::string second("@B47@C95@b@t");
    const vtkstd::string third("@B49@C0@.");

    col.assign(first);
    col += name;
    col.push_back('\t');
    col += second;
    col += ext;
    col.push_back('\t');
    col += third;
    col += vtkstd::string(absPath);

    if ((browser = aGUI->GetSourceBrowser()) != NULL)
      {
      browser->add(col.c_str(), (void *) inputSource);
      }
    else
      {
      svvError(<< "Failed to get SourceBrowser!");
      }

    vtkActor*   actor;

    if ((actor = svv.MakeSurfaceActor(inputSource, name.c_str())) != NULL)
      {
      node = AddPropNode(aGUI, actor, name.c_str());
      actor->GetProperty()->SetSpecular(0.333f);
      actor->GetProperty()->SetSpecularPower(24.f);

      svvStatus(<< "Rendered source: '" << name << "'");

      // references are retained in the list so decrement ref count here
      actor->Delete();
      }
    else
      {
      svvError(<< "Failed to render source: '" << name << "' ("
               << inputSource << ")");
      }

    // references are retained in the list so decrement ref count here
    inputSource->Delete();
    }
  else
    {
    svvError(<< "Failed to load source: '" << aPathName << "'");
    }

  return node;
}

// ----------------------------------------------------------------------------
static bool
get_next_input_file (ifstream& aFileStream,
                     vtkstd::string& aName,
                     SvvColorWord& aColor)
{
  vtkstd::string line;

  do
    {
    if(!getline(aFileStream,line)) return false;
    }
  while(line[0]=='!' || line[0]=='#' || line[0]=='\n');

  vtkstd::istringstream stringStream(line);
  stringStream >> aName;

  return ((aFileStream.eof() || !aFileStream.good()) ? false : true);
}

// ----------------------------------------------------------------------------
// load a bunch of polygonal data files contained in an ASCII list
// ----------------------------------------------------------------------------
void
SvvLoadListCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI* gui;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;

  char* pathName = NULL;

  if ( (pathName = svv_singlefile_chooser(
          "Load ASCII list of polygonal data files?",
          "ASCII Text Files (*.{txt,asc,dave})" )) == NULL )
    {
    return;
    }

  if (!can_read_file(pathName))
    {
    svvError(<< "Unable to read '" << pathName << "' for Dave.");
    return;
    }

  // Opens the file and positions the stream pointer at EOF ...
  ifstream ifs( pathName, ios::in | ios::ate
#ifdef _WIN32
                | ios::binary
#endif
    );

  if (ifs)
    {
    svvDebug(<< "SvvLoadListCb: opened '" << pathName << "'");

    const char* fileName = fl_filename_name(pathName);

    svvInfo(<< "Loading for Dave: '" << pathName << "'");
    svvStatus(<< "Loading for Dave: '" << fileName << "'");

    // Rewind the stream ... and start reading the items.
    ifs.seekg(0, ios::beg);
    
    SvvColorWord color;
    vtkstd::string fullName;
    int total = 0, failures = 0;

    while (get_next_input_file(ifs, fullName, color))
      {
      if (AddSourceItemAndPropNode(gui, fullName.c_str()) == NULL) failures++;
      total++;
      }

    svvInfo(<< "Dave Loaded " << (total - failures) << " of " << total
            << " sources (" << failures << " failures).");
    }
  else
    {
    svvError(<< "Failed to load '" << pathName << "' for Dave.");
    }
  
  gui->GetView()->Update();
}

void
SvvLoadListCb (Fl_Button* aButton, void* aPtr)
{ SvvLoadListCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
SvvLoadListCb (Fl_Menu_* aMenu, void*)
{ SvvLoadListCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvSaveListCb (Fl_Widget*, void*)
{
  char* pathName = NULL;

  if ( (pathName = svv_newfile_chooser(
          "Save ASCII list of polygonal data files?",
          "ASCII Text Files (*.{txt,asc,dave})" )) == NULL )
    {
    return;
    }

  if (!can_write_file(pathName))
    {
    svvError(<< "Unable to write '" << pathName << "' for Dave.");
    return;
    }

  // Opens the file and positions the stream pointer at EOF ...
  ofstream ofs( pathName, ios::out | ios::trunc
#ifdef _WIN32
                | ios::binary
#endif
    );

  if (ofs.is_open())
    {
    svvDebug(<< "SvvSaveListCb opened '" << pathName << "'");

    const char* fileName = fl_filename_name(pathName);

    svvInfo(<< "Saving for Dave: '" << pathName << "'");
    svvStatus(<< "Saving for Dave: '" << fileName << "'");

    vtkInputSourceCollection* inputSources;
    vtkInputSource*           inputSource;
    const char*               inputName;

    int total = 0, successes = 0;

    if ((inputSources = svv.GetInputSources()) != NULL)
      {
      for ( inputSources->InitTraversal();
            inputSource = inputSources->GetNextSource(); )
        {
        if ((inputName = inputSource->GetFileName()) != NULL)
          {
          ofs << inputName << endl;
          successes++;
          }
        total++;
        }

      svvInfo(<< "Dave Saved " << successes << " of " << total
              << " sources (" << (total - successes) << " failures).");
      }
    else
      {
      svvError(<< "Failed to get input sources for Dave to save.");
      }
    }
}

void
SvvSaveListCb (Fl_Button* aButton, void* aPtr)
{ SvvSaveListCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
SvvSaveListCb (Fl_Menu_* aMenu, void*)
{ SvvSaveListCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }

// ----------------------------------------------------------------------------
// Add a polygonal surface or 3d volume
// ----------------------------------------------------------------------------
void
SvvLoadSurfaceCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI*      gui;
  Fl_Toggle_Tree* toggleTree;
  Fl_Toggle_Node* selectedNode;

  char*           pathName = NULL;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;

  int count;

  if ( ( count = svv_multifile_chooser(
           "Read 3D Polygonal Surface Data to the Current View?",
           "Surface Files (*.{vtk,vtp,pvtp,obj,stl,ply,byu,g,s,t,v,prt"
           ",cgm,tri,lim,txt,asc})" ) ) == 0 )
    return;

  int failures = 0;

  for (int i=0; i<count; i++)
    {
    pathName = svv_file_chooser_value(i);

    // make the "standard" surface
    if (AddSourceItemAndPropNode(gui, pathName) == NULL) failures++;
    }

  svvInfo(<< "Added " << (count - failures) << " of " << count
          << " surface sources (" << failures << " failures).");

  gui->GetView()->Update();
}

void
SvvLoadSurfaceCb (Fl_Button* aButton, void* aPtr)
{ SvvLoadSurfaceCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
SvvLoadSurfaceCb (Fl_Menu_* aMenu, void*)
{ SvvLoadSurfaceCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvLoadVolumeCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI*      gui;
  Fl_Toggle_Tree* toggleTree;
  Fl_Toggle_Node* selectedNode;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;

  if ((selectedNode = toggleTree->selected_node()) == NULL)
    {
    toggleTree->add_next_sibling("untitled volume", 0, gVolumeIcon);
    }
  else
    {
    toggleTree->traverse_start(selectedNode);
    if (selectedNode->opened() && selectedNode->openable())
      {
      toggleTree->add_next_child("untitled volume", 0, gVolumeIcon);
      }
    else
      {
      toggleTree->add_next_sibling("untitled volume", 0, gVolumeIcon);
      }
    }
}

void
SvvLoadVolumeCb (Fl_Button* aButton, void* aPtr)
{ SvvLoadVolumeCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
SvvLoadVolumeCb (Fl_Menu_* aMenu, void*)
{ SvvLoadVolumeCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvNodeGroupCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI* gui;
  Fl_Toggle_Tree*       toggleTree;
  Fl_Toggle_Node*       selectedNode;
  vtkAssembly*          assembly;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;

  assembly = vtkAssembly::New();

  AddPropNode(gui, assembly);

  // references are retained in the renderer, so decrement ref count here
  assembly->Delete();

  gui->GetView()->Update();
}

void
SvvNodeGroupCb (Fl_Button* aButton, void* aPtr)
{ SvvNodeGroupCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
SvvNodeGroupCb (Fl_Menu_* aMenu, void*)
{ SvvNodeGroupCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
AddRendererCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI*      gui;
  Fl_Toggle_Tree* toggleTree;
  Fl_Toggle_Node* toggleNode;
  vtkRenderer*    renderer;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;

  renderer = vtkRenderer::New();

  toggleNode = AddRendererNode(gui, renderer);
  toggleTree->traverse_start(toggleNode);

  // References are retained in the RenderWindow, so decrement ref count here.
  renderer->Delete();

  gui->GetView()->Update();
}

void
AddRendererCb (Fl_Button* aButton, void* aPtr)
{ AddRendererCb(static_cast<Fl_Widget*>(aButton),aPtr); }

void
AddRendererCb (Fl_Menu_* aMenu, void*)
{ AddRendererCb(static_cast<Fl_Widget*>(aMenu),aMenu->user_data()); }


// ----------------------------------------------------------------------------
#if 0
static inline void
AddProp3Ds (vtkProp3DCollection* aProp3DCollection, vtkAbstractToolBarUI* aGUI)
{
  if (aProp3DCollection == NULL)
    return;
  if (aGUI == NULL)
    return;

  Fl_Toggle_Tree* toggleTree;

  if ((toggleTree = aGUI->GetNodeBrowser()) == NULL)
    return;

  vtkProp3D*   prop3D;
  vtkActor*    actor;
  vtkVolume*   volume;
  vtkAssembly* assembly;
  
  while ((prop3D = aProp3DCollection->GetNextProp3D()) != NULL)
    {
    if      ((actor = vtkActor::SafeDownCast(prop3D)) != NULL)
      {
      AddProp3DNode(aGUI, actor);
      }
    else if ((volume = vtkVolume::SafeDownCast(prop3D)) != NULL)
      {
      AddProp3DNode(aGUI, volume);
      }
    else if ((assembly = vtkAssembly::SafeDownCast(prop3D)) != NULL)
      {
      toggleTree->traverse_start(AddProp3DNode(aGUI, assembly));
      // recurse
      AddProp3Ds(assembly->GetParts(), aGUI);
      }
    }
}
#endif /* 0 */

static inline void
AddImportedProps (vtkAbstractToolBarUI* aGUI, vtkRenderer* aRenderer)
{
  if (aGUI == NULL || aRenderer == NULL)
    {
    return;
    }

  Fl_Toggle_Tree*       toggleTree;
  Fl_VTK_View*          view;
  vtkRenderingContext*  renderingContext;
  vtkImporter*          importer;
  vtkPropCollection*    propCollection;
  Fl_Toggle_Node*       rendererNode;

  if ((toggleTree = aGUI->GetNodeBrowser()) == NULL)
    return;
  if ((view = aGUI->GetView()) == NULL)
    return;
  if ((renderingContext = aGUI->GetClientRenderingContext()) == NULL)
    return;
  if ((importer = renderingContext->GetImporter()) == NULL)
    return;
  if ((propCollection = aRenderer->GetProps()) == NULL)
    return;
  if ((rendererNode = AddRendererNode(aGUI,aRenderer)) == NULL)
    return;

#ifdef SVV_SUPPORT_MRML
  vtkMrmlImporter* mrmlImporter  = vtkMrmlImporter::SafeDownCast(importer);
  vtkMrmlTree*     mrmlColorTree = NULL;
  vtkMrmlTree*     mrmlDataTree  = NULL;

  if (mrmlImporter != NULL)
    {
    mrmlColorTree = mrmlImporter->GetColorTree();
    mrmlDataTree  = mrmlImporter->GetDataTree();
    }
#endif /* SVV_SUPPORT_MRML */

  vtkProp*              prop;
  vtkActor*             actor;
  vtkVolume*            volume;
  vtkAssembly*          assembly;

  Fl_Toggle_Node*       selectedNode;
  Fl_Toggle_Node*       node;
  Fl_Toggle_Node*       parentNode     = NULL;
  vtkAssembly*          parentAssembly = NULL;

  int                   canOpen;
  Fl_Image*             pIcon;
  void*                 data;

  int i = 0;

  // for each Prop in the Renderer ...
  for ( propCollection->InitTraversal();
        (prop = propCollection->GetNextProp()) != NULL; )
    {
    void* data = reinterpret_cast<void*>(prop);

    // If the Prop cannot be found within the ToggleTree ...
    if ((node = toggleTree->find(data)) == NULL)
      {
      svvDebug(<< "AddImportedProps: New Prop (" << data << ")");

      const char* name = NULL;

      assembly = NULL;

#  ifdef SVV_SUPPORT_MRML
      if (mrmlDataTree != NULL)
        {
        vtkMrmlNode* mrmlNode;
        
        if ( (mrmlNode = mrmlDataTree->GetNodeByID(SVV_PTR_TO_INT32(prop)))
             != NULL )
          {
          name = mrmlNode->GetTitle();
          }
        }
#  endif /* SVV_SUPPORT_MRML */

      if ((node = AddPropNode(aGUI, prop, name)) == NULL)
        {
        Fl::error("AddImportedProps() Failed to add imported prop node.");
        }
      }
    // If the Prop was found within the ToggleTree ...
    else
      {
      svvDebug(<< "AddImportedProps: Old Prop (" << data << ")");

      if ((assembly = sgGetNodeAssembly(toggleTree, node)) != NULL)
        {
        parentNode     = node;
        parentAssembly = assembly;
        }
      else
        {
        parentNode     = node->parent_node();
        parentAssembly = sgGetNodeAssembly(toggleTree,parentNode);
        }

      view->SetCurrentAssembly(parentAssembly);

      if (!parentNode->opened())
        {
        toggleTree->open(parentNode);
        }

      toggleTree->traverse_start(node);
      }

    } // for each Prop
}

// ----------------------------------------------------------------------------
void
SvvImportCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI*    gui;
  Fl_VTK_View*  view;

  if ( (gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL ||
       (view = gui->GetView()) == NULL )
    {
    return;
    }

  char*         pathName = NULL;
  const char*   fileName = NULL;

  if ( ( pathName = svv_singlefile_chooser(
           "Import 3D Scene to the Current View?",
           "Importable Files (*.{mrml,xml,3ds,wrl})") ) != NULL )
    {
    vtkRenderer* renderer;

    fileName = fl_filename_name(pathName);

    svvInfo(<< "Importing from: '" << pathName << "'");
    svvStatus(<< "Importing from: '" << fileName << "'");

    int debug = view->GetDebug();
    view->DebugOn();

    if ((renderer = view->Import(pathName, 0/*type*/)) == NULL)
      {
      svvWarning(<< "Failed to import 3D scene file: '" << pathName << "'");
      fl_alert("Failed to import 3D scene file: '%s'",pathName);
      }
    else
      {
      // Sort through returned Renderer inserting its Props into the tree.
      AddImportedProps(gui, renderer);
      }

    view->SetDebug(debug);
    }
}

void
SvvImportCb (Fl_Button* aButton, void* aPtr)
{ SvvImportCb(static_cast<Fl_Widget*>(aButton), aPtr); }

void
SvvImportCb (Fl_Menu_* aMenu, void*)
{ SvvImportCb(static_cast<Fl_Widget*>(aMenu), aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvExportCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI*    gui;
  Fl_VTK_View*  view;

  if ( (gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL ||
       (view = gui->GetView()) == NULL )
    {
    return;
    }

  char*         pathName = NULL;
  const char*   fileName = NULL;

  if ( ( pathName = svv_newfile_chooser(
           "Export 3D Scene from the Current View?",
           "Exportable Files (*.{mrml,xml,iv,obj,oogl,rib,wrl})") ) != NULL )
    {

    fileName = fl_filename_name(pathName);

    svvInfo(<< "Exporting View to: '" << pathName << "'");
    svvStatus(<< "Exporting View to: '" << fileName << "'");

    int debug = view->GetDebug();
    view->DebugOn();

#ifdef SVV_SUPPORT_MRML
    if ( vtkFileType::SuffixToFileType(pathName, true)
         == VTK_EXTENSIONS_NAMESPACE_QUALIFIER eMRMLFileType )
      {
      vtkMrml* impl;

      if ((impl = vtkMrmlRegistry::GetImplementation()) != NULL)
        {
        
        }
      }
#endif /* SVV_SUPPORT_MRML */

    if (view->Export(pathName, 0/*type*/) == 0)
      {
      svvWarning(<< "Failed to export 3D scene file: '" << pathName << "'");
      fl_alert("Failed to export 3D scene file: '%s'",pathName);
      }

    view->SetDebug(debug);
    }
}

void
SvvExportCb (Fl_Button* aButton, void* aPtr)
{ SvvExportCb(static_cast<Fl_Widget*>(aButton), aPtr); }

void
SvvExportCb (Fl_Menu_* aMenu, void*)
{ SvvExportCb(static_cast<Fl_Widget*>(aMenu), aMenu->user_data()); }


// ----------------------------------------------------------------------------
void
SvvNodeAttributesCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI* gui;
  Fl_Toggle_Tree*       toggleTree;
  Fl_Toggle_Node*       selectedNode;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;
  if ((selectedNode = toggleTree->selected_node()) == NULL)
    return;

  Fl::warning("SvvNodeAttributesCb() not yet implemented.");
}

void
SvvNodeAttributesCb (Fl_Button* aButton, void* aPtr)
{ SvvNodeAttributesCb(static_cast<Fl_Widget*>(aButton), aPtr); }

void
SvvNodeAttributesCb (Fl_Menu_* aMenu, void*)
{ SvvNodeAttributesCb(static_cast<Fl_Widget*>(aMenu), aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvNodeFilteringCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI* gui;
  Fl_Toggle_Tree*       toggleTree;
  Fl_Toggle_Node*       selectedNode;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;
  if ((selectedNode = toggleTree->selected_node()) == NULL)
    return;

  Fl::warning("SvvNodeFilteringCb() not yet implemented.");
}

void
SvvNodeFilteringCb (Fl_Button* aButton, void* aPtr)
{ SvvNodeFilteringCb(static_cast<Fl_Widget*>(aButton), aPtr); }

void
SvvNodeFilteringCb (Fl_Menu_* aMenu, void*)
{ SvvNodeFilteringCb(static_cast<Fl_Widget*>(aMenu), aMenu->user_data()); }


// ----------------------------------------------------------------------------
void
SvvNodeRemoveCb (Fl_Widget*, void* aPtr)
{
  vtkAbstractToolBarUI* gui;
  Fl_VTK_View*          view;
  Fl_Toggle_Tree*       toggleTree;
  Fl_Toggle_Node*       selectedNode;
  vtkRenderer*          renderer;
  vtkAssembly*          assembly;
  vtkActor*             actor;
  vtkObject*            object;
  vtkProp*              prop;
  vtkProp3D*            prop3D;

  if ((gui = reinterpret_cast<vtkAbstractToolBarUI*>(aPtr)) == NULL)
    return;
  if ((view = gui->GetView()) == NULL)
    return;
  if ((toggleTree = gui->GetNodeBrowser()) == NULL)
    return;

#ifdef SVV_SUPPORT_MRML
  vtkMrml* mrmlImpl = vtkMrmlRegistry::GetImplementation(); 
  vtkMrmlNode* mrmlNode;
#endif /* SVV_SUPPORT_MRML */

  // If multiple nodes are selected, then selected() returns 0,
  // and selection() should be checked.
  if ((selectedNode = toggleTree->selected_node()) != NULL)
    {
    if ((prop = sgGetNodeProp(toggleTree, selectedNode)) != NULL)
      {
      vtkActor*    actor;

      if ((actor = sgGetNodeActor(toggleTree, selectedNode)) != NULL)
        {
        vtkUpstream*    upstream = vtkUpstream::New();
        vtkInputSource* inputSource;
        vtkSource*      source;

        upstream->SetSource(actor);

        if ((source = upstream->GetSource()) != NULL)
          {
          if ((inputSource = vtkInputSource::SafeDownCast(source)) != NULL)
            {
            svv.RemoveSurfaceDataSource(inputSource);
            }
          }
        else
          {
          while ((source = upstream->GetInputSource()) != NULL)
            {
            if ((inputSource = vtkInputSource::SafeDownCast(source)) != NULL)
              {
              svv.RemoveSurfaceDataSource(inputSource);
              break;
              }
            upstream->SetSource(source);
            }
          }
        // This is a no-op if not an actor subclass.
        svv.RemoveSurfaceActor(vtkActor::SafeDownCast(prop));
        }
      view->RemoveProp(prop);
#ifdef SVV_SUPPORT_MRML
      if ((mrmlNode = mrmlImpl->GetNodeByID(SVV_PTR_TO_INT32(prop))) != NULL)
        {
        mrmlImpl->DeleteNode(mrmlNode);
        }
#endif /* SVV_SUPPORT_MRML */
      }
    svvInfo(<< "removing: " << selectedNode->label());
    svvStatus(<< "removing: " << selectedNode->label());

    toggleTree->remove(selectedNode);
    }
  else
    {
    Fl_Toggle_Node* selection;

    while ((selection = toggleTree->selection_next()) != NULL)
      {
      if ((prop = sgGetNodeProp(toggleTree, selection)) != NULL)
        {
        vtkActor*    actor;

        if ((actor = sgGetNodeActor(toggleTree, selection)) != NULL)
          {
          vtkUpstream*    upstream = vtkUpstream::New();
          vtkInputSource* inputSource;
          vtkSource*      source;

          upstream->SetSource(actor);

          if ((source = upstream->GetSource()) != NULL)
            {
            if ((inputSource = vtkInputSource::SafeDownCast(source)) != NULL)
              {
              svv.RemoveSurfaceDataSource(inputSource);
              }
            }
          else
            {
            while ((source = upstream->GetInputSource()) != NULL)
              {
              if ((inputSource = vtkInputSource::SafeDownCast(source)) != NULL)
                {
                svv.RemoveSurfaceDataSource(inputSource);
                break;
                }
              upstream->SetSource(source);
              }
            }
          // This is a no-op if not an actor subclass.
          svv.RemoveSurfaceActor(vtkActor::SafeDownCast(prop));
          }
        view->RemoveProp(prop);
#ifdef SVV_SUPPORT_MRML
        if ((mrmlNode = mrmlImpl->GetNodeByID(SVV_PTR_TO_INT32(prop))) != NULL)
          {
          mrmlImpl->DeleteNode(mrmlNode);
          }
#endif /* SVV_SUPPORT_MRML */
        }
      svvInfo(<< "removing: " << selection->label());
      svvStatus(<< "removing: " << selection->label());

      toggleTree->remove(selection);
      }
    }

  toggleTree->redraw();
  view->Update();
}

void
SvvNodeRemoveCb (Fl_Button* aButton, void* aPtr)
{ SvvNodeRemoveCb(static_cast<Fl_Widget*>(aButton), aPtr); }

void
SvvNodeRemoveCb (Fl_Menu_* aMenu, void*)
{ SvvNodeRemoveCb(static_cast<Fl_Widget*>(aMenu), aMenu->user_data()); }

// ----------------------------------------------------------------------------
void
SvvNodeBrowserBrowserCb (Fl_Toggle_Tree* aTree, void* aPtr)
{
  if (aTree == NULL)
    return;

  vtkRenderingContext* renderingContext = NULL;
  vtkAbstractUI* gui;

  if ((gui = reinterpret_cast<vtkAbstractUI*>(aPtr)) != NULL)
    {
    renderingContext = gui->GetClientRenderingContext();
    }

  switch (aTree->state())
    {

    case FL_TOGGLE_NONE:
      {
      svvStatus(<< "state: FL_TOGGLE_NONE");

      if (renderingContext != NULL)
        {
        //renderingContext->GetCurrentRenderer()->PickProp(-1,-1);
	//renderingContext->SetCurrentRenderer(0);
	renderingContext->SetCurrentAssembly(0);
	renderingContext->SetCurrentProp(0);
        }

      } break;

    case FL_TOGGLE_SELECT:
      {
      svvStatus(<< "state: FL_TOGGLE_SELECT");

      Fl_Toggle_Node* selectedNode;
      vtkProp* prop;

      if (renderingContext)
        {
	if ((selectedNode = aTree->selected_node()) != NULL)
          {
	  if (prop = reinterpret_cast<vtkProp*>(selectedNode->user_data()))
            {
	    renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
	else
          {
	  Fl_Toggle_Node* selection;
	  Fl_Toggle_Node* last;

	  while ((selection = aTree->selection_next()) != NULL)
            last = selection;

          if ((prop = reinterpret_cast<vtkProp*>(last->user_data())) != NULL)
            {
            renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
        }

      } break;

    case FL_TOGGLE_RESELECT:
      {
      svvStatus(<< "state: FL_TOGGLE_RESELECT");

      Fl_Toggle_Node* selectedNode;
      vtkProp*        prop;

      if (renderingContext != NULL)
        {
	if ((selectedNode = aTree->selected_node()) != NULL)
          {
          if ( (prop = reinterpret_cast<vtkProp*>(selectedNode->user_data()))
               != NULL )
            {
	    renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
	else
          {
	  Fl_Toggle_Node* selection;
	  Fl_Toggle_Node* last;

	  while ((selection = aTree->selection_next()) != NULL)
            last = selection;

          if ((prop = reinterpret_cast<vtkProp*>(last->user_data())) != NULL)
            {
            renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
        }

      } break;

    case FL_TOGGLE_SELECT_MASK:
      {
      svvStatus(<< "state: FL_TOGGLE_SELECT_MASK");
      break;
      }

    case FL_TOGGLE_OPENED:
      {
      svvStatus(<< "state: FL_TOGGLE_OPENED");
      break;
      }

    case FL_TOGGLE_CLOSED:
      {
      svvStatus(<< "state: FL_TOGGLE_CLOSED");
      break;
      }

    case FL_TOGGLE_HIT:
      {
      svvStatus(<< "state: FL_TOGGLE_HIT");

      Fl_Toggle_Node* selectedNode;
      vtkProp*        prop;

      if (renderingContext != NULL)
        {
	if ((selectedNode = aTree->selected_node()) != NULL)
          {
	  if ( (prop = reinterpret_cast<vtkProp*>(selectedNode->user_data()))
               != NULL )
            {
	    renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
	else
          {
	  Fl_Toggle_Node* selection;
	  Fl_Toggle_Node* last;

	  while ((selection = aTree->selection_next()) != NULL)
            last = selection;

          if ((prop = reinterpret_cast<vtkProp*>(last->user_data())) != NULL)
            {
            renderingContext->SetCurrentProp(prop);
            //prop->Pick();
            }
          }
        }

      } break;

    } // switch (aTree->state())

  Fl_Toggle_Node* s;

  if ((s = aTree->selected_node()) != NULL)
    {
    svvStatus(<< "selected: " << s->label());
    //aTree->Print(vtkstd::clog);
    }
  else
    {
    // if selected_node()==NULL then we have a multiple selections
    svvDebug(<< "selected MULTIPLE ...\n");
    //aTree->Print(vtkstd::clog);

    int n = aTree->selection_count();

    svvStatus(<< n << " selections");
    svvDebug(<< n << " selections: ");

    for (int i=0; i<n; i++)
      {
      s = aTree->selection_node(i);
      svvDebug(<< "\t" << i << " " << s->label());
      }
    }
}

// ----------------------------------------------------------------------------
void
SvvProcessMouseEvents (vtkObject*    aObject,
                       unsigned long aEvent,
                       void*         aClientData, 
                       void*         aCallData)
{
  vtkInteractorStyle* style =
    reinterpret_cast<vtkInteractorStyle*>(aClientData);

  vtkstd::clog << "SvvProcessMouseEvents(" << aObject << ", " << aEvent
               << ", " << aClientData << ", " << aCallData << ")"
               << endl;

  switch (aEvent)
    {
    case vtkCommand::MouseMoveEvent: 
      break;
    case vtkCommand::LeftButtonPressEvent:
      if (style->GetInteractor()->GetShiftKey()) style->OnLeftButtonDown();
      break;
    case vtkCommand::LeftButtonReleaseEvent:
      break;
    case vtkCommand::MiddleButtonPressEvent:
      break;
    case vtkCommand::MiddleButtonReleaseEvent:
      break;
    case vtkCommand::RightButtonPressEvent:
      break;
    case vtkCommand::RightButtonReleaseEvent: 
      break;
    } // switch (aEvent)
}

void
SvvProcessKeyEvents (vtkObject*    aObject, 
                     unsigned long aEvent,
                     void*         aClientData, 
                     void*         aCallData)
{
  vtkInteractorObserver* style 
    = reinterpret_cast<vtkInteractorObserver*>(aClientData);

  vtkstd::clog << "SvvProcessKeyEvents(" << aObject << ", " << aEvent
               << ", " << aClientData << ", " << aCallData << ")"
               << endl;

  switch (aEvent)
    {
    case vtkCommand::KeyPressEvent:
      break;
    case vtkCommand::KeyReleaseEvent: 
      break;
    case vtkCommand::CharEvent:  
      break;
    } // switch (aEvent)

  vtkstd::clog << "" << endl;
}

// ----------------------------------------------------------------------------
void
SvvNodeBrowserStatusCb (Fl_Progress*, void*)
{
}

// ----------------------------------------------------------------------------
void
SvvPaletteCb (Fl_Button*, void*)
{
}

/* 
 * End of: $Id: vtkTreeCb.cxx,v 1.1.1.1 2006/12/19 22:59:51 christianh Exp $.
 * 
 */
