/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkSessionExporter.cxx,v 1.1.1.1 2006/12/19 23:00:08 christianh Exp $
 * 
 */
#include "vtkSessionExporter.h"
#include <sys/stat.h>  // stat()
#include <unistd.h>    // stat() and access()
#include "vtkByteSwap.h"
#include "vtkBase64OutputStream.h"
#include "vtkDataCompressor.h"
#include "vtkAssemblyNode.h"
#include "vtkCamera.h"
#include "vtkGeometryFilter.h"
#include "vtkImageData.h"
#include "vtkLight.h"
#include "vtkLightCollection.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkRenderWindow.h"
#include "vtkRendererCollection.h"
#include "vtkTexture.h"
#include "vtkTransform.h"
#include "vtkUnsignedCharArray.h"


#define VTK_INDENT_MORE         this->Indent += 4
#define VTK_INDENT_LESS         this->Indent -= 4


// ----------------------------------------------------------------------------
vtkCxxRevisionMacro(vtkSessionExporter, "$Revision: 1.1.1.1 $");
vtkStandardNewMacro(vtkSessionExporter);
vtkCxxSetObjectMacro(vtkSessionExporter, Compressor, vtkDataCompressor);
vtkCxxSetObjectMacro(vtkSessionExporter, DataStream, vtkOutputStream);

// ----------------------------------------------------------------------------
vtkSessionExporter::vtkSessionExporter(void)
{
  this->RenderingContext = 0;

  this->StartWrite          = 0;
  this->StartWriteArgDelete = 0;
  this->StartWriteArg       = 0;
  this->EndWrite            = 0;
  this->EndWriteArgDelete   = 0;
  this->EndWriteArg         = 0;

  this->FileName   = 0;
  this->FilePrefix = 0;
  this->SetFileInfix(".");
  this->SetFileSuffix("svs");
  this->BaseName   = 0;

  this->Stream     = 0;
  this->FileStream = 0;

  // Default binary data mode is base-64 encoding.
  this->DataStream = vtkBase64OutputStream::New();
  
  // Byte order defaults to that of machine.
#ifdef VTK_WORDS_BIGENDIAN
  this->BigEndian = 1;
#else
  this->BigEndian = 0;
#endif  

  // Initialize compression data.
  this->BlockSize         = 32768;
  this->Compressor        = 0;
  this->CompressionHeader = 0;
}

// ----------------------------------------------------------------------------
vtkSessionExporter::~vtkSessionExporter(void)
{
  this->SetRenderingContext(0);
  
  if ((this->StartWriteArg) && (this->StartWriteArgDelete))
    {
    (*this->StartWriteArgDelete)(this->StartWriteArg);
    }
  if ((this->EndWriteArg) && (this->EndWriteArgDelete))
    {
    (*this->EndWriteArgDelete)(this->EndWriteArg);
    }

  this->SetFileName(0);
  this->SetFilePrefix(0);
  this->SetFileInfix(0);
  this->SetFileSuffix(0);
  this->SetBaseName(0);
  if (this->FileStream) delete this->FileStream;
  this->DataStream->Delete();
  this->SetCompressor(0);
}

//----------------------------------------------------------------------------
void
vtkSessionExporter::SetByteOrderToBigEndian(void)
{
  this->SetBigEndian(1);
}

void
vtkSessionExporter::SetByteOrderToLittleEndian(void)
{
  this->SetBigEndian(0);
}


// ----------------------------------------------------------------------------
void
vtkSessionExporter::Write(void)
{
  // Write data to output. Method executes subclasses WriteData() method, as 
  // well as StartWrite() and EndWrite() methods.

  if (!this->IsSafeToWrite())
    {
    return;
    }

  if (this->StartWrite)
    {
    (*this->StartWrite)(this->StartWriteArg);
    }

  this->WriteData();

  if (this->EndWrite)
    {
    (*this->EndWrite)(this->EndWriteArg);
    }
}

void
vtkSessionExporter::Update(void)
{
  this->Write();
}

// ----------------------------------------------------------------------------
void
vtkSessionExporter::SetStartWriteArgDelete(void (*f)(void*))
{
  if (f != this->StartWriteArgDelete)
    {
    this->StartWriteArgDelete = f;
    this->Modified();
    }
}

void
vtkSessionExporter::SetStartWrite(void (*f)(void*), void* arg)
{
  if (f != this->StartWrite)
    {
    // delete the current arg if there is one and a delete meth
    if ((this->StartWriteArg) && (this->StartWriteArgDelete))
      {
      (*this->StartWriteArgDelete)(this->StartWriteArg);
      }
    this->StartWrite    = f;
    this->StartWriteArg = arg;
    this->Modified();
    }
}

void
vtkSessionExporter::SetEndWriteArgDelete(void (*f)(void*))
{
  if (f != this->EndWriteArgDelete)
    {
    this->EndWriteArgDelete = f;
    this->Modified();
    }
}

void
vtkSessionExporter::SetEndWrite(void (*f)(void*), void* arg)
{
  if (f != this->EndWrite)
    {
    // delete the current arg if there is one and a delete meth
    if ((this->EndWriteArg) && (this->EndWriteArgDelete))
      {
      (*this->EndWriteArgDelete)(this->EndWriteArg);
      }
    this->EndWrite    = f;
    this->EndWriteArg = arg;
    this->Modified();
    }
}

#if 0
// ----------------------------------------------------------------------------
void
vtkSessionExporter::WriteData(void)
{
  vtkDebugMacro(<< "Commencing XML export.");

  // Make sure that the output stream is ready for writing.
  if (!this->IsSafeToWrite())
    {
    return;
    }

  vtkRenderer* ren;
  FILE* fp;
  vtkActorCollection* ac;
  vtkActor* anActor;
  vtkActor* aPart;
  vtkLightCollection* lc;
  vtkLight* aLight;
  vtkCamera* cam;
  float* tempf;

  // make sure the user specified a filename
  if (!this->BaseName)
    {
    vtkErrorMacro(<< "Please specify BaseName to use");
    return;
    }
  
  // first make sure there is only one renderer in this rendering window
  if (this->RenderWindow->GetRenderers()->GetNumberOfItems() > 1)
    {
    vtkErrorMacro(<< "Only one renderer per window supported.");
    return;
    }

  // get the renderer
  this->RenderWindow->GetRenderers()->InitTraversal();
  ren = this->RenderWindow->GetRenderers()->GetNextItem();
  
  // make sure it has at least one actor
  if (ren->GetActors()->GetNumberOfItems() < 1)
    {
    vtkErrorMacro(<< "No actors found for writing file.");
    return;
    }
    
  // try opening the files
  if (!(fp = fopen(this->BaseName,"w")))
    {
    vtkErrorMacro(<< "Unable to open file " << this->BaseName);
    return;
    }
  
  //
  //  Write header
  //
  vtkDebugMacro(<< "Writing OpenInventor file");
  fprintf(fp,"#Inventor V2.0 ascii\n");
  fprintf(fp,"# OpenInventor file written by the visualization toolkit\n\n");

  fprintf(fp, "Separator {\n");
  VTK_INDENT_MORE;

  // do the camera
  cam = ren->GetActiveCamera();
  if (cam->GetParallelProjection()) 
    {
    fprintf(fp,"%sOrthographicCamera\n%s{\n", indent, indent);
    }
  else
    {
    // this assumes the aspect ratio is 1
    fprintf(fp,"%sPerspectiveCamera\n%s{\n%s    heightAngle %f\n",
            indent, indent, indent,
            cam->GetViewAngle()*3.1415926/180.0);
    }
  VTK_INDENT_MORE;
  fprintf(fp,"%snearDistance %f\n",indent, cam->GetClippingRange()[0]);
  fprintf(fp,"%sfarDistance %f\n",indent, cam->GetClippingRange()[1]);
  fprintf(fp,"%sfocalDistance %f\n",indent, cam->GetDistance());
  fprintf(fp,"%sposition %f %f %f\n", indent, cam->GetPosition()[0],
          cam->GetPosition()[1], cam->GetPosition()[2]);
  tempf = cam->GetOrientationWXYZ();
  fprintf(fp,"%sorientation %g %g %g %g\n%s}\n", indent,
          tempf[1], tempf[2], tempf[3], tempf[0]*3.1415926/180.0, indent);
  VTK_INDENT_LESS;

  // do the lights first the ambient then the others
  fprintf(fp,"# The following environment information is disabled\n");
  fprintf(fp,"# because a popular viewer (Template Graphics Software SceneViewer) has\n");
  fprintf(fp,"# trouble (access violations under Windows NT) with it.\n");
  fprintf(fp,"#%sEnvironment {\n", indent);
  // couldn't figure out a way to do headlight -- seems to be a property of the
  // viewer not the model
  VTK_INDENT_MORE;
  fprintf(fp,"#%sambientIntensity 1.0 # ambient light\n", indent);
  fprintf(fp,"#%sambientColor %f %f %f }\n\n", indent,
          ren->GetAmbient()[0], ren->GetAmbient()[1], ren->GetAmbient()[2]);
  VTK_INDENT_LESS;

  // make sure we have a default light
  // if we dont then use a headlight
  lc = ren->GetLights();
  for (lc->InitTraversal(); (aLight = lc->GetNextItem()); )
    {
    this->WriteALight(aLight, fp);
    }
  
  // do the actors now
  ac = ren->GetActors();
  vtkAssemblyPath *apath;
  for (ac->InitTraversal(); (anActor = ac->GetNextActor()); )
    {
    for (anActor->InitPathTraversal(); (apath=anActor->GetNextPath()); )
      {
      aPart=(vtkActor *)apath->GetLastNode()->GetProp();
      this->WriteAnActor(aPart, fp);
      }
    }
  
  VTK_INDENT_LESS;
  fprintf(fp, "}\n"); // close Separator
  
  fclose(fp);
}

void vtkSessionExporter::WriteALight(vtkLight *aLight, FILE *fp)
{
  float *pos, *focus, *color;
  float dir[3];
  
  pos = aLight->GetPosition();
  focus = aLight->GetFocalPoint();
  color = aLight->GetColor();

  dir[0] = focus[0] - pos[0];
  dir[1] = focus[1] - pos[1];
  dir[2] = focus[2] - pos[2];
  vtkMath::Normalize(dir);
    
  if (aLight->GetPositional())
    {
    float *attn;
    
    if (aLight->GetConeAngle() >= 180.0)
      {
      fprintf(fp,"%sPointLight {\n", indent);
      VTK_INDENT_MORE;
      }
    else
      { 
      fprintf(fp,"%sSpotLight {\n", indent);
      VTK_INDENT_MORE;
      fprintf(fp,"%sdirection %f %f %f\n", indent, dir[0], dir[1], dir[2]);
      fprintf(fp,"%scutOffAngle %f\n", indent, aLight->GetConeAngle());
      // the following ignores linear and quadratic attenuation values
      attn = aLight->GetAttenuationValues();
      fprintf(fp,"%sdropOffRate %f\n", indent, attn[0]);
      }
    fprintf(fp,"%slocation %f %f %f\n", indent, pos[0], pos[1], pos[2]);
    }
  else
    {
    fprintf(fp,"%sDirectionalLight {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp,"%sdirection %f %f %f\n", indent, dir[0], dir[1], dir[2]);
    }
  
  fprintf(fp,"%scolor %f %f %f\n", indent, color[0], color[1], color[2]);
  fprintf(fp,"%sintensity %f\n", indent, aLight->GetIntensity());
  if (aLight->GetSwitch())
    {
    fprintf(fp,"%son TRUE\n%s}\n", indent, indent);
    }
  else
    {
    fprintf(fp,"%son FALSE\n%s}\n", indent, indent);
    }
  VTK_INDENT_LESS;
}

void vtkSessionExporter::WriteAnActor(vtkActor *anActor, FILE *fp)
{
  vtkDataSet* ds;
  vtkPolyData* pd;
  vtkGeometryFilter* gf = 0;
  vtkPointData* pntData;
  vtkPoints* points = 0;
  vtkDataArray* normals = 0;
  vtkDataArray* tcoords = 0;
  int i;
  vtkProperty* prop;
  float* tempf;
  vtkCellArray* cells;
  vtkIdType npts = 0;
  vtkIdType* indx = 0;
  float tempf2;
  vtkPolyDataMapper* pm;
  vtkUnsignedCharArray* colors;
  float* p;
  unsigned char* c;
  vtkTransform* trans;
  
  // see if the actor has a mapper. it could be an assembly
  if (!anActor->GetMapper())
    {
    return;
    }

  fprintf(fp,"%sSeparator {\n", indent);
  VTK_INDENT_MORE;

  // first stuff out the transform
  trans = vtkTransform::New();
  trans->SetMatrix(anActor->vtkProp3D::GetMatrix());
  
  fprintf(fp,"%sTransform {\n", indent);
  VTK_INDENT_MORE;
  tempf = trans->GetPosition();
  fprintf(fp,"%stranslation %g %g %g\n", indent, tempf[0], tempf[1], tempf[2]);
  tempf = trans->GetOrientationWXYZ();
  fprintf(fp,"%srotation %g %g %g %g\n", indent, tempf[1], tempf[2], 
          tempf[3], tempf[0]*3.1415926/180.0);
  tempf = trans->GetScale();
  fprintf(fp,"%sscaleFactor %g %g %g\n", indent, tempf[0], tempf[1], tempf[2]);
  fprintf(fp,"%s}\n", indent);
  VTK_INDENT_LESS;
  trans->Delete();
  
  // get the mappers input and matrix
  ds = anActor->GetMapper()->GetInput();
  
  // we really want polydata
  if (ds->GetDataObjectType() != VTK_POLY_DATA)
    {
    gf = vtkGeometryFilter::New();
    gf->SetInput(ds);
    gf->Update();
    pd = gf->GetOutput();
    }
  else
    {
    ds->Update();
    pd = (vtkPolyData *)ds;
    }

  pm = vtkPolyDataMapper::New();
  pm->SetInput(pd);
  pm->SetScalarRange(anActor->GetMapper()->GetScalarRange());
  pm->SetScalarVisibility(anActor->GetMapper()->GetScalarVisibility());
  pm->SetLookupTable(anActor->GetMapper()->GetLookupTable());

  points = pd->GetPoints();
  pntData = pd->GetPointData();
  normals = pntData->GetNormals();
  tcoords = pntData->GetTCoords();
  colors  = pm->MapScalars(1.0);
  
  fprintf(fp,"%sMaterial {\n", indent);
  VTK_INDENT_MORE;
  
  // write out the material properties to the mat file
  prop = anActor->GetProperty();
  // the following is based on a guess about how VTK's GetAmbient
  // property corresponds to SoMaterial's ambientColor
  tempf2 = prop->GetAmbient();
  tempf = prop->GetAmbientColor();
  fprintf(fp,"%sambientColor %g %g %g\n", indent,
          tempf[0]*tempf2, tempf[1]*tempf2, tempf[2]*tempf2);
  tempf2 = prop->GetDiffuse();
  tempf = prop->GetDiffuseColor();
  fprintf(fp,"%sdiffuseColor %g %g %g\n", indent,
          tempf[0]*tempf2, tempf[1]*tempf2, tempf[2]*tempf2);
  tempf2 = prop->GetSpecular();
  tempf = prop->GetSpecularColor();
  fprintf(fp,"%sspecularColor %g %g %g\n", indent,
          tempf[0]*tempf2, tempf[1]*tempf2, tempf[2]*tempf2);
  fprintf(fp,"%sshininess %g\n", indent,prop->GetSpecularPower()/128.0);
  fprintf(fp,"%stransparency %g\n", indent,1.0 - prop->GetOpacity());
  fprintf(fp,"%s}\n", indent); // close matrial
  VTK_INDENT_LESS;

  // is there a texture map
  if (anActor->GetTexture())
    {
    vtkTexture *aTexture = anActor->GetTexture();
    int* size;
    int  xsize, ysize, bpp;
    vtkDataArray* scalars;
    vtkUnsignedCharArray* mappedScalars;
    unsigned char* txtrData;
    int totalValues;
    
    // make sure it is updated and then get some info
    if (!aTexture->GetInput())
      {
      vtkErrorMacro(<< "texture has no input!\n");
      return;
      }
    aTexture->GetInput()->Update();
    size = aTexture->GetInput()->GetDimensions();
    scalars = aTexture->GetInput()->GetPointData()->GetScalars();

    // make sure scalars are non null
    if (!scalars)
      {
      vtkErrorMacro(<< "No scalar values found for texture input!\n");
      return;
      }

    // make sure using unsigned char data of color scalars type
    if ( aTexture->GetMapColorScalarsThroughLookupTable() ||
        (scalars->GetDataType() != VTK_UNSIGNED_CHAR) )
      {
      mappedScalars = aTexture->GetMappedScalars ();
      }
    else
      {
      mappedScalars = static_cast<vtkUnsignedCharArray*>(scalars);
      }

    // we only support 2d texture maps right now
    // so one of the three sizes must be 1, but it 
    // could be any of them, so lets find it
    if (size[0] == 1)
      {
      xsize = size[1];
      ysize = size[2];
      }
    else
      {
      xsize = size[0];
      if (size[1] == 1)
        {
        ysize = size[2];
        }
      else
        {
        ysize = size[1];
        if (size[2] != 1)
          {
          vtkErrorMacro(<< "3D texture maps currently are not supported!\n");
          return;
          }
        }
      }

    fprintf(fp, "%sTexture2 {\n", indent);
    VTK_INDENT_MORE;
    bpp = mappedScalars->GetNumberOfComponents();
    fprintf(fp, "%simage %d %d %d\n", indent, xsize, ysize, bpp);
    VTK_INDENT_MORE;
    txtrData = static_cast<vtkUnsignedCharArray *>(mappedScalars)->GetPointer(0);
    totalValues = xsize*ysize;
    fprintf(fp,"%s",indent);
    for (i = 0; i < totalValues; i++)
      {
      fprintf(fp,"%.2x",*txtrData);
      txtrData++;
      if (bpp > 1)
        {
        fprintf(fp,"%.2x",*txtrData);
        txtrData++;
        }
      if (bpp > 2)
        {
        fprintf(fp,"%.2x",*txtrData);
        txtrData++;
        }
      if (bpp > 3)
        {
        fprintf(fp,"%.2x",*txtrData);
        txtrData++;
        }
      if (i%8 == 0)
        {
        fprintf(fp,"\n%s    ", indent);
        }
      else
        {
        fprintf(fp," ");
        }
      }
    VTK_INDENT_LESS;
    fprintf(fp, "%s}\n", indent);
    VTK_INDENT_LESS;
    }

  // write out point data if any
  this->WritePointData(points, normals, tcoords, colors, fp);

  // write out polys if any
  if (pd->GetNumberOfPolys() > 0)
    {
    fprintf(fp,"%sIndexedFaceSet {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp,"%scoordIndex  [\n", indent);
    VTK_INDENT_MORE;
    
    cells = pd->GetPolys();
    for (cells->InitTraversal(); cells->GetNextCell(npts,indx); )
      {
      fprintf(fp,"%s", indent);
      for (i = 0; i < npts; i++)
        {
        // treating vtkIdType as int
        fprintf(fp,"%i, ",(int)indx[i]);
        if (((i+1)%10) == 0)
          {
          fprintf(fp, "\n%s    ", indent);
          }
        }
      fprintf(fp,"-1,\n");
      }
    fprintf(fp,"%s]\n", indent);
    VTK_INDENT_LESS;
    fprintf(fp,"%s}\n", indent);
    VTK_INDENT_LESS;
    }

  // write out tstrips if any
  if (pd->GetNumberOfStrips() > 0)
    {
    fprintf(fp,"%sIndexedTriangleStripSet {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp,"%scoordIndex  [\n", indent);
    VTK_INDENT_MORE;
    cells = pd->GetStrips();
    for (cells->InitTraversal(); cells->GetNextCell(npts,indx); )
      {
      fprintf(fp,"%s", indent);
      for (i = 0; i < npts; i++)
        {
        // treating vtkIdType as int
        fprintf(fp,"%i, ", (int)indx[i]);
        if (((i+1)%10) == 0)
          {
          fprintf(fp, "\n%s    ", indent);
          }
        }
      fprintf(fp,"-1,\n");
      }
    fprintf(fp,"%s]\n", indent);
    VTK_INDENT_LESS;
    fprintf(fp,"%s}\n", indent);
    VTK_INDENT_LESS;
    }
  
  // write out lines if any
  if (pd->GetNumberOfLines() > 0)
    {
    fprintf(fp,"%sIndexedLineSet {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp,"%scoordIndex  [\n", indent);
    VTK_INDENT_MORE;
    cells = pd->GetLines();
    for (cells->InitTraversal(); cells->GetNextCell(npts,indx); )
      {
      fprintf(fp,"%s", indent);
      for (i = 0; i < npts; i++)
        {
        // treating vtkIdType as int
        fprintf(fp,"%i, ", (int)indx[i]);
        if (((i+1)%10) == 0)
          {
          fprintf(fp, "\n%s    ", indent);
          }
        }
      fprintf(fp,"-1,\n");
      }
    fprintf(fp,"%s]\n", indent);
    VTK_INDENT_LESS;
    fprintf(fp,"%s}\n", indent);
    VTK_INDENT_LESS;
    }
  
  // write out verts if any
  // (more complex because there is no IndexedPointSet)
  if (pd->GetNumberOfVerts() > 0)
    {
    fprintf(fp, "%sSeparator {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp, "%sCoordinate3 {\n", indent);
    VTK_INDENT_MORE;
    fprintf(fp,"%spoint [", indent);
    VTK_INDENT_MORE;
    cells = pd->GetVerts();
    for (cells->InitTraversal(); cells->GetNextCell(npts,indx); )
      {
      for (i = 0; i < npts; i++)
        {
        p = points->GetPoint(indx[i]);
        fprintf (fp,"%s%g %g %g,\n", indent, p[0], p[1], p[2]);
        }
      }
    fprintf(fp,"%s]\n", indent);
    VTK_INDENT_LESS;
    fprintf(fp,"%s}\n", indent);
    VTK_INDENT_LESS;
    if (colors)
      {
      fprintf(fp,"%sPackedColor {", indent);
      VTK_INDENT_MORE;
      fprintf(fp,"%srgba [\n", indent);
      VTK_INDENT_MORE;
      for (cells->InitTraversal(); cells->GetNextCell(npts,indx); )
        {
        fprintf(fp,"%s", indent);
        for (i = 0; i < npts; i++)
          {
          c = colors->GetPointer(4*indx[i]);
          fprintf( fp,"%#lx, ", 
                   ((unsigned long)c[3] << 24) |
                   (((unsigned long)c[2])<<16) |
                   (((unsigned long)c[1])<<8) |
                   ((unsigned long)c[0]) );

          if (((i+1)%5) == 0)
            {
            fprintf(fp, "\n%s", indent);
            }
          }
        }
      fprintf(fp,"\n%s]\n", indent);
      VTK_INDENT_LESS;
      fprintf(fp,"%s}\n", indent);
      VTK_INDENT_LESS;
      fprintf(fp,"%sMaterialBinding { value PER_VERTEX_INDEXED }\n", indent);
      }
    
    fprintf(fp, "%sPointSet {\n", indent);
    VTK_INDENT_MORE;
    // treating vtkIdType as int
    fprintf(fp, "%snumPoints %d\n", indent, (int)npts);
    VTK_INDENT_MORE;
    fprintf(fp, "%s}\n", indent);
    VTK_INDENT_LESS;
    fprintf(fp,"%s}\n", indent); // close the Separator
    VTK_INDENT_LESS;
    }
  fprintf(fp, "%s}\n", indent);
  VTK_INDENT_LESS;
  if (gf)
    {
    gf->Delete();
    }
  pm->Delete();
}

void
vtkSessionExporter::WritePointData(vtkPoints* points,
                                   vtkDataArray* normals,
                                   vtkDataArray* tcoords, 
                                   vtkUnsignedCharArray* colors,
                                   std::ostream& os)
{
  float* p;
  int i;
  unsigned char* c;
  
  // write out the points
  os << this->Indent << "Coordinate3 {\n";
  VTK_INDENT_MORE;
  os << this->Indent << "point [\n";
  VTK_INDENT_MORE;
  for (i=0; i<points->GetNumberOfPoints(); i++)
    {
    p = points->GetPoint(i);
    os << this->Indent
       << p[0] << " " << p[1] << " " << p[2] << ",\n";
    }
  os << this->Indent << "]\n";
  VTK_INDENT_LESS;
  os << this->Indent << "}\n";
  VTK_INDENT_LESS;
  
  // write out the point data
  if (normals)
    {
    os << this->Indent << "Normal {\n";
    VTK_INDENT_MORE;
    os << this->Indent << "vector [\n";
    VTK_INDENT_MORE;
    for (i=0; i<normals->GetNumberOfTuples(); i++)
      {
      p = normals->GetTuple(i);
      os << this->Indent
         << p[0] << " " << p[1] << " " << p[2] << ",\n";
      }
    os << this->Indent << "]\n";
    VTK_INDENT_LESS;
    os << this->Indent << "}\n";
    VTK_INDENT_LESS;
    }

  // write out the point data
  if (tcoords)
    {
    os << this->Indent << "TextureCoordinateBinding  {\n";
    VTK_INDENT_MORE;
    os << this->Indent << "value PER_VERTEX_INDEXED\n";
    VTK_INDENT_LESS;
    os << this->Indent << "}\n";
    os << this->Indent << "TextureCoordinate2 {\n";
    VTK_INDENT_MORE;
    os << this->Indent << "point [\n";
    VTK_INDENT_MORE;
    for (i=0; i<tcoords->GetNumberOfTuples(); i++)
      {
      p = tcoords->GetTuple(i);
      os << this->Indent << p[0] << " " << p[1] << ",\n";
      }
    os << this->Indent << "]\n";
    VTK_INDENT_LESS;
    os << this->Indent << "}\n";
    VTK_INDENT_LESS;
    }
  
  // write out the point data
  if (colors)
    {
    os << this->Indent << "PackedColor {\n";
    VTK_INDENT_MORE;
    os << this->Indent << "rgba [\n";
    VTK_INDENT_MORE;
    os << this->Indent;
    for (i=0; i<colors->GetNumberOfTuples(); i++)
      {
      c = colors->GetPointer(4*i);
      fprintf( fp,"%#lx, ", 
               ( ((unsigned long)c[3] << 24) |
                 (((unsigned long)c[2])<<16) |
                 (((unsigned long)c[1])<<8) |
                 ((unsigned long)c[0]) ) );

      if (((i+1)%5)==0)
        {
        os << "\n" << this->Indent;
        }
      }
    os << "\n" << this->Indent << "]\n";
    VTK_INDENT_LESS;
    os << this->Indent << "}\n";
    VTK_INDENT_LESS;
    os << this->Indent
       << "MaterialBinding { value PER_VERTEX_INDEXED }\n";
    }
}
#endif /* 0 */

// ----------------------------------------------------------------------------
int
vtkSessionExporter::IsSafeToWrite(void)
{
  // Make sure input is available.
  if (!this->RenderingContext)
    {
    vtkErrorMacro(<< "No rendering context provided!");
    return 0;
    }

  return this->CheckStream();
}

int
vtkSessionExporter::CheckStream(void)
{
  // Make sure that there is a writable stream.
  if (!this->Stream)
    {

    if (this->FilePrefix)
      {
      std::string pathName;
      
      pathName.assign(this->FilePrefix);
      if (this->BaseName && ((this->FilePrefix)[0] != '/'))
        {
        if (this->BaseName[strlen(this->BaseName)-1] != '/')
          {
          pathName.insert(0,"/");
          }
        pathName.insert(0,this->BaseName);
        }
      pathName.append(this->FileInfix);
      pathName.append(this->FileSuffix);
      this->SetFileName(pathName.c_str());
      }
    else if (this->FileName)
      {
      if (this->BaseName && ((this->FileName)[0] != '/'))
        {
        std::string pathName;
        
        pathName.assign(this->BaseName);
        if (this->BaseName[strlen(this->BaseName)-1] != '/')
          {
          pathName.append("/");
          }
        pathName.append(this->FileName);
        this->SetFileName(pathName.c_str());
        }
      }
    else
      {
      vtkErrorMacro("Neither Stream nor FileName or FilePrefix set.");
      return 0;
      }
    
    struct stat fs;
    
    if (stat(this->FileName, &fs) != 0)
      {
      vtkErrorMacro("Cannot open output file: " << this->FileName
                    << "( "<< strerror(errno) << " )");
      return 0;
      }
    else if (access(this->FileName,W_OK))
      {
      vtkErrorMacro("Output file is not writable: " << this->FileName);
      return 0;
      }
    else if (S_ISDIR(fs.st_mode))
      {
      vtkErrorMacro("Output pathname is a directory: " << this->FileName);
      return 0;
      }

    // Opens the file stream and positions the stream pointer at EOF ...
#ifdef _WIN32
    this->FileStream =
      new std::ofstream(this->FileName, ios::binary | ios::out | ios::ate);
#else
    this->FileStream =
      new std::ofstream(this->FileName, ios::out | ios::ate);
#endif

    if (!(*(this->FileStream)))
      {
      vtkErrorMacro("Cannot open output file stream: " << this->FileName);
      delete this->FileStream;
      this->FileStream = 0;
      return 0;
      }

    // The read position at EOF is the file size.
    std::ofstream::pos_type size = this->FileStream->tellg();
    // Rewind the stream.
    this->FileStream->seekg(0, ios::beg);

    this->Stream = this->FileStream;
    }

  this->DataStream->SetStream(this->Stream);

  return 1;
}

// ----------------------------------------------------------------------------
unsigned long
vtkSessionExporter::GetMTime(void)
{
  unsigned long mTime = this->Superclass::GetMTime();
  unsigned long time;

  if (this->RenderingContext)
    {
    time = this->RenderingContext->GetMTime();
    mTime = (time>mTime ? time : mTime);
    }

  return mTime;
}

// ----------------------------------------------------------------------------
void
vtkSessionExporter::PrintSelf(std::ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);

  if (this->RenderingContext)
    {
    os << indent << "Rendering Context: (" << (void *)this->RenderingContext
       << ")\n";
    }
  else
    {
    os << indent << "Rendering Context: (none)\n";
    }

  if (this->StartWrite)
    {
    os << indent << "Start Write: (" << (void *)this->StartWrite << ")\n";
    }
  else
    {
    os << indent << "Start Write: (none)\n";
    }

  if (this->EndWrite)
    {
    os << indent << "End Write: (" << (void *)this->EndWrite << ")\n";
    }
  else
    {
    os << indent << "End Write: (none)\n";
    }

  os << indent << "FileName: " 
     << (this->FileName ? this->FileName : "(none)") << std::endl;
  os << indent << "BaseName: " 
     << (this->BaseName ? this->BaseName : "(none)") << std::endl;
  os << indent << "BigEndian: (" << (this->BigEndian?"true":"false") << ")\n";
  if (this->Compressor)
    {
    os << indent << "Compressor: " << this->Compressor << "\n";
    }
  else
    {
    os << indent << "Compressor: (none)\n";
    }
  os << indent << "BlockSize: " << this->BlockSize << std::endl;
}

/* 
 * End of: $Id: vtkSessionExporter.cxx,v 1.1.1.1 2006/12/19 23:00:08 christianh Exp $.
 * 
 */
