/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMain.cxx,v 1.1.1.1 2006/12/19 22:58:33 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
// SVV
#include "svvMain.h"
#include "svvController.h"
// IOStreams
#include <iostream>

// ----------------------------------------------------------------------------
#ifdef SVV_UNIX
#  include <signal.h>
#  ifdef _sigargs
#    define SIGARG _sigargs
#  else
#    ifdef __sigargs
#      define SIGARG __sigargs
#    else
#      define SIGARG int // you may need to fix this for older systems
#    endif
#  endif

SVV_BEGIN_EXTERN_C
extern void     SvvSignalHandler (SIGARG sigNum);
SVV_END_EXTERN_C
#endif /* SVV_UNIX */

// ----------------------------------------------------------------------------
extern void     SvvPrintUsage (std::ostream& os, int& argc, char**& argv);
extern int      SvvParseArgs (int argc, char** argv, int& i);

// ----------------------------------------------------------------------------
int
main (int argc, char* argv[], char* envp[])
{
  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

#ifdef SVV_UNIX
  signal(SIGILL, SvvSignalHandler);
  signal(SIGFPE, SvvSignalHandler);
  signal(SIGSEGV, SvvSignalHandler);
  signal(SIGBUS, SvvSignalHandler);
  signal(SIGINT, SvvSignalHandler);
#endif /* SVV_UNIX */

  // Note that this will create a vtkMPIController if MPI
  // is configured, vtkThreadedController otherwise.
  vtkMultiProcessController* processController;

  processController = vtkMultiProcessController::New();
  processController->Initialize(&argc, &argv);

  // When using MPI, the number of processes is determined
  // by the external program which launches this application.
  // However, when using threads, we need to set it ourselves.
  if (processController->IsA("vtkThreadedController"))
    {
    // Set the number of processes to 4 for this example.
    processController->SetNumberOfProcesses(4);
    // Forces processController to use deep copies instead of shallow. This is 
    // needed when asynch. processing occurs (i.e. pipeline parallelism).
    processController->ForceDeepCopyOn();
    }

  if (processController->GetNumberOfProcesses() != 4)
    {
    std::cerr << "SV requires four processes." << std::endl;
    processController->Finalize();
    processController->Delete();
    return 1;
    }

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if ((ret = svvController::Args(&argc, &argv, &envp, i, sgParseArgs)) == 0)
    {
    sgPrintUsage(std::cerr, argc, argv[0]);
    processController->Finalize();
    processController->Delete();
    return 1;
    }
  
  processController->SetMultipleMethod(0, SvvProcess0, &inputArgs);
  processController->SetMultipleMethod(1, SvvProcess1, &inputArgs);
  processController->SetMultipleMethod(2, SvvProcess2, &inputArgs);
  processController->SetMultipleMethod(3, SvvProcess3, &inputArgs);

  processController->MultipleMethodExecute();

  processController->Finalize();
  processController->Delete();
  
  return 0;
}

/* 
 * End of: $Id: svvMain.cxx,v 1.1.1.1 2006/12/19 22:58:33 christianh Exp $.
 * 
 */
