/* -*- C++ -*-
 * 
 * $Id: svPolyDataFilter.H,v 1.1.1.1 2006/12/19 22:58:37 christianh Exp $
 * 
 * 
 * Copyright (c) 2002 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

// .NAME svPolyDataFilter - Aggregate filter object abstract base class.
// .SECTION Description
// Aggregate filter object abstract base class.

#ifndef SV_POLYDATAFILTER_H_
#  define SV_POLYDATAFILTER_H_

// VTK
#  include "vtkObject.h"
#  include "vtkPolyData.h"
#  include "vtkPolyDataToPolyDataFilter.h"
// SV
#  include "svMacros.H" /* includes svSystem */


// 
// .SECTION See Also
// vtkPolyDataToPolyDataFilter vtkPolyData vtkObject

class svPolyDataFilter : public vtkObject
{
public:
  vtkTypeMacro(svPolyDataFilter,vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);

  // Description:
  // pure virtual base class methods
  virtual void InstallFilter(void)   = 0;
  virtual void ResetParameters(void) = 0;
  virtual void ApplyParameters(void) = 0;

  // Description:
  // Set/Get this filter's active or inactive status.
  void   SetActive(int arg);
  int    GetActive(void) const {return this->Active;}
  void   Activate(void)        {this->SetActive(1);}
  void   Deactivate(void)      {this->SetActive(0);}

  // Description:
  // Set/Get a pointer to the vtkPolyDataToPolyDataFilter.
  void SetFilter(vtkPolyDataToPolyDataFilter* arg);
  vtkPolyDataToPolyDataFilter* GetFilter() const {return this->Filter;}

  // Description:
  // Methods for chaining up the VTK filtering pipeline.
  virtual void         SetInput(vtkPolyData*);
  virtual vtkPolyData* GetOutput(void);
  virtual void         Insert(vtkPolyData* input, vtkPolyData* output);

protected:
  // constructor
  svPolyDataFilter(void) : Filter(0) { }
  // destructor
  virtual ~svPolyDataFilter(void) { }

  vtkPolyDataToPolyDataFilter* Filter;
  vtkPolyData*                 Input;

  int                          Active;

private:
  svPolyDataFilter(const svPolyDataFilter&);  // Not implemented.
  void operator=(const svPolyDataFilter&);  // Not implemented.
};

typedef svPolyDataFilter* svPolyDataFilterPtr;


inline void
svPolyDataFilter::SetActive(int arg)
{
  this->Active = int(arg!=0);
  if(this->Active) {
    if(!this->Filter) this->InstallFilter();
    if(this->Input) this->Filter->SetInput(this->Input);
    this->ApplyParameters();
  }
}

inline void
svPolyDataFilter::SetFilter(vtkPolyDataToPolyDataFilter* arg)
{
  if(this->Filter != arg) {
    // to avoid destructor recursion
    vtkPolyDataToPolyDataFilter* tmp   = this->Filter;
    vtkPolyData*                 input = 0;
    this->Filter = arg;
    if(tmp) {
      input = tmp->GetInput();
      tmp->Delete();
    }
    if(this->Filter) {
      this->Filter->Register(0/*NULL*/);
      if(input) {
        this->Filter->SetInput(input);
      }
    }
  }
}

inline void
svPolyDataFilter::SetInput(vtkPolyData* input)
{
  this->Input = input;
  if(this->Filter && this->Active) this->Filter->SetInput(this->Input);
}

inline vtkPolyData*
svPolyDataFilter::GetOutput(void)
{
  if(this->Filter && this->Active) return this->Filter->GetOutput();
  return this->Input;
}

inline void
svPolyDataFilter::Insert(vtkPolyData* input, vtkPolyData* output)
{
  this->ApplyParameters();
  this->SetInput(input);
  output = this->GetOutput();
}

#endif /* SV_POLYDATAFILTER_H_ */
/* 
 * End of $Id: svPolyDataFilter.H,v 1.1.1.1 2006/12/19 22:58:37 christianh Exp $.
 * 
 */
