/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvColorNameMap.cxx,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvColorNameMap.h"
// C++ forwarding standard C headers.
#include <cmath>   /* floor() */
#include <cctype>  /* isspace() */
#include <cstdio>  /* sscanf() */
// STL
#include <string>
#include <map>
#include <set>
#include <list>
#include <algorithm> // for find_if
#include <sstream>
// IOStreams
#include <fstream>
// VTK Common
#include "vtkObjectFactory.h"

SVV_NAMESPACE_BEGIN

// ----------------------------------------------------------------------------
//      s v v C o l o r N a m e M a p
// ----------------------------------------------------------------------------

vtkCxxRevisionMacro (svvColorNameMap, "$Revision: 1.1.1.1 $");
vtkStandardNewMacro (svvColorNameMap);


// ----------------------------------------------------------------------------
typedef vtkstd::pair<vtkstd::string, SvvColorWord>      _SvvColorNamePair;
typedef vtkstd::map<vtkstd::string, SvvColorWord>       _SvvColorNameMap;
typedef vtkstd::set<vtkstd::string>                     _SvvColorNameSet;
typedef vtkstd::list<vtkstd::string>                    _SvvStringList;


// ----------------------------------------------------------------------------
class svvColorNameMapInternal
{
public:
  friend class svvColorNameMap;

  _SvvColorNameMap              Map;
  _SvvColorNameSet              Names;
  _SvvColorNameMap::iterator    MapIT;
  _SvvColorNameSet::iterator    NameIT;

  vtkstd::string                Title;

  static _SvvColorNameMap       Defaults;

  static bool                   Initialized;
  static bool                   ReadingFloats;

  /** \internal
   * Static search path along which to seach for file(s) of RGB-name pairs.
   */
  static _SvvStringList         SearchPath;

  /** \internal
   * Static name of the file(s) of RGB-name pairs.
   */
  static _SvvStringList         ColorFileNames;

  svvColorNameMapInternal (void)
    : Map(),
      Names(),
      MapIT(Map.end()),
      NameIT(Names.end()),
      Title("color name map")
    {}
};


// ----------------------------------------------------------------------------
const char* const
svvColorNameMap::NamedColorPathEnvVar = "SVV_COLORFILE_PATHS";

const char* const
svvColorNameMap::NamedColorFileEnvVar = "SVV_COLORFILE_NAMES";

///////////////////////////////////////////////////////////////////////////////
// Begin class static initialization.
///////////////////////////////////////////////////////////////////////////////
static const std::string s_paths_[] =
{
  "/usr/X11R6/lib/X11",
  "/usr/openwin/lib/X11",
  "/usr/lib/X11",
  "/usr/local/share/svv",
  "/usr/share/svv",
  ".",
};

static const std::string s_files_[] =
{
  "rgb.txt",
};

static const _SvvColorNamePair s_pairs_[] =
{
  _SvvColorNamePair("default", SVV_MAKE_RGB(204,204,204)),
};

// ----------------------------------------------------------------------------
bool svvColorNameMapInternal::Initialized   = false;
bool svvColorNameMapInternal::ReadingFloats = false;

_SvvStringList
svvColorNameMapInternal::SearchPath =
_SvvStringList(&s_paths_[0], &s_paths_[4]);

_SvvStringList
svvColorNameMapInternal::ColorFileNames =
_SvvStringList(&s_files_[0], &s_files_[1]);

_SvvColorNameMap
svvColorNameMapInternal::Defaults =
_SvvColorNameMap(&s_pairs_[0], &s_pairs_[1]);

// ----------------------------------------------------------------------------
static bool
s_get_next_color (std::ifstream& ifs,
                  std::string&   aName,
                  SvvColorWord&  aColor)
{
  std::string line;
  float r, g, b;
  SvvColorWord color;

  do
    {
    if (!getline(ifs,line)) return false;
    }
  while (line[0]=='!' || line[0]=='#' || line[0]=='\n');

  std::istringstream ist(line);
  ist >> r >> g >> b;

  //while (ist.good())
  //  {
  //  ist >> aName;
  //  if (ist.eof()) break;
  //  aName.append(" ");
  //  }

  if (!ist.good()) return false;

  SvvColorByte rb, gb, bb;

  if      (svvColorNameMapInternal::ReadingFloats)
    {
    rb = SvvColorByte(rint(r*float(SVV_RGB_MAX)));
    gb = SvvColorByte(rint(g*float(SVV_RGB_MAX)));
    bb = SvvColorByte(rint(b*float(SVV_RGB_MAX)));
    aColor = SVV_MAKE_RGB( SVV_CLAMPBYTE(rb),
                           SVV_CLAMPBYTE(gb),
                           SVV_CLAMPBYTE(bb) );
    }
  else if (floor(r)!=r || floor(g)!=g || floor(b)!=b)
    {
    svvColorNameMapInternal::ReadingFloats = true;
    rb = SvvColorByte(rint(r*float(SVV_RGB_MAX)));
    gb = SvvColorByte(rint(g*float(SVV_RGB_MAX)));
    bb = SvvColorByte(rint(b*float(SVV_RGB_MAX)));
    aColor = SVV_MAKE_RGB( SVV_CLAMPBYTE(rb),
                           SVV_CLAMPBYTE(gb),
                           SVV_CLAMPBYTE(bb) );
    }
  else
    {
    rb = SVV_CLAMPBYTE(r);
    gb = SVV_CLAMPBYTE(g);
    bb = SVV_CLAMPBYTE(b);
    aColor = SVV_MAKE_RGB(rb,gb,bb);
    }

  ist.clear();
  aName.erase();

  std::string tmp;
  const std::string ws(" ");

  while (ist.good())
    {
    ist >> tmp;
    aName.append(tmp);
    if (ist.eof()) break;
    aName.append(ws);
    }
  
  return ((ifs.eof() || !ifs.good()) ? false : true);
}

void
s_load_colors (void)
{
  if (svvColorNameMapInternal::Initialized)
    {
    return;
    }

  const std::string delims(" \t:;");
  const std::string dirsep("/");
  std::string concat, pathSub, nameSub;
  std::string::size_type beg, end;
  char* cptr;

  if (cptr = getenv(svvColorNameMap::NamedColorPathEnvVar))
    {
    //std::cerr << NamedColorPathEnvVar << "='" << cptr << "'" << std::endl;

    concat = cptr;
    end = 0;

    while ((beg = concat.find_first_not_of(delims,end)) != std::string::npos)
      {
      if ((end = concat.find_first_of(delims,beg)) == std::string::npos)
        {
        end = concat.length(); // End of word is end of line.
        }
      pathSub = concat.substr(beg,end-beg);
      //std::cerr << "path='" << pathSub << "'" << std::endl;
      svvColorNameMapInternal::SearchPath.push_back(pathSub);
      }
    }
  svvColorNameMapInternal::SearchPath.unique();


  if (cptr = getenv(svvColorNameMap::NamedColorFileEnvVar))
    {
    concat = cptr;
    end = 0;

    while ((beg = concat.find_first_not_of(delims,end)) != std::string::npos)
      {
      if ((end = concat.find_first_of(delims,beg)) == std::string::npos)
        {
        end = concat.length(); // End of word is end of line.
        }
      nameSub = concat.substr(beg,end-beg);
      //std::cerr << "name='" << nameSub << "'" << std::endl;
      svvColorNameMapInternal::ColorFileNames.push_back(nameSub);
      }
    }
  svvColorNameMapInternal::ColorFileNames.unique();

  //std::cerr << "loading colors" << std::endl;

  std::string basename, pathname;
  std::string name;
  SvvColorWord color;

  std::pair<_SvvColorNameMap::iterator, bool> rp;

  _SvvStringList::iterator pathIT;
  _SvvStringList::iterator fileIT;

  for ( pathIT  = svvColorNameMapInternal::SearchPath.begin();
        pathIT != svvColorNameMapInternal::SearchPath.end();
        ++pathIT )
    {
    for ( fileIT  = svvColorNameMapInternal::ColorFileNames.begin();
          fileIT != svvColorNameMapInternal::ColorFileNames.end();
          ++fileIT )
      {
      pathname = *pathIT + dirsep + *fileIT;

      std::ifstream ifs;

      // Opens the file and positions the stream pointer at EOF ...
      ifs.open(pathname.c_str(), std::ios::in | std::ios::ate);

      if (ifs.is_open())
        {
        // Rewind the stream ... and start reading the items.
        ifs.seekg(0, std::ios::beg);

        while (s_get_next_color(ifs, name, color))
          {
          //std::cerr << "< name = \"" << name << "\", agbr = "
          //          << std::hex << color << " >" << std::endl;
          rp =
            svvColorNameMapInternal::Defaults.
            insert(_SvvColorNamePair(name,color));
          }
        }

      ifs.close();
      svvColorNameMapInternal::ReadingFloats = false;
      }
    }

  //std::cerr << svvColorNameMapInternal::Defaults.size()
  //          << " named colors mapped" << std::endl;

  svvColorNameMapInternal::Initialized = true;
}

///////////////////////////////////////////////////////////////////////////////
// End of class static internals
///////////////////////////////////////////////////////////////////////////////

svvColorNameMap::svvColorNameMap (void)
  : Internal(new svvColorNameMapInternal)
{
}

svvColorNameMap::~svvColorNameMap()
{
  delete this->Internal;
  this->Internal = NULL;
}

// ----------------------------------------------------------------------------
void
svvColorNameMap::SetTitle (const char* const& aName)
{
  if (aName != NULL && *aName != '\0')
    {
    this->Internal->Title.assign(aName);
    }
  else
    {
    this->Internal->Title.assign("map");
    }
}

void
svvColorNameMap::GetTitle (const char* & aName)
{
  aName = this->Internal->Title.c_str();
}

// ----------------------------------------------------------------------------
void
svvColorNameMap::InsertDefaults (void)
{
  if (!svvColorNameMapInternal::Initialized)
    {
    s_load_colors();
    }

  this->Internal->Map = svvColorNameMapInternal::Defaults; // copy
}

// ----------------------------------------------------------------------------
bool
svvColorNameMap::AddColor (const char* const&  aName,
                           const SvvColorWord& aColor)
{
  return (this->Internal->Map.insert(_SvvColorNamePair(aName,aColor))).second;
}

bool
svvColorNameMap::FindColor (const char* const& aName,
                            SvvColorWord&      aColor)
{
  _SvvColorNameMap::iterator pos;

  if ((pos = this->Internal->Map.find(aName)) != this->Internal->Map.end())
    {
    aColor = pos->second;
    return true;
    }

  return false;
}

bool
svvColorNameMap::EraseColor (const char* const& aName)
{
  return (this->Internal->Map.erase(aName) > 0);
}

// ----------------------------------------------------------------------------
class ColorEq
{
  SvvColorWord color_;

public:
  ColorEq (const SvvColorWord& aColor) : color_(aColor) {}

  bool operator() (std::pair<const std::string, SvvColorWord> a)
    { return a.second == this->color_; }
};

bool
svvColorNameMap::FindName (const SvvColorWord& aColor,
                           const char* &       aName)
{
  _SvvColorNameMap::iterator pos = this->Internal->Map.begin();

  if (!this->Internal->Names.empty())
    {
    this->Internal->Names.clear();
    }

  while ( (pos = vtkstd::find_if( pos, this->Internal->Map.end(),
                                  ColorEq(aColor) ) )
          != this->Internal->Map.end() )
    {
    this->Internal->Names.insert(pos->first);
    }

  if ( (this->Internal->NameIT = this->Internal->Names.begin())
       != this->Internal->Names.end() )
    {
    aName = this->Internal->NameIT->c_str();
    return true;
    }

  return false;
}

bool
svvColorNameMap::FindNextName (const char* & aName)
{
  if (++(this->Internal->NameIT) != this->Internal->Names.end())
    {
    aName = this->Internal->NameIT->c_str();
    return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
void
svvColorNameMap::GetSize (SvvSize& a)
{
  a = SvvSize(this->Internal->Map.size());
}

// ----------------------------------------------------------------------------
void
svvColorNameMap::InitTraversal (const char* const&   aName)
{
  if      (this->Internal->Map.empty())
    {
    this->Internal->MapIT = this->Internal->Map.end();
    }
  else if (aName != NULL && *aName != '\0')
    {
    this->Internal->MapIT = this->Internal->Map.find(aName);
    }
  else
    {
    this->Internal->MapIT = this->Internal->Map.begin();
    }
}

bool
svvColorNameMap::TraverseForward (const char* &      aName,
                                  SvvColorWord&      aColor)
{
  if (this->Internal->MapIT == this->Internal->Map.end())
    {
    return false;
    }

  aName         = this->Internal->MapIT->first.c_str();
  aColor        = this->Internal->MapIT->second;

  ++(this->Internal->MapIT);

  return true;
}

/* ----------------------------------------------------------------------------
 *
 *  HOW TO RETURN hsl.to.rgb(h, s, l): 
 *     SELECT: 
 *        l<=0.5: PUT l*(s+1) IN m2
 *        ELSE: PUT l+s-l*s IN m2
 *     PUT l*2-m2 IN m1
 *     PUT hue.to.rgb(m1, m2, h+1/3) IN r
 *     PUT hue.to.rgb(m1, m2, h    ) IN g
 *     PUT hue.to.rgb(m1, m2, h-1/3) IN b
 *     RETURN (r, g, b)
 *
 *  HOW TO RETURN hue.to.rgb(m1, m2, h):
 *     IF h<0: PUT h+1 IN h
 *     IF h>1: PUT h-1 IN h
 *     IF h*6<1: RETURN m1+(m2-m1)*h*6
 *     IF h*2<1: RETURN m2
 *     IF h*3<2: RETURN m1+(m2-m1)*(2/3-h)*6
 *     RETURN m1
 */
static inline SvvColorByte
s_hue_to_rgb_component (float aM1, float aM2, float aHue)
{
  if (aHue<0.f) aHue += 1.f;
  if (aHue>0.f) aHue -= 1.f;

  int v;

  if      ((aHue*6.f)<1.f) v = int(rint(aM1+(aM2-aM1)*aHue*6.f));
  else if ((aHue*2.f)<1.f) v = int(rint(aM2));
  else if ((aHue*3.f)<2.f) v = int(rint(aM1+(aM2-aM1)*(2.f/3.f-aHue)*6.f));
  else                     v = int(rint(aM1));

  return SvvColorByte(v<0 ? 0 : (v>255 ? 255 : v));
}

static inline float
s_hsl_to_rgb (float         aHue,
              float         aSat,
              float         aLight,
              SvvColorByte& aRed,
              SvvColorByte& aGrn,
              SvvColorByte& aBlu)
{
  float m2 = (aLight<=0.5 ? aLight*(aSat + 1.f) : aLight + aSat - aLight*aSat);
  float m1 = aLight*2.f - m2;

  aRed = s_hue_to_rgb_component(m1, m2, aHue + 1.f/3.f);
  aGrn = s_hue_to_rgb_component(m1, m2, aHue          );
  aBlu = s_hue_to_rgb_component(m1, m2, aHue - 1.f/3.f);

  return m1;
}

/** Returns the integral value of the hexidecimal char (0-9,a-f,A-F). */
static inline SvvColorByte
s_get_hex_digit_value (char c)
{
  if (c >= '0' && c <= '9')
    return (c - '0');

  if (c >= 'a' && c <= 'f')
    return (c - 'a' + 10);

  if (c >= 'A' && c <= 'F')
    return (c - 'A' + 10);

  return 0;
}

/* 
 * SkipLeadingBlanks skips all standard white-space characters at the
 * beginning of the string and returns a pointer to the new position. 
 */
static inline const char*
s_skip_leading_blanks (const char* aPtr)
{
  while (isspace(*aPtr)) aPtr++;

  return aPtr;
}

// ----------------------------------------------------------------------------
const char*
svvColorNameMap::StringToColor (const char*   aValue,
                                SvvColorByte& aRed,
                                SvvColorByte& aGrn,
                                SvvColorByte& aBlu,
                                SvvColorByte& aAlpha)
{
  char  colorName[128];
  int   i;
  bool  failed = true;
  const char* ptr = aValue;

  aRed   = 0;
  aGrn   = 0;
  aBlu   = 0;
  aAlpha = 255;

  if (*ptr=='#')
    {
    // 
    // We expect an hexa encoding like F0F or FF00FF.
    // 
    if (isxdigit(ptr[1]) && isxdigit(ptr[2]) && isxdigit(ptr[3]))
      {
      if (!isxdigit(ptr[4]))
        {
        // 
        // encoded on 3 digits #F0F
        // 
        aRed = s_get_hex_digit_value(ptr[1]) * 16 +
          s_get_hex_digit_value(ptr[1]);
        aGrn = s_get_hex_digit_value(ptr[2]) * 16 +
          s_get_hex_digit_value(ptr[2]);
        aBlu = s_get_hex_digit_value(ptr[3]) * 16 +
          s_get_hex_digit_value(ptr[3]);
        ptr = &ptr[4];
        failed = false;
        }
      else if (isxdigit(ptr[5]) && isxdigit(ptr[6]))
        {
        // 
        // encoded on 6 digits #FF00FF
        // 
        aRed = s_get_hex_digit_value(ptr[1]) * 16 +
          s_get_hex_digit_value(ptr[2]);
        aGrn = s_get_hex_digit_value(ptr[3]) * 16 +
          s_get_hex_digit_value(ptr[4]);
        aBlu = s_get_hex_digit_value(ptr[5]) * 16 +
          s_get_hex_digit_value(ptr[6]);
        ptr = &ptr[7];
        failed = false;
        }
      }

    }
  else if (!strncasecmp(ptr,"rgb",3))
    {
    //
    // Encoded as 'rgb(r, g, b)' or 'rgb(r%, g%, b%)'.
    //
    bool parseAlpha = false;

    ptr += 3;
    if (*ptr=='a' || *ptr=='A')
      {
      //
      // Encoded as 'rgba(r, g, b, [0..1])' or 'rgba(r%, g%, b%, [0..1])'.
      //
      parseAlpha = true;
      ptr++;
      }

    ptr  = s_skip_leading_blanks(ptr);
    if (*ptr=='(')
      {
      ptr++;
      ptr = s_skip_leading_blanks(ptr);
      failed = false;

      int r, g, b, rr, gg, bb;

      sscanf(ptr,"%d",&r);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        rr = (r*255 / 100);
        aRed = SvvColorByte(rr<0 ? 0 : (rr>255 ? 255 : rr));
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        aRed = SvvColorByte(r<0 ? 0 : (r>255 ? 255 : r));
        }
      ptr++;

      sscanf(ptr,"%d",&g);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        gg = (g*255 / 100);
        aGrn = SvvColorByte(gg<0 ? 0 : (gg>255 ? 255 : gg));
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        aGrn = SvvColorByte(g<0 ? 0 : (g>255 ? 255 : g));
        }
      ptr++;

      sscanf(ptr,"%d",&b);
      if (parseAlpha)
        while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      else
        while (*ptr!='\0' && *ptr!=')' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        bb = (b*255 / 100);
        aBlu = SvvColorByte(bb<0 ? 0 : (bb>255 ? 255 : bb));
        if (parseAlpha)
          while (*ptr!='\0' && *ptr!=',') ptr++;
        else
          while (*ptr!='\0' && *ptr!=')') ptr++;
        }
      else
        {
        aBlu = SvvColorByte(b<0 ? 0 : (b>255 ? 255 : b));
        }

      if (parseAlpha)
        {
        float a;

        sscanf(ptr,"%f",&a);
        while (*ptr!='\0' && *ptr!=')') ptr++;
        aAlpha = SvvColorByte(a<0.f ? 0 : (a>1.f ? 255 : rint(a*255)));
        }

      // Search the rgb end.
      if (*ptr==')') ptr++;
      }

    }
  else if (!strncasecmp(ptr,"hsl",3))
    {

    //
    // Encoded as 'hsl(h, s, l)' or 'hsl(h%, s%, l%)'.
    //

    bool parseAlpha = false;

    ptr += 3;
    if (*ptr=='a' || *ptr=='A')
      {
      //
      // Encoded as 'hsla(h, s, l, [0..1])' or 'hsla(h%, s%, l%, [0..1])'.
      //
      parseAlpha = true;
      ptr++;
      }

    ptr = s_skip_leading_blanks(ptr);
    if (*ptr=='(')
      {
      ptr++;
      ptr = s_skip_leading_blanks(ptr);
      failed = false;

      int h, s, l;
      float hh, ss, ll;

      sscanf(ptr,"%d",&h);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        hh = float(h) / 100.f;
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        hh = float(h) / 255.f;
        }
      ptr++;

      sscanf(ptr,"%d",&s);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        ss = float(s) / 100.f;
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        ss = float(s) / 255.f;
        }
      ptr++;

      sscanf(ptr,"%d",&l);
      if (parseAlpha)
        while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      else
        while (*ptr!='\0' && *ptr!=')' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        ll = float(l) / 100.f;
        if (parseAlpha)
          while (*ptr!='\0' && *ptr!=',') ptr++;
        else
          while (*ptr!='\0' && *ptr!=')') ptr++;
        }
      else
        {
        ll = float(l) / 255.f;
        }

      float v = s_hsl_to_rgb(hh,ss,ll, aRed,aGrn,aBlu);

      if (parseAlpha)
        {
        float a;

        sscanf(ptr,"%f",&a);
        while (*ptr!='\0' && *ptr!=')') ptr++;
        aAlpha = SvvColorByte(a<0.f ? 0 : (a>1.f ? 255 : rint(a*255)));
        }

      // Search the hsl end.
      if (*ptr==')') ptr++;
      }
    
    }
  else if (isalpha(*ptr) || isxdigit(*ptr))
    {
    // 
    // We expect a color name like "red". Store it in colorName.
    // 
    int   len  = (sizeof(colorName) / sizeof(char)) - 1;
    const char* sptr = ptr;

    for(i=0; i<len && isalnum(*ptr); i++)
      {
      colorName[i] = *ptr;
      ptr++;
      }
    colorName[i] = '\0';
      
    // Look for the color name in our own color name database.
    if (failed = !(this->FindColor(colorName, aRed, aGrn, aBlu, aAlpha)))
      {
      // 
      // It may be a sequence of 3 or 6 hex. digits missing the leading '#'.
      // 
      ptr = sptr;
      if (isxdigit (ptr[0]) && isxdigit (ptr[1]) && isxdigit (ptr[2]))
        {
        // We expect an hexa encoding like F0F or FF00FF.
        if (!isxdigit (ptr[3]))
          {
          // Encoded on 3 digits e.g. '#F0F'.
          aRed = s_get_hex_digit_value(ptr[0]) * 16 +
            s_get_hex_digit_value(ptr[0]);
          aGrn = s_get_hex_digit_value(ptr[1]) * 16 +
            s_get_hex_digit_value(ptr[1]);
          aBlu = s_get_hex_digit_value(ptr[2]) * 16 +
            s_get_hex_digit_value(ptr[2]);
          ptr = &ptr[3];
          failed = false;
          }
        else if (isxdigit (ptr[4]) && isxdigit (ptr[5]))
          {
          // Encoded on 6 digits e.g. '#FF00FF'.
          aRed = s_get_hex_digit_value(ptr[0]) * 16 +
            s_get_hex_digit_value(ptr[1]);
          aGrn = s_get_hex_digit_value(ptr[2]) * 16 +
            s_get_hex_digit_value(ptr[3]);
          aBlu = s_get_hex_digit_value(ptr[4]) * 16 +
            s_get_hex_digit_value(ptr[5]);
          ptr = &ptr[6];
          failed = false;
          }
        }
      }
    }

  if (failed)
    {
    return aValue;
    }
  else
    {
    return ptr;
    }
}

SVV_NAMESPACE_END

/* 
 * End of: $Id: svvColorNameMap.cxx,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $.
 * 
 */
