/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvColorNameMap.h,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_COLOR_NAME_MAP_H_
#  define SVV_COLOR_NAME_MAP_H_
// SVV color typedefs
#  include "svvTypes.h"
// C++ forwarding ANSI C
#  include <cmath>
// VTK Common
#  include "vtkObject.h"

SVV_NAMESPACE_BEGIN

class svvColorNameMapInternal;

/** \class   svvColorNameMap
 *  \brief   Defined RGB colors (initially parsed from rgb.txt).
 * 
 * svvColorNameMap provides an interface to an associative array as 
 * well as a list of defined RGB colors.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:58:34 $
 */
class svvColorNameMap : public vtkObject
{
public:
  static svvColorNameMap* New (void);
  vtkTypeRevisionMacro (svvColorNameMap, vtkObject);

  void          SetTitle (const char* const&    name);
  void          GetTitle (const char* &         name);

  void          InsertDefaults (void);

  /*@{*/
  /** Adds an associated name-color pair to the map.
   * \return \c true on success, \c false otherwise.
   */
  bool          AddColor (const char* const&    name,
                          const SvvColorWord&   rgba);

  inline bool   AddColor (const char* const&    name,
                          const float&          red,
                          const float&          green,
                          const float&          blue,
                          const float&          alpha);

  inline bool   AddColor (const char* const&    name,
                          const float&          red,
                          const float&          green,
                          const float&          blue);

  inline bool   AddColor (const char* const&    name,
                          const float           rgba[4]);

  inline bool   AddColor (const char* const&    name,
                          SvvConstColorByte&    red,
                          SvvConstColorByte&    green,
                          SvvConstColorByte&    blue,
                          SvvConstColorByte&    alpha);

  inline bool   AddColor (const char* const&    name,
                          SvvConstColorByte&    red,
                          SvvConstColorByte&    green,
                          SvvConstColorByte&    blue);

  inline bool   AddColor (const char* const&    name,
                          SvvConstColorByte     rgba[4]);
  /*@}*/

  /*@{*/
  /** Given a name, find an associated color in the map.
   * \return \c true on success, \c false otherwise.
   */
  bool          FindColor (const char* const&   name,
                           SvvColorWord&        rgba);

  inline bool   FindColor (const char* const&   name,
                           float&               red,
                           float&               green,
                           float&               blue,
                           float&               alpha);

  inline bool   FindColor (const char* const&   name,
                           float&               red,
                           float&               green,
                           float&               blue);

  inline bool   FindColor (const char* const&   name,
                           float                rgba[4]);

  inline bool   FindColor (const char* const&   name,
                           SvvColorByte&        red,
                           SvvColorByte&        green,
                           SvvColorByte&        blue,
                           SvvColorByte&        alpha);

  inline bool   FindColor (const char* const&   name,
                           SvvColorByte&        red,
                           SvvColorByte&        green,
                           SvvColorByte&        blue);
  
  inline bool   FindColor (const char* const&   name,
                           SvvColorByte         rgba[4]);
  /*@}*/

  /** Erase an associated name - color pair from the map. */
  bool          EraseColor (const char* const&  name);

  /*@{*/
  /** Find the first \c name in the map associated with the given color.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool          FindName (const SvvColorWord&   rgba,
                          const char* &         name);

  inline bool   FindName (const SvvColorByte&   red,
                          const SvvColorByte&   green,
                          const SvvColorByte&   blue,
                          const char* &         name);

  inline bool   FindName (const SvvColorByte&   red,
                          const SvvColorByte&   green,
                          const SvvColorByte&   blue,
                          const SvvColorByte&   alpha,
                          const char* &         name);
  /*@}*/

  /**
   * Find the next \c name in the map associated with the color 
   * specified in a previous call to \c FindName.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool          FindNextName (const char* &     name);

  /** Get the number of entries in the map. */
  void          GetSize (SvvSize&               n);

  /** Initiates traversal through the map.
   *
   * \param name  The name of the material at which the traversal
   *              iterator will be positioned. If \name is \c NULL or
   *              empty, traversal starts at the first pair in the map.
   */
  void          InitTraversal (const char* const& name = NULL);

  /**
   * Get the current \c name - \c color association in the map
   * and iterate forward to the next pair.
   * 
   * \return \c false if either the map is empty, traversal was initialized
   *         with a name not associated with any color in the map, or
   *         traversal has reached the end of the map. Otherwise \c true
   *         is returned.
   */
  bool          TraverseForward (const char* &  name,
                                 SvvColorWord&  rgba);

  /**
   * Returns the RGB of the color and the pointer to the following text in
   * value if the parsing was finished.
   */
  const char*   StringToColor (const char*      name,
                               SvvColorByte&    red,
                               SvvColorByte&    green,
                               SvvColorByte&    blue,
                               SvvColorByte&    alpha);

  /*@{*/
  /** Converts an RGBA color between internal formats. */
  static inline void    ConvertColor (const SvvColorByte&       red,
                                      const SvvColorByte&       green,
                                      const SvvColorByte&       blue,
                                      const SvvColorByte&       alpha,
                                      SvvColorWord&             rgba);

  static inline void    ConvertColor (const float&              red,
                                      const float&              green,
                                      const float&              blue,
                                      const float&              alpha,
                                      SvvColorWord&             rgba);

  static inline void    ConvertColor (const SvvColorWord&       rgba,
                                      float&                    red,
                                      float&                    green,
                                      float&                    blue,
                                      float&                    alpha);

  static inline void    ConvertColor (const SvvColorWord&       rgba,
                                      SvvColorByte&             red,
                                      SvvColorByte&             green,
                                      SvvColorByte&             blue,
                                      SvvColorByte&             alpha);
  /*@}*/

  /*@{*/
  /** Name of environment variables to hold surface and volume data paths. */
  static const char* const NamedColorPathEnvVar;
  static const char* const NamedColorFileEnvVar;
  /*@}*/

protected:
  svvColorNameMap (void);
  ~svvColorNameMap();

  svvColorNameMapInternal* Internal;

private:
  svvColorNameMap (const svvColorNameMap&); // Not implemented.
  void operator= (const svvColorNameMap&); // Not implemented.
};

// ----------------------------------------------------------------------------
inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           const float&       aR,
                           const float&       aG,
                           const float&       aB,
                           const float&       aA)
{
  SvvColorWord color;
  svvColorNameMap::ConvertColor(aR, aG, aB, aA, color);
  return this->AddColor(aName, color);
}

inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           const float&       aR,
                           const float&       aG,
                           const float&       aB)
{
  float alpha = 1.f;
  return this->AddColor(aName, aR, aG, aB, alpha);
}

inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           const float        aColor[4])
{
  return this->AddColor(aName, aColor[0], aColor[1], aColor[2], aColor[3]);
}

inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           SvvConstColorByte& aR,
                           SvvConstColorByte& aG,
                           SvvConstColorByte& aB,
                           SvvConstColorByte& aA)
{
  SvvColorWord color = SVV_MAKE_RGBA(aR, aG, aB, aA);
  return this->AddColor(aName,color);
}

inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           SvvConstColorByte& aR,
                           SvvConstColorByte& aG,
                           SvvConstColorByte& aB)
{
  SvvColorByte a = SVV_RGB_MAX;
  return this->AddColor(aName, aR, aG, aB, a);
}

inline bool
svvColorNameMap::AddColor (const char* const& aName,
                           SvvConstColorByte  aColor[4])
{
  return this->AddColor(aName, aColor[0], aColor[1], aColor[2], aColor[3]);
}

// ----------------------------------------------------------------------------
inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            float&             aR,
                            float&             aG,
                            float&             aB,
                            float&             aA)
{
  SvvColorWord color;

  if (this->FindColor(aName, color))
    {
    svvColorNameMap::ConvertColor(color, aR, aG, aB, aA);
    return true;
    }

  return false;
}

inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            float&             aR,
                            float&             aG,
                            float&             aB)
{
  float alpha;
  return this->FindColor(aName, aR, aG, aB, alpha);
}

inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            float              aColor[4])
{
  return this->FindColor(aName, aColor[0], aColor[1], aColor[2], aColor[3]);
}

inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            SvvColorByte&      aR,
                            SvvColorByte&      aG,
                            SvvColorByte&      aB,
                            SvvColorByte&      aA)
{
  SvvColorWord color;

  if (this->FindColor(aName, color))
    {
    svvColorNameMap::ConvertColor(color, aR, aG, aB, aA);
    return true;
    }

  return false;
}

inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            SvvColorByte&      aR,
                            SvvColorByte&      aG,
                            SvvColorByte&      aB)
{
  SvvColorByte alpha;
  return this->FindColor(aName, aR, aG, aB, alpha);
}

inline bool
svvColorNameMap::FindColor (const char* const& aName,
                            SvvColorByte       aColor[4])
{
  return this->FindColor(aName, aColor[0], aColor[1], aColor[2], aColor[3]);
}

// ----------------------------------------------------------------------------
inline bool
svvColorNameMap::FindName (const SvvColorByte& aR,
                           const SvvColorByte& aG,
                           const SvvColorByte& aB,
                           const char* &       aName)
{
  SvvColorWord color = SVV_MAKE_RGB(aR,aG,aB);
  return this->FindName(color, aName);
}

inline bool
svvColorNameMap::FindName (const SvvColorByte& aR,
                           const SvvColorByte& aG,
                           const SvvColorByte& aB,
                           const SvvColorByte& aA,
                           const char* &       aName)
{
  SvvColorWord color = SVV_MAKE_RGBA(aR,aG,aB,aA);
  return this->FindName(color, aName);
}

// ----------------------------------------------------------------------------
// static methods
// ----------------------------------------------------------------------------
inline void
svvColorNameMap::ConvertColor (const SvvColorByte& aR,
                               const SvvColorByte& aG,
                               const SvvColorByte& aB,
                               const SvvColorByte& aA,
                               SvvColorWord&       aColor)
{
  aColor = SVV_MAKE_RGBA(aR,aG,aB,aA);
}

inline void
svvColorNameMap::ConvertColor (const float&  aR,
                               const float&  aG,
                               const float&  aB,
                               const float&  aA,
                               SvvColorWord& aColor)
{
  SvvColorByte r = SvvColorByte(rint((aR>1.f?1.f:aR) * float(SVV_RGB_MAX)));
  SvvColorByte g = SvvColorByte(rint((aG>1.f?1.f:aG) * float(SVV_RGB_MAX)));
  SvvColorByte b = SvvColorByte(rint((aB>1.f?1.f:aB) * float(SVV_RGB_MAX)));
  SvvColorByte a = SvvColorByte(rint((aA>1.f?1.f:aA) * float(SVV_RGB_MAX)));

  svvColorNameMap::ConvertColor(r, g, b, a, aColor);
}

inline void
svvColorNameMap::ConvertColor (const SvvColorWord& aColor,
                               SvvColorByte&       aR,
                               SvvColorByte&       aG,
                               SvvColorByte&       aB,
                               SvvColorByte&       aA)
{
  aR = SVV_GET_RGBA_R(aColor);
  aG = SVV_GET_RGBA_G(aColor);
  aB = SVV_GET_RGBA_B(aColor);
  aA = SVV_GET_RGBA_A(aColor);
}

inline void
svvColorNameMap::ConvertColor (const SvvColorWord& aColor,
                               float&              aR,
                               float&              aG,
                               float&              aB,
                               float&              aA)
{
  SvvColorByte r, g, b, a;

  svvColorNameMap::ConvertColor(aColor, r, g, b, a);

  aR = float(r) / float(SVV_RGB_MAX);
  aG = float(g) / float(SVV_RGB_MAX);
  aB = float(b) / float(SVV_RGB_MAX);
  aA = float(a) / float(SVV_RGB_MAX);
}

SVV_NAMESPACE_END

#endif /* SVV_COLOR_NAME_MAP_H_ */
/* 
 * End of: $Id: svvColorNameMap.h,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $.
 * 
 */
