/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvController.cxx,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
// STL
#include <new>
#include <string>
// from VTK Common
#include "vtkCommand.h"
#include "vtkImplicitFunction.h"
#include "vtkTransform.h"
#include "vtkProperty2D.h"
#include "vtkDataSet.h"
#include "vtkPolyData.h"
#include "vtkImageData.h"
#include "vtkObjectFactory.h"
// from VTK Graphics
#include "vtkCubeSource.h"
#include "vtkSphereSource.h"
#include "vtkCylinderSource.h"
#include "vtkOutlineSource.h"
#include "vtkAppendPolyData.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkPolyDataNormals.h"
#include "vtkClipPolyData.h"
#include "vtkTubeFilter.h"
// from VTK Hybrid
#include "vtkDepthSortPolyData.h"
#include "vtkCubeAxesActor2D.h"
// from VTK Rendering
#include "vtkTextProperty.h"
#include "vtkAxisActor2D.h"
#include "vtkPolyDataMapper.h"
#include "vtkVolumeMapper.h"
#include "vtkRenderWindow.h"
#include "vtkRendererCollection.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkLODActor.h"
#include "vtkProperty.h"
#include "vtkAssembly.h"
#include "vtkActor.h"
#include "vtkVolume.h"
#include "vtkRenderer.h"
#include "vtkActorCollection.h"
#include "vtkVolumeCollection.h"
// from vtkExtensions
#include "vtkGeosphereSource.h"
#include "vtkSpiralShellSource.h"
#include "vtkPolyDataPipeline.h"
#include "vtkPolyDataPipelineCollection.h"
#include "vtkInputSource.h"
#include "vtkInputSourceCollection.h"
#include "vtkTraversal.h"
// SVV
#include "svvController.h"
#include "svvMaterialNameMap.h"
#include "svvColorNameMap.h"

SVV_NAMESPACE_BEGIN

/** \internal
 * this is for fake initialization
 */
template<class T>
union _aligned_buffer_t
{
  char buf[sizeof(T)];
  struct { double a; double b; } padding;
};


int*    svvController::s_argc_ptr_    = NULL;
char*** svvController::s_argv_ptr_    = NULL;
char*** svvController::s_envp_ptr_    = NULL;
int     svvController::s_args_called_ = 0;

// Singleton static space.
static _aligned_buffer_t<svvController> fake;
// Singleton instance reference.
svvController& svv = *(reinterpret_cast<svvController*>(&fake));


vtkCxxRevisionMacro(svvController, "$Revision: 1.1.1.1 $");

///////////////////////////////////////////////////////////////////////////////
// Begin class singleton internals
///////////////////////////////////////////////////////////////////////////////

svvController*
svvController::New (void)
{
  return &svv;
}

svvController::svvController (void)
{
  if (this != &svv)
    {
    vtkErrorMacro(<<"Violation of singleton policy should not be possible!");
    }

  //this->DebugOn();

  this->RenderModePointsThreshold = 65536;
  this->RenderModeCellsThreshold  = 65536;

  this->MaterialNameMap = svvMaterialNameMap::New();
  this->MaterialNameMap->InsertDefaults();

  this->ColorNameMap = svvColorNameMap::New();
  this->ColorNameMap->InsertDefaults();

  this->InputSources     = vtkInputSourceCollection::New();
  this->SurfaceActors    = vtkActorCollection::New();
  this->SurfacePipelines = vtkPolyDataPipelineCollection::New();

  this->Interactor = NULL;

  this->BoundingBox             = NULL;
  this->BoundingBoxRenderer     = NULL;

  this->MaterialsDemo           = NULL;
  this->MaterialsDemoRadii      = 2;
  this->MaterialsDemoResolution = 2;
  this->MaterialsDemoRenderer   = NULL;

  this->SetDataPathsEnvVar("SVV_DATA_PATHS");
}

svvController::~svvController()
{
  if (this != &svv)
    {
    vtkErrorMacro(<<"Violation of singleton policy should not be possible!");
    }

  this->MaterialNameMap->UnRegister(this);
  this->MaterialNameMap = NULL;

  this->ColorNameMap->UnRegister(this);
  this->ColorNameMap = NULL;

  if (this->InputSources)
    this->SetInputSources(NULL);

  if (this->SurfaceActors)
    this->SetSurfaceActors(NULL);

  if (this->SurfacePipelines)
    this->SetSurfacePipelines(NULL);

  if (this->Interactor)
    this->SetInteractor(NULL);

  if (this->BoundingBox)
    this->SetBoundingBox(NULL);

  if (this->BoundingBoxRenderer)
    this->SetBoundingBoxRenderer(NULL);

  if (this->MaterialsDemo)
    this->SetMaterialsDemo(NULL);

  if (this->MaterialsDemoRenderer)
    this->SetMaterialsDemoRenderer(NULL);
}

svvController::Init::Init (void)
{
  if (svvController::Init::TranslationUnits++ == 0)
    {
    svvController::ClassInitialize ();
    }
}

svvController::Init::~Init()
{
  if (--svvController::Init::TranslationUnits == 0)
    {
    svvController::ClassFinalize();
    }
}

// Must NOT be initialized.  Default initialization to zero is necessary.
long svvController::Init::TranslationUnits;

void
svvController::ClassInitialize (void)
{
  svvController* pSvvController = new((void *) &fake) svvController;
}

void
svvController::ClassFinalize (void)
{
  svvController* pSvvController = reinterpret_cast<svvController*>(&svv);
  // The superclass constructor sets the reference count to 1. Since the
  // instance owns itself, the count remains at 1 until it's time to die.
  pSvvController->SetReferenceCount(0);
  // EXPLICIT DESTRUCTOR CALL AHEAD ... think thrice!!!
  pSvvController->~svvController();
}

///////////////////////////////////////////////////////////////////////////////
// End of class singleton internals
///////////////////////////////////////////////////////////////////////////////

vtkCxxSetObjectMacro (svvController, BoundingBox, vtkProp3D);
vtkCxxSetObjectMacro (svvController, BoundingBoxRenderer, vtkRenderer);
vtkCxxSetObjectMacro (svvController, MaterialsDemo, vtkProp3D);
vtkCxxSetObjectMacro (svvController, MaterialsDemoRenderer, vtkRenderer);
vtkCxxSetObjectMacro (svvController, InputSources, vtkInputSourceCollection);
vtkCxxSetObjectMacro (svvController, SurfacePipelines,
                      vtkPolyDataPipelineCollection);
vtkCxxSetObjectMacro(svvController, SurfaceActors, vtkActorCollection);

int
svvController::Args (int* c, char** v[], char** p[], int& i,
                     int (*cb)(int, char**, int&))
{
  svvController::s_argc_ptr_    = c;
  svvController::s_argv_ptr_    = v;
  svvController::s_envp_ptr_    = p;
  svvController::s_args_called_ = 1;
  
  i = 1; // skip argv[0]
  while (i < *c) if (!cb || !cb(*c, *v, i)) return 0;

  return i;
}

// ----------------------------------------------------------------------------
svvMaterialNameMap*
svvController::GetMaterialNameMap (void) const
{
  return this->MaterialNameMap;
}

svvColorNameMap*
svvController::GetColorNameMap (void) const
{
  return this->ColorNameMap;
}

// ----------------------------------------------------------------------------
bool
svvController::FindMaterial (const char* const& aName, SvvMaterial& aMaterial)
{
  return this->MaterialNameMap->FindMaterial(aName, aMaterial);
}

bool
svvController::AddMaterial (const char* const& aName, const SvvMaterial& aMaterial)
{
  return this->MaterialNameMap->AddMaterial(aName, aMaterial);
}

void
svvController::PrintMaterials (ostream& aTarget)
{
  this->MaterialNameMap->PrintAll(aTarget);
}

// ----------------------------------------------------------------------------
bool
svvController::FindColor (const char* const& aName,
                          float& r, float& g, float& b, float& a)
{
  return this->ColorNameMap->FindColor(aName,r,g,b,a);
}

bool
svvController::FindColor (const char* const& aName,
                          float& r, float& g, float& b)
{
  return this->ColorNameMap->FindColor(aName,r,g,b);
}

bool
svvController::FindColor (const char* const& aName, float aColor[4])
{
  return this->ColorNameMap->FindColor(aName,aColor);
}

bool
svvController::FindColor (const char* const& aName,
                          SvvColorByte& r, SvvColorByte& g, SvvColorByte& b,
                          SvvColorByte& a)
{
  return this->ColorNameMap->FindColor(aName,r,g,b,a);
}

bool
svvController::FindColor (const char* const& aName,
                          SvvColorByte& r, SvvColorByte& g, SvvColorByte& b)
{
  return this->ColorNameMap->FindColor(aName,r,g,b);
}

bool
svvController::FindColor (const char* const& aName, SvvColorByte aColor[4])
{
  return this->ColorNameMap->FindColor(aName,aColor);
}

bool
svvController::AddColor (const char* const& aName,
                         const float& r, const float& g, const float& b,
                         const float& a)
{
  return this->ColorNameMap->AddColor(aName,r,g,b,a);
}

bool
svvController::AddColor (const char* const& aName,
                         const float& r, const float& g, const float& b)
{
  return this->ColorNameMap->AddColor(aName,r,g,b);
}

bool
svvController::AddColor (const char* const& aName, const float aColor[4])
{
  return this->AddColor(aName, aColor[0], aColor[1], aColor[2], aColor[3]);
}

// ----------------------------------------------------------------------------
vtkProperty*
svvController::MakeProperty (SvvMaterial& m, float as, float ds, float ss)
{
  vtkProperty* p;

  p = vtkProperty::New();

  p->SetAmbient(as);
  p->SetDiffuse(ds);
  p->SetSpecular(ss);
  p->SetAmbientColor(m.Ka);
  p->SetDiffuseColor(m.Kd);
  p->SetSpecularColor(m.Ks);
  p->SetSpecularPower(m.Se);
  p->SetOpacity(m.Kd[3]);

  return p;
}

vtkProperty*
svvController::MakeProperty (float aRGBA[4], float as, float ds, float ss)
{
  vtkProperty* pProperty = vtkProperty::New();

  pProperty->SetAmbient(as);
  pProperty->SetDiffuse(ds);
  pProperty->SetSpecular(ss);
  pProperty->SetColor(aRGBA);
  pProperty->SetOpacity(aRGBA[3]);

  return pProperty;
}

vtkProperty*
svvController::MakeProperty (const char* const& aName)
{
  SvvMaterial material;
  float rgba[4];

  if (this->FindMaterial(aName,material))
    {
    return this->MakeProperty(material);
    }
  else if (this->FindColor(aName,rgba))
    {
    return this->MakeProperty(rgba);
    }
  else
    {
    vtkWarningMacro(<< "Could not find defined material or color: " << aName);
    return vtkProperty::New();
    }
}

// ----------------------------------------------------------------------------
void
svvController::AddSurfaceDataSource (vtkInputSource* aSource)
{
  this->InputSources->AddSource(aSource);
}

void
svvController::RemoveSurfaceDataSource (vtkInputSource* aSource)
{
  this->InputSources->RemoveSource(aSource);
}

vtkInputSource*
svvController::MakeSurfaceDataSource (const char* aFileName,
                                      int         aType,
                                      bool        aUnlistedFlag)
{
  vtkInputSource* source;

  source = vtkInputSource::New();

  source->SetFileName(aFileName);

  if (aType)
    {
    source->SetFileType(aType);
    }

  if (!aUnlistedFlag)
    {
    this->AddSurfaceDataSource(source);
    }

  return source;
}

// ----------------------------------------------------------------------------
void
svvController::UpdateRenderModes (void)
{
  vtkActorCollection* ac;
  vtkActor*           actor;
  vtkPolyDataMapper*  mapper;

  ac = this->SurfaceActors;
  ac->InitTraversal();
  while ((actor = ac->GetNextActor()) != NULL)
    {
    if ((mapper = vtkPolyDataMapper::SafeDownCast(actor->GetMapper())) != NULL)
      {
      int nCells  = mapper->GetInput()->GetNumberOfCells();
      int nPoints = mapper->GetInput()->GetNumberOfPoints();

      if ( nPoints > this->RenderModePointsThreshold ||
           nCells > this->RenderModeCellsThreshold )
        {
        vtkDebugMacro(<< "Actor (" << actor << ") : " << nPoints
                      << " points (>" << this->RenderModePointsThreshold
                      << ") or " << nCells << " cells (>"
                      << this->RenderModeCellsThreshold << ") exceeds "
                      << "threshold for display list use." << endl
                      << " ... Using immediate mode rendering.\n");
        mapper->ImmediateModeRenderingOn();
        }
      else
        {
        vtkDebugMacro(<< "Actor (" << actor << ") : " << nPoints
                      << " points (<" << this->RenderModePointsThreshold
                      << ") and " << nCells << " cells (<"
                      << this->RenderModeCellsThreshold << ") below "
                      << "threshold for display list use.\n");
        mapper->ImmediateModeRenderingOff();
        }
      }
    }
}

// ----------------------------------------------------------------------------
void
svvController::AddSurfaceActor (vtkActor* aActor)
{
  if (aActor == NULL)
    {
    vtkDebugMacro(<< "AddSurfaceActor: received NULL pointer to actor.");
    return;
    }

  this->SurfaceActors->AddItem(aActor);
}

void
svvController::RemoveSurfaceActor (vtkActor* aActor)
{
  if (aActor == NULL)
    {
    vtkDebugMacro(<< "RemoveSurfaceActor: received NULL pointer to actor.");
    return;
    }

  this->SurfaceActors->RemoveItem(aActor);
}

// ----------------------------------------------------------------------------
vtkActor*
svvController::MakeSurfaceActor (vtkPolyDataSource* aSource,
                                 vtkProperty*       aProperty,
                                 bool               aUnlistedFlag)
{
  if (aSource == NULL)
    {
    vtkErrorMacro(<< "MakeSurfaceActor: received NULL pointer to source.");
    return NULL;
    }

  float bounds[6], length, scale;

  // get extents
  aSource->GetOutput()->GetBounds(bounds);
  length = aSource->GetOutput()->GetLength();
  scale  = length / 4.f;
  
  vtkLODActor* lodActor = vtkLODActor::New();
    {
    if (aProperty != NULL)
      {
      lodActor->SetProperty(aProperty);
      }
    vtkPolyDataMapper* polyDataMapper = vtkPolyDataMapper::New();
      {
#if DEPTH_SORT_TRANSLUCENT_POLYS
      vtkDepthSortPolyData* depthSorter = 0;

      if (aProperty->GetOpacity() < 1.f && this->Interactor != NULL)
        {
        vtkRenderWindow* rw;

        if ((rw = this->Interactor->GetRenderWindow()) != NULL)
          {
          vtkRendererCollection* rc;

          if ((rc = rw->GetRenderers()) != NULL)
            {
            rc->InitTraversal();
            vtkRenderer* renderer;
            if ((renderer = rc->GetNextItem()) != NULL)
              {
              depthSorter = vtkDepthSortPolyData::New();
              depthSorter->SetInput(aSource->GetOutput());
              depthSorter->SetCamera(renderer->GetActiveCamera());
              depthSorter->SetProp3D(lodActor);
              depthSorter->SetDepthSortModeToParametricCenter();
              depthSorter->SetDirectionToBackToFront();
              }
            }
          }
        }

      if (depthSorter != NULL)
        {
        polyDataMapper->SetInput(depthSorter->GetOutput());
        depthSorter->Delete();
        }
      else
        {
        polyDataMapper->SetInput(aSource->GetOutput());
        }
#else
      polyDataMapper->SetInput(aSource->GetOutput());
#endif /* DEPTH_SORT_TRANSLUCENT_POLYS */

      polyDataMapper->ScalarVisibilityOff();
      // Optimization: Only create display lists when actor is below size limit.
      polyDataMapper->GetInput()->Update();
      int nCells  = polyDataMapper->GetInput()->GetNumberOfCells();
      int nPoints = polyDataMapper->GetInput()->GetNumberOfPoints();
      if ( nPoints > this->RenderModePointsThreshold ||
           nCells > this->RenderModeCellsThreshold )
        {
        polyDataMapper->ImmediateModeRenderingOn();
        }
      }
      lodActor->SetMapper(polyDataMapper);
      polyDataMapper->Delete();
    }

    if (!aUnlistedFlag)
      {
      this->AddSurfaceActor(lodActor);
      }

    return vtkActor::SafeDownCast(lodActor);
}

vtkActor*
svvController::MakeSurfaceActor (vtkPolyDataSource* aSource,
                                 SvvMaterial&          aMaterial,
                                 bool               aUnlistedFlag)
{
  vtkProperty* property;
  vtkActor*    actor;

  property = this->MakeProperty(aMaterial);
  actor    = this->MakeSurfaceActor(aSource, property, aUnlistedFlag);
  property->Delete();

  return actor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkPolyDataSource* aSource,
                                 float              aRGBA[4],
                                 bool               aUnlistedFlag)
{
  vtkProperty* property;
  vtkActor*    actor;

  property = this->MakeProperty(aRGBA);
  actor    = this->MakeSurfaceActor(aSource, property, aUnlistedFlag);
  property->Delete();

  return actor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkPolyDataSource* aSource,
                                 const char*        aName,
                                 bool               aUnlistedFlag)
{
  vtkProperty* property;
  vtkActor*    actor;

  property = this->MakeProperty(aName);
  actor    = this->MakeSurfaceActor(aSource, property, aUnlistedFlag);
  property->Delete();

  return actor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkInputSource* aInputSource,
                                 vtkProperty*    aProperty,
                                 bool            aUnlistedFlag)
{
  if (aInputSource->GetOutputType() != VTK_POLY_DATA)
    {
    vtkWarningMacro(<< "Cannot create surface actor from source output"
                    << " of type " << aInputSource->GetOutputType() << ".");
    return NULL;
    }

  float bounds[6], length;

  // get extents
  aInputSource->GetPolyDataOutput()->GetBounds(bounds);
  length = aInputSource->GetPolyDataOutput()->GetLength();
  
  vtkLODActor* lodActor = vtkLODActor::New();
    {
    if (aProperty)
      {
      lodActor->SetProperty(aProperty);
      }
    vtkPolyDataMapper* polyDataMapper = vtkPolyDataMapper::New();
      {
      polyDataMapper->SetInput(aInputSource->GetPolyDataOutput());
      polyDataMapper->ScalarVisibilityOff();
      // Optimization: Only create display lists when actor is below size limit.
      polyDataMapper->GetInput()->Update();
      int nCells  = polyDataMapper->GetInput()->GetNumberOfCells();
      int nPoints = polyDataMapper->GetInput()->GetNumberOfPoints();
      if ( nPoints > this->RenderModePointsThreshold ||
           nCells > this->RenderModeCellsThreshold )
        {
        polyDataMapper->ImmediateModeRenderingOn();
        }
      }
      lodActor->SetMapper(polyDataMapper);
      polyDataMapper->Delete();
    }

  if (!aUnlistedFlag)
    {
    this->AddSurfaceActor(lodActor);
    }

  return lodActor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkInputSource* aInputSource,
                                 SvvMaterial&       aMaterial,
                                 bool            aUnlistedFlag)
{
  vtkProperty* property = this->MakeProperty(aMaterial);
  vtkActor* actor = this->MakeSurfaceActor(aInputSource,property,aUnlistedFlag);
  property->Delete();

  return actor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkInputSource* aInputSource,
                                 float           aRGBA[4],
                                 bool            aUnlistedFlag)
{
  vtkProperty* property = this->MakeProperty(aRGBA);
  vtkActor* actor = this->MakeSurfaceActor(aInputSource,property,aUnlistedFlag);
  property->Delete();

  return actor;
}

vtkActor*
svvController::MakeSurfaceActor (vtkInputSource* aInputSource,
                                 const char*     aName,
                                 bool            aUnlistedFlag)
{
  vtkProperty* property = this->MakeProperty(aName);
  vtkActor* actor = this->MakeSurfaceActor(aInputSource,property,aUnlistedFlag);
  property->Delete();

  return actor;
}

// ----------------------------------------------------------------------------
vtkActor*
svvController::MakeClipSelectionActor (vtkPolyDataSource*   aSource,
                                       vtkImplicitFunction* aFunction,
                                       const char*          aName)
{
  if (aSource == NULL)
    {
    vtkErrorMacro(<< "MakeClipSelectionActor:"
                  << " received NULL pointer to source.");
    return NULL;
    }

  vtkClipPolyData* clipper = vtkClipPolyData::New();
    {
    clipper->SetInput(aSource->GetOutput());
    clipper->SetClipFunction(aFunction);
    clipper->InsideOutOn();
    }

  vtkActor* actor = this->MakeSurfaceActor(clipper, aName, true/*unlisted*/);
    {
    actor->VisibilityOff();
    actor->SetScale(1.01, 1.01, 1.01);
    }

  return actor;
}

// ----------------------------------------------------------------------------
void
svvController::AddSurfacePipeline (vtkPolyDataPipeline* aPipeline)
{
  if (aPipeline == NULL)
    {
    vtkDebugMacro(<< "AddSurfacePipeline:"
                  << " received NULL pointer to pipeline.");
    return;
    }

  this->SurfacePipelines->AddItem(aPipeline);
}

void
svvController::RemoveSurfacePipeline (vtkPolyDataPipeline* aPipeline)
{
  if (aPipeline == NULL)
    {
    vtkDebugMacro(<< "RemoveSurfacePipeline:"
                  << " received NULL pointer to pipeline.");
    return;
    }

  this->SurfacePipelines->RemoveItem(aPipeline);
}

vtkPolyDataPipeline*
svvController::MakeSurfacePipeline (vtkInputSource* aInputSource,
                                    vtkProperty*    aProperty)
{
  vtkPolyDataPipeline* surface;

  surface = vtkPolyDataPipeline::New();

  if (aProperty)
    {
    surface->SetProperty(aProperty);
    }

  return surface;
}

vtkPolyDataPipeline*
svvController::MakeSurfacePipeline (vtkInputSource* aInputSource,
                                    SvvMaterial&       aMaterial)
{
  vtkProperty*         property = this->MakeProperty(aMaterial);
  vtkPolyDataPipeline* surface  = this->MakeSurfacePipeline( aInputSource,
                                                             property );
  property->Delete();

  return surface;
}

vtkPolyDataPipeline*
svvController::MakeSurfacePipeline (vtkInputSource* aInputSource,
                                    float           aRGBA[4])
{
  vtkProperty*         property = this->MakeProperty(aRGBA);
  vtkPolyDataPipeline* surface  = this->MakeSurfacePipeline( aInputSource,
                                                             property );
  property->Delete();

  return surface;
}

vtkPolyDataPipeline*
svvController::MakeSurfacePipeline (vtkInputSource* aInputSource,
                                    const char*     aName)
{
  vtkProperty*         property = this->MakeProperty(aName);
  vtkPolyDataPipeline* surface  = this->MakeSurfacePipeline( aInputSource,
                                                             property );
  property->Delete();

  return surface;
}

// ----------------------------------------------------------------------------
vtkCxxSetObjectMacro (svvController, Interactor, vtkRenderWindowInteractor);

// ----------------------------------------------------------------------------
static vtkActor*
make_cube_actor (float bounds[6], vtkCamera* camera = 0)
{
  vtkActor* cubeActor = vtkActor::New();
    {
    vtkPolyDataMapper* cubeMapper = vtkPolyDataMapper::New();
      {
      vtkDepthSortPolyData* depthSorter = vtkDepthSortPolyData::New();
        {
	vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
          {
	  vtkCubeSource* cubeSource = vtkCubeSource::New();
            {
	    cubeSource->SetBounds(bounds);
            }
            normals->SetInput(cubeSource->GetOutput());
            cubeSource->Delete();
          }
        depthSorter->SetInput(normals->GetOutput());
        if (camera)
          {
          depthSorter->SetCamera(camera);
          depthSorter->SetDirectionToBackToFront();
          }
        else
          {
          depthSorter->SetVector(0.0, 0.0, 1.0);
          depthSorter->SetDirectionToSpecifiedVector();
          }
        depthSorter->SetProp3D(cubeActor);
        depthSorter->SetDepthSortModeToParametricCenter();
        normals->Delete();
        }
      cubeMapper->SetInput(depthSorter->GetOutput());
      cubeMapper->SetResolveCoincidentTopologyToPolygonOffset();
      depthSorter->Delete();
      }
    cubeActor->SetMapper(cubeMapper);
    cubeMapper->Delete();
    }
  return cubeActor;
}

// ----------------------------------------------------------------------------
bool
svvController::GetRenderedBounds (float aBounds[6], vtkRenderer* aRenderer)
{
  if (aBounds != NULL)
    {
    if (aRenderer != NULL)
      {
      aBounds[0] = VTK_LARGE_FLOAT; aBounds[1] = -VTK_LARGE_FLOAT;
      aBounds[2] = VTK_LARGE_FLOAT; aBounds[3] = -VTK_LARGE_FLOAT;
      aBounds[4] = VTK_LARGE_FLOAT; aBounds[5] = -VTK_LARGE_FLOAT;

      float btmp[6];
      int   count = 0;

      vtkActor* actor;
      vtkActorCollection* actorCollection = aRenderer->GetActors();

      for ( actorCollection->InitTraversal();
            (actor = actorCollection->GetNextActor()) != NULL; )
        {
        actor->GetBounds(btmp);
        if (btmp[0] < aBounds[0]) aBounds[0] = btmp[0];
        if (btmp[1] > aBounds[1]) aBounds[1] = btmp[1];
        if (btmp[2] < aBounds[2]) aBounds[2] = btmp[2];
        if (btmp[3] > aBounds[3]) aBounds[3] = btmp[3];
        if (btmp[4] < aBounds[4]) aBounds[4] = btmp[4];
        if (btmp[5] > aBounds[5]) aBounds[5] = btmp[5];
        ++count;
        }

      vtkVolume* volume;
      vtkVolumeCollection* volumeCollection = aRenderer->GetVolumes();

      for ( volumeCollection->InitTraversal();
            (volume = volumeCollection->GetNextVolume()) != NULL; )
        {
        volume->GetBounds(btmp);
        if (btmp[0] < aBounds[0]) aBounds[0] = btmp[0];
        if (btmp[1] > aBounds[1]) aBounds[1] = btmp[1];
        if (btmp[2] < aBounds[2]) aBounds[2] = btmp[2];
        if (btmp[3] > aBounds[3]) aBounds[3] = btmp[3];
        if (btmp[4] < aBounds[4]) aBounds[4] = btmp[4];
        if (btmp[5] > aBounds[5]) aBounds[5] = btmp[5];
        ++count;
        }

      return (count > 0 ? true : false);
      }
    }

  return false;
}

bool
svvController::GetInputBounds (float aBounds[6], vtkRenderer* aRenderer)
{
  int count = 0;

  if (aBounds != NULL)
    {
    if (aRenderer != NULL)
      {
      aBounds[0] = VTK_LARGE_FLOAT; aBounds[1] = -VTK_LARGE_FLOAT;
      aBounds[2] = VTK_LARGE_FLOAT; aBounds[3] = -VTK_LARGE_FLOAT;
      aBounds[4] = VTK_LARGE_FLOAT; aBounds[5] = -VTK_LARGE_FLOAT;

      float btmp[6];

      vtkDataSet* input;
      vtkDataSet* top;

      vtkActor* actor;
      vtkActorCollection* actorCollection = aRenderer->GetActors();
      vtkMapper*  mapper;

      for ( actorCollection->InitTraversal();
            (actor = actorCollection->GetNextActor()) != NULL; )
        {
        if ((mapper = actor->GetMapper()) != NULL)
          {
          if ((input = mapper->GetInput()) != NULL)
            {
            if ( (top = vtkDataSet::SafeDownCast(vtkTraversal::GetTop(input)))
                 != NULL )
              {
              top->GetBounds(btmp);
              if (btmp[0] < aBounds[0]) aBounds[0] = btmp[0];
              if (btmp[1] > aBounds[1]) aBounds[1] = btmp[1];
              if (btmp[2] < aBounds[2]) aBounds[2] = btmp[2];
              if (btmp[3] > aBounds[3]) aBounds[3] = btmp[3];
              if (btmp[4] < aBounds[4]) aBounds[4] = btmp[4];
              if (btmp[5] > aBounds[5]) aBounds[5] = btmp[5];
              ++count;
              }
            }
          }
        }

      vtkVolume* volume;
      vtkVolumeCollection* volumeCollection = aRenderer->GetVolumes();
      vtkVolumeMapper* volumeMapper;

      for ( volumeCollection->InitTraversal();
            (volume = volumeCollection->GetNextVolume()) != NULL; )
        {
        if ((volumeMapper = volume->GetMapper()) != NULL)
          {
          if ((input = volumeMapper->GetInput()) != NULL)
            {
            if ( (top = vtkDataSet::SafeDownCast(vtkTraversal::GetTop(input)))
                 != NULL )
              {
              top->GetBounds(btmp);
              if (btmp[0] < aBounds[0]) aBounds[0] = btmp[0];
              if (btmp[1] > aBounds[1]) aBounds[1] = btmp[1];
              if (btmp[2] < aBounds[2]) aBounds[2] = btmp[2];
              if (btmp[3] > aBounds[3]) aBounds[3] = btmp[3];
              if (btmp[4] < aBounds[4]) aBounds[4] = btmp[4];
              if (btmp[5] > aBounds[5]) aBounds[5] = btmp[5];
              ++count;
              }
            }
          }
        }
      }
    else
      {
      this->GetInputSources()->GetBounds(aBounds);
      count = this->GetInputSources()->GetNumberOfItems();
      }
    }

  return (count > 0 ? true : false);
}

// ----------------------------------------------------------------------------
vtkProp*
svvController::GetAxes (vtkRenderer* aRenderer)
{
  vtkCubeAxesActor2D* actor2D = NULL;

  if (aRenderer != NULL)
    {
    vtkProp* prop;
    vtkPropCollection* propCollection = aRenderer->GetProps();

    for ( propCollection->InitTraversal();
          (prop = propCollection->GetNextProp()) != NULL; )
      if ((actor2D = vtkCubeAxesActor2D::SafeDownCast(prop)) != NULL)
        break;
    }

  return actor2D;
}

vtkProp*
svvController::UpdateAxes (vtkRenderer* aRenderer)
{
  vtkCubeAxesActor2D* actor2D = NULL;

  if (aRenderer != NULL)
    {
    float bounds[6];

    actor2D = vtkCubeAxesActor2D::SafeDownCast(this->GetAxes(aRenderer));

    if (this->GetRenderedBounds(bounds, aRenderer))
      {
      vtkDebugMacro(<< "Overall axes bounds = { " << bounds[0] << ", "
                    << bounds[1] << ", " << bounds[2] << ", " << bounds[3]
                    << ", " << bounds[4] << ", " << bounds[5] << " }");

      if (actor2D == NULL)
        {
        actor2D = vtkCubeAxesActor2D::New();
          {
          actor2D->SetFlyModeToOuterEdges();
          actor2D->ScalingOff();
          actor2D->SetCornerOffset(0.1);
          actor2D->SetInertia(4);
          actor2D->SetLabelFormat("%.1f");
          actor2D->SetFontFactor(0.8);
          actor2D->GetAxisTitleTextProperty()->ShadowOn();
          actor2D->GetAxisLabelTextProperty()->ShadowOn();
          actor2D->GetAxisTitleTextProperty()->AntiAliasingOn();
          actor2D->GetAxisLabelTextProperty()->AntiAliasingOn();
          actor2D->GetXAxisActor2D()->GetTitleTextProperty()->SetColor(1,0,0);
          actor2D->GetYAxisActor2D()->GetTitleTextProperty()->SetColor(1,1,0);
          actor2D->GetZAxisActor2D()->GetTitleTextProperty()->SetColor(0,0,1);
          }
        aRenderer->AddProp(actor2D);
        actor2D->Delete();
        }

      if (actor2D != NULL)
        {
        actor2D->SetCamera(aRenderer->GetActiveCamera());
        actor2D->SetBounds(bounds);
        }
      }
    else
      {
      if (actor2D != NULL)
        {
        aRenderer->RemoveProp(actor2D);
        actor2D = NULL;
        }
      }
    }

  return actor2D;
}

void
svvController::RemoveAxes (vtkRenderer* aRenderer, vtkProp** aRemoved)
{
  if (aRenderer != NULL)
    {
    vtkProp* prop;

    if ((prop = this->GetAxes(aRenderer)) != NULL)
      {
      if (aRemoved != NULL)
        {
        *aRemoved = prop;
        if (*aRemoved != NULL) (*aRemoved)->Register(NULL);
        }
      aRenderer->RemoveProp(prop);
      }
    }
}

bool
svvController::ToggleAxes (vtkRenderer* aRenderer, vtkProp** aProp)
{
  if (aRenderer != NULL)
    {
    vtkProp* prop;

    if ((prop = this->GetAxes(aRenderer)) == NULL)
      {
      prop = this->UpdateAxes(aRenderer);

      if (aProp != NULL)
        {
        *aProp = prop;
        if (*aProp != NULL) (*aProp)->Register(NULL);
        }

      return (prop != NULL ? true : false);
      }

    if (aProp != NULL)
      {
      *aProp = prop;
      if (*aProp != NULL) (*aProp)->Register(NULL);
      }

    aRenderer->RemoveProp(prop);
    }

  return false;
}

// ----------------------------------------------------------------------------
void
svvController::UpdateBoundingBox (vtkRenderer* aRenderer)
{
  float bounds[6] = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                      VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                      VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

  if (aRenderer)
    {
    this->SetBoundingBoxRenderer(aRenderer);

    float btmp[6];

    vtkActor* actor;
    vtkActorCollection* actorCollection = aRenderer->GetActors();

    for ( actorCollection->InitTraversal();
          (actor = actorCollection->GetNextActor()) != NULL; )
      {
      actor->GetBounds(btmp);
      if (btmp[0] < bounds[0]) bounds[0] = btmp[0];
      if (btmp[1] > bounds[1]) bounds[1] = btmp[1];
      if (btmp[2] < bounds[2]) bounds[2] = btmp[2];
      if (btmp[3] > bounds[3]) bounds[3] = btmp[3];
      if (btmp[4] < bounds[4]) bounds[4] = btmp[4];
      if (btmp[5] > bounds[5]) bounds[5] = btmp[5];
      }

    vtkVolume* volume;
    vtkVolumeCollection* volumeCollection = aRenderer->GetVolumes();

    for ( volumeCollection->InitTraversal();
          (volume = volumeCollection->GetNextVolume()) != NULL; )
      {
      volume->GetBounds(btmp);
      if (btmp[0] < bounds[0]) bounds[0] = btmp[0];
      if (btmp[1] > bounds[1]) bounds[1] = btmp[1];
      if (btmp[2] < bounds[2]) bounds[2] = btmp[2];
      if (btmp[3] > bounds[3]) bounds[3] = btmp[3];
      if (btmp[4] < bounds[4]) bounds[4] = btmp[4];
      if (btmp[5] > bounds[5]) bounds[5] = btmp[5];
      }
    }
  else
    {
    this->GetInputSources()->GetBounds(bounds);
    }

  vtkDebugMacro(<< "Overall bounding box bounds = { " << bounds[0] << ", "
		<< bounds[1] << ", " << bounds[2] << ", " << bounds[3]
		<< ", " << bounds[4] << ", " << bounds[5] << " }");

  // If no axes prop exists at this point, try to create a simple one.
  float length = float(svvMath::BoundingLength(bounds));
  float radius = length * 0.0036f;

  if (this->BoundingBox == NULL && length > 0.f)
    {
    vtkAssembly* assembly = vtkAssembly::New();
      {
      vtkActor* outlineActor = vtkActor::New();
        {
        vtkPolyDataMapper* outlineMapper = vtkPolyDataMapper::New();
          {
          vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
            {
            vtkAppendPolyData* appender = vtkAppendPolyData::New();
              {
              // Edges
              vtkTubeFilter* outlineTubes = vtkTubeFilter::New();
                {
                vtkOutlineSource* outlineSource = vtkOutlineSource::New();
                  {
                  outlineSource->SetBounds(bounds);
                  }
                  outlineTubes->SetInput(outlineSource->GetOutput());
                  outlineTubes->SetRadius(length * 0.0012f);
                  outlineTubes->SetNumberOfSides(12);
                  outlineTubes->CappingOff();
                  //outlineTubes->SidesShareVerticesOn(); /* VTK > 4.1 */
                  outlineSource->Delete();
                }
                appender->AddInput(outlineTubes->GetOutput());
                outlineTubes->Delete();
                // Corners
                for (int k=0; k<2; k++)
                  {
                  for (int i=0; i<2; i++)
                    {
                    for (int j=0; j<2; j++)
                      {
                      vtkSphereSource* sphereSource = vtkSphereSource::New();
                        {
                        sphereSource->SetRadius(radius);
                        sphereSource->SetCenter( bounds[j],
                                                 bounds[i+2],
                                                 bounds[k+4] );
                        sphereSource->SetThetaResolution(12);
                        sphereSource->SetPhiResolution(12);
                        }
                        appender->AddInput(sphereSource->GetOutput());
                        sphereSource->Delete();
                      }
                    }
                  }
              }
              normals->SetInput(appender->GetOutput());
              appender->Delete();
            }
            outlineMapper->SetInput(normals->GetOutput());
            normals->Delete();
          }
          outlineActor->SetMapper(outlineMapper);
          outlineMapper->Delete();
        }
        vtkProperty* property;
        property = this->MakeProperty("polished bronze");
        property->BackfaceCullingOn();
        outlineActor->SetProperty(property);
        property->Delete();
        assembly->AddPart(outlineActor);
        outlineActor->Delete();
        // BoundingBox
        vtkCamera* pCamera = 0;
        if (this->MaterialsDemoRenderer)
          {
          pCamera = this->MaterialsDemoRenderer->GetActiveCamera();
          }
        vtkActor* cubeActor = make_cube_actor(bounds,pCamera);
        property = this->MakeProperty("pearl");
        property->SetOpacity(0.08);
        cubeActor->SetProperty(property);
        property->Delete();
        assembly->AddPart(cubeActor);
        cubeActor->Delete();
      }
      this->SetBoundingBox(assembly);
      assembly->Delete();
    }

  if (this->BoundingBox)
    {
    float padded[6], pad = radius/2.f;

    padded[0] = bounds[0] - pad;
    padded[1] = bounds[1] + pad;
    padded[2] = bounds[2] - pad;
    padded[3] = bounds[3] + pad;
    padded[4] = bounds[4] - pad;
    padded[5] = bounds[5] + pad;

    vtkAssembly* pAssembly;
    if (pAssembly = vtkAssembly::SafeDownCast(this->BoundingBox))
      {

      float btmp[6];
      pAssembly->GetBounds(btmp);

      if ( btmp[0]==padded[0] && btmp[0]==padded[1] &&
           btmp[0]==padded[2] && btmp[0]==padded[3] &&
           btmp[0]==padded[4] && btmp[0]==padded[5] ) return;

      pAssembly->SetOrigin(padded[0], padded[2], padded[4]);

      float stmp[3];
      pAssembly->GetScale(stmp);

      float scale[3];
      scale[0] = ((padded[1]-padded[0]) / (btmp[1]-btmp[0])) * stmp[0];
      scale[1] = ((padded[3]-padded[2]) / (btmp[3]-btmp[2])) * stmp[1];
      scale[2] = ((padded[5]-padded[4]) / (btmp[5]-btmp[4])) * stmp[2];

      pAssembly->SetScale(scale);
      }
    }
}

vtkProp3D*
svvController::GetBoundingBox (vtkRenderer* aRenderer)
{
  this->UpdateBoundingBox(aRenderer);
  return this->BoundingBox;
}

// ----------------------------------------------------------------------------
void
svvController::UpdateMaterialsDemo (vtkRenderer* aRenderer)
{
  if (aRenderer != NULL)
    {
    this->SetMaterialsDemoRenderer(aRenderer);
    }

  if (this->MaterialsDemo == NULL)
    {
    SvvSize nMaterials;

    this->MaterialNameMap->GetSize(nMaterials);

    vtkAssembly* assembly = vtkAssembly::New();
      {
      int   width = int(floor(sqrt(nMaterials)));
      int   col   = 0;
      int   row   = 0;
      float incr  = this->MaterialsDemoRadii * 2.5;

      // now map the geometry as a graphic entity
      vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
        {
        vtkDepthSortPolyData* depthSorter = vtkDepthSortPolyData::New();
          {
          vtkCleanPolyData* cleaner = vtkCleanPolyData::New();
            {
#if 1
            vtkSpiralShellSource* source = vtkSpiralShellSource::New();
              {
              if (this->GetDebug()) source->DebugOn();
              source->SetThetaResolution(16*this->MaterialsDemoResolution);
              source->SetPhiResolution(48*this->MaterialsDemoResolution);
              source->SetNumberOfSpirals(1.5);
              source->SetFinalRadius(this->MaterialsDemoRadii * 0.75);
              source->SetInnerRadius(this->MaterialsDemoRadii * 0.25);
              source->SetHeight(this->MaterialsDemoRadii * 2.0);
              source->TriangleTessellationOn();
              source->Update();
              }
#else
              vtkGeosphereSource* source = vtkGeosphereSource::New();
                {
                if (this->GetDebug()) source->DebugOn();
                source->SetRadius(this->MaterialsDemoRadii);
                source->SetNumberOfSubdivisions(this->MaterialsDemoResolution);
                source->SetCenter(0.f, 0.f, 0.f);
                source->Update();
                }
#endif /* 0 */
                cleaner->SetInput(source->GetOutput());
                cleaner->PointMergingOn();
                source->Delete();
            }
            depthSorter->SetInput(cleaner->GetOutput());
            if (this->GetDebug()) depthSorter->DebugOn();
            if (this->MaterialsDemoRenderer)
              {
              vtkCamera* pCamera =
                this->MaterialsDemoRenderer->GetActiveCamera();
              depthSorter->SetCamera(pCamera);
              depthSorter->SetDirectionToBackToFront();
              }
            else
              {
              depthSorter->SetVector(0.0, 0.0, 1.0);
              depthSorter->SetDirectionToSpecifiedVector();
              }
            depthSorter->SetDepthSortModeToParametricCenter();
            depthSorter->SetSortScalars(0/*OFF*/);
            depthSorter->Update();
            cleaner->Delete();
          }
          mapper->SetInput(depthSorter->GetOutput());
          depthSorter->Delete();
        }
      
        const char* name = NULL;
        SvvMaterial material;

        for ( this->MaterialNameMap->InitTraversal();
              this->MaterialNameMap->TraverseForward(name, material); )
          {
          vtkActor* actor = vtkActor::New();
            {
            // make the material property
            vtkProperty* property = svv.MakeProperty(name);

            actor->SetProperty(property);
            property->Delete();
            actor->SetMapper(mapper);
#if 1
            actor->RotateY(45.0);
            actor->RotateX(90.0);
#endif /* 0 */
            actor->SetPosition(col*incr, row*incr, 0.f);
            }
            assembly->AddPart(actor);
            actor->Delete();

            if (++col > width)
              {
              col = 0;
              row++;
              }
          }
        mapper->Delete();
      }
      this->SetMaterialsDemo(assembly);
      assembly->Delete();
    }

  if (this->MaterialsDemo)
    {
    // ...
    }
}

vtkProp3D*
svvController::GetMaterialsDemo (vtkRenderer* aRenderer)
{
  this->UpdateMaterialsDemo(aRenderer);
  return this->MaterialsDemo;
}

// ----------------------------------------------------------------------------
void
svvController::PrintSelf (std::ostream& aTarget, vtkIndent aIndent)
{
  this->vtkObject::PrintSelf(aTarget,aIndent);

  aTarget << aIndent << "MaterialNameMap:"
          << " (" << this->MaterialNameMap << ")" << endl;

  aTarget << aIndent << "ColorNameMap:"
          << " (" << this->ColorNameMap << ")" << endl;

  aTarget << aIndent << "InputSources:"
          << " (" << this->InputSources << ")" << endl;

  aTarget << aIndent << "SurfaceActors:"
          << " (" << this->SurfaceActors << ")" << endl;

  aTarget << aIndent << "SurfacePipelines:"
          << " (" << this->SurfacePipelines << ")" << endl;

  aTarget << aIndent << "Interactor:"
          << " (" << this->Interactor << ")" << endl;

  aTarget << aIndent << "BoundingBox:"
          << " (" << this->BoundingBox << ")" << endl;

  aTarget << aIndent << "BoundingBoxRenderer:"
          << " (" << this->BoundingBoxRenderer << ")" << endl;

  aTarget << aIndent << "MaterialsDemo:"
          << " (" << this->MaterialsDemo << ")" << endl;

  aTarget << aIndent << "MaterialsDemoRadii: "
          << this->MaterialsDemoRadii << endl;

  aTarget << aIndent << "MaterialsDemoResolution: "
          << this->MaterialsDemoResolution << endl;

  aTarget << aIndent << "MaterialsDemoRenderer:"
          << " (" << this->MaterialsDemoRenderer << ")" << endl;
}

SVV_NAMESPACE_END

/* 
 * End of: $Id: svvController.cxx,v 1.1.1.1 2006/12/19 22:58:34 christianh Exp $.
 * 
 */
