/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMath.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_MATH_H_
#  define SVV_MATH_H_
#  include <cmath>
#  include "svvConsts.h" /* includes svvTypes */

#  ifdef SVV_DEBUG
#    include <iostream>
#  endif /* SVV_DEBUG */

SVV_NAMESPACE_BEGIN

/** \class   svvMath
 *  \brief   math routines
 * 
 * svvMath class is composed of inline static math routines (mostly vector and
 * geometric).
 * 
 * \author  $Author: christianh $
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:58:35 $
 */
class svvMath
{
public:
  /** Useful constants. */
  static SvvFloat64 DegreesToRadians(void) { return DegToRad; }
  static SvvFloat64 RadiansToDegrees(void) { return RadToDeg; }
  /** Set the elements of a 3-vector to zero. */
  static void ZeroVec3(float [3]);
  /** Set the elements of a 3-vector to ( x, y, z ). */
  static void SetVec3(float [3], float x, float y, float z);
  /** Subtract one 3-vector from another. */
  static void SubVec3(const float [3], const float [3], float dst[3]);
  /** Copy a 3-vector to a second 3-vector. */
  static void CopyVec3(const float [3], float [3]);
  /** Cross product of two 3-vectors. Result vector in dst[3]. */
  static void CrossVec3(const float [3], const float [3], float x[3]);
  /** Compute the norm of 3-vector. */
  static SvvFloat64 NormVec3(const float v[3]);
  /** product of a 3-vector and a scalar (in place). */
  static void ScaleVec3(float [3], const float c);
  /** product of a 3-vector and a scalar (result in vout). */
  static void ScaleVec3(const float [3], const float c, float dst[3]);
  /** Normalize (in place) a 3-vector. Returns norm of vector. */
  static void NormalizeVec3(float [3]);
  /** Normalize into dst. */
  static void NormalizeVec3(const float [3], float dst[3]);
  /** Dot product of two 3-vectors. */
  static SvvFloat64 DotVec3(const float [3], const float [3]);
  /** Add one 3-vector to another. */
  static void AddVec3(const float [3], const float [3], float dst[3]);
  /** Compute distance squared between two points (represented as 3-vectors). */
  static SvvFloat64 ComputeDistanceVec3(const float [3], const float [3]);
  /** Given an axis and angle, compute quaternion. */
  static void AxisRotationToQuat(float a[3], float phi, float q[4]);
  /** Project an x,y pair onto a sphere of radius r OR a hyperbolic sheet
   * if we are away from the center of the sphere.
   */
  static SvvFloat64 ScreenCoordinatesToSphere(float r, float x, float y);
  /** Simulates a track-ball.
   * Projects the points onto the virtual trackball, then figures out the
   * axis of rotation, which is the cross product of P1 P2 and O P1 (O is
   * the center of the ball, 0,0,0).
   *
   * \note
   * This is a deformed trackball. i.e. it is a trackball in the center, but
   * is deformed into a hyperbolic sheet of rotation away from the center.
   * This particular function was chosen after trying out several variations.
   * It is assumed that the arguments to this routine are in the range [-1..1].
   */
  static void ScreenToQuat(float q[4], float, float, float, float);
  /**
   * Given two rotations, e1 and e2, expressed as quaternion rotations,
   * figure out the equivalent single rotation and stuff it into dest.
   *
   * This routine also normalizes the result every RENORMCOUNT times it is
   * called, to keep error from creeping in.
   *
   * \note
   * This routine is written so that q1 or q2 may be the same as dest
   * (or each other).
   */
  static void AddQuats(float [4], float [4], float dst[4]);
  /** Add two quaternions placing the result into \c dst.
   * Quaternions always obey:  a^2 + b^2 + c^2 + d^2 = 1.0
   * If they don't add up to 1.0, dividing by their magnitude will
   * renormalize them.
   * \note
   * See the following for more information on quaternions:
   *   - Shoemake, K., Animating rotation with quaternion curves, Computer
   *     Graphics 19, No 3 (Proc. SIGGRAPH'85), 245-254, 1985.
   *   - Pletinckx, D., Quaternion calculus as a basic tool in computer
   *     graphics, The Visual Computer 5, 2-13, 1989.
   */
  static void NormalizeQuat(float [4]);
  /** Build a rotation matrix, given a quaternion rotation. */
  static void QuatToMatrix(float m[4][4], float q[4]);
  /** Compute a (unnormalized) triangle normal direction from three points. */
  static void TriangleNormalDirection(float [3], float [3], float [3],
                                      SvvFloat64 n[3]);
  /** Compute a (unnormalized) triangle normal direction from three points.
   * Warning: returned pointer utilizes internal static space.
   */
  static SvvFloat64* TriangleNormalDirection(float [3], float [3], float [3]);
  /** Compute a triangle normal from three points (represented as 3-vectors). */
  static void TriangleNormal(float [3], float [3], float [3], SvvFloat64 n[3]);
  /** Compute a triangle normal from three points (represented as 3-vectors).
   * Warning: returned pointer utilizes internal static space.
   */
  static SvvFloat64* TriangleNormal(float [3], float [3], float [3]);
  /** Compute the center of a triangle. */
  static void TriangleCenter(float [3], float [3], float [3], SvvFloat64 c[3]);
  /** Compute the center of a triangle.
   * Warning: returned pointer utilizes internal static space.
   */
  static SvvFloat64* TriangleCenter(float [3],float [3],float [3]);
  /** Compute the area of a triangle in 3D. */
  static SvvFloat64 TriangleArea(float [3], float [3], float [3]);
  /** Get the length of the diagonal of a bounding box. */
  static SvvFloat64 BoundingLength(float [6]);
  /** Compute the floor of the log base 2 of a unsigned integer
   * (used mostly for computing log2(2^n)).
   */
  static unsigned int FloorLog2(unsigned int);
  /** floor outside of libc. */
  static SvvFloat64 Floor(SvvFloat64);

private:
  svvMath(void);  // Not implemented.
  ~svvMath(void);  // Not implemented.
  svvMath(const svvMath&);  // Not implemented.
  void operator=(const svvMath&);  // Not implemented.
};

// ----------------------------------------------------------------------------
inline void
svvMath::ZeroVec3 (float aVec[3])
{
  aVec[0] = 0.0;
  aVec[1] = 0.0;
  aVec[2] = 0.0;
}

// ----------------------------------------------------------------------------
inline void
svvMath::SetVec3 (float aVec[3], float aX, float aY, float aZ)
{
  aVec[0] = aX;
  aVec[1] = aY;
  aVec[2] = aZ;
}

// ----------------------------------------------------------------------------
inline void
svvMath::SubVec3 (const float* aVec1, const float* aVec2, float* aDstVec)
{
  aDstVec[0] = aVec1[0] - aVec2[0];
  aDstVec[1] = aVec1[1] - aVec2[1];
  aDstVec[2] = aVec1[2] - aVec2[2];
}

// ----------------------------------------------------------------------------
inline void
svvMath::CopyVec3 (const float aVec1[3], float aVec2[3])
{
  aVec2[0] = aVec1[0];
  aVec2[1] = aVec1[1];
  aVec2[2] = aVec1[2];
}

// ----------------------------------------------------------------------------
inline void
svvMath::CrossVec3 (const float aVec1[3], const float aVec2[3], float aCross[3])
{
  SvvFloat64 x = aVec1[1] * aVec2[2] - aVec1[2] * aVec2[1];
  SvvFloat64 y = aVec1[2] * aVec2[0] - aVec1[0] * aVec2[2];
  SvvFloat64 z = aVec1[0] * aVec2[1] - aVec1[1] * aVec2[0];
  aCross[0] = x;
  aCross[1] = y;
  aCross[2] = z; 
}

// ----------------------------------------------------------------------------
inline SvvFloat64
svvMath::NormVec3 (const float aVec[3])
{
  return SvvFloat64(sqrt(aVec[0]*aVec[0] + aVec[1]*aVec[1] + aVec[2]*aVec[2]));
}

inline void
svvMath::ScaleVec3 (float aVec[3], const float aFactor)
{
  aVec[0] *= aFactor;
  aVec[1] *= aFactor;
  aVec[2] *= aFactor;
}

// ----------------------------------------------------------------------------
inline void
svvMath::ScaleVec3 (const float aVec[3], const float aFactor, float aDstVec[3])
{
  aDstVec[0] = aFactor * aVec[0];
  aDstVec[1] = aFactor * aVec[1];
  aDstVec[2] = aFactor * aVec[2];
}

// ----------------------------------------------------------------------------
// Normalize aVec in place.
inline void
svvMath::NormalizeVec3 (float aVec[3])
{
  SvvFloat64 length = aVec[0]*aVec[0] + aVec[1]*aVec[1] + aVec[2]*aVec[2];
  
  if (length <= Zero)
    {
    aVec[0] = Zero;
    aVec[1] = Zero;
    aVec[2] = Zero;
    return;
    }
  else if (length == One)
    {
    return;
    }
  else
    {
#  if 1
    union { unsigned int i; float f; } seed;
    float xy, subexp;
    
    /*
     * This code calculates a reciprocal square root accurate to well over
     * 16 bits using Newton-Raphson approximation.
     *
     * To calculate the seed, the shift compresses the floating-point
     * range just as sqrt() does, and the subtract inverts the range
     * like reciprocation does.  The constant was chosen by trial-and-error
     * to minimize the maximum error of the iterated result for all values
     * over the range .5 to 2.
     */
    seed.f = length;
    seed.i = 0x5f375a00u - (seed.i >> 1);
    
    /*
     * The Newton-Raphson iteration to approximate X = 1/sqrt(Y) is:
     *
     *	X[1] = .5*X[0]*(3 - Y*X[0]^2)
     *
     * A double iteration is:
     *
     *	X[2] = .0625*X[0]*(3 - Y*X[0]^2)*[12 - (Y*X[0]^2)*(3 - Y*X[0]^2)^2]
     *
     */
    xy     = length * seed.f * seed.f;
    subexp = 3.f - xy;
    length = 0.0625f * seed.f * subexp * (12.f - xy * subexp * subexp);
#  else
    length = SvvFloat64(One / sqrt(length));
#  endif /* 0 */
    aVec[0] *= length;
    aVec[1] *= length;
    aVec[2] *= length;
    return;
    }
}

// ----------------------------------------------------------------------------
// Normalize aVec into aDstVec.
inline void
svvMath::NormalizeVec3 (const float aVec[3], float aDstVec[3])
{
  SvvFloat64 length = aVec[0]*aVec[0] + aVec[1]*aVec[1] + aVec[2]*aVec[2];
  
  if (length <= Zero)
    {
    aDstVec[0] = Zero;
    aDstVec[1] = Zero;
    aDstVec[2] = Zero;
    return;
    }
  else if (length == One)
    {
    aDstVec[0] = aVec[0];
    aDstVec[1] = aVec[1];
    aDstVec[2] = aVec[2];
    return;
    }
  else
    {
#  if 1
    union { unsigned int i; float f; } seed;
    float xy, subexp;
    
    /*
     * This code calculates a reciprocal square root accurate to well over
     * 16 bits using Newton-Raphson approximation.
     *
     * To calculate the seed, the shift compresses the floating-point
     * range just as sqrt() does, and the subtract inverts the range
     * like reciprocation does.  The constant was chosen by trial-and-error
     * to minimize the maximum error of the iterated result for all values
     * over the range .5 to 2.
     */
    seed.f = length;
    seed.i = 0x5f375a00u - (seed.i >> 1);
    
    /*
     * The Newton-Raphson iteration to approximate X = 1/sqrt(Y) is:
     *
     *	X[1] = .5*X[0]*(3 - Y*X[0]^2)
     *
     * A double iteration is:
     *
     *	X[2] = .0625*X[0]*(3 - Y*X[0]^2)*[12 - (Y*X[0]^2)*(3 - Y*X[0]^2)^2]
     *
     */
    xy     = length * seed.f * seed.f;
    subexp = 3.f - xy;
    length = 0.0625f * seed.f * subexp * (12.f - xy * subexp * subexp);
#  else
    length = SvvFloat64(One / sqrt(length));
#  endif /* 0 */
    aDstVec[0] = aVec[0] * length;
    aDstVec[1] = aVec[1] * length;
    aDstVec[2] = aVec[2] * length;
    return;
    }
}

// ----------------------------------------------------------------------------
inline SvvFloat64
svvMath::DotVec3 (const float aVec1[3], const float aVec2[3])
{
  return ( SvvFloat64(aVec1[0] * aVec2[0]) +
	   SvvFloat64(aVec1[1] * aVec2[1]) +
	   SvvFloat64(aVec1[2] * aVec2[2]) );
}

// ----------------------------------------------------------------------------
inline void
svvMath::AddVec3 (const float aVec1[3], const float aVec2[3], float aDstVec[3])
{
  aDstVec[0] = aVec1[0] + aVec2[0];
  aDstVec[1] = aVec1[1] + aVec2[1];
  aDstVec[2] = aVec1[2] + aVec2[2];
}

// ----------------------------------------------------------------------------
inline SvvFloat64
svvMath::ComputeDistanceVec3 (const float aVec1[3], const float aVec2[3])
{
  return ( SvvFloat64(aVec1[0] - aVec2[0]) * SvvFloat64(aVec1[0] - aVec2[0]) +
           SvvFloat64(aVec1[1] - aVec2[1]) * SvvFloat64(aVec1[1] - aVec2[1]) +
           SvvFloat64(aVec1[2] - aVec2[2]) * SvvFloat64(aVec1[2] - aVec2[2]) );
}

// ----------------------------------------------------------------------------
// Given an axis and angle, compute quaternion.
inline void
svvMath::AxisRotationToQuat (float aAxis[3], float aPhi, float aQuat[4])
{
  svvMath::NormalizeVec3(aAxis);
  svvMath::CopyVec3(aAxis,aQuat);
  svvMath::ScaleVec3(aQuat,sin(aPhi/2.0));
  aQuat[3] = cos(aPhi/2.0);
}

// ----------------------------------------------------------------------------
// Project an x,y pair onto a sphere of radius r OR a hyperbolic sheet
//   if we are away from the center of the sphere.
inline SvvFloat64
svvMath::ScreenCoordinatesToSphere (float aRadius, float aX, float aY)
{
  SvvFloat64 d = SvvFloat64(sqrt(aX*aX + aY*aY));
  SvvFloat64 z;

  if (d < (SvvFloat64(aRadius) * 0.70710678118654752440))
    { /* Inside sphere */
    z = SvvFloat64(sqrt(aRadius*aRadius - d*d));
    }
  else
    { /* On hyperbola */
    SvvFloat64 t = SvvFloat64(aRadius) / 1.41421356237309504880;
    z = t*t / d;
    }

  return z;
}

// This size should really be based on the distance from the center of
//   rotation to the point on the object underneath the mouse.  That
//   point would then track the mouse as closely as possible.  This is a
//   simple example, though, so that is left as an Exercise for the
//   Programmer.
#  define TRACKBALLSIZE  (0.8f)

// ----------------------------------------------------------------------------
// Simulates a track-ball. Projects the points onto the virtual trackball, 
//   then figures out the axis of rotation, which is the cross product of 
//   P1 P2 and O P1 (O is the center of the ball, 0,0,0).
// Note: This is a deformed trackball. i.e. it is a trackball in the center,
//   but is deformed into a hyperbolic sheet of rotation away from the center. 
//   This particular function was chosen after trying out several variations.
// It is assumed that the arguments to this routine are in the range [-1..1].
inline void
svvMath::ScreenToQuat (float aQuat[4], float aX1,float aY1, float aX2,float aY2)
{
  float axis[3];        /* Axis of rotation */
  float phi;            /* how much to rotate about axis */
  float p1[3];
  float p2[3];
  float d[3];
  float t;

  if (aX1 == aX2 && aY1 == aY2)
    {
    /* Zero rotation */
    svvMath::ZeroVec3(aQuat);
    aQuat[3] = 1.0;
    return;
    }

  // First, figure out z-coord. for projection of P1 and P2 to deformed sphere.
  svvMath::SetVec3( p1,aX1,aY1,
                    svvMath::ScreenCoordinatesToSphere(TRACKBALLSIZE,aX1,aY1) );
  svvMath::SetVec3( p2,aX2,aY2,
                    svvMath::ScreenCoordinatesToSphere(TRACKBALLSIZE,aX2,aY2) );

  // Now, we want the cross product of P1 and P2
  svvMath::CrossVec3(p2,p1,axis);

  // Figure out how much to rotate around that axis.
  svvMath::SubVec3(p1,p2,d);
  t = svvMath::NormVec3(d) / (2.0*TRACKBALLSIZE);

  // Avoid problems with out-of-control values...
  if (t >  1.0) t =  1.0;
  if (t < -1.0) t = -1.0;
  phi = 2.0 * asin(t);

  svvMath::AxisRotationToQuat(axis, phi, aQuat);
}

// ----------------------------------------------------------------------------
#  define RENORMCOUNT 97
// ----------------------------------------------------------------------------
// Given two rotations, e1 and e2, expressed as quaternion rotations,
//   figure out the equivalent single rotation and stuff it into dest.
// This routine also normalizes the result every RENORMCOUNT times it is
//   called, to keep error from creeping in.
// NOTE: This routine is written so that q1 or q2 may be the same
//       as dest (or each other).
inline void
svvMath::AddQuats (float aQuat1[4], float aQuat2[4], float aDstQuat[4])
{
  static int count = 0;
  float t1[4];
  float t2[4];
  float t3[4];
  float tf[4];

#  ifdef SVV_DEBUG
  std::cerr << "q1 = " << aQuat1[0] << " " << aQuat1[1] << " " << aQuat1[2]
            << " " << aQuat1[3] << std::endl;
  std::cerr << "q2 = " << aQuat2[0] << " " << aQuat2[1] << " " << aQuat2[2]
            << " " << aQuat2[3] << std::endl;
#  endif /* SVV_DEBUG */

  svvMath::CopyVec3(aQuat1,t1);
  svvMath::ScaleVec3(t1,aQuat2[3]);

  svvMath::CopyVec3(aQuat2,t2);
  svvMath::ScaleVec3(t2,aQuat1[3]);

  svvMath::CrossVec3(aQuat2,aQuat1,t3);
  svvMath::AddVec3(t1,t2,tf);
  svvMath::AddVec3(t3,tf,tf);
  tf[3] = aQuat1[3] * aQuat2[3] - svvMath::DotVec3(aQuat1,aQuat2);

#  ifdef SVV_DEBUG
  std::cerr << "tf = " << tf[0] << " " << tf[1] << " " << tf[2]
            << " " << tf[3] << std::endl;
#  endif /* SVV_DEBUG */

  aDstQuat[0] = tf[0];
  aDstQuat[1] = tf[1];
  aDstQuat[2] = tf[2];
  aDstQuat[3] = tf[3];

  if (++count > RENORMCOUNT)
    {
    count = 0;
    svvMath::NormalizeQuat(aDstQuat);
    }
}

// ----------------------------------------------------------------------------
// Quaternions always obey:  a^2 + b^2 + c^2 + d^2 = 1.0
//   If they don't add up to 1.0, dividing by their magnitude will
//   renormalize them.
// Note: See the following for more information on quaternions:
//   - Shoemake, K., Animating rotation with quaternion curves, Computer
//     Graphics 19, No 3 (Proc. SIGGRAPH'85), 245-254, 1985.
//   - Pletinckx, D., Quaternion calculus as a basic tool in computer
//     graphics, The Visual Computer 5, 2-13, 1989.
inline void
svvMath::NormalizeQuat (float aQuat[4])
{
  float magnitude = sqrt( aQuat[0]*aQuat[0] +
                          aQuat[1]*aQuat[1] +
                          aQuat[2]*aQuat[2] +
                          aQuat[3]*aQuat[3] );

  for (int i=0; i<4; i++) aQuat[i] /= magnitude;
}

// ----------------------------------------------------------------------------
// Build a rotation matrix, given a quaternion rotation.
inline void
svvMath::QuatToMatrix (float aMatrix[4][4],
                       float aQuat[4])
{
  aMatrix[0][0] = 1.0 - 2.0 * (aQuat[1] * aQuat[1] + aQuat[2] * aQuat[2]);
  aMatrix[0][1] = 2.0 * (aQuat[0] * aQuat[1] - aQuat[2] * aQuat[3]);
  aMatrix[0][2] = 2.0 * (aQuat[2] * aQuat[0] + aQuat[1] * aQuat[3]);
  aMatrix[0][3] = 0.0;

  aMatrix[1][0] = 2.0 * (aQuat[0] * aQuat[1] + aQuat[2] * aQuat[3]);
  aMatrix[1][1] = 1.0 - 2.0 * (aQuat[2] * aQuat[2] + aQuat[0] * aQuat[0]);
  aMatrix[1][2] = 2.0 * (aQuat[1] * aQuat[2] - aQuat[0] * aQuat[3]);
  aMatrix[1][3] = 0.0;

  aMatrix[2][0] = 2.0 * (aQuat[2] * aQuat[0] - aQuat[1] * aQuat[3]);
  aMatrix[2][1] = 2.0 * (aQuat[1] * aQuat[2] + aQuat[0] * aQuat[3]);
  aMatrix[2][2] = 1.0 - 2.0 * (aQuat[1] * aQuat[1] + aQuat[0] * aQuat[0]);
  aMatrix[2][3] = 0.0;

  aMatrix[3][0] = 0.0;
  aMatrix[3][1] = 0.0;
  aMatrix[3][2] = 0.0;
  aMatrix[3][3] = 1.0;
}

// 
// Triangle stuff
// 

// ----------------------------------------------------------------------------
inline void
svvMath::TriangleNormalDirection (float aVertex1[3],
                                  float aVertex2[3],
                                  float aVertex3[3],
                                  SvvFloat64 aNormal[3])
{
  // order is important!!! maintain consistency with triangle vertex order 
  register SvvFloat64 ax = SvvFloat64(aVertex3[0] - aVertex2[0]);
  register SvvFloat64 ay = SvvFloat64(aVertex3[1] - aVertex2[1]);
  register SvvFloat64 az = SvvFloat64(aVertex3[2] - aVertex2[2]);
  register SvvFloat64 bx = SvvFloat64(aVertex1[0] - aVertex2[0]);
  register SvvFloat64 by = SvvFloat64(aVertex1[1] - aVertex2[1]);
  register SvvFloat64 bz = SvvFloat64(aVertex1[2] - aVertex2[2]);

  aNormal[0] = (ay * bz - az * by);
  aNormal[1] = (az * bx - ax * bz);
  aNormal[2] = (ax * by - ay * bx);
}

// ----------------------------------------------------------------------------
// Warning: returned pointer utilizes internal static space.
inline SvvFloat64*
svvMath::TriangleNormalDirection (float aVertex1[3],
                                  float aVertex2[3],
                                  float aVertex3[3])
{
  static SvvFloat64   n[3] = { Zero, Zero, Zero };

  // order is important!!! maintain consistency with triangle vertex order 
  register SvvFloat64 ax = SvvFloat64(aVertex3[0] - aVertex2[0]);
  register SvvFloat64 ay = SvvFloat64(aVertex3[1] - aVertex2[1]);
  register SvvFloat64 az = SvvFloat64(aVertex3[2] - aVertex2[2]);
  register SvvFloat64 bx = SvvFloat64(aVertex1[0] - aVertex2[0]);
  register SvvFloat64 by = SvvFloat64(aVertex1[1] - aVertex2[1]);
  register SvvFloat64 bz = SvvFloat64(aVertex1[2] - aVertex2[2]);

  n[0] = (ay * bz - az * by);
  n[1] = (az * bx - ax * bz);
  n[2] = (ax * by - ay * bx);

  return n;
}

// ----------------------------------------------------------------------------
inline void
svvMath::TriangleNormal (float aVertex1[3],
                         float aVertex2[3],
                         float aVertex3[3],
                         SvvFloat64 aNormal[3])
{
  register SvvFloat64 length;
  
  svvMath::TriangleNormalDirection(aVertex1, aVertex2, aVertex3, aNormal);
  
  if ( ( length = sqrt( aNormal[0]*aNormal[0] +
                        aNormal[1]*aNormal[1] +
                        aNormal[2]*aNormal[2] ) ) != 0.0 )
    {
    aNormal[0] /= length;
    aNormal[1] /= length;
    aNormal[2] /= length;
    }
}

// ----------------------------------------------------------------------------
// Warning: returned pointer utilizes internal static space.
inline SvvFloat64*
svvMath::TriangleNormal (float aVertex1[3],
                         float aVertex2[3],
                         float aVertex3[3])
{
  static SvvFloat64 n[3] = { Zero, Zero, Zero };

  register SvvFloat64 length;
  
  svvMath::TriangleNormalDirection(aVertex1, aVertex2, aVertex3, n);
  
  if ((length = sqrt(n[0]*n[0] + n[1]*n[1] + n[2]*n[2])) != 0.0)
    {
    n[0] /= length;
    n[1] /= length;
    n[2] /= length;
    }

  return n;
}

// ----------------------------------------------------------------------------
inline void
svvMath::TriangleCenter (float aVertex1[3],
                         float aVertex2[3],
                         float aVertex3[3],
                         SvvFloat64 aCenter[3])
{
  aCenter[0] = SvvFloat64(aVertex1[0] + aVertex2[0] + aVertex3[0]) / 3.0;
  aCenter[1] = SvvFloat64(aVertex1[1] + aVertex2[1] + aVertex3[1]) / 3.0;
  aCenter[2] = SvvFloat64(aVertex1[2] + aVertex2[2] + aVertex3[2]) / 3.0;
}

// ----------------------------------------------------------------------------
// Warning: returned pointer utilizes internal static space.
inline SvvFloat64*
svvMath::TriangleCenter (float aVertex1[3],
                         float aVertex2[3],
                         float aVertex3[3])
{
  static SvvFloat64 center[3] = { Zero, Zero, Zero };

  center[0] = SvvFloat64(aVertex1[0] + aVertex2[0] + aVertex3[0]) / 3.0;
  center[1] = SvvFloat64(aVertex1[1] + aVertex2[1] + aVertex3[1]) / 3.0;
  center[2] = SvvFloat64(aVertex1[2] + aVertex2[2] + aVertex3[2]) / 3.0;

  return center;
}

// ----------------------------------------------------------------------------
inline SvvFloat64
svvMath::TriangleArea (float aVertex1[3],
                       float aVertex2[3],
                       float aVertex3[3])
{
  register SvvFloat64 a = svvMath::ComputeDistanceVec3(aVertex1, aVertex2);
  register SvvFloat64 b = svvMath::ComputeDistanceVec3(aVertex2, aVertex3);
  register SvvFloat64 c = svvMath::ComputeDistanceVec3(aVertex3, aVertex1);

  return SvvFloat64(0.25* sqrt(fabs((double)4.0*a*c - (a-b+c)*(a-b+c))));
}

// ----------------------------------------------------------------------------
// Get the length of the diagonal of a bounding box.
inline SvvFloat64
svvMath::BoundingLength (float aBounds[6])
{
  double diff;
  double length = 0.0;

  for (int i=0; i<3; i++)
    {
    diff    = aBounds[2*i+1] - aBounds[2*i];
    length += (diff * diff);
    }
 
  return SvvFloat64(sqrt(length));
}

// ----------------------------------------------------------------------------
// Compute the floor of the log base 2 of a unsigned integer
//   (used mostly for computing log2(2^n)).
inline unsigned int
svvMath::FloorLog2 (unsigned int a)
{
  int i = 1;
  while((a >> i) > 0) i++;
  return i-1;
}

// ----------------------------------------------------------------------------
// floor outside of libc.
inline SvvFloat64
svvMath::Floor (SvvFloat64 a)
{
  SvvFloat64 retf;

  retf = (int)a;
  if (a < 0.0)
    {
    retf -= 1.0;
    }

  return retf;
}

SVV_NAMESPACE_END

#endif /* SVV_MATH_H_ */
/* 
 * End of: $Id: svvMath.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $.
 * 
 */
