/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvModelSource.cxx,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvModelSource.h"
#include "models/teapot.h"
#include "models/cow.h"
#include "models/face.h"
#include "models/mechanical.h"
#include "models/motor.h"
#include "models/pialsurf.h"
#include "models/spheresurf.h"
#include "models/inflatedsurf.h"
#include "models/smoothwmsurf.h"
// VTK
#include "vtkPoints.h"
#include "vtkPointData.h"
#include "vtkCellArray.h"
#include "vtkFloatArray.h"
#include "vtkPolyData.h"
#include "vtkObjectFactory.h"

SVV_NAMESPACE_BEGIN

// ----------------------------------------------------------------------------
//      s v v M o d e l S o u r c e
// ----------------------------------------------------------------------------
vtkCxxRevisionMacro (svvModelSource, "$Revision: 1.1.1.1 $");
vtkStandardNewMacro (svvModelSource);

// ----------------------------------------------------------------------------
svvModelSource::svvModelSource (void)
{
  this->Model = VTK_EXT_TEAPOT;
  this->Scale = 1.f;
  this->Center[0] = this->Center[1] = this->Center[2] = 0.f;
  this->Normals = 1;
}

svvModelSource::~svvModelSource()
{
}

// ----------------------------------------------------------------------------
void
svvModelSource::ExecuteData(vtkDataObject* vtkNotUsed(output))
{
  vtkPoints*     newPoints; 
  vtkCellArray*  newPolys;
  vtkFloatArray* newNormals;
  vtkPolyData*   pPolyData = this->GetOutput();

  vtkIdType nVertices, nIndices;
  GLfloat  (*pVertices)[6];
  GLushort (*pIndices)[3];

  switch (this->Model)
    {
    case VTK_EXT_COW:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(cow.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(cow.indices.pointer);
      nVertices = cow.vertices.count;
      nIndices  = cow.indices.count;
      break;
    case VTK_EXT_FACE:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(face.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(face.indices.pointer);
      nVertices = face.vertices.count;
      nIndices  = face.indices.count;
      break;
    case VTK_EXT_BOILER:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(mechanical.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(mechanical.indices.pointer);
      nVertices = mechanical.vertices.count;
      nIndices  = mechanical.indices.count;
      break;
    case VTK_EXT_MOTOR:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(motor.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(motor.indices.pointer);
      nVertices = motor.vertices.count;
      nIndices  = motor.indices.count;
      break;
    case VTK_EXT_PIALSURF:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(pialsurf.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(pialsurf.indices.pointer);
      nVertices = pialsurf.vertices.count;
      nIndices  = pialsurf.indices.count;
      break;
    case VTK_EXT_SPHERESURF:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(spheresurf.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(spheresurf.indices.pointer);
      nVertices = spheresurf.vertices.count;
      nIndices  = spheresurf.indices.count;
      break;
    case VTK_EXT_INFLATEDSURF:
      pVertices =
        reinterpret_cast<GLfloat (*)[6]>(inflatedsurf.vertices.pointer);
      pIndices  =
        reinterpret_cast<GLushort (*)[3]>(inflatedsurf.indices.pointer);
      nVertices = inflatedsurf.vertices.count;
      nIndices  = inflatedsurf.indices.count;
      break;
    case VTK_EXT_CORTEXSURF:
      pVertices =
        reinterpret_cast<GLfloat (*)[6]>(smoothwmsurf.vertices.pointer);
      pIndices  =
        reinterpret_cast<GLushort (*)[3]>(smoothwmsurf.indices.pointer);
      nVertices = smoothwmsurf.vertices.count;
      nIndices  = smoothwmsurf.indices.count;
      break;
    case VTK_EXT_TEAPOT:
    default:
      pVertices = reinterpret_cast<GLfloat (*)[6]>(teapot.vertices.pointer);
      pIndices  = reinterpret_cast<GLushort (*)[3]>(teapot.indices.pointer);
      nVertices = teapot.vertices.count;
      nIndices  = teapot.indices.count;
      break;
    }

  // Specify number of points for object. Does allocation and sets MaxId ivar.
  newPoints = vtkPoints::New();
  newPoints->SetNumberOfPoints(nVertices);

  if (this->Normals)
    {
    // allocate the vertex normals array
    newNormals = vtkFloatArray::New();
    newNormals->SetNumberOfComponents(3);
    newNormals->Allocate(3*nVertices);
    }

  // allocate the cells
  newPolys  = vtkCellArray::New();
  newPolys->Allocate(newPolys->EstimateSize(nIndices,3));

  //
  // Create model
  //

  vtkDebugMacro(<< "Creating points ...");

  // create point coordinates
  float pts[3];
  for (vtkIdType id=0; id<nVertices; id++)
    {
    // Create point coordinates at the transformed position.
    pts[0] = pVertices[id][3] * this->Scale + this->Center[0];
    pts[1] = pVertices[id][4] * this->Scale + this->Center[1];
    pts[2] = pVertices[id][5] * this->Scale + this->Center[2];
    newPoints->SetPoint(id, pts);
    }
  this->UpdateProgress(0.333);

  if (this->Normals)
    {
    for (vtkIdType id=0; id<nVertices; id++)
      {
      newNormals->InsertNextTuple(&pVertices[id][0]);
      }
    }
  this->UpdateProgress(0.667);

  vtkDebugMacro(<< "Generating mesh connectivity ...");

  // create cells
  vtkIdType cellIds[3];
  for (vtkIdType id=0; id<nIndices; id++)
    {
    for (vtkIdType n=0; n<3; n++) cellIds[n] = vtkIdType(pIndices[id][n]);
    newPolys->InsertNextCell(3, cellIds);
    }

  vtkDebugMacro(<< "Updating the PolyData and releasing memory.");

  // Update the PolyData and release memeory
  newPoints->Squeeze();
  pPolyData->SetPoints(newPoints);
  newPoints->Delete();

  if (this->Normals)
    {
    newNormals->Squeeze();
    pPolyData->GetPointData()->SetNormals(newNormals);
    newNormals->Delete();
    }

  newPolys->Squeeze();
  pPolyData->SetPolys(newPolys);
  newPolys->Delete();

  vtkDebugMacro(<< pPolyData->GetNumberOfPoints() << " points, "
                << pPolyData->GetNumberOfCells() << " cells, "
                << pPolyData->GetNumberOfVerts() << " verts, "
                << pPolyData->GetNumberOfLines() << " lines, "
                << pPolyData->GetNumberOfPolys() << " polys, "
                << pPolyData->GetNumberOfStrips() << " strips");

  this->UpdateProgress(1.0);
}

// ----------------------------------------------------------------------------
const char*
svvModelSource::GetModelAsString (void)
{
  switch (this->Model)
    {
    case VTK_EXT_TEAPOT:       return "teapot";
    case VTK_EXT_COW:          return "cow";
    case VTK_EXT_FACE:         return "face";
    case VTK_EXT_BOILER:       return "boiler";
    case VTK_EXT_MOTOR:        return "motor";
    case VTK_EXT_PIALSURF:     return "pialsurf";
    case VTK_EXT_SPHERESURF:   return "spheresurf";
    case VTK_EXT_INFLATEDSURF: return "inflatedsurf";
    case VTK_EXT_CORTEXSURF:   return "cortexsurf";
    default:                   return "default";
    }
}

// ----------------------------------------------------------------------------
void
svvModelSource::PrintSelf (ostream& aTarget, vtkIndent aIndent)
{
  this->Superclass::PrintSelf(aTarget, aIndent);

  aTarget << aIndent << "Scale:   " << this->Scale << endl;

  aTarget << aIndent << "Center:  (" << this->Center[0] << ", " 
          << this->Center[1] << ", " << this->Center[2] << ")" << endl;

  aTarget << aIndent << "Model:   " << this->GetModelAsString() << endl;

  aTarget << aIndent << "Normals: " << (this->Normals?"ON":"OFF") << endl;
}

SVV_NAMESPACE_END
/* 
 * End of: $Id: svvModelSource.cxx,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $.
 * 
 */
