/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvTypes.h,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_TYPES_H_
#  define SVV_TYPES_H_
/* First include the local configuration for this machine. */
#  include "svvConfiguration.h"
#  include "vtkExtensionsTypes.h"
/* system */
#  include <limits.h>
#  include <float.h>
/* OpenGL */
#  include <GL/gl.h>

SVV_BEGIN_EXTERN_C

/*@{*/
/** vtkSizeType is the unsigned integral type. */
typedef vtkSizeType             SvvSize;
typedef vtkSSizeType            SvvSSize;
#  define SVV_SIZE_MAX          VTK_EXTENSIONS_SIZE_MAX
#  define SVV_SSIZE_MAX         VTK_EXTENSIONS_SSIZE_MAX
/*@}*/

/*@{*/
/** The maximum and minimum values of a vtkInt8Type or vtkUint8Type. */
typedef vtkUint8Type            SvvUint8;
typedef vtkInt8Type             SvvInt8;
#  define SVV_INT8_MAX          VTK_EXTENSIONS_INT8_MAX
#  define SVV_INT8_MIN          VTK_EXTENSIONS_INT8_MIN
#  define SVV_UINT8_MAX         VTK_EXTENSIONS_UINT8_MAX
/*@}*/

/*@{*/
/** The maximum and minimum values of a vtkInt16Type or vtkUint16Type. */
typedef vtkUint16Type           SvvUint16;
typedef vtkInt16Type            SvvInt16;
#  define SVV_INT16_MAX         VTK_EXTENSIONS_INT16_MAX
#  define SVV_INT16_MIN         VTK_EXTENSIONS_INT16_MIN
#  define SVV_UINT16_MAX        VTK_EXTENSIONS_UINT16_MAX
/*@}*/

/*@{*/
/** The int32 types are known to be 32 bits each. */
typedef vtkUint32Type           SvvUint32;
typedef vtkInt32Type            SvvInt32;
#  define SVV_INT32(_x)         VTK_EXTENSIONS_INT32(_x)
#  define SVV_UINT32(_x)        VTK_EXTENSIONS_UINT32(_x)
/*@}*/

/*@{*/
/** The maximum and minimum values of a vtkInt32Type or vtkUint32Type. */
#  define SVV_INT32_MAX         VTK_EXTENSIONS_INT32_MAX
#  define SVV_INT32_MIN         VTK_EXTENSIONS_INT32_MIN
#  define SVV_UINT32_MAX        VTK_EXTENSIONS_UINT32_MAX
/*@}*/


/*@{*/
/** Bit masking macros (n must be <= 31 to be portable). */
#  define SVV_BIT(_n)           VTK_EXTENSIONS_BIT(_n)
#  define SVV_BITMASK(_n)       VTK_EXTENSIONS_BITMASK(_n)
/*@}*/

/*@{*/
/** The vtkIntnType types are most appropriate for automatic variables. 
 * They are guaranteed to be at least 16 bits, though various architectures 
 * may define them to be wider (e.g., 32 or even 64 bits). These types are
 * never valid for fields of a structure. 
 */
typedef vtkIntnType             SvvIntn;
typedef vtkUintnType            SvvUintn;
/*@}*/

/*@{*/
/** vtkExtensions uses both 32 and 64 bit floating point types. */
typedef vtkFloat32Type          SvvFloat32;
typedef vtkFloat64Type          SvvFloat64;
/*@}*/

/*@{*/
/** A type for representing byte offsets from some location. */
typedef vtkOffset32Type         SvvOffset32;
typedef vtkOffset64Type         SvvOffset64;
/*@}*/

/*@{*/
/** A type for pointer difference.
 * Variables of this type are suitable for storing a pointer or pointer
 * sutraction. 
 */
typedef vtkPtrdiffType          SvvPtrdiff;
typedef vtkUptrdiffType         SvvUptrdiff;
/*@}*/

/*@{*/
/** Use SvvBool for variables and parameter types.
 * Use SVV_FALSE and SVV_TRUE for clarity of target
 * type in assignments and actual arguments. Use 'if (bool)',
 * 'while (!bool)', '(bool) ? x : y', etc. to test booleans just as you
 * would C int-valued conditions. 
 */
typedef vtkBoolType             SvvBool;
#  define SVV_TRUE      VTK_EXTENSIONS_TRUE
#  define SVV_FALSE     VTK_EXTENSIONS_FALSE
/*@}*/

/** Use SvvPackedBool within structs.
 * Use where bitfields are not desireable but minimum and consistant
 * overhead matters.
 */
typedef vtkPackedBoolType       SvvPackedBool;

/**
 * Used by routines that have single point of failure or special status return.
 */
typedef enum
{
  SVV_FAILURE = -1,
  SVV_SUCCESS = 0
} SvvStatusType;

/** Generic result data type. */
typedef vtkResultType           SvvResult;

/*@{*/
/** Commonly used macros for operations on compatible types. */
#  define SVV_MAX(_a,_b)                VTK_EXTENSIONS_MAX(_a,_b)
#  define SVV_MIN(_a,_b)                VTK_EXTENSIONS_MIN(_a,_b)
#  define SVV_CLAMP(_v,_min,_max)       VTK_EXTENSIONS_CLAMP(_v,_min,_max)
#  define SVV_ABS(_x)                   VTK_EXTENSIONS_ABS(_x)
#  define SVV_ROUNDUP(_x,_y)            VTK_EXTENSIONS_ROUNDUP(_x,_y)
/*@}*/

/*@{*/
/** Use these macros to do 64bit safe pointer conversions. */
#  define SVV_PTR_TO_INT32(_x)          VTK_EXTENSIONS_PTR_TO_INT32(_x)
#  define SVV_INT32_TO_PTR(_x)          VTK_EXTENSIONS_INT32_TO_PTR(_x)
/*@}*/

/*@{*/
/** \name Color data types. */
/*@{*/
/** A color is a 32 bit unsigned integer with four components: R, G, B, and A.*/
typedef SvvUint32               SvvColorWord;
typedef SvvColorWord*           SvvColorWordPtr;
typedef SvvColorWord&           SvvColorWordRef;
typedef const SvvColorWord      SvvConstColorWord;
typedef const SvvColorWordPtr   SvvConstColorWordPtr;
typedef const SvvColorWordRef   SvvConstColorWordRef;
/*@}*/
/*@{*/
/** the four components (channels) are 8 bit unsigned untegers. */
typedef SvvUint8                SvvColorByte;
typedef SvvColorByte*           SvvColorBytePtr;
typedef SvvColorByte&           SvvColorByteRef;
typedef const SvvColorByte      SvvConstColorByte;
typedef const SvvColorBytePtr   SvvConstColorBytePtr;
typedef const SvvColorByteRef   SvvConstColorByteRef;
/*@}*/
/*@}*/

/** brightest value for R,G,B chars. */
#  define SVV_RGB_MAX   SVV_UINT8_MAX

/** make an RGBA SvvColorWord. */
#  define SVV_MAKE_RGBA(_r,_g,_b,_a) \
        ((SvvColorWord)(((_r) << 24) | ((_g) << 16) | ((_b) << 8) | (_a)))
/** make an RGB SvvColorWord. */
#  define SVV_MAKE_RGB(_r,_g,_b)    SVV_MAKE_RGBA(_r,_g,_b,SVV_RGB_MAX)

/** make an ABGR SvvColorWord. */
#  define SVV_MAKE_ABGR(_a,_b,_g,_r) \
        ((SvvColorWord)(((_a) << 24) | ((_b) << 16) | ((_g) << 8) | (_r)))
/** make an BGR SvvColorWord. */
#  define SVV_MAKE_BGR(_b,_g,_r)    SVV_MAKE_ABGR(SVV_RGB_MAX,_b,_g,_r)

/*@{*/
/** get one byte value from an RGBA SvvColorWord. */
#  define SVV_GET_RGBA_R(_rgba)     ((SvvColorByte)(((_rgba) >> 24) & 0xff))
#  define SVV_GET_RGBA_G(_rgba)     ((SvvColorByte)(((_rgba) >> 16) & 0xff))
#  define SVV_GET_RGBA_B(_rgba)     ((SvvColorByte)(((_rgba) >> 8) & 0xff))
#  define SVV_GET_RGBA_A(_rgba)     ((SvvColorByte)((_rgba) & 0xff))
/*@}*/

/*@{*/
/** get one byte value from an ABGR SvvColorWord. */
#  define SVV_GET_ABGR_R(_rgba)     ((SvvColorByte)((_rgba) & 0xff))
#  define SVV_GET_ABGR_G(_rgba)     ((SvvColorByte)(((_rgba) >> 8) & 0xff))
#  define SVV_GET_ABGR_B(_rgba)     ((SvvColorByte)(((_rgba) >> 16) & 0xff))
#  define SVV_GET_ABGR_A(_rgba)     ((SvvColorByte)(((_rgba) >> 24) & 0xff))
/*@}*/

/** get value of SvvColorWord's RGB component exclusive of alpha component. */
#  define SVV_RGBONLY(_rgba)        ((_rgba) & 0xffffff00)

/** get value of SvvColorWord's RGB component exclusive of alpha component. */
#  define SVV_BGRONLY(_abgr)        ((_abgr) & 0x00ffffff)

#  define SVV_GET_BYTE(_rgba,_n) \
        ((SvvColorByte)(((_rgba) >> ((3-_n)<<3)) & 0xff))

/** reference a byte within an RGBA SvvColorWord. */
#  define SVV_BYTE_REF(_word,_n)    (((SvvColorBytePtr)(&_word))[_n])

/** get a pointer to a SvvColorByte within an RGBA SvvColorWord. */
#  define SVV_BYTE_PTR(_word,_n)    (&(SVV_BYTE_REF(_word,_n)))

#  define SVV_CLAMPBYTE(_c) \
  ((_c)<=0.0 ? 0 : ((_c)>(double)(SVV_RGB_MAX)?SVV_RGB_MAX:(SvvColorByte)(_c)))

/************************************************************************/

/*
 * Implementation data types.  The implementation is designed to run in
 * single precision or double precision mode, all of which is controlled
 * by an ifdef and the following typedef's.
 *
 * Use SVV_MAX_GL_REAL to get the largest representable Real
 */
#  ifdef SVV_GL_DOUBLE_PRECISION
typedef vtkFloat64Type  GLreal;
#    define SVV_MAX_GL_REAL DBL_MAX
#  else
typedef vtkFloat32Type  GLreal;
#    define SVV_MAX_GL_REAL FLT_MAX
#  endif /* SVV_GL_DOUBLE_PRECISION */

typedef GLenum          SvvEnum;
typedef GLboolean       SvvBoolean;
typedef GLbitfield      SvvBitfield;
typedef GLbyte          SvvByte;
typedef GLubyte         SvvUByte;
typedef GLshort         SvvShort;
typedef GLushort        SvvUShort;
typedef GLint           SvvInt;
typedef GLuint          SvvUInt;
typedef GLsizei         SvvSizei;
typedef GLfloat         SvvFloat;
typedef GLclampf        SvvClampf;
typedef GLdouble        SvvDouble;
typedef GLclampd        SvvClampd;
typedef GLreal          SvvReal;
typedef GLvoid          SvvVoid;

/*@{*/
/** Material data types. */
typedef struct _SvvGLMaterialRec
{
  SvvReal       Ka[4];      /* [00..15] (scaled) */
  SvvReal       Kd[4];      /* [16..31] (scaled) */
  SvvReal       Ks[4];      /* [32..47] (scaled) */
  SvvReal       Ke[4];      /* [48..63] (scaled) */
  SvvReal       Se;         /* [64..67] */
} SvvGLMaterial;        /* 68 bytes */

typedef struct _SvvMaterialRec
{
  SvvFloat32    Ka[3];  /* [00..11] (scaled) */
  SvvFloat32    Kd[4];  /* [12..27] (scaled) */
  SvvFloat32    Ks[3];  /* [28..39] (scaled) */
  SvvFloat32    Se;     /* [40..43] */
} SvvMaterial;          /* 44 bytes */

typedef struct _SvvPackedMaterialRec
{
  SvvColorWord  Ka;     /* [00..03] (4th component of Ka undefined) */
  SvvColorWord  Kd;     /* [04..07] (store alpha as 4th component of Kd) */
  SvvColorWord  Ks;     /* [08..11] (store Se as 4th component of Ks) */
} SvvPackedMaterial;    /* 12 bytes */
/*@}*/

/*
 * Number of texture units.  Each texture unit includes a complete
 * set of texture bindings and a complete set of environment bindings.
 */
#  define SVV_NUM_TEXTURE_UNITS_GUESS 4

/************************************************************************/

/*
 * Coordinate structure.  Coordinates contain x, y, z and w.
 */
typedef struct __SvvCoordRec
{
  SvvReal x, y, z, w;
} SvvCoord;

/*
 * Color structure.  Colors are composed of red, green, blue and alpha.
 */
typedef struct __SvvColorRec
 {
   SvvReal r, g, b, a;
 } SvvColor;

/************************************************************************
 * Lighting                                                             *
 ************************************************************************/

/* Values for cmParam */
#  define SVV_EMISSION			0
#  define SVV_AMBIENT			1
#  define SVV_SPECULAR			2
#  define SVV_AMBIENT_AND_DIFFUSE	3
#  define SVV_DIFFUSE			4

/*
 * Light state.  Contains all the user controllable lighting state.
 * Most of the colors kept in user state are scaled to match the
 * drawing surfaces color resolution.
 */

typedef struct __SvvMaterialStateRec
{
  SvvColor              ambient;  /* unscaled */
  SvvColor              diffuse;  /* unscaled */
  SvvColor              specular; /* unscaled */
  SvvColor              emissive; /* scaled */
  SvvReal               specularExponent;
  SvvReal               cmapa;
  SvvReal               cmaps;
  SvvReal               cmapd;
} SvvMaterialState;

/************************************************************************/

typedef struct __SvvLightModelStateRec
{
  SvvColor              ambient;  /* scaled */
  SvvBoolean            localViewer;
  SvvBoolean            twoSided;
  SvvEnum               colorControl;
} SvvLightModelState;

typedef struct __SvvLightSourceStateRec
{
  SvvColor              ambient;  /* scaled */
  SvvColor              diffuse;  /* scaled */
  SvvColor              specular; /* scaled */
  SvvCoord              position;
  SvvCoord              positionEye;
  SvvCoord              direction;
  SvvReal               spotLightExponent;
  SvvReal               spotLightCutOffAngle;
  SvvReal               constantAttenuation;
  SvvReal               linearAttenuation;
  SvvReal               quadraticAttenuation;
} SvvLightSourceState;

typedef struct __SvvLightStateRec
{
  SvvEnum               colorMaterialFace;
  SvvEnum               colorMaterialParam;
  SvvEnum               shadingModel;
  SvvLightModelState    model;
  SvvMaterialState      front;
  SvvMaterialState      back;
  SvvLightSourceState*  source;
} SvvLightState;

/************************************************************************/

/*
 * Per light source per material computed state.
 */
typedef struct __SvvLightSourcePerMaterialMachineRec
{
  SvvColor              ambient;  /* light ambient times material ambient */
  SvvColor              diffuse;  /* light diffuse times material diffuse */
  SvvColor              specular; /* light specular times material specular */
} SvvLightSourcePerMaterialMachine;

/*
 * Per light source computed state.
 */
typedef struct __SvvLightSourceMachineRec
{
  /*
   * ambient, diffuse and specular are each pre-multiplied by the
   * material ambient, material diffuse and material specular.
   * We use the face being lit to pick between the two sets.
   */
  SvvLightSourcePerMaterialMachine      front;
  SvvLightSourcePerMaterialMachine      back;
  
  SvvLightSourceState*  state;

  SvvReal               constantAttenuation;
  SvvReal               linearAttenuation;
  SvvReal               quadraticAttenuation;
  SvvReal               spotLightExponent;
  
  /* Position of the light source in eye coordinates */
  SvvCoord              position;
  
  /* Direction of the light source in eye coordinates, normalize */
  SvvCoord              direction;
  
  /* Cosine of the spot light cutoff angle */
  SvvReal               cosCutOffAngle;
  
  /* Precomputed attenuation, only when k1 and k2 are zero */
  SvvReal               attenuation;
  
  /* This will be set when the cut off angle != 180 */
  SvvBoolean            isSpot;
  
  /* When possible, the normalized "h" value from the spec is pre-computed */
  SvvCoord              hHat;
  
  /* Unit vector VPpli pre-computed (only when light is at infinity) */
  SvvCoord              unitVPpli;
  
  /* As above, but in model coordinates */
  SvvCoord              hHatModel;
  SvvCoord              unitVPpliModel;
  
  /* Link to next active light */
  __SvvLightSourceMachineRec*   next;
  
  /* Spot light exponent lookup table */
  SvvReal*               spotTable;
  
  /* Values used to avoid pow function during spot computations */
  SvvReal               threshold;
  SvvReal               scale;
  
  /* Set to GL_TRUE if slow processing path is needed */
  SvvBoolean            slowPath;
} SvvLightSourceMachine;

#  if 1
/* 
 * Per material computed state.
 */
typedef struct __SvvMaterialMachineRec
{
  /*
   * Sum of:
   *	material emissive color
   *  material ambient color * scene ambient color
   *
   * This sum is carefully kept scaled.
   */
  SvvColor              sceneColor;

  /*
   * Combination of the current material and first light source
   * One light being optimized. one for specular/diffuse & ambient
   */
  SvvColor              firstLightMatS;
  SvvColor              firstLightMatD;
  SvvColor              firstLightMatA;

  /* Specular exponent */
  SvvReal               specularExponent;

  /* Specular exponent lookup table */
  SvvReal*               specTable;

  /* Values used to avoid pow function during specular computations */
  SvvReal               threshold;
  SvvReal               scale;

  /* cache entry where this data came from */
  void*                 cache;

  /* Scaled and clamped form of material diffuse alpha */
  SvvReal               alpha;

  /* For color index mode lighting */
  SvvReal               cmapd_minus_cmapa;  /* = cmapd - cmapa */
  SvvReal               cmaps_minus_cmapa;  /* = cmaps - cmapa */
} SvvMaterialMachine;

typedef struct __SvvLightMachineRec
{
  SvvLightSourceMachine*        source;
  SvvMaterialMachine            front;
  SvvMaterialMachine            back;
  
  /* List of enabled light sources */
  SvvLightSourceMachine*        sources;
  
  SvvInt                numEnabledLights;
  SvvInt                fastLightMode;
  
  /* Current material color material (iff one material is being updated) */
  SvvMaterialState*     cm;
  SvvMaterialMachine*   cmm;
  
  /* Cache of lookup tables for spot lights and specular highlights */
  void*                 lutCache;

  /* Computed starting color for vertex lighting */
  SvvColor              baseColor;
} SvvLightMachine;
#  endif /* 0 */

/************************************************************************
 * Vertex                                                               *
 ************************************************************************/

/* Indicies for colors[] array in vertex */
#  define SVV_FRONTFACE                 0
#  define SVV_BACKFACE                  1
#  define SVV_FRONTFACE_SPECULAR        2
#  define SVV_BACKFACE_SPECULAR         3

/*
 * Definitions of bits for flags
 */

/* Bits for has */
#  define SVV_HAS_SHIFT         0
#  define SVV_HAS_FRONT_COLOR   (0x0001 << SVV_HAS_SHIFT)
#  define SVV_HAS_BACK_COLOR    (0x0002 << SVV_HAS_SHIFT)
#  define SVV_HAS_TEXTURE       (0x0004 << SVV_HAS_SHIFT)
#  define SVV_HAS_NORMAL        (0x0008 << SVV_HAS_SHIFT)
#  define SVV_HAS_EYE           (0x0010 << SVV_HAS_SHIFT)
#  define SVV_HAS_CLIP          (0x0020 << SVV_HAS_SHIFT)
#  define SVV_HAS_FOG           (0x0040 << SVV_HAS_SHIFT)
#  define SVV_HAS_WINDOW        (0x0080 << SVV_HAS_SHIFT)
#  define SVV_HAS_MASK          (0x00ff << SVV_HAS_SHIFT)

#  define SVV_HAS_BOTH          (SVV_HAS_FRONT_COLOR | SVV_HAS_BACK_COLOR)
#  define SVV_HAS_LIGHTING      (SVV_HAS_EYE | SVV_HAS_NORMAL)

#  define SVV_NEEDS_RASTERPOS   (SVV_HAS_FRONT_COLOR | \
                                 SVV_HAS_TEXTURE | \
                                 SVV_HAS_LIGHTING)

/* Bits for general */
#  define SVV_VERTEX_GENERAL_SHIFT      12
#  define SVV_VERTEX_EDGE_FLAG          (0x1 << SVV_VERTEX_GENERAL_SHIFT)
#  define SVV_VERTEX_CULL_FLAG          (0x2 << SVV_VERTEX_GENERAL_SHIFT)
#  define SVV_VERTEX_GENERAL_MASK       (0x3 << SVV_VERTEX_GENERAL_SHIFT)


#  define SVV_MAX_USER_CLIP_PLANES_GUESS        10

/************************************************************************/

/*
 * Total number of clipping planes supported by this gl.  This includes
 * the frustum's six clipping planes.
 */
#  define SVV_MAX_CLIP_PLANES_GUESS     (6 + SVV_MAX_USER_CLIP_PLANES_GUESS)

/************************************************************************/

/*
 * Vertex structure.  Each vertex contains enough state to properly
 * render the active primitive.
 */
typedef struct __SvvVertexStateRec
{
  /*
   * Keep this data structure aligned: have all vectors start on 
   * 4-word boundary, and sizeof this struct should be kept at 
   * a multiple of 4 words. Also helps to bunch together most
   * frequently used items, helps cache.
   */
  
  /*
   * Coordinates straight from client. These fields may not be
   * set depending on the active modes.  The normal and texture
   * coordinate are used by lighting and texturing.  These cells
   * may be overwritten by the eyeNormal and the generated texture
   * coordinate, depending on the active modes.
   */
  SvvCoord              obj;
  SvvCoord              normal;
  
  /*
   * Projected clip coordinate.  This field is filled in when the users
   * object coordinate has been multiplied by the combined modelview
   * and projection matrix.
   */
  SvvCoord              clip;

  /*
   * This contains lots of different kind of flags:
   *   has: Which members of the vertex are valid
   *   general: General flags for the vertex (like edge flags, etc.)
   *   vertexsize: 2/3/4D vertex
   *   clipcodes: for frustum and user clipplane clipping.
   */
  SvvUInt               flagBits;

  /*
   * Pointer to current face color in use. Additionally, when we
   * need to access the secondary face color, dereference the address
   * (color + 2), which will point to the respective secondary face
   * color.  Currently, the secondary color is only accessed for use
   * with separate specular color.
   */
  SvvColor*             color;

  /*
   * For value for this vertex.  Used only for cheap fogging.
   */
  SvvReal               fog;

  SvvUInt               pad0;

  /*
   * Window and eye coordinate. This field is filled in when the window 
   * clip coordinate is converted to a drawing surface relative "window" 
   * coordinate.
   * NOTE: the window.w coordinate contains 1/clip.w.
   */
  SvvCoord              window;
  SvvCoord              eye;

  /*
   * Coordinates straight from client. These fields may not be
   * set depending on the active modes.  The normal and texture
   * coordinate are used by lighting and texturing.  These cells
   * may be overwritten by the eyeNormal and the generated texture
   * coordinate, depending on the active modes.
   */
  SvvCoord              texture[SVV_NUM_TEXTURE_UNITS_GUESS];

  /*
   * Colors.  colors[0] is the "front" color, colors[1] is the "back" color.
   * The color pointer points to which color is current for this
   * vertex.  Verticies can have more than one color when two sided
   * lighting is enabled. (note color pointer moved up top).
   */
  SvvColor              colors[4];
} SvvVertexState;

/*
 * used for window rect clipping
 */
typedef struct __SvvClipVertexStateRec
{
  SvvVertexState        v;
  SvvUInt               outcode;
} SvvClipVertexState;

/************************************************************************
 * Pixel                                                                *
 ************************************************************************/

/* Bitmasks for pixelModeFlags */
#  define SVV_PIXEL_MODE_INDEX_LOOKUP                   (1 <<  0)
#  define SVV_PIXEL_MODE_STENCIL_LOOKUP                 (1 <<  1)
#  define SVV_PIXEL_MODE_INDEX_SHIFT_OFFSET             (1 <<  2)
#  define SVV_PIXEL_MODE_RGBA_LOOKUP                    (1 <<  3)
#  define SVV_PIXEL_MODE_RGBA_SCALE_BIAS                (1 <<  4)
#  define SVV_PIXEL_MODE_DEPTH_SCALE_BIAS               (1 <<  5)
#  define SVV_PIXEL_MODE_COLOR_TABLE                    (1 <<  6)
#  define SVV_PIXEL_MODE_CONVOLUTION                    (1 <<  7)
#  define SVV_PIXEL_MODE_POST_CONVOLUTION_SCALE_BIAS    (1 <<  8)
#  define SVV_PIXEL_MODE_POST_CONVOLUTION_COLOR_TABLE   (1 <<  9)
#  define SVV_PIXEL_MODE_COLOR_MATRIX                   (1 << 10)
#  define SVV_PIXEL_MODE_POST_COLOR_MATRIX_SCALE_BIAS   (1 << 11)
#  define SVV_PIXEL_MODE_POST_COLOR_MATRIX_COLOR_TABLE  (1 << 12)
#  define SVV_PIXEL_HISTOGRAM                           (1 << 13)
#  define SVV_PIXEL_MINMAX                              (1 << 14)

/************************************************************************/

/*
 * Some pixel code depends upon this constant in deciding what the longest
 * horizontal span of a DrawPixels command can possibly be.  With 2048, it
 * is allocating an array of size 32K.  If this constant increases too much,
 * then the pixel code might need to be revisited.
 */
#  define SVV_MAX_MAX_VIEWPORT_GUESS    2048
#  define SVV_MAX_SPAN_WIDTH    (SVV_MAX_MAX_VIEWPORT_GUESS)
#  define SVV_MAX_SPAN_SIZE     (SVV_MAX_SPAN_WIDTH * 4 * sizeof(SvvReal))

/*
 * State settable by glPixelTransfer*() and glPixelZoom()
 */
typedef struct __SvvPixelTransferModeRec
{
  SvvColor              scale;      /* Pixel scale and bias */
  SvvColor              bias;
  
  SvvReal               depthScale;    /* Depth scale and bias */
  SvvReal               depthBias;
  
  SvvColor              postConvolutionScale;
  SvvColor              postConvolutionBias;
  
  SvvColor              postColorMatrixScale;
  SvvColor              postColorMatrixBias;
  
  SvvReal               zoomX;
  SvvReal               zoomY;
} SvvPixelTransferMode;

/*
 * State settable with glPixelStore*()
 */
typedef struct __SvvPixelPackModeRec
{
  SvvBoolean            swapEndian;
  SvvBoolean            lsbFirst;
  
  SvvUInt               lineLength;
  SvvUInt               imageHeight;
  SvvUInt               skipImages;
  SvvUInt               skipLines;
  SvvUInt               skipPixels;
  SvvUInt               alignment;
} SvvPixelPackMode;

typedef struct __SvvPixelUnpackModeRec
{
  SvvBoolean            swapEndian;
  SvvBoolean            lsbFirst;
  
  SvvUInt               lineLength;
  SvvUInt               imageHeight;
  SvvUInt               skipImages;
  SvvUInt               skipLines;
  SvvUInt               skipPixels;
  SvvUInt               alignment;
} SvvPixelUnpackMode;

/*
 * Stackable pixel state
 */
typedef struct __SvvPixelStateRec
{
  SvvPixelTransferMode  transferMode;

  /*
   * Read buffer.  Where pixel reads come from.
   */
  SvvEnum               readBuffer;

  /*
   * Read buffer specified by user.  May be different from readBuffer
   * above.  If the user specifies GL_FRONT_LEFT, for example, then 
   * readBuffer is set to GL_FRONT, and readBufferReturn to 
   * GL_FRONT_LEFT if the context does not support stereo.
   */
  SvvEnum               readBufferReturn;
} SvvPixelState;

/*
 * Stackable client side pixel state
 */
typedef struct __SvvClientPixelStateRec
{
  SvvPixelPackMode      packModes;
  SvvPixelUnpackMode    unpackModes;
} SvvClientPixelState;

/************************************************************************/

typedef struct __SvvPixelSpanInfoRec
{
  SvvEnum               srcFormat, srcType; /* Form of source image */
  const SvvVoid*        srcImage;           /* The source image */
  const SvvVoid*        srcCurrent;         /* current ptr into the src data */
  SvvInt                srcRowIncrement;    /* Add this to get to next row */
  SvvInt                srcGroupIncrement;  /* Add this to get to next group */
  SvvInt                srcImageIncrement;  /* Add this to get to next image */
  SvvInt                srcComponents;	    /* (RGBA=4, ALPHA=1, etc.) */
  SvvInt                srcElementSize;	    /* Size of one element (BYTE=1) */
  SvvInt                srcSwapBytes;		
  SvvInt                srcLsbFirst;
  SvvInt                srcSkipPixels, srcSkipLines, srcSkipImages;
  SvvInt                srcLineLength, srcImageHeight;
  SvvInt                srcAlignment;
  SvvInt                srcStartBit;        /* After applying skipPixels */

  SvvEnum               dstFormat, dstType; /* Form of destination image */
  SvvVoid*              dstImage;           /* The destination image */
  SvvVoid*              dstCurrent;         /* current ptr into the dst data */
  SvvInt                dstRowIncrement;    /* Add this to get to next row */
  SvvInt                dstGroupIncrement;  /* Add this to get to next group */
  SvvInt                dstImageIncrement;  /* Add this to get to next image */
  SvvInt                dstComponents;      /* (RGBA=4, ALPHA=1, etc.) */
  SvvInt                dstElementSize;     /* Size of one element (BYTE=1) */
  SvvInt                dstSwapBytes;		
  SvvInt                dstLsbFirst;
  SvvInt                dstSkipPixels, dstSkipLines, dstSkipImages;
  SvvInt                dstLineLength, dstImageHeight;
  SvvInt                dstAlignment;
  SvvInt                dstStartBit;        /* After applying skipPixels */

  SvvReal               zoomx, zoomy;
  SvvInt                width, height, depth; /* Size of image */
  SvvInt                dim;   /* dimensionality of calling routine */

  SvvReal               readX; /* Reading X coord (CopyPixels, ReadPixels) */
  SvvReal               readY; /* Reading Y coord (CopyPixels, ReadPixels) */
  SvvReal               x, y;               /* Effective raster coordinates */
  SvvInt                startCol, startRow; /* First actual pixel goes here */
  SvvInt                endCol;             /* Last col rendered ( -coladd ) */
  SvvInt                columns, rows;	    /* Takes zoomx,zoomy into account */
  SvvBoolean            overlap;    /* Do CopyPixels src/dest region overlap? */
  SvvInt                rowsUp;     /* for overlapping CopyPixels regions */
  SvvInt                rowsDown;   /* for overlapping CopyPixels regions */
  SvvInt                rowadd;     /* Adders for incrementing the row */
  SvvInt                coladd;     /* Adders for incrementing the col */
  SvvReal               rendZoomx;  /* effective zoomx for render span */
  SvvUInt               fragz;      /* save this computation in the span walker */
  SvvReal               rpyUp, rpyDown;
  SvvInt                startUp, startDown;
  SvvInt                readUp, readDown;
  SvvVoid*              redMap;
  SvvVoid*              greenMap;
  SvvVoid*              blueMap;
  SvvVoid*              alphaMap;
  SvvVoid*              indexMap;
  SvvShort*             pixelArray; /* Array of pixel replication counts
                                     * (if zoomx < -1 or zoomx > 1)
                                     * or pixels to skip
                                     * (if zoomx < 1 and zoomx > -1).
                                     */

  SvvColor              scale;      /* generic scale values */
  SvvColor              bias;       /* generic bias values */

  /*
   * A pile of span routines used by the DrawPixels, ReadPixels, and 
   * CopyPixels functions.
   */
  SvvInt                numSpanMods; /* # of span modifiers currently in use */

#  if 0
  void (*spanReader)(__GLcontext*       gc,
                     SvvPixelSpanInfo*  spanInfo,
                     GLvoid*            outspan);

  void (*(spanModifier[13]))(__GLcontext*       gc,
                             SvvPixelSpanInfo*  spanInfo,
                             const GLvoid*      inspan,
                             GLvoid*            outspan);

  void (*spanRender)(__GLcontext*       gc,
                     SvvPixelSpanInfo*  spanInfo,
                     const GLvoid*      inspan);
#  else
  void*                 ptr0;
  void*                 ptr1;
  void*                 ptr2;
#  endif /* 0 */

  /*
   * These fields are used only for selecting spanModifiers.  The
   * span modifier routines themselves do not use them.  They are
   * in this record because they need to be passed through copyImage.
   */
  SvvBoolean            applySrcClamp;  /* clamp source data */
  SvvBoolean            applyDstClamp;  /* clamp destination data */
  SvvBoolean            applyFbScale;   /* apply frame buffer scale/unscale */
  SvvBoolean            applyPixelTransfer; /* apply pixel transfer ops */
  SvvBoolean            applyGenericScaleBias; /* apply nonpixxfer scale&bias */
  SvvBoolean            nonColorComp;   /* not color components (eg hgram) */

  /*
  ** These fields are used by convolution operators
  */
  SvvInt                spanCount;          /* running counter */
  SvvInt                convInitialSpans;
  SvvInt                convFinalSpans;
  SvvInt                convModifierIndex;  /* convolution span Modifier idx */

  /* __GLconvolutionFilter* filter; */
  void*                 filter;
  SvvVoid**             spanCache;  /* cache of spans for convolution */
#  if 0
  void (*convolveRows)(__GLcontext*     gc,
                       GLint            spanCount,
                       void*            filter,
                       GLint            firstRow,
                       GLint            lastRow,
                       GLint            width,
                       GLint            height,
                       const GLfloat*   inspan,
                       GLint            cacheRow,
                       GLfloat**        spanCache);
#  else
  void*                 ptr3;
#  endif /* 0 */
} SvvPixelSpanInfo;

/************************************************************************
 * Attrib                                                               *
 ************************************************************************/

/*
 * Point state.
 */
typedef struct __SvvPointStateRec
{
  SvvReal               requestedSize;
  SvvReal               smoothSize;
  SvvInt                aliasedSize;
} SvvPointState;

/************************************************************************/

/*
 * Line state.  Contains all the client controllable line state.
 */
typedef struct __SvvLineStateRec
{
  SvvReal               requestedWidth;
  SvvReal               smoothWidth;
  SvvInt                aliasedWidth;
  SvvUShort             stipple;
  SvvShort              stippleRepeat;
} SvvLineState;

/************************************************************************/

/*
 * Polygon state.  Contains all the user controllable polygon state
 * except for the stipple state.
 */
typedef struct __SvvPolygonStateRec
{
  SvvEnum               frontMode;
  SvvEnum               backMode;

  /*
   * Culling state.  Culling can be enabled/disabled and set to cull
   * front or back faces.  The FrontFace call determines whether clockwise
   * or counter-clockwise oriented vertices are front facing.
   */
  SvvEnum               cull;
  SvvEnum               frontFaceDirection;

  /*
   * Polygon offset state. 
   */
  SvvReal               factor;
  SvvReal               units;
} SvvPolygonState;

/************************************************************************/

/*
 * Polygon stipple state.
 */
typedef struct __SvvPolygonStippleStateRec
{
  SvvUByte              stipple[4*32];
} SvvPolygonStippleState;

/************************************************************************/

/*
 * Transform state.
 */
typedef struct __SvvTransformStateRec
{
  /*
   * Current mode of the matrix stack.  This determines what effect
   * the various matrix operations (load, mult, scale) apply to.
   */
  SvvEnum               matrixMode;

  /*
   * User clipping planes in eye space.  These are the user clip planes
   * projected into eye space.  
   */
/* XXX BUG! stacking of eyeClipPlanes is busted! */
  SvvCoord*             eyeClipPlanes;

  /*
   * EXT_cull_vertex attributes.
   */
  SvvCoord              eyePos;          /* eye space eye location */
  SvvCoord              eyePosObj;       /* object space eye location */
  SvvBoolean            objEyeSpecified; /* true if user specified obj space */
} SvvTransformState;

/************************************************************************/

typedef struct __SvvAttributeStateRec
{
  /*
   * Mask of which fields in this structure are valid.
   */
  SvvUInt                       mask;

  SvvPointState                 point;
  SvvLineState                  line;
  SvvPolygonState               polygon;
  SvvPolygonStippleState        polygonStipple;
  SvvPixelState                 pixel;
  SvvLightState                 light;
  /* __fogState                    fog; */
  /* __depthState                  depth; */
  /* __accumState                  accum; */
  /* __stencilState                stencil; */
  /* __viewport                    viewport; */
  SvvTransformState             transform;
  /* __enableState                 enables; */
  /* __rasterState                 raster; */
  /* __hintState                   hints; */
  /* __evaluatorState              evaluator; */
  /* __dlistState                  list; */
  /* __textureState                texture; */
  /* __scissor                     scissor; */
  /* __colorTableState             colorTable; */
} SvvAttributeState;

typedef struct __SvvClientAttributeStateRec
{
  SvvUInt                       mask;

  SvvClientPixelState           clientPixel;
  /* __clientTextureState          clientTexture; */
  /* __clientVertArrayState        vertexArray; */
} SvvClientAttributeState;

SVV_END_EXTERN_C

#endif /* SVV_TYPES_H_ */
/* 
 * End of: $Id: svvTypes.h,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $.
 * 
 */
