/* 
 * $Id: Pipeline.cxx,v 1.1.1.1 2006/12/19 22:59:32 christianh Exp $
 * 
 */
// C++
#include <iostream>
#include "vtkPolyData.h"
// VTK Rendering
#include "vtkPolyDataMapper.h"
// VTK Graphics
#include "vtkGlyph3D.h"
#include "vtkSphereSource.h"
#include "vtkConeSource.h"
#include "vtkAppendPolyData.h"
#include "vtkTransformTextureCoords.h"
#include "vtkTextureMapToPlane.h"
#include "vtkTextureMapToSphere.h"
#include "vtkTextureMapToCylinder.h"
#include "vtkImplicitTextureCoords.h"
#include "vtkHull.h"
#include "vtkCleanPolyData.h"
#include "vtkFeatureEdges.h"
#include "vtkPolyDataNormals.h"
#include "vtkCleanPolyData.h"
#include "vtkDecimatePro.h"
#include "vtkLoopSubdivisionFilter.h"
#include "vtkButterflySubdivisionFilter.h"
#include "vtkLinearSubdivisionFilter.h"
// 
#include "vtkUpstream.h"
// 
#include "Pipeline.H"

using std::cerr;


// ----------------------------------------------------------------------------
vtkPolyDataSource*
MakeHullSource(vtkSource* pSource, int steps)
{
  vtkPolyData* pPolyData;
  vtkUpstream* pUpstream = vtkUpstream::New();
  
  pUpstream->SetSource(pSource);
  if(!(pPolyData = pUpstream->GetOutputPolyData())) return 0;
  pUpstream->Delete();

  vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
  {
    vtkHull* hull = vtkHull::New();
    {
      vtkCleanPolyData* cleaner = vtkCleanPolyData::New();
      {
	cleaner->SetInput(pPolyData);
	cleaner->ConvertLinesToPointsOn();
	cleaner->ConvertPolysToLinesOn();
	cleaner->ConvertStripsToPolysOn();
	cleaner->PointMergingOn();
	cleaner->Update();
      }
      hull->AddRecursiveSpherePlanes(steps);
      hull->SetInput(cleaner->GetOutput());
      hull->Update();
      cleaner->Delete();
    }
    normals->SetInput(hull->GetOutput());
    normals->Update();
    hull->Delete();
  }

  return normals;
}

// ----------------------------------------------------------------------------
vtkPolyDataSource*
MakeFeatureEdgesSource(vtkSource* pSource)
{
  vtkPolyData* pPolyData;
  vtkUpstream* pUpstream = vtkUpstream::New();
  
  pUpstream->SetSource(pSource);
  if(!(pPolyData = pUpstream->GetOutputPolyData())) return 0;
  pUpstream->Delete();

  // get the feature edges of the poly data source
  vtkFeatureEdges* featureEdges = vtkFeatureEdges::New();
  {
    featureEdges->SetInput(pPolyData);
    featureEdges->SetFeatureAngle(30.f);  // VTK default = 30.0
    featureEdges->SetBoundaryEdges(1);    // VTK default = 1
    featureEdges->SetFeatureEdges(1);     // VTK default = 1
    featureEdges->SetNonManifoldEdges(1); // VTK default = 1
    featureEdges->SetManifoldEdges(0);    // VTK default = 0
    featureEdges->SetColoring(0);         // VTK default = 1
    featureEdges->Update();
  }
  
  return featureEdges;
}

// ----------------------------------------------------------------------------
vtkPolyDataSource*
MakeMaceSource(void)
{
  // Create the mace geometry actor.
  vtkAppendPolyData* append = vtkAppendPolyData::New();
  {
    vtkSphereSource* sphereSource = vtkSphereSource::New();
    {
      sphereSource->SetThetaResolution(16);
      sphereSource->SetPhiResolution(16);
    }
    vtkGlyph3D* glyph = vtkGlyph3D::New();
    {
      vtkConeSource* coneSource = vtkConeSource::New();
      {
	coneSource->SetResolution(12);
      }
      glyph->SetInput(sphereSource->GetOutput());
      glyph->SetSource(coneSource->GetOutput());
      glyph->SetVectorModeToUseNormal();
      glyph->SetScaleModeToScaleByVector();
      glyph->SetScaleFactor(0.1);
      coneSource->Delete();
    }
    append->AddInput(glyph->GetOutput());
    append->AddInput(sphereSource->GetOutput());
    sphereSource->Delete();
    glyph->Delete();
  }

  return append;
}

vtkTransformTextureCoords*
InsertTextureMapToPlane(vtkActor* pActor)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkTransformTextureCoords* transform =vtkTransformTextureCoords::New();
    {
      vtkTextureMapToPlane* mapToPlane = vtkTextureMapToPlane::New();
      {
        mapToPlane->SetInput(pdMapper->GetInput());
        mapToPlane->AutomaticPlaneGenerationOn();
      }
      transform->SetInput(mapToPlane->GetOutput());
      mapToPlane->Delete();
    }
    pdMapper->SetInput(transform->GetPolyDataOutput());
    transform->Update();
    return transform;
  }
  return 0;
}

vtkTransformTextureCoords*
InsertTextureMapToSphere(vtkActor* pActor, bool seamless)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkTransformTextureCoords* transform =vtkTransformTextureCoords::New();
    {
      vtkTextureMapToSphere* mapToSphere = vtkTextureMapToSphere::New();
      {
        mapToSphere->SetInput(pdMapper->GetInput());
        mapToSphere->AutomaticSphereGenerationOn();
        mapToSphere->SetPreventSeam(int(seamless));
      }
      transform->SetInput(mapToSphere->GetOutput());
      mapToSphere->Delete();
    }
    pdMapper->SetInput(transform->GetPolyDataOutput());
    transform->Update();
    return transform;
  }
  return 0;
}

vtkTransformTextureCoords*
InsertTextureMapToCylinder(vtkActor* pActor, bool seamless)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkTransformTextureCoords* transform =vtkTransformTextureCoords::New();
    {
      vtkTextureMapToCylinder* mapToCylinder = vtkTextureMapToCylinder::New();
      {
        mapToCylinder->SetInput(pdMapper->GetInput());
        mapToCylinder->AutomaticCylinderGenerationOn();
        mapToCylinder->SetPreventSeam(int(seamless));
      }
      transform->SetInput(mapToCylinder->GetOutput());
      mapToCylinder->Delete();
    }
    pdMapper->SetInput(transform->GetPolyDataOutput());
    transform->Update();
    return transform;
  }
  return 0;
}

vtkTransformTextureCoords*
InsertTextureMapToImplicit(vtkActor* pActor,
                           vtkImplicitFunction* functionR,
                           vtkImplicitFunction* functionS,
                           vtkImplicitFunction* functionT)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkTransformTextureCoords* transform =vtkTransformTextureCoords::New();
    {
      vtkImplicitTextureCoords* mapToImplicit = vtkImplicitTextureCoords::New();
      {
        mapToImplicit->SetInput(pdMapper->GetInput());
        if(functionR) mapToImplicit->SetRFunction(functionR);
        if(functionS) mapToImplicit->SetRFunction(functionS);
        if(functionT) mapToImplicit->SetRFunction(functionT);
        mapToImplicit->SetFlipTexture(0);
      }
      transform->SetInput(mapToImplicit->GetOutput());
      mapToImplicit->Delete();
    }
    pdMapper->SetInput(transform->GetPolyDataOutput());
    transform->Update();
    return transform;
  }
  return 0;
}

void
InsertNormals(vtkActor* pActor)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
    {
      vtkCleanPolyData* cleaner = vtkCleanPolyData::New();
      {
        cleaner->SetInput(pdMapper->GetInput());
        cleaner->PointMergingOn();
      }
      normals->SetInput(cleaner->GetOutput());
      normals->SetFeatureAngle(45.0);   // default is 30.0
      normals->SplittingOn();           // default is 1
      normals->ConsistencyOn();         // default is 1
      normals->ComputeCellNormalsOff(); // default is 0
      normals->ComputePointNormalsOn(); // default is 1
      cleaner->Delete();
    }
    pdMapper->SetInput(normals->GetOutput());
    normals->Update();
    normals->Delete();
  }
}

void
InsertDecimation(vtkActor* pActor, float reduction)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkDecimatePro* pDecimator = vtkDecimatePro::New();
    {
      vtkCleanPolyData* pCleanPolyData = vtkCleanPolyData::New();
      {
        pCleanPolyData->SetInput(pdMapper->GetInput());
        pCleanPolyData->PointMergingOn();
        pCleanPolyData->ConvertLinesToPointsOn();
        pCleanPolyData->ConvertPolysToLinesOn();
        pCleanPolyData->ConvertStripsToPolysOn();
      }
      pDecimator->SetInput(pCleanPolyData->GetOutput());
      pDecimator->SetTargetReduction(reduction);
      pDecimator->SetFeatureAngle(15.0);  // default is 15.0
      pDecimator->SetPreserveTopology(0); // default is 0
      pDecimator->SetSplitAngle(75.0);    // default is 75.0
      pDecimator->SetSplitting(1);        // default is 1
      pDecimator->SetPreSplitMesh(0);     // default is 0
      pDecimator->SetDegree(25);          // default is 25
      pDecimator->SetBoundaryVertexDeletion(1);  // default is 1
      pDecimator->SetInflectionPointRatio(10.0); // default is 10.0
      pCleanPolyData->Delete();
    }
    pdMapper->SetInput(pDecimator->GetOutput());
    pDecimator->Update();
    pDecimator->Delete();
  }
}

void
InsertLoopSubdivision(vtkActor* pActor, int subdivisions)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkLoopSubdivisionFilter* filter = vtkLoopSubdivisionFilter::New();
    {
      filter->SetInput(pdMapper->GetInput());
      filter->SetNumberOfSubdivisions(subdivisions);
    }
    pdMapper->SetInput(filter->GetOutput());
    filter->Update();
    filter->Delete();
  }
}

void
InsertButterflySubdivision(vtkActor* pActor, int subdivisions)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkButterflySubdivisionFilter* filter =vtkButterflySubdivisionFilter::New();
    {
      filter->SetInput(pdMapper->GetInput());
      filter->SetNumberOfSubdivisions(subdivisions);
    }
    pdMapper->SetInput(filter->GetOutput());
    filter->Update();
    filter->Delete();
  }
}

void
InsertLinearSubdivision(vtkActor* pActor, int subdivisions)
{
  vtkPolyDataMapper* pdMapper;
  if(pdMapper = vtkPolyDataMapper::SafeDownCast(pActor->GetMapper())) {
    vtkLinearSubdivisionFilter* filter = vtkLinearSubdivisionFilter::New();
    {
      filter->SetInput(pdMapper->GetInput());
      filter->SetNumberOfSubdivisions(subdivisions);
    }
    pdMapper->SetInput(filter->GetOutput());
    filter->Update();
    filter->Delete();
  }
}

/* 
 * End of: $Id: Pipeline.cxx,v 1.1.1.1 2006/12/19 22:59:32 christianh Exp $.
 * 
 */
