/* 
 * $Id: vtkTester.cxx,v 1.1.1.1 2006/12/19 22:59:32 christianh Exp $
 * 
 */
// standard C++
#include <iostream>
// Rendering
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkCamera.h"
#include "vtkPolyDataMapper.h"
#include "vtkActor.h"
#include "vtkProperty.h"
#include "vtkAssembly.h"
// Graphics
#include "vtkCubeSource.h"
#include "vtkSphereSource.h"
#include "vtkOutlineSource.h"
#include "vtkSuperquadricSource.h"
#include "vtkFeatureEdges.h"
#include "vtkPolyDataNormals.h"
#include "vtkAppendPolyData.h"
#include "vtkCleanPolyData.h"
#include "vtkTubeFilter.h"
// Hybrid
#include "vtkDepthSortPolyData.h"
// SVV
#include "svvController.h"

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);

static vtkActor*   MakeCube(float [6], vtkCamera* = 0);
static vtkActor*   MakeSphere(float, float [3], vtkCamera* = 0);
static vtkProp3D*  MakeTestGeometry(vtkCamera* = 0);

extern vtkProp* MakeVolumetricText(vtkActor*, vtkCamera*);

int
main(int argc, char* argv[])
{
  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  svv.DebugOn();

  // Create the renderers, render window, and interactor
  vtkRenderWindow*           renderWindow = vtkRenderWindow::New();
  vtkRenderWindowInteractor* interactor   = vtkRenderWindowInteractor::New();
  vtkRenderer*               renderer     = vtkRenderer::New();
  vtkCamera*                 camera       = renderer->GetActiveCamera();

  interactor->SetRenderWindow(renderWindow);
  renderWindow->AddRenderer(renderer);

  vtkProp3D* tester;
  tester = MakeTestGeometry(camera);
  renderer->AddProp(tester);
  tester->Delete();

  float center[3] = { 1.0, 1.0, 0.0 };
  float radius    = 2.f / 3.f;
  float bounds[6];

  bounds[0] = center[0] - radius; bounds[1] = center[0] + radius;
  bounds[2] = center[1] - radius; bounds[3] = center[1] + radius;
  bounds[4] = center[2] - radius; bounds[5] = center[2] + radius;

  vtkActor* cube = MakeCube(bounds);
  vtkProperty* cproperty = svv.MakeProperty("red plastic");
  cube->SetProperty(cproperty);
  cproperty->Delete();
  renderer->AddProp(cube);
  vtkProp* cubeCaption = MakeVolumetricText(cube,camera);
  renderer->AddProp(cubeCaption);
  cube->Delete();
  cubeCaption->Delete();

  vtkActor* sphere = MakeSphere(radius, center);
  vtkProperty* sproperty = svv.MakeProperty("polished copper");
  sphere->SetProperty(sproperty);
  sproperty->Delete();
  renderer->AddProp(sphere);
  vtkProp* sphereCaption = MakeVolumetricText(sphere,camera);
  renderer->AddProp(sphereCaption);
  sphere->Delete();
  sphereCaption->Delete();

  // configure renderer
  renderer->SetBackground(0.1f, 0.1f, 0.1f);
  renderer->SetTwoSidedLighting(1);
  renderer->ResetCamera();
  //renderWindow->LineSmoothingOn();
  renderWindow->SetSize(1024, 1024);
  renderWindow->Render();
  
  // Interact with the data at 3 frames per second
  interactor->SetDesiredUpdateRate(3.0);
  interactor->SetStillUpdateRate(0.001);

  interactor->Start();

  // Clean up
  renderer->Delete();
  renderWindow->Delete();
  interactor->Delete();

  return 0;
}


static vtkActor*
MakeCube(float bounds[6], vtkCamera* camera)
{
  float  width, height, depth, length, radius;
  int    resolution = 16;

  width  = bounds[1] - bounds[0];
  height = bounds[3] - bounds[2];
  depth  = bounds[5] - bounds[4];
  length = float(sqrt(width*width+height*height+depth*depth));
  radius = length*0.01f;

  vtkActor* actor = vtkActor::New();
  {
    vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
    {
      vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
      {
        vtkAppendPolyData* appender = vtkAppendPolyData::New();
        {
#if 0
          vtkCubeSource* cube = vtkCubeSource::New();
          {
            cube->SetBounds(bounds);
            cube->Update();
          }
	  appender->AddInput(cube->GetOutput());
          cube->Delete();
#endif /* 0 */
          vtkTubeFilter* outlineTubes = vtkTubeFilter::New();
          {
            vtkOutlineSource* outline = vtkOutlineSource::New();
            {
              outline->SetBounds(bounds);
              outline->Update();
            }
            outlineTubes->SetInput(outline->GetOutput());
            outlineTubes->SetRadius(radius);
            outlineTubes->SetNumberOfSides(resolution);
            outline->Delete();
          }
	  appender->AddInput(outlineTubes->GetOutput());
          outlineTubes->Delete();
          // Corners
          for(int k=0; k<2; k++) {
            for(int i=0; i<2; i++) {
              for(int j=0; j<2; j++) {
                vtkSphereSource* sphereSource = vtkSphereSource::New();
                {
                  sphereSource->SetRadius(radius+(radius*(i+j+k)));
                  sphereSource->SetCenter(bounds[j],bounds[i+2],bounds[k+4]);
                  sphereSource->SetThetaResolution(resolution);
                  sphereSource->SetPhiResolution(resolution);
                }
                appender->AddInput(sphereSource->GetOutput());
                sphereSource->Delete();
              }
            }
          }
        }
	normals->SetInput(appender->GetOutput());
	appender->Delete();
      }
      mapper->SetResolveCoincidentTopologyToPolygonOffset();
      // depth sort if we have a reference to a camera
      vtkDepthSortPolyData* sorter = 0;
      if(camera) {
	sorter = vtkDepthSortPolyData::New();
	sorter->SetInput(normals->GetOutput());
	sorter->SetDirectionToBackToFront();
	sorter->SetDepthSortModeToParametricCenter();
	sorter->SetVector(1.0, 1.0, 1.0);
	sorter->SetCamera(camera);
	sorter->SetProp3D(actor);
	sorter->SetSortScalars(1/*On*/);
	sorter->Update();
	mapper->SetInput(sorter->GetOutput());
	mapper->SetScalarVisibility(0);
	sorter->Delete();
      }
      else {
	mapper->SetInput(normals->GetOutput());
      }
      normals->Delete();
    }
    actor->SetMapper(mapper);
    mapper->Delete();
  }

  return actor;
}

static vtkActor*
MakeSphere(float radius, float center[3], vtkCamera* camera)
{
  float  length, tubeRadius;
  int    resolution = 16;

  length     = float(sqrt(3.0*radius*radius*4.0));
  tubeRadius = length*0.01f;

  vtkActor* actor = vtkActor::New();
  {
    vtkPolyDataMapper* mapper = vtkPolyDataMapper::New();
    {
      vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
      {
	vtkSphereSource* sphereSource = vtkSphereSource::New();
	{
	  sphereSource->SetRadius(radius);
	  sphereSource->SetCenter(center);
	  sphereSource->SetThetaResolution(32);
	  sphereSource->SetPhiResolution(32);
	  sphereSource->Update();
	}
	normals->SetInput(sphereSource->GetOutput());
	sphereSource->Delete();
      }
      // depth sort if we have a reference to a camera
      vtkDepthSortPolyData* sorter = 0;
      if(camera) {
	sorter = vtkDepthSortPolyData::New();
	sorter->SetInput(normals->GetOutput());
	sorter->SetDirectionToBackToFront();
	sorter->SetDepthSortModeToParametricCenter();
	sorter->SetVector(1.0, 1.0, 1.0);
	sorter->SetCamera(camera);
	sorter->SetProp3D(actor);
	sorter->SetSortScalars(1/*On*/);
	sorter->Update();
	mapper->SetInput(sorter->GetOutput());
	mapper->SetScalarVisibility(0);
	sorter->Delete();
      }
      else {
	mapper->SetInput(normals->GetOutput());
      }
      normals->Delete();
    }
    actor->SetMapper(mapper);
    mapper->Delete();
  }

  return actor;
}

static vtkProp3D*
MakeTestGeometry(vtkCamera* camera)
{
  float  scale1 = 1.f, scale2 = 0.8f;
  int    resolution = 6;

  vtkAssembly* assembly = vtkAssembly::New();
  {
    // add the mapped poly data and feature edges to actors
    vtkActor* opaqueActor      = vtkActor::New();
    vtkActor* translucentActor = vtkActor::New();
    {
      // map poly data and feature edges
      vtkPolyDataMapper* opaqueMapper      = vtkPolyDataMapper::New();
      vtkPolyDataMapper* translucentMapper = vtkPolyDataMapper::New();
      {
	vtkAppendPolyData* opaquePolyData      = vtkAppendPolyData::New();
	vtkAppendPolyData* translucentPolyData = vtkAppendPolyData::New();
	{
	  // Create test geometry source 1
	  vtkSuperquadricSource* source1 = vtkSuperquadricSource::New();
	  {
	    source1->ToroidalOff();
	    source1->SetThetaResolution(32);
	    source1->SetPhiResolution(32);
	    source1->SetPhiRoundness(0.3);
	    source1->SetThetaRoundness(0.85);
	    source1->SetScale(scale1, scale1, scale1);
	    source1->SetCenter(0.0, 0.0, 0.0);
	    source1->Update();
	  }
	  translucentPolyData->AddInput(source1->GetOutput());
          source1->Delete();
	  // get the feature edges of the poly data source
          vtkFeatureEdges* featureEdges1 = vtkFeatureEdges::New();
          {
            featureEdges1->SetInput(source1->GetOutput());
            featureEdges1->BoundaryEdgesOn();
            featureEdges1->ColoringOn();
            featureEdges1->ManifoldEdgesOff();
            featureEdges1->Update();
          }
          opaquePolyData->SetInput(featureEdges1->GetOutput());
          featureEdges1->Delete();
	  // Create test geometry source 2
	  vtkSuperquadricSource* source2 = vtkSuperquadricSource::New();
	  {
	    source2->ToroidalOff();
	    source2->SetThetaResolution(32);
	    source2->SetPhiResolution(32);
	    source2->SetPhiRoundness(0.7);
	    source2->SetThetaRoundness(1.25);
	    source2->SetScale(scale2, scale2, scale2);
	    source2->SetCenter(-0.3, 0.3, 0.0);
	    source2->Update();
	  }
	  translucentPolyData->AddInput(source2->GetOutput());
          source2->Delete();
	  // get the feature edges of the poly data source
          vtkFeatureEdges* featureEdges2 = vtkFeatureEdges::New();
          {
            featureEdges2->SetInput(source2->GetOutput());
            featureEdges2->BoundaryEdgesOn();
            featureEdges2->ColoringOff();
            featureEdges2->ManifoldEdgesOff();
            featureEdges2->Update();
          }
	  opaquePolyData->AddInput(featureEdges2->GetOutput());
          featureEdges2->Delete();
	  // Create test geometry source 3
	  vtkSuperquadricSource* source3 = vtkSuperquadricSource::New();
	  {
	    source3->ToroidalOn();
	    source3->SetThetaResolution(32);
	    source3->SetPhiResolution(32);
	    source3->SetPhiRoundness(1.5);
	    source3->SetThetaRoundness(1.0);
	    source3->SetScale(1.5, 1.5, 1.5);
	    source3->SetCenter(0.0, -0.2, 0.0);
	    source3->Update();
	  }
	  opaquePolyData->AddInput(source3->GetOutput());
          source3->Delete();
	}
	// depth sort if we have a reference to a camera
	vtkDepthSortPolyData* depthSorter = 0;
	if(camera) {
	  depthSorter = vtkDepthSortPolyData::New();
	  depthSorter->SetInput(translucentPolyData->GetOutput());
	  depthSorter->SetDirectionToBackToFront();
	  depthSorter->SetDepthSortModeToParametricCenter();
	  depthSorter->SetVector(1.0, 1.0, 1.0);
	  depthSorter->SetCamera(camera);
	  depthSorter->SetSortScalars(1/*On*/);
	  depthSorter->Update();
	  translucentMapper->SetInput(depthSorter->GetOutput());
	  translucentMapper->SetScalarVisibility(0);
	  depthSorter->Delete();
	}
	else {
	  translucentMapper->SetInput(translucentPolyData->GetOutput());
	}
	// map the opaque data
	opaqueMapper->SetInput(opaquePolyData->GetOutput());
	opaqueMapper->SetResolveCoincidentTopologyToPolygonOffset();
	translucentPolyData->Delete();
	opaquePolyData->Delete();
      }
      vtkProperty* translucentProperty = svv.MakeProperty("emerald");
      vtkProperty* backfaceProperty    = svv.MakeProperty("ruby");
      vtkProperty* opaqueProperty      = svv.MakeProperty("polished gold");
      opaqueProperty->SetLineWidth(2.0);
      translucentActor->SetMapper(translucentMapper);
      translucentActor->SetProperty(translucentProperty);
      translucentActor->SetBackfaceProperty(backfaceProperty);
      opaqueActor->SetMapper(opaqueMapper);
      opaqueActor->SetProperty(opaqueProperty);
      translucentMapper->Delete();
      opaqueMapper->Delete();
      translucentProperty->Delete();
      backfaceProperty->Delete();
      opaqueProperty->Delete();
    }
    assembly->AddPart(opaqueActor);
    assembly->AddPart(translucentActor);
    opaqueActor->Delete();
    translucentActor->Delete();
  }

  return assembly;
}

/* 
 * End of: $Id: vtkTester.cxx,v 1.1.1.1 2006/12/19 22:59:32 christianh Exp $.
 * 
 */
