/* -*- C++ -*-
 * 
 * $Id: sv.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $
 * 
 * 
 * Copyright (c) 2002 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

// illustrates the use of vtkFLTK classes.

// the FLTK-specialized object factory
#include "vtkFLTKObjectFactory.h"

// here we have all the usual VTK stuff that we need for our pipeline
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkStructuredPoints.h"
#include "vtkFloatArray.h"
#include "vtkContourFilter.h"
#include "vtkPolyDataMapper.h"
#include "vtkAppendPolyData.h"
#include "vtkCamera.h"
#include "vtkDepthSortPolyData.h"
#include "vtkActor.h"
#include "vtkActorCollection.h"

#include "sv.H"

// and of course some fltk stuff
#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/gl.h>
#include "svUI.H"

// Surface
#include "Surface.H"

#include <cmath>

static float colors[4][4] = { {0.8f,  0.f,  0.f,  1.f},
                              {0.3f, 0.6f, 0.f,  0.75f},
                              {0.f,  0.3f, 0.6f, 0.5f},
                              {0.8f, 0.f,  0.6f, 0.25f} };

static char names1[4][32] = { "lh.white.asc",
                              "lh.inflated.asc",
                              "lh.sphere.asc",
                              "lh.orig.asc" };

static char names2[4][32] = { "rh.white.asc",
                              "rh.inflated.asc",
                              "rh.sphere.asc",
                              "rh.orig.asc" };

static char names3[4][32] = { "lh.white.asc",
                              "rh.white.asc",
                              "lh.inflated.asc",
                              "rh.inflated.asc" };


static vtkActor*
GetSurfaceActor(char* name, float rgba[4], bool filter = false);

static vtkActor*
MakeVolume(int width, int height, int depth);

static bool filter = true;

int
main(int argc, char* argv[])
{
  // register with VTK to produce FLTK-specific objects
  vtkObjectFactory::RegisterFactory(vtkFLTKObjectFactory::New());

  fl_open_display();
  Fl::gl_visual(FL_RGB8|FL_DOUBLE|FL_ACCUM|FL_DEPTH|FL_MULTISAMPLE);

  svUI* gui = new svUI;

  // map the interface windows
  gui->Show();

  // get pointers to the interactors
  vtkRenderWindowInteractor* interactor =
    reinterpret_cast<vtkRenderWindowInteractor*>(gui->viewArea->user_data());
  // create a render window
  vtkRenderWindow* renderWindow = vtkRenderWindow::New();
  // add the render window to the interactor
  interactor->SetRenderWindow(renderWindow);
  gui->SetRenderWindow(renderWindow);
  // create a renderer
  vtkRenderer* renderer = vtkRenderer::New();
  // configure the renderer and add the volume actor
  renderer->SetBackground(0.1f, 0.05f, 0.05f);
  // add the renderer to the render window
  renderWindow->AddRenderer(renderer);
  gui->SetRenderer(renderer);

  // this is the standard way of "starting" a fltk application
  int fl_ret = Fl::run();

  // We can now delete all our references to the VTK pipeline as the objects
  // themselves will stick around until we dereference the Fl_VTK_Window (and
  // its associated Interactor.
  renderer->Delete();
  renderWindow->Delete();

  delete gui;

  return fl_ret;
}


static vtkActor*
GetSurfaceActor(char* name, float rgba[4], bool filter = false)
{
  vtkPolyData* surface      = 0;
  vtkActor*    surfaceActor = 0;

  // read data
  if(!(surface = Surface::ReadAsciiToPolys(name))) return 0;
  // construct pipeline for spherical volume
  surfaceActor = Surface::MakeSurfaceActor(surface, rgba, filter);
  // clean up references
  surface->Delete();

  return surfaceActor;
}

static vtkActor*
MakeVolume(int width, int height, int depth)
{
  vtkActor* volumeActor = vtkActor::New();
  {
    vtkPolyDataMapper* volMapper = vtkPolyDataMapper::New();
    {
      vtkContourFilter* contour = vtkContourFilter::New();
      {
        vtkStructuredPoints* vol = vtkStructuredPoints::New();
        {
          // XYZ Dimensions
          vol->SetDimensions(width, height, depth);
          // Origin
          vol->SetOrigin(-0.5, -0.5, -0.5);
          // Spacing
          float xres = 1.f/float(width-1);
          float yres = 1.f/float(height-1);
          float zres = 1.f/float(depth-1);
          vol->SetSpacing(xres, yres, zres);

          int   kOffset, jOffset, offset;
          float x, y, z, s;
          vtkDataArray* scalars = vtkFloatArray::New();
          for(int k=0; k<depth; k++) {
            z = -0.5 + k*zres;
            kOffset = k * height * width;
            for(int j=0; j<height; j++) {
              y = -0.5 + j*yres;
              jOffset = j * width;
              for(int i=0; i<width; i++) {
                x = -0.5 + i*xres;
                s = x*x + y*y + z*z - (0.4*0.4);
                offset = i + jOffset + kOffset;
                scalars->InsertTuple1(offset, s);
              }
            }
          }
          (vol->GetPointData())->SetScalars(scalars);
          // clean-up
          scalars->Delete();
        }
        contour->SetInput(vol);
        contour->SetValue(0, 0.0);
        // clean-up
        vol->Delete();
      }
      volMapper->SetInput(contour->GetOutput());
      volMapper->ScalarVisibilityOff();
      // clean-up
      contour->Delete();
    }
    volumeActor->SetMapper(volMapper);
    // clean-up
    volMapper->Delete();
  }

  return volumeActor;
} // MakeVolume()
