// 
// $Id: svFacePropertiesView.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $
// 

#include <iostream>

#include <GL/glu.h>

// includes needed by the event handler
#include <FL/Fl.H>
#include <FL/x.H>

#include "svFacePropertiesView.H"

// render Icosohedron (possibly recursively subdivide approximating a sphere)
extern void
svIcosohedron(GLint lod=0, GLboolean colored=GL_FALSE, GLboolean mono=GL_FALSE);

void
animate(void* p)
{
  static double period = 1.0 / 30.0;
  svFacePropertiesView* view = reinterpret_cast<svFacePropertiesView*>(p);

  if(view->visible()) {
    view->Trackball->IncrementRotation();
    view->ReconfigureModel();
    Fl::repeat_timeout(period, animate, p);
  }
}

svModels svFacePropertiesView::ModelMap;

svFacePropertiesView::svFacePropertiesView(int x,int y,int w,int h,
                                           const char* l=0)
  : Fl_Gl_Window(x,y,w,h,l),
    FaceProperties(new svFaceProperties),
    Trackball(new svTrackball),
    spinmode_(GL_FALSE),
    spinning_(GL_FALSE),
    scale_(1.f),
    TestGeometry(0),
    ContrastGeometry(0),
    PointSmoothing(GL_FALSE),
    LineSmoothing(GL_FALSE),
    PolygonSmoothing(GL_FALSE),
    TwoSidedLighting(GL_FALSE),
    PositionalLighting(GL_TRUE),
    MonoBackground(GL_FALSE),
    ConfiguredSmoothing(GL_FALSE),
    ConfiguredLight(GL_FALSE),
    ConfiguredLightPosition(GL_FALSE),
    ConfiguredLightModel(GL_FALSE),
    ConfiguredModel(GL_FALSE),
    Debug(1)
{
  // Get an RGB visual with depth buffer that supports double buffering.
  this->mode(FL_RGB|FL_DEPTH|FL_DOUBLE|FL_MULTISAMPLE);
  // 
  this->translate_[0] = 0.f;
  this->translate_[1] = 0.f;
  this->translate_[2] = 0.f;
  // GL default spot direction: (0, 0, -1)
  this->LightFocalPoint[0] = 0.f;
  this->LightFocalPoint[1] = 0.f;
  this->LightFocalPoint[2] = 0.f;
  // GL default position is (0, 0, 1, 0)
  this->LightPosition[0] = 0.f;
  this->LightPosition[1] = 0.f;
  this->LightPosition[2] = 1.f;
  // GL default: ambient=(0,0,0,1), diffuse=(1,1,1,1), specular=(1,1,1,1)
  this->LightColor[0]  = 1.f;
  this->LightColor[1]  = 1.f;
  this->LightColor[2]  = 1.f;
  this->LightIntensity = 1.f;
  // Lighting quadratic attenuation constants.
  this->LightAttenuation[0] = 1.f;  // GL default is 1 (constant)
  this->LightAttenuation[1] = 0.f;  // GL default is 0 (linear)
  this->LightAttenuation[2] = 0.f;  // GL default is 0 (quadratic)
  // spotlight parameters
  this->LightSpotCutoff     = 30.f; // GL default is 180
  this->LightSpotExponent   = 1.f;  // GL default is 0
  // GL default: (0.2, 0.2, 0.2, 1.0)
  this->AmbientLighting[0] = 0.2f;
  this->AmbientLighting[1] = 0.2f;
  this->AmbientLighting[2] = 0.2f;
  this->AmbientLighting[3] = 1.f;
}

svFacePropertiesView::~svFacePropertiesView()
{
  if(this->FaceProperties) delete this->FaceProperties;
}

static void
context_init(void)
{
  static GLboolean initialized = GL_FALSE;
  if(initialized) return; // do once

  glClearColor(0.0, 0.0, 0.0, 0.0);

  // Make sure that stuff that we do NOT need is Disabled.
  glDisable(GL_SCISSOR_TEST);   // 1 ... fragment operations order
  glDisable(GL_ALPHA_TEST);     // 2
  glDisable(GL_STENCIL_TEST);   // 3
  glEnable(GL_DEPTH_TEST);      // 4
  glEnable(GL_BLEND);           // 5 
  glDisable(GL_DITHER);         // 6

  glDepthFunc(GL_LESS);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  // make the projection matrix stack current and make it the identity matrix
  glMatrixMode(GL_PROJECTION);
  gluPerspective( 30.0/* FOV in degrees */, 1.0/* aspect ratio */,
                  1.0/* Z near */, 60.0/* Z far */ );
  // make the modelview matrix stack current and make it the identity matrix
  glMatrixMode(GL_MODELVIEW);
  gluLookAt( 0.0, 0.0, 4.0,  /* eye is at (0,0,40) */
             0.0, 0.0, 0.0,   /* center is at (0,0,0) */
             0.0, 1.0, 0.0 ); /* up is in positive Y direction */
  // dummy push so we can pop on model recalc
  glPushMatrix();

  // enable lighting state
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_NORMALIZE);
}

void
svFacePropertiesView::ConfigureSmoothing(void)
{
  if(this->PointSmoothing) glEnable(GL_POINT_SMOOTH);
  else                     glDisable(GL_POINT_SMOOTH);

  if(this->LineSmoothing) glEnable(GL_LINE_SMOOTH);
  else                    glDisable(GL_LINE_SMOOTH);

  if(this->PolygonSmoothing) glEnable(GL_POLYGON_SMOOTH);
  else                       glDisable(GL_POLYGON_SMOOTH);

  this->ConfiguredSmoothing = GL_TRUE;
}

#if 0
void 
svFacePropertiesView::DrawLightCone(void)
{
}
#else
int   drawCone = 1;

float theta = (float)(M_PI) / 3.f;
float phi   = (float)(M_PI) / 3.f;
float angle = (float)(M_PI) / 12.f; /* GL_SPOT_CUTOFF */

#define rad2deg(_r) (180.0 * (_r) / M_PI)
#define deg2rad(_d) (M_PI * (_d) / 180.0)

// Draw a cone
#define numSides 30
void 
svFacePropertiesView::DrawLightCone(void)
{
  float angle;
  if(this->PositionalLighting)
    angle = 180.f;
  else
    angle = (this->LightSpotCutoff < 180.f ? this->LightSpotCutoff : 180.f);

  float s = 2 * sin(angle);
  float c = 2 * cos(angle);

  // set up the light
  GLfloat color[4];
  
  for(int n=0; n<3; n++) color[n] = this->LightIntensity * this->LightColor[n];
  color[3] = 0.75f;

  glPushMatrix();
  {
    glTranslatef(1.0, 0.0, 1.0);
    glRotatef(45.0, 0.0, 1.0, 0.0);
    glPushAttrib(GL_ENABLE_BIT);
    {
      glDisable(GL_LIGHTING);
      
      // The light
      glEnable(GL_POINT_SMOOTH);
      glPointSize(6.0);
      glColor4f(1.0, 1.0, 1.0, 1.0);
      glBegin(GL_POINTS);
      {
        glVertex3f(0.0, 0.0, 0.0);
      }
      glEnd();
      
      glDisable(GL_DEPTH_TEST);

      // The cone
      if(drawCone) {
        glEnable(GL_CULL_FACE);

        glCullFace(GL_FRONT);
        glBegin(GL_TRIANGLE_FAN);
        {
          glColor4fv(color);
          glVertex3f(0.0, 0.0, 0.0);
          
          glColor4f(1.0, 1.0, 1.0, 0.25);
          for(int i=0; i<=numSides; i++) {
            glVertex3f(s * cos(2*M_PI * i/numSides),
                       s * sin(2*M_PI * i/numSides), -c);
          }
        }
        glEnd();

        glCullFace(GL_BACK);
        glBegin(GL_TRIANGLE_FAN);
        {
          glColor4fv(color);
          glVertex3f(0.0, 0.0, 0.0);
          color[3] = 0.25;
          glColor4fv(color);
          for(int i=0; i<=numSides; i++) {
            glVertex3f(s * cos(2 * i * M_PI / numSides),
                       s * sin(2 * i * M_PI / numSides), -c);
          }
        }
        glEnd();
      }
    }
    glPopAttrib();
  }
  glPopMatrix();
}
#endif /* 0 */

void
svFacePropertiesView::ConfigureLight(void)
{
  // set up the light
  GLfloat color[4];
  
  for(int n=0; n<3; n++) color[n] = this->LightIntensity * this->LightColor[n];
  color[3] = 1.f;
  
  glLightfv(GL_LIGHT0, GL_DIFFUSE, color);
  glLightfv(GL_LIGHT0, GL_SPECULAR, color);
  
  if(this->PositionalLighting) {
    glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 0.f);
    glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 180.f);
  }
  else {
    glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION,  this->LightAttenuation[0]);
    glLightf(GL_LIGHT0, GL_LINEAR_ATTENUATION,    this->LightAttenuation[1]);
    glLightf(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, this->LightAttenuation[2]);
    
    // set up spot parameters if neccesary
    if(this->LightSpotCutoff < 180.f) {
      glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, this->LightSpotExponent);
      glLightf(GL_LIGHT0, GL_SPOT_CUTOFF,   this->LightSpotCutoff);
    }
    else {
      glLighti(GL_LIGHT0, GL_SPOT_CUTOFF, 180);
    }
  }

  this->ConfiguredLight = GL_TRUE;
}

void
svFacePropertiesView::ConfigureLightPosition(void)
{
  // set up the light
  GLfloat position[4];
  GLfloat direction[3];

  for(int n=0; n<3; n++)
    direction[n] = this->LightFocalPoint[n] - this->LightPosition[n];
  
  if(this->PositionalLighting) {
    for(int n=0; n<3; n++) position[n] = -direction[n];
    position[3] = 0.f;
    
    glLightfv(GL_LIGHT0, GL_POSITION, position);
  }
  else {
    for(int n=0; n<3; n++) position[n] = this->LightPosition[n];
    position[3] = 1.f;
    
    glLightfv(GL_LIGHT0, GL_POSITION, position);
    // set up spot direction if neccesary
    if(this->LightSpotCutoff < 180.f) {
      glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, direction);
    }
  }

  this->ConfiguredLightPosition = GL_TRUE;
}

void
svFacePropertiesView::ConfigureLightModel(void)
{
  // light both front and back faces
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, this->TwoSidedLighting);
  // intensity of ambient light from no particular source
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, this->AmbientLighting);

  this->ConfiguredLight = GL_TRUE;
}

void
svFacePropertiesView::ConfigureModel(void)
{
  GLfloat m[4][4];
  
  glPopMatrix();
  glPushMatrix();

  if(!this->ConfiguredLightPosition) this->ConfigureLightPosition();
  
  glTranslatef(this->translate_[0], this->translate_[1], this->translate_[2]);
  this->Trackball->GetMatrixFromQuaternion(m);
  
  glMultMatrixf(&m[0][0]);
  glScalef(this->scale_, this->scale_, this->scale_);
  
  this->ConfiguredModel = GL_TRUE;
}

#if 0
static GLuint
make_cojones_thing(void)
{
  static GLuint listhead = 0;

  // allocate display list
  if(!listhead) listhead = glGenLists(6);
  
  GLdouble eqn0[4] = { 1.0,  0.0,  0.0, 0.0};    // clip left side
  GLdouble eqn1[4] = {-0.5, -0.5,  0.5, 0.8125}; // upper-right-front hole
  GLdouble eqn2[4] = {-0.4, -0.2, -0.5, 0.6};    // upper-right-back hole
  
  glNewList(listhead+5, GL_COMPILE);
  {
    svIcosohedron(4/*1280 faces*/);
  }
  glEndList();

  glNewList(listhead+4, GL_COMPILE);
  {
    glClipPlane(GL_CLIP_PLANE0,eqn0);
    glClipPlane(GL_CLIP_PLANE1,eqn1);
    glClipPlane(GL_CLIP_PLANE2,eqn2);
    glEnable(GL_CLIP_PLANE0);
    glEnable(GL_CLIP_PLANE1);
    glEnable(GL_CLIP_PLANE2);
    glCallList(listhead+5);
    glDisable(GL_CLIP_PLANE0);
    glDisable(GL_CLIP_PLANE1);
    glDisable(GL_CLIP_PLANE2);
  }
  glEndList();
  
  glNewList(listhead+3, GL_COMPILE);
  {
    // first, render the back faces of the translucent geometry
    glCullFace(GL_FRONT);
    glCallList(listhead+4);
    // next, render the front faces of the translucent geometry
    glCullFace(GL_BACK);
    glCallList(listhead+4);
  }
  glEndList();
  
  glNewList(listhead+1, GL_COMPILE);
  {
    glPushMatrix();
    {
      glRotatef(225.f, 0.f, 0.f, 1.f);
      glCallList(listhead+3);
    }
    glPopMatrix();
  }
  glEndList();
  
  glNewList(listhead+2, GL_COMPILE);
  {
    // make the z-buffer read-only for translucent geometry
    glDepthMask(GL_FALSE);
    
    glPushMatrix();
    {
      glRotatef(45.f, 0.f, 0.f, 1.f);
      glCallList(listhead+3);
    }
    glPopMatrix();
  }
  glEndList();

  return listhead;
}
#endif /* 0 */

GLuint
make_opaque_cojones(void)
{
  static GLuint listhead = 0;

  // allocate display list
  if(!listhead) listhead = glGenLists(1);
  
  glNewList(listhead, GL_COMPILE);
  {
    glPushAttrib(GL_LIGHTING_BIT|GL_POLYGON_BIT);
    {
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      glShadeModel(GL_SMOOTH);
      glPushMatrix();
      {
        glScalef(0.5f, 0.5f, 0.5f);
        svIcosohedron(2, GL_TRUE/*color material*/, GL_FALSE);
      }
      glPopMatrix();
    }
    glPopAttrib();
  }
  glEndList();

  return listhead;
}

void
svFacePropertiesView::init(void)
{
  context_init();

  this->ConfigureSmoothing();
  this->ConfigureLight();
  this->ConfigureLightPosition();
  this->ConfigureLightModel();
  this->ConfigureModel();

  if(!this->TestGeometry) {
    this->Model        = svFacePropertiesView::ModelMap["teapot"];
    this->TestGeometry = this->Model->GetDisplayList();
  }
  if(!this->ContrastGeometry)
    this->ContrastGeometry = make_opaque_cojones();

  if(this->Debug) {
    cerr << "svFacePropertiesView(" << this << ")::init()\n";
    svProperty::PrintLighting(cerr);
    svProperty::PrintMaterials(cerr);
  }
}

void
svFacePropertiesView::reshape()
{
  glViewport(0, 0, GLsizei(this->w()), GLsizei(this->h()));
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective( 30.0/* FOV in degrees */,
                  GLdouble(this->w())/GLdouble(this->h())/* aspect ratio */,
                  1.0/* Z near */, 60.0/* Z far */ );
  glMatrixMode(GL_MODELVIEW);
}

void
svFacePropertiesView::draw()
{
  if(!this->valid()) {
    this->init();
    this->reshape();
  }

  if(!this->ConfiguredSmoothing)  this->ConfigureSmoothing();
  if(!this->ConfiguredLight)      this->ConfigureLight();
  if(!this->ConfiguredLightModel) this->ConfigureLightModel();
  if(!this->ConfiguredModel)      this->ConfigureModel();

  // write the opaque geometry depth values
  glDepthMask(GL_TRUE);
  // clear attributes
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  // do face culling
  glEnable(GL_CULL_FACE);
  // cull the back faces of the opaque geometry
  glCullFace(GL_BACK);

  this->DrawLightCone();

  glCallList(this->ContrastGeometry);

  // 
  // Now draw some geometry using our material properties
  // 
  glPushAttrib(GL_LIGHTING_BIT|GL_POLYGON_BIT);
  {
    //this->FaceProperties->Apply();
    //glCallList(this->TestGeometry+1);
    this->FaceProperties->Apply();
    glCallList(this->TestGeometry+1);
  }
  glPopAttrib();

} // draw()

static const char*
GetEventAsString(int event)
{
  switch (event)
    {
    case FL_NO_EVENT:
      switch (fl_xevent->type)
        {
        case KeyPress:          return "FL_NO_EVENT (KeyPress)";
        case KeyRelease:        return "FL_NO_EVENT (KeyRelease)";
        case ButtonPress:       return "FL_NO_EVENT (ButtonPress)";
        case ButtonRelease:     return "FL_NO_EVENT (ButtonRelease)";
        case MotionNotify:      return "FL_NO_EVENT (MotionNotify)";
        case EnterNotify:       return "FL_NO_EVENT (EnterNotify)";
        case LeaveNotify:       return "FL_NO_EVENT (LeaveNotify)";
        case FocusIn:           return "FL_NO_EVENT (FocusIn)";
        case FocusOut:          return "FL_NO_EVENT (FocusOut)";
        case KeymapNotify:      return "FL_NO_EVENT (KeymapNotify)";
        case Expose:            return "FL_NO_EVENT (Expose)";
        case GraphicsExpose:    return "FL_NO_EVENT (GraphicsExpose)";
        case NoExpose:          return "FL_NO_EVENT (NoExpose)";
        case VisibilityNotify:  return "FL_NO_EVENT (VisibilityNotify)";
        case CreateNotify:      return "FL_NO_EVENT (CreateNotify)";
        case DestroyNotify:     return "FL_NO_EVENT (DestroyNotify)";
        case UnmapNotify:       return "FL_NO_EVENT (UnmapNotify)";
        case MapNotify:         return "FL_NO_EVENT (MapNotify)";
        case MapRequest:        return "FL_NO_EVENT (MapRequest)";
        case ReparentNotify:    return "FL_NO_EVENT (ReparentNotify)";
        case ConfigureNotify:   return "FL_NO_EVENT (ConfigureNotify)";
        case ConfigureRequest:  return "FL_NO_EVENT (ConfigureRequest)";
        case GravityNotify:     return "FL_NO_EVENT (GravityNotify)";
        case ResizeRequest:     return "FL_NO_EVENT (ResizeRequest)";
        case CirculateNotify:   return "FL_NO_EVENT (CirculateNotify)";
        case CirculateRequest:  return "FL_NO_EVENT (CirculateRequest)";
        case PropertyNotify:    return "FL_NO_EVENT (PropertyNotify)";
        case SelectionClear:    return "FL_NO_EVENT (SelectionClear)";
        case SelectionRequest:  return "FL_NO_EVENT (SelectionRequest)";
        case SelectionNotify:   return "FL_NO_EVENT (SelectionNotify)";
        case ColormapNotify:    return "FL_NO_EVENT (ColormapNotify)";
        case ClientMessage:     return "FL_NO_EVENT (ClientMessage)";
        case MappingNotify:     return "FL_NO_EVENT (MappingNotify)";
        case LASTEvent:         return "FL_NO_EVENT (LASTEvent)";
        default:                return "FL_NO_EVENT (???)";
        } // switch (fl_xevent->type)
      break;
    case FL_PUSH:           return "FL_PUSH";             break;
    case FL_RELEASE:        return "FL_RELEASE";          break;
    case FL_ENTER:          return "FL_ENTER";            break;
    case FL_LEAVE:          return "FL_LEAVE";            break;
    case FL_DRAG:           return "FL_DRAG";             break;
    case FL_FOCUS:          return "FL_FOCUS";            break;
    case FL_UNFOCUS:        return "FL_UNFOCUS";          break;
#if FL_MAJOR_VERSION == 1
#  if FL_MINOR_VERSION > 0
    case FL_KEYDOWN:        return "FL_KEYDOWN";          break;
    case FL_KEYUP:          return "FL_KEYUP";            break;
#  else
    case FL_KEYBOARD:       return "FL_KEYBOARD";         break;
#  endif /* FL_MINOR_VERSION */
#else
    case FL_KEYBOARD:       return "FL_KEYBOARD";         break;
#endif /* FL_MAJOR_VERSION */
    case FL_CLOSE:          return "FL_CLOSE";            break;
    case FL_MOVE:           return "FL_MOVE";             break;
    case FL_SHORTCUT:       return "FL_SHORTCUT";         break;
    case FL_DEACTIVATE:     return "FL_DEACTIVATE";       break;
    case FL_ACTIVATE:       return "FL_ACTIVATE";         break;
    case FL_HIDE:           return "FL_HIDE";             break;
    case FL_SHOW:           return "FL_SHOW";             break;
    case FL_PASTE:          return "FL_PASTE";            break;
    case FL_SELECTIONCLEAR: return "FL_SELECTIONCLEAR";   break;
#if FL_MAJOR_VERSION == 1
#  if FL_MINOR_VERSION > 0
    case FL_MOUSEWHEEL:     return "FL_MOUSEWHEEL";       break;
#  endif /* FL_MINOR_VERSION */
#endif /* FL_MAJOR_VERSION */
    } // switch (event)

  return "ERROR unknown event";
} // GetEventAsString()

int
svFacePropertiesView::handle(int event)
{
  if(this->shown()) this->make_current();

  int ex = Fl::event_x();
  int ey = Fl::event_y();

  GLfloat width  = GLfloat(this->w());
  GLfloat height = GLfloat(this->h());

  static double period = 1.0 / 30.0;

  static bool dragging = false;
  static int  pushed, released;
  // cursor location in widget coord inside this widget's x,y,w,h
  static int  pushCol, pushRow, dragCol, dragRow, releaseCol, releaseRow;
  int focused = 0;

  switch (event)
    {

    case FL_ENTER: {
      focused = this->take_focus();
    } break;

    //
    // MOUSE events
    //

    case FL_PUSH: { // mouse button clicked inside this widget
      dragging = false;
      pushed   = Fl::event_button();
      pushCol  = ex;
      pushRow  = ey;

      switch (pushed)
        {

        case 1: { // BUTTON 1 pushed
          if(this->spinning_) {
            Fl::remove_timeout(animate,(void*)this);
            this->spinning_ = GL_FALSE;
          }
        } break;

        case 2: { // BUTTON 2 pushed
          // ...
        } break;

        case 3: { // BUTTON 3 pushed
          // ...
        } break;

        } // switch (pushed)

      return 1; // handled FL_PUSH

    } break; // FL_PUSH

    case FL_DRAG: { // some mouse button is clicked and the mouse is moving
      dragging = true;
      dragCol  = ex; 
      dragRow  = ey;

      switch (pushed)
        {

        case 1: { // BUTTON 1 dragging
          GLfloat p1x = (2.f * GLfloat(pushCol) - width) / width;
          GLfloat p1y = (height - 2.f * GLfloat(pushRow)) / height;
          GLfloat p2x = (2.f * GLfloat(dragCol) - width) / width;
          GLfloat p2y = (height - 2.f * GLfloat(dragRow)) / height;
          this->Trackball->Set(p1x, p1y, p2x, p2y);
          pushCol = dragCol;
          pushRow = dragRow;
          if(this->spinmode_) {
            if(!this->spinning_) {
              this->spinning_ = GL_TRUE;
              Fl::add_timeout(period,animate,(void*)this);
            }
          }
          else {
            this->Trackball->IncrementRotation();
            this->ReconfigureModel();
          }
        } break;

        case 2: { // BUTTON 2 dragging
          // ...
          this->translate_[0] += (GLfloat(dragCol - pushCol)*2.f / width);
          this->translate_[1] += (GLfloat(pushRow - dragRow)*2.f / height);
          pushCol = dragCol;
          pushRow = dragRow;
          this->ReconfigureModel();
        } break;

        case 3: { // BUTTON 3 dragging
          this->scale_ *= (1.0 + GLfloat(pushRow - dragRow) / height);
          pushCol = dragCol;
          pushRow = dragRow;
          this->ReconfigureModel();
        } break;

        } // switch (pushed)

      return 1; // handled FL_DRAG

    } break; // FL_DRAG

    case FL_RELEASE: { // some mouse button is released (after click or drag)
      released   = Fl::event_button();
      releaseCol = ex;
      releaseRow = ey;

      if(dragging) { // might be a drag, man, might not
        switch (released) {

        case 1:  // BUTTON 1 released
          // ...
          break;

        case 2:  // BUTTON 2 released
          // ...
          break;

        case 3:  // BUTTON 3 released
          // ...
          break;

        } // switch (released)
      }
      else {
        switch (released) {

        case 1:  // BUTTON 1 released
          // ...
          break;

        case 2:  // BUTTON 2 released
          // ...
          break;

        case 3:  // BUTTON 3 released
          // ...
          break;

        } // switch (released)
      }

      return 1; // handled FL_RELEASE

    } break; // FL_RELEASE

    //
    // KEYBOARD events
    //

    case FL_FOCUS: { // FL_FOCUS
      return 1; // must do this to get keyboard events in general
    } break; // FL_FOCUS

    case FL_UNFOCUS: { // FL_UNFOCUS
      // ... 
    } break; // FL_UNFOCUS:

    case FL_KEYBOARD: { // FL_KEYBOARD: mouse button clicked inside this widget

      char key     = Fl::event_text()[0];

      if(this->Debug) {
        if(Fl::event_state(FL_ALT))
          cerr << "    <Alt>  + '" << int(key) << "'\n";
        if(Fl::event_state(FL_META))
          cerr << "    <Meta> + '" << int(key) << "'\n";
        if(Fl::event_state(FL_CTRL))
          cerr << "    <Ctrl> + '" << int(key) << "'\n";
      }

      switch (key) {

      case 't': {
        if(this->spinning_) {
          Fl::remove_timeout(animate,(void*)this);
          this->spinning_ = GL_FALSE;
        }
        this->spinmode_ = GL_FALSE;
        return 1;
      } break;

      case 'j': {
        this->spinmode_ = GL_TRUE;
        return 1;
      } break;

      } // switch (key)

    } // FL_KEYBOARD

    } // switch (event)

  // pass other events to the base class...
  return Fl_Gl_Window::handle(event);
} // handle()

// 
// End of $Id: svFacePropertiesView.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $.
// 
