// 
// $Id: svSurface.cxx,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $
// 

#include <cstdio>
#include <sys/param.h>
#include <cstring>
#include <cstdlib> // getenv()
#include <iostream>
#include <cctype>

#include "vtkUnstructuredGrid.h"
#include "vtkPoints.h"
#include "vtkFloatArray.h"
#include "vtkCellArray.h"
#include "vtkPolyDataMapper.h"
#include "vtkContourFilter.h"
#include "vtkGaussianSplatter.h"
#include "vtkPolyDataNormals.h"
#include "vtkDecimatePro.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkTubeFilter.h"
#include "vtkAxes.h"
#include "vtkCubeAxesActor2D.h"
#include "vtkCamera.h"

#include "svSurface.H"
#include "svSurface.H"

static char* fgetl(char* s, int n, FILE* fp);

vtkDataSet*
Surface::ReadAsciiToUGrid(char* fname)
{
  char filename[MAXPATHLEN+1];
  const char* temp = getenv(SV_SURFACE_DATA_ROOT);
  if(!temp) {
    cerr << "ERROR: Please define environment variable "
         << SV_SURFACE_DATA_ROOT << "\n";
    return 0;
  }
  strcpy(filename,temp);
  strcat(filename,"/");
  strcat(filename,fname);
  
  FILE* fp = 0;
  if(!(fp=fopen(filename,"r"))) {
    cerr << "ERROR: Can't read file: '" << filename << "'\n";
    return 0;
  }

  char  line[512];
  int   nvertices = 0, nfaces = 0;
  char* cp = fgetl(line, 512, fp);
  sscanf(cp,"%d %d\n",&nvertices, &nfaces) ;

  cerr << "ReadData(nvertices=" << nvertices << ", nfaces=" << nfaces << ")\n";

  vtkPoints*     surfacePoints  = vtkPoints::New();
  surfacePoints->SetNumberOfPoints(vtkIdType(nvertices));

  vtkFloatArray* surfaceScalars = vtkFloatArray::New();
  surfaceScalars->SetNumberOfComponents(1);
  surfaceScalars->SetNumberOfTuples(nvertices);

  // get the vertices
  register float vertex[3];
  for(int vindex=0; vindex<nvertices; vindex++) {
    fscanf(fp,"%f  %f  %f  %*d\n", &vertex[0], &vertex[1], &vertex[2]);
    surfacePoints->SetPoint(vindex, vertex);
    surfaceScalars->InsertTuple1(vindex, 1.f);
  }
    
  vtkUnstructuredGrid* surfaceData = vtkUnstructuredGrid::New();
  surfaceData->SetPoints(surfacePoints);
  (surfaceData->GetPointData())->SetScalars(surfaceScalars);
    
  // clean up references
  surfacePoints->Delete();
  surfaceScalars->Delete();

  return surfaceData;
}

vtkPolyData*
Surface::ReadAsciiToPolys(const char* fname)
{
  char filename[2048];
  const char* temp = getenv("SURFACE_DATA_ROOT");
  if(!temp) {
    cerr << "ERROR: Please define environment variable SURFACE_DATA_ROOT\n";
    return 0;
  }
  strcpy(filename,temp);
  strcat(filename,"/");
  strcat(filename,fname);
  
  FILE* fp = 0;
  if(!(fp=fopen(filename,"r"))) {
    cerr << "ERROR: Can't read file: '" << filename << "'\n";
    return 0;
  }

  char  line[512];
  int   nvertices = 0, nfaces = 0;
  char* cp = fgetl(line, 512, fp);
  sscanf(cp,"%d %d\n",&nvertices, &nfaces) ;

  cerr << "ReadData(nvertices=" << nvertices << ", nfaces=" << nfaces << ")\n";
  
  vtkPolyData*   surface = vtkPolyData::New();
  vtkPoints*     points  = vtkPoints::New();
  vtkFloatArray* scalars = vtkFloatArray::New();
  vtkCellArray*  polys   = vtkCellArray::New();

  points->SetNumberOfPoints(vtkIdType(nvertices));
  scalars->SetNumberOfValues(vtkIdType(nvertices));
  polys->Allocate(polys->EstimateSize(nfaces,VERTICES_PER_FACE));

  register float vertex[3];
  // get the vertices
  for(int vindex=0; vindex<nvertices; vindex++) {
    fscanf(fp,"%f  %f  %f  %*d\n",&vertex[0],&vertex[1],&vertex[2]);
    // must call SetNumberOfPoints() prior to using SetPoint()
    points->SetPoint(vindex, vertex);
    // must call SetNumberOfValues() prior to using SetValue()
    scalars->SetValue(vindex, 1.f);
  }

  int tmp;
  register vtkIdType pts[VERTICES_PER_FACE];
  // get the faces
  for(int findex=0; findex<nfaces; findex++) {
    for(int n=0; n<VERTICES_PER_FACE; n++) {
      fscanf(fp,"%d ", &tmp);
      pts[n] = vtkIdType(tmp);
    }
    fscanf(fp,"%*d\n");
    // incert the face vertex Ids
    polys->InsertNextCell(VERTICES_PER_FACE, pts);
  }
  
  // 
  // We now assign the pieces to the vtkPolyData.
  // 

  // vertices
  surface->SetPoints(points);
  // faces
  surface->SetPolys(polys);
  // scalars
  (surface->GetPointData())->SetScalars(scalars);

  surface->BuildCells();
  surface->BuildLinks();
  // probabbly do not need this as the polydata is built from known 
  //   number vertices and faces
  surface->Squeeze();

  // clean-up
  points->Delete();
  scalars->Delete();
  polys->Delete();

  cerr << "PolyData size: " << surface->GetActualMemorySize() << " (kb)\n";

  return surface;
}

vtkActor*
Surface::MakeSurfaceActor(vtkDataSet* surfaceData, float rgba[4])
{
  vtkGaussianSplatter* surfaceSplatter = vtkGaussianSplatter::New();
  surfaceSplatter->SetInput(surfaceData);
  surfaceSplatter->SetSampleDimensions(50, 50, 50);
  surfaceSplatter->SetRadius(0.01);
  surfaceSplatter->ScalarWarpingOff();

  vtkContourFilter* surfaceContourFilter = vtkContourFilter::New();
  surfaceContourFilter->SetInput(surfaceSplatter->GetOutput());
  surfaceContourFilter->SetValue(0,0.01);

  vtkPolyDataMapper* surfaceMapper = vtkPolyDataMapper::New();
  surfaceMapper->SetInput(surfaceContourFilter->GetOutput());
  surfaceMapper->ScalarVisibilityOff();

  vtkActor* surfaceActor = vtkActor::New();
  surfaceActor->SetMapper(surfaceMapper);
  (surfaceActor->GetProperty())->SetColor(rgba);
  (surfaceActor->GetProperty())->SetOpacity(rgba[3]);

  // clean-up
  surfaceSplatter->Delete();
  surfaceContourFilter->Delete();
  surfaceMapper->Delete();

  return surfaceActor;
}

vtkActor*
Surface::MakeSurfaceActor(vtkPolyData* data, float rgba[4], bool filter=false)
{
  vtkPolyDataMapper* surfaceMapper = vtkPolyDataMapper::New();

  if(filter) {
    
    // vtkWindowedSincPolyDataFilter* smoother =
    //   vtkWindowedSincPolyDataFilter::New();
    // smoother->SetNumberOfIterations(20);
    // smoother->SetPassBand(0.5f);
    // smoother->SetInput(data);

    // vtkDecimatePro* decimator = vtkDecimatePro::New();
    // decimator->SetTargetReduction(0.9);
    // decimator->SplittingOn();
    // decimator->PreserveTopologyOn();
    // decimator->SetInput(data);

    vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
    normals->SplittingOn();
    normals->ComputePointNormalsOn();
    normals->ComputeCellNormalsOff();
    normals->FlipNormalsOn();
    // normals->NonManifoldTraversalOn();
    // normals->SetInput(decimator->GetOutput());
    // normals->SetInput(smoother->GetOutput());
    // normals->SetInput(smoother->GetOutput());
    normals->SetInput(data);
    surfaceMapper->SetInput(normals->GetOutput());
    // clean up references
    normals->Delete();
  }
  else {
    surfaceMapper->SetInput(data);
  }

  surfaceMapper->ScalarVisibilityOff();

  vtkActor* surfaceActor = vtkActor::New();
  surfaceActor->SetMapper(surfaceMapper);
  (surfaceActor->GetProperty())->SetColor(rgba);
  (surfaceActor->GetProperty())->SetOpacity(rgba[3]);
  if(filter) {
    (surfaceActor->GetProperty())->SetInterpolationToGouraud();
  }

  // clean up references
  surfaceMapper->Delete();

  return surfaceActor;
}

vtkActor*
Surface::MakeAxesActor(vtkDataSet* dataSet)
{
  // create axes
  vtkActor* axesActor = vtkActor::New();
  {
    vtkPolyDataMapper* axesMapper = vtkPolyDataMapper::New();
    {
      vtkTubeFilter* axesTubes = vtkTubeFilter::New();
      {
        vtkAxes* axes = vtkAxes::New();
        {
          float bounds[6];
          dataSet->GetBounds(bounds);
          axes->SetOrigin(bounds[0], bounds[2], bounds[4]);
          axes->SetScaleFactor(dataSet->GetLength()/4.0);
        }
        axesTubes->SetInput(axes->GetOutput());
        axesTubes->SetRadius(axes->GetScaleFactor()/100.0);
        axesTubes->SetNumberOfSides(10);
        // clean-up
        axes->Delete();
      }
      axesMapper->SetInput(axesTubes->GetOutput());
      // clean-up
      axesTubes->Delete();
    }
    axesActor->SetMapper(axesMapper);
    // clean-up
    axesMapper->Delete();
  }

  return axesActor;
}

vtkActor*
Surface::MakeAxesActor(float bounds[6])
{
  // create axes
  vtkActor* axesActor = vtkActor::New();
  {
    vtkPolyDataMapper* axesMapper = vtkPolyDataMapper::New();
    {
      vtkTubeFilter* axesTubes = vtkTubeFilter::New();
      {
        vtkAxes* axes = vtkAxes::New();
        {
          axes->SetOrigin(bounds[0], bounds[2], bounds[4]);
          axes->SetScaleFactor(1.0);
        }
        axesTubes->SetInput(axes->GetOutput());
        axesTubes->SetRadius(axes->GetScaleFactor()/100.0);
        axesTubes->SetNumberOfSides(10);
        // clean-up
        axes->Delete();
      }
      axesMapper->SetInput(axesTubes->GetOutput());
      // clean-up
      axesTubes->Delete();
    }
    axesActor->SetMapper(axesMapper);
    // clean-up
    axesMapper->Delete();
  }

  return axesActor;
}

void
Surface::MakeCubeAxes(vtkProp* axes, float bounds[6], vtkCamera* camera)
{
  vtkCubeAxesActor2D* cubeAxes = vtkCubeAxesActor2D::SafeDownCast(axes);

  if(!cubeAxes) {
    cubeAxes = vtkCubeAxesActor2D::New();
  }

  cubeAxes->SetBounds(bounds);
  cubeAxes->SetCamera(camera);
  cubeAxes->SetLabelFormat("%6.4g");
  cubeAxes->ShadowOn();
  cubeAxes->SetFlyModeToOuterEdges();
  cubeAxes->SetFontFactor(0.8);
  (cubeAxes->GetProperty())->SetColor(1.f, 1.f, 1.f);
}

static char*
fgetl(char* s, int n, FILE* fp)
{
  char* cp;

  do
  {
    cp = fgets(s, n, fp) ;
    if(!cp) return 0;

    while(isspace(int(*cp))) cp++;

  } while(((*cp) == '#') || ((*cp) == '\n') || ((*cp) == 0));

  for(char* cp2=cp; *cp2; cp2++) {
    if(*cp2 == '#') *cp2 = '\0';
  }

  int len = strlen(cp);
  if(cp[len-1] == '\n') cp[len-1] = '\0'; // strip newline

  return cp;
}

// 
// End of: $Id: svSurface.cxx,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $.
// 
