//
// "$Id: svType.cxx,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $"
//
// C function type code for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2002 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "fltk-bugs@fltk.org".
//

#include <FL/Fl.H>
#include "svBrowserObject.h"
#include <FL/fl_show_input.H>
#include <string.h>
#include <stdio.h>
#include <ctype.h>

extern int i18n_type;
extern const char* i18n_include;
extern const char* i18n_function;
extern const char* i18n_file;
extern const char* i18n_set;
extern char i18n_program[];

// return current indentation:
static const char* spaces = "                ";
static int indentation;
const char*
indent()
{
  int i = indentation;
  if(i>16) i = 16;
  return spaces+16-i;
}

// storestring()
// Replaces string pointer with new value, maybe strip leading/trailing blanks.
// Returns whether the destination was changed (true|false)
bool
storestring(const char* n, const char * & p, bool strip=true)
{
  if(n == p) return false;
  int length = 0;

  if(n) {
    // strip leading blanks
    if(strip) while(isspace(*n)) n++;
    // strip trailing blanks
    const char* end = n + strlen(n);
    if(strip) while(end > n && isspace(*(end-1))) end--;
    length = end - n;
    if(!length) n = 0; // NULL
  }    

  if(n == p) return false;

  // if n and p are >0 strlen and match over `length' and strlen(p) is `length'.
  if(n && p && !strncmp(n,p,length) && !p[length]) return false;

  if(p) free((void*)p);

  if(!n || !(*n)) {
    // if NULL pointer or nul string
    p = 0;
  } else {
    // else strdup `n' to `q' and assign `q' address to `p'.
    char* q = (char*)malloc(length+1);
    strncpy(q,n,length);
    q[length] = '\0';
    p = q;
  }
  modflag = 1;

  return true;
}

// 
// svBrowserObject
// 

svBrowserObject* svBrowserObject::first;
svBrowserObject* svBrowserObject::last;

// protected constructor
svBrowserObject::svBrowserObject()
  : name_(0), label_(0), callback_(0), user_data_(0), user_data_type_(0),
  parent(0),
  new_selected(0), selected(0), visible(0), rtti(0), level(0),
  next(0), prev(0),
  factory(0)
{ }

// destructor
svBrowserObject::~svBrowserObject()
{
  // warning: destructor only works for widgets that have been add()ed.
  if(widget_browser) widget_browser->deleting(this);

  if(this->prev) this->prev->next = this->next;
  else svBrowserObject::first = this->next;

  if(this->next) this->next->prev = this->prev;
  else svBrowserObject::last = this->prev;

  if(svBrowserObject::current == this) svBrowserObject::current = 0;

  modflag = 1;

  if(this->parent) this->parent->remove_child(this);
}

// 
static void
fixvisible(svBrowserObject* p)
{
  svBrowserObject* t = p;
  for(;;) {
    t->visible = (t->parent ? t->parent->visible && t->parent->open_ : 1);
    t = t->next;
    if(!t || t->level <= p->level) break;
  }
}

void
svBrowserObject::setlabel(const char*) {}

// add a list of widgets as a new child of p:
void
svBrowserObject::add(svBrowserObject* theParent)
{
  if(theParent && this->parent == theParent) return;

  this->parent = theParent;

  svBrowserObject* end = this;
  while(end->next) end = end->next;

  svBrowserObject* q;
  int     newlevel;
  if(theParent) {
    for(q = theParent->next; q && q->level > theParent->level; q = q->next);
    newlevel = theParent->level + 1;
  } else {
    q = 0;
    newlevel = 0;
  }

  for(svBrowserObject* t = this->next; t; t = t->next)
    t->level += (newlevel - this->level);

  this->level = newlevel;
  if(q) {
    this->prev = q->prev;
    this->prev->next = this;
    q->prev = end;
    end->next = q;
  } else if(svBrowserObject::first) {
    this->prev = svBrowserObject::last;
    this->prev->next = this;
    end->next = 0;
    svBrowserObject::last = end;
  } else {
    svBrowserObject::first = this;
    svBrowserObject::last = end;
    this->prev = end->next = 0;
  }

  if(theParent) theParent->add_child(this,0);

  this->open_ = 1;
  fixvisible(this);
  modflag = 1;

  widget_browser->redraw();
}

// add to a parent before another widget:
void
svBrowserObject::insert(svBrowserObject* g)
{
  svBrowserObject* end = this;

  while(end->next) end = end->next;

  this->parent  = g->parent;
  int newlevel  = g->level;
  this->visible = g->visible;

  for(svBrowserObject* t = this->next; t; t = t->next) t->level += newlevel-this->level;

  this->level = newlevel;
  this->prev  = g->prev;

  if(this->prev) this->prev->next = this;
  else svBrowserObject::first = this;

  end->next = g;
  g->prev   = end;

  fixvisible(this);

  if(this->parent) this->parent->add_child(this, g);

  widget_browser->redraw();
}

// delete from parent:
svBrowserObject*
svBrowserObject::remove()
{
  svBrowserObject* end = this;

  for(;;) {
    if(!end->next || end->next->level <= this->level) break;
    end = end->next;
  }

  if(this->prev) this->prev->next = end->next;
  else svBrowserObject::first = end->next;

  if(end->next) end->next->prev = this->prev;
  else svBrowserObject::last = this->prev;

  svBrowserObject* r = end->next;
  this->prev = end->next = 0;

  if(this->parent) this->parent->remove_child(this);
  this->parent = 0;

  widget_browser->redraw();
  selection_changed(0);

  return r;
}

// move f (and it's children) into list before g:
// returns pointer to whatever is after f & children
void
svBrowserObject::move_before(svBrowserObject* g)
{
  if(this->level != g->level)
    cout << "move_before levels don't match! " << this->level
         << " != " << g->level << "\n";

  svBrowserObject* n;
  for(n = this->next; n && n->level > this->level; n = n->next);

  if(n == g) return;

  svBrowserObject* l = (n ? n->prev : svBrowserObject::last);
  this->prev->next = n;

  if(n) n->prev = this->prev;
  else svBrowserObject::last = this->prev;

  this->prev = g->prev;
  l->next = g;

  if(this->prev) this->prev->next = this;
  else svBrowserObject::first = this;

  g->prev = l;

  if(this->parent) this->parent->move_child(this,g);

  widget_browser->redraw();
}

// Generate a descriptive text for this item, to put in browser & window titles
const char*
svBrowserObject::title()
{
  const char* c = this->name();
  if(c) return c;
  return this->type_name();
}

void
svBrowserObject::name(const char* n)
{
  if(storestring(n,name_)) {
    if(this->visible) widget_browser->redraw();
  }
}

void
svBrowserObject::label(const char* n)
{
  if(storestring(n,this->label_,false)) {
    this->setlabel(this->label_);
    if(this->visible && !this->name_) widget_browser->redraw();
  }
}

void
svBrowserObject::callback(const char* n)
{
  storestring(n,this->callback_);
}

void
svBrowserObject::user_data(const char* n)
{
  storestring(n,this->user_data_);
}

void
svBrowserObject::user_data_type(const char* n)
{
  storestring(n,this->user_data_type_);
}

// turn a click at x,y on this into the actual picked object:
svBrowserObject*
svBrowserObject::click_test(int, int)
{
  return 0;
}

void
svBrowserObject::add_child(svBrowserObject*, svBrowserObject*) {}

void
svBrowserObject::move_child(svBrowserObject*, svBrowserObject*) {}

void
svBrowserObject::remove_child(svBrowserObject*) {}

void
svBrowserObject::open()
{
  cout << "Open of '" << this->type_name() << "' is not yet implemented\n";
}

// write a widget and all it's children:
void
svBrowserObject::write()
{
  write_indent(this->level);
  write_word(this->type_name());
  write_word(this->name());
  write_open(this->level);
  write_close(this->level);
  if(!this->is_parent()) return;
  // now do children:
  write_open(this->level);
  svBrowserObject* child;
  for(child = next; child && child->level>this->level; child = child->next) {
    if(child->level == this->level+1) child->write();
  }
  write_close(this->level);
}

// Return message number for I18N...
int
svBrowserObject::msgnum()
{
  int count;
  svBrowserObject* p;

  for(count=0,p=this; p; ) {
    if(p->label()) count++;
    if( p!=this &&
        p->is_widget() &&
        (reinterpret_cast<svWidgetType*>(p))->tooltip() ) count++;

    p = (p->prev ? p->prev : p->parent);
  }

  return count;
}

const char*
svBrowserObject::class_name(const int need_nest) const
{
  svBrowserObject* p = this->parent;
  while(p) {
    if(p->is_class()) {
      // see if we are nested in another class, we must fully-qualify name:
      // this is lame but works...
      const char* q = 0;
      if(need_nest) q = p->class_name(need_nest);
      if(q) {
	static char buffer[256];
	if(q != buffer) strlcpy(buffer, q, 256);
	strlcat(buffer, "::", 256);
	strlcat(buffer, p->name(), 256);
	return buffer;
      }
      return p->name();
    }
    p = p->parent;
  }
  return 0;
}


////////////////////////////////////////////////////////////////

// move selected widgets in their parent's list:
void
earlier_cb(Fl_Widget*, void*)
{
  svBrowserObject* f;
  for(f = svBrowserObject::first; f; ) {
    svBrowserObject* nxt = f->next;
    if(f->selected) {
      svBrowserObject* g;
      for(g = f->prev; g && g->level > f->level; g = g->prev);
      if(g && g->level == f->level && !g->selected) f->move_before(g);
    }
    f = nxt;
  }
}

void
later_cb(Fl_Widget*, void*)
{
  svBrowserObject* f;
  for(f = svBrowserObject::last; f; ) {
    svBrowserObject* prv = f->prev;
    if(f->selected) {
      svBrowserObject* g;
      for(g = f->next; g && g->level > f->level; g = g->next);
      if(g && g->level == f->level && !g->selected) g->move_before(f);
    }
    f = prv;
  }
}

////////////////////////////////////////////////////////////////


svBrowserObject*
svBrowserRenderer::make()
{
  svBrowserObject* p = svBrowserObject::current;
  while(p) p = p->parent;

  svBrowserRenderer* o = new svBrowserRenderer();
  o->name("Renderer");
  o->add(p);
  o->factory = this;
  o->public_ = 1;

  return o;
}

#include "function_panel.h"
#include <FL/fl_ask.H>

void
svBrowserRenderer::open()
{
  if(!function_panel) make_function_panel();

  // update initial input values
  f_name_input->static_value(this->name());
  f_public_button->value(this->public_);

  // show the input window
  function_panel->show();

  const char* message = 0;
  for(;;) {
    // repeat as long as there are errors

    if(message) fl_alert(message);

    for(;;) {
      Fl_Widget* w = Fl::readqueue();
      if(w == f_panel_cancel) goto BREAK2;
      else if(w == f_panel_ok) break;
      else if(!w) Fl::wait();
    }

    const char*c = f_name_input->value();
    while(isspace(*c)) c++; // strip blanks
    // parse for C code syntax errors
    message = c_check(c); if(message) continue;
    const char* d = c;
    for(; *d != '('; d++) if(isspace(*d) || !*d) break;
    if(*c && *d != '(') {
      message = "must be name(arguments), try again:"; continue;
    }

    // update member data
    this->name(f_name_input->value());
    this->public_ = f_public_button->value();

    break;
  }

  BREAK2:
  function_panel->hide();
}




////////////////////////////////////////////////////////////////

int
svClassType::is_public() const
{
  return public_;
}

svBrowserObject*
svClassType::make()
{
  svBrowserObject* p = svBrowserObject::current;
  while(p && !p->is_decl_block()) p = p->parent;
  svClassType* o = new svClassType();
  o->name("UserInterface");
  o->subclass_of = 0;
  o->public_ = 1;
  o->add(p);
  o->factory = this;
  return o;
}

void
svClassType::open()
{
  if(!class_panel) make_class_panel();
  c_name_input->static_value(name());
  c_subclass_input->static_value(subclass_of);
  c_public_button->value(public_);
  class_panel->show();
  const char* message = 0;
  for(;;) { // repeat as long as there are errors
    if(message) fl_alert(message);
    for(;;) {
      Fl_Widget* w = Fl::readqueue();
      if(w == c_panel_cancel) goto BREAK2;
      else if(w == c_panel_ok) break;
      else if(!w) Fl::wait();
    }
    const char*c = c_name_input->value();
    while(isspace(*c)) c++;
    if(!*c) goto OOPS;
    while(is_id(*c)) c++;
    while(isspace(*c)) c++;
    if(*c) {OOPS: message = "class name must be C++ identifier"; continue;}
    c = c_subclass_input->value();
    message = c_check(c); if(message) continue;
    name(c_name_input->value());
    storestring(c, subclass_of);
    public_ = c_public_button->value();
    break;
  }
  BREAK2:
  class_panel->hide();
}


//
// Widget type code for the Fast Light Tool Kit (FLTK).
//

#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <FL/Fl_Input.H>
#include "svWidgetType.H"
#include "alignment_panel.h"
#include <FL/fl_message.H>
#include <FL/Fl_Slider.H>
#include <FL/Fl_Window.H>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

// Make an svWidgetType subclass instance.
// It figures out the automatic size and parent of the new widget,
// creates the Fl_Widget (by calling the virtual function _make),
// adds it to the Fl_Widget hierarchy, creates a new svBrowserObject
// instance, sets the widget pointers, and makes all the display
// update correctly...

extern int         reading_file;
int                force_parent;
extern int         gridx;
extern int         gridy;
extern int         i18n_type;
extern const char* i18n_include;
extern const char* i18n_function;
extern const char* i18n_file;
extern const char* i18n_set;


const char*
subclassname(svBrowserObject* l)
{
  if(l->is_widget()) {
    svWidgetType* p = reinterpret_cast<svWidgetType*>(l);
    const char* c = p->subclass();
    if(c) return c;
  }
  return l->type_name();
}

svBrowserObject*
svWidgetType::make()
{
  // Find the current widget, or widget to copy:
  svBrowserObject* qq = svBrowserObject::current;
  while(qq && !qq->is_widget()) qq = qq->parent;
  if(!qq) {
    fl_message("Please select a widget");
    return 0;
  }
  svWidgetType* q = reinterpret_cast<svWidgetType*>(qq);
  // find the parent widget:
  svWidgetType* p = q;
  if((force_parent || !p->is_group()) && p->parent->is_widget())
    p = reinterpret_cast<svWidgetType*>(p->parent);
  force_parent = 0;

  // Figure out a border between widget and window:
  int B = p->o->w()/2;
  if(p->o->h()/2 < B) B = p->o->h()/2;
  if(B>25) B = 25;

  int ULX,ULY; // parent's origin in window
  if(!p->is_window()) { // if it is a group, add corner
    ULX = p->o->x();
    ULY = p->o->y();
  } else {
    ULX = ULY = 0;
  }

  // Figure out a position and size for the widget
  int  X, Y, W, H;
  if(this->is_group()) { // fill the parent with the widget
    X = ULX + B;
    W = p->o->w() - B;
    Y = ULY + B;
    H = p->o->h() - B;
  }
  else if(q != p) { // copy position and size of current widget
    W = q->o->w();
    H = q->o->h();
    X = q->o->x() + W;
    Y = q->o->y();
    if((X + W) > (ULX + p->o->w())) {
      X = q->o->x();
      Y = q->o->y() + H;
      if((Y + H) > (ULY + p->o->h())) Y = ULY + B;
    }
  }
  else { // just make it small and square...
    X = ULX + B;
    Y = ULY + B;
    W = H = B;
  }

  // satisfy the grid requirements (otherwise it edits really strangely):
  if(gridx>1) {
    X = (X/gridx)*gridx;
    W = ((W-1)/gridx+1)*gridx;
  }
  if(gridy>1) {
    Y = (Y/gridy)*gridy;
    H = ((H-1)/gridy+1)*gridy;
  }

  // Construct the svBrowserObject:
  svWidgetType* t = _make();

  if(!o) o = widget(0,0,100,100); // create template widget

  t->factory = this;
  // Construct the Fl_Widget:
  t->o = widget(X,Y,W,H);

  if(reading_file) t->o->label(0);
  else if(t->o->label()) t->label(t->o->label()); // allow editing

  t->o->user_data((void*)t);

  // Put it in the parent:
  //  ((Fl_Group *)(p->o))->add(t->o); (done by svBrowserObject::add())
  // add to browser:
  t->add(p);
  t->redraw();

  return t;
}


#include "Fluid_Image.h"

void
svWidgetType::setimage(Fluid_Image* i)
{
  if(i == this->image) return;

  if(this->image) this->image->decrement();
  if(i) i->increment();

  this->image = i;

  if(i) i->image(this->o);
  else  this->o->image(0);

  this->redraw();
}

void
svWidgetType::setinactive(Fluid_Image* i)
{
  if(i == this->inactive) return;

  if(this->inactive) this->inactive->decrement();
  if(i) i->increment();

  this->inactive = i;

  if(i) i->deimage(this->o);
  else  this->o->deimage(0);

  this->redraw();
}

void svWidgetType::setlabel(const char* n)
{
  o->label(n);
  redraw();
}

svWidgetType::svWidgetType() {
  for(int n=0; n<NUM_EXTRA_CODE; n++) {extra_code_[n] = 0; subclass_ = 0;}
  tooltip_ = 0;
  image_name_ = 0;
  inactive_name_ = 0;
  image = 0;
  inactive = 0;
  xclass = 0;
  o = 0;
  public_ = 1;
}

svWidgetType::~svWidgetType() {
  if(o) {
    o->hide();
    if(o->parent()) ((Fl_Group*)o->parent())->remove(*o);
    delete o;
  }
}

void svWidgetType::extra_code(int m,const char* n) {
  storestring(n,extra_code_[m]);
}

extern void redraw_browser();
void svWidgetType::subclass(const char* n) {
  if(storestring(n,subclass_) && visible)
    redraw_browser();
}

void svWidgetType::tooltip(const char* n) {
  storestring(n,tooltip_);
  o->tooltip(n);
}

void svWidgetType::image_name(const char* n) {
  setimage(Fluid_Image::find(n));
  storestring(n,image_name_);
}

void svWidgetType::inactive_name(const char* n) {
  setinactive(Fluid_Image::find(n));
  storestring(n,inactive_name_);
}

void
svWidgetType::redraw()
{
  svBrowserObject* t = this;
  while(t->parent && t->parent->is_widget()) t = t->parent;
  ((svWidgetType*)t)->o->redraw();
}

// the recursive part sorts all children, returns pointer to next:
svBrowserObject* sort(svBrowserObject* parent) {
  svBrowserObject* f,*n=0;
  for(f = parent ? parent->next : svBrowserObject::first; ; f = n) {
    if(!f || parent && f->level <= parent->level) return f;
    n = sort(f);
    if(!f->selected || !f->is_widget()) continue;
    Fl_Widget* fw = ((svWidgetType*)f)->o;
    svBrowserObject* g; // we will insert before this
    for(g = parent->next; g != f; g = g->next) {
      if(!g->selected || g->level > f->level) continue;
      Fl_Widget* gw = ((svWidgetType*)g)->o;
      if(gw->y() > fw->y()) break;
      if(gw->y() == fw->y() && gw->x() > fw->x()) break;
    }
    if(g != f) f->move_before(g);
  }
}


////////////////////////////////////////////////////////////////

// turn number to string or string to number for saving to file:
// does not work for hierarchial menus!

const char*
item_name(Fl_Menu_Item* m, int i)
{
  if(m) {
    while(m->label()) {
      if(m->argument() == i) return m->label();
      m++;
    }
  }

  static char buffer[20];
  (void)snprintf(buffer,20,"%d", i);

  return buffer;
}

int
item_number(Fl_Menu_Item* m, const char* i)
{
  if(m && i) {
    if(i[0]=='F' && i[1]=='L' && i[2]=='_') i += 3;
    while(m->label()) {
      if(!strcmp(m->label(), i)) return int(m->argument());
      m++;
    }
  }

  return atoi(i);
}


void visible_cb(Fl_Light_Button* i, void* v) {
  if(v == LOAD) {
    i->value(current_widget->o->visible());
    if(current_widget->is_window()) i->deactivate();
    else i->activate();
  } else {
    int n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	n ? q->o->show() : q->o->hide();
	q->redraw();
      }
  }
}

////////////////////////////////////////////////////////////////

void color_cb(Fl_Button* i, void* v) {
  Fl_Color c = current_widget->o->color();
  if(v == LOAD) {
    i->activate();
  } else {
    Fl_Color d = fl_show_colormap(c);
    if(d == c) return;
    c = d;
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	q->o->color(c); q->o->redraw();
      }
  }
  i->color(c); i->labelcolor(fl_contrast(FL_BLACK,c)); i->redraw();
}

void user_data_cb(Fl_Input* i, void* v) {
  if(v == LOAD) {
    i->static_value(current_widget->user_data());
  } else {
    const char* c = i->value();
    const char* d = c_check(c);
    if(d) {fl_message("Error in user_data: %s",d); haderror = 1; return;}
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next) if(o->selected) {
      o->user_data(c);
    }
  }
}

void user_data_type_cb(Fl_Input* i, void* v) {
  static const char* dflt = "void*";
  if(v == LOAD) {
    const char* c = current_widget->user_data_type();
    if(!c) c = dflt;
    i->static_value(c);
  } else {
    const char* c = i->value();
    const char* d = c_check(c);
    if(!*c) i->value(dflt);
    else if(!strcmp(c,dflt)) c = 0;
    if(!d) {
      if(c && *c && c[strlen(c)-1] != '*' && strcmp(c,"long"))
	d = "must be pointer or long";
    }
    if(d) {fl_message("Error in type: %s",d); haderror = 1; return;}
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next) if(o->selected) {
      o->user_data_type(c);
    }
  }
}


void subclass_cb(Fl_Input* i, void* v) {
  if(v == LOAD) {
    i->activate();
    i->static_value(current_widget->subclass());
  } else {
    const char* c = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* t = (svWidgetType*)o;
	t->subclass(c);
      }
  }
}

////////////////////////////////////////////////////////////////

// textstuff: set textfont, textsize, textcolor attributes:

// default widget returns 0 to indicate not-implemented:
int svWidgetType::textstuff(int, Fl_Font&, int&, Fl_Color&) {return 0;}

void textfont_cb(Fl_Choice* i, void* v) {
  Fl_Font n; int s; Fl_Color c;
  if(v == LOAD) {
    if(!current_widget->textstuff(0,n,s,c)) {i->deactivate(); return;}
    i->activate();
    if(n > 15) n = FL_HELVETICA;
    i->value(n);
  } else {
    n = (Fl_Font)i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	q->textstuff(1,n,s,c);
	q->o->redraw();
      }
  }
}

void textsize_cb(Fl_Value_Input* i, void* v) {
  Fl_Font n; int s; Fl_Color c;
  if(v == LOAD) {
    if(!current_widget->textstuff(0,n,s,c)) {i->deactivate(); return;}
    i->activate();
  } else {
    s = int(i->value());
    if(s <= 0) s = FL_NORMAL_SIZE;
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	q->textstuff(2,n,s,c);
	q->o->redraw();
      }
  }
  i->value(s);
}

void textcolor_cb(Fl_Button* i, void* v) {
  Fl_Font n; int s; Fl_Color c;
  if(v == LOAD) {
    if(!current_widget->textstuff(0,n,s,c)) {i->deactivate(); return;}
    i->activate();
  } else {
    c = i->color();
    Fl_Color d = fl_show_colormap(c);
    if(d == c) return;
    c = d;
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	q->textstuff(3,n,s,c); q->o->redraw();
      }
  }
  i->color(c); i->labelcolor(fl_contrast(FL_BLACK,c)); i->redraw();
}

////////////////////////////////////////////////////////////////
// Kludges to the panel for subclasses:

void slider_size_cb(Fl_Value_Input* i, void* v) {
  if(v == LOAD) {
    if(current_widget->is_valuator()!=2) {i->deactivate(); return;}
    i->activate();
    i->value(((Fl_Slider*)(current_widget->o))->slider_size());
  } else {
    double n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->is_valuator()==2) {
	  ((Fl_Slider*)(q->o))->slider_size(n);
	  q->o->redraw();
	}
      }
  }
}

void min_cb(Fl_Value_Input* i, void* v) {
  if(v == LOAD) {
    if(!current_widget->is_valuator()) {i->deactivate(); return;}
    i->activate();
    i->value(((Fl_Valuator*)(current_widget->o))->minimum());
  } else {
    double n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->is_valuator()) {
	  ((Fl_Valuator*)(q->o))->minimum(n);
	  q->o->redraw();
	}
      }
  }
}

void max_cb(Fl_Value_Input* i, void* v) {
  if(v == LOAD) {
    if(!current_widget->is_valuator()) {i->deactivate(); return;}
    i->activate();
    i->value(((Fl_Valuator*)(current_widget->o))->maximum());
  } else {
    double n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->is_valuator()) {
	  ((Fl_Valuator*)(q->o))->maximum(n);
          q->o->redraw();
	}
      }
  }
}

void step_cb(Fl_Value_Input* i, void* v) {
  if(v == LOAD) {
    if(!current_widget->is_valuator()) {i->deactivate(); return;}
    i->activate();
    i->value(((Fl_Valuator*)(current_widget->o))->step());
  } else {
    double n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->is_valuator()) {
	  ((Fl_Valuator*)(q->o))->step(n);
	  q->o->redraw();
	}
      }
  }
}

void value_cb(Fl_Value_Input* i, void* v) {
  if(v == LOAD) {
    if(current_widget->is_valuator()) {
      i->activate();
      i->value(((Fl_Valuator*)(current_widget->o))->value());
    } else if(current_widget->is_button()) {
      i->activate();
      i->value(((Fl_Button*)(current_widget->o))->value());
    } else 
      i->deactivate();
  } else {
    double n = i->value();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->is_valuator()) {
	  ((Fl_Valuator*)(q->o))->value(n);
	} else if(q->is_button()) {
	  ((Fl_Button*)(q->o))->value(n != 0);
	}
      }
  }
}

////////////////////////////////////////////////////////////////

// subtypes:

Fl_Menu_Item* svWidgetType::subtypes() {return 0;}

void subtype_cb(Fl_Choice* i, void* v) {
  if(v == LOAD) {
    Fl_Menu_Item* m = current_widget->subtypes();
    if(!m) {i->deactivate(); return;}
    i->menu(m);
    int j;
    for(j = 0;; j++) {
      if(!m[j].text) {j = 0; break;}
      if(m[j].argument() == current_widget->o->type()) break;
    }
    i->value(j);
    i->activate();
    i->redraw();
  } else {
    int n = int(i->mvalue()->argument());
    Fl_Menu_Item* m = current_widget->subtypes();
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next)
      if(o->selected && o->is_widget()) {
	svWidgetType* q = (svWidgetType*)o;
	if(q->subtypes()==m) {
	  q->o->type(n);
	  q->redraw();
	}
      }
  }
}

////////////////////////////////////////////////////////////////

static Fl_Window* the_panel;

void
propagate_load(Fl_Group* g, void* v)
{
  if(v == LOAD) {
    Fl_Widget*const* a = g->array();
    for(int i=g->children(); i--;) {
      Fl_Widget* o = *a++;
      o->do_callback(o,LOAD);
    }
  }
}

void
set_cb(Fl_Button*, void*)
{
  haderror = 0;
  Fl_Widget*const* a = the_panel->array();
  for(int i=the_panel->children(); i--;) {
    Fl_Widget* o = *a++;
    if(o->changed()) {
      o->do_callback();
      if(haderror) return;
      o->clear_changed();
    }
  }
}

void ok_cb(Fl_Return_Button* o, void* v) {
  set_cb(o,v);
  if(!haderror) the_panel->hide();
}

void revert_cb(Fl_Button*, void*) {
  // We have to revert all dynamically changing fields:
  // but for now only the first label works...
  if(numselected == 1) current_widget->label(oldlabel);
  propagate_load(the_panel, LOAD);
}

void cancel_cb(Fl_Button* o, void* v) {
  revert_cb(o,v);
  the_panel->hide();
}

void toggle_overlays(Fl_Widget* ,void* ); // in Fl_Window_Type.C
void overlay_cb(Fl_Button*o,void* v) {
  toggle_overlays(o,v);
}

// update the panel according to current widget set:
static void
load_panel()
{
  if(!the_panel) return;

  // find all the Fl_Widget subclasses currently selected:
  numselected = 0;
  current_widget = 0;
  if(svBrowserObject::current) {
    if(svBrowserObject::current->is_widget())
      current_widget=(svWidgetType*)svBrowserObject::current;
    for(svBrowserObject* o = svBrowserObject::first; o; o = o->next) {
      if(o->is_widget() && o->selected) {
	numselected++;
	if(!current_widget) current_widget = (svWidgetType*)o;
      }
    }
  }
  if(numselected)
    propagate_load(the_panel, LOAD);
  else
    the_panel->hide();
}

// This is called when user double-clicks an item, open or update the panel:
void svWidgetType::open() {
  if(!the_panel) the_panel = make_widget_panel();
  load_panel();
  if(numselected) the_panel->show();
}

svBrowserObject* svBrowserObject::current;

extern void redraw_overlays();
extern void redraw_browser();

// Called when ui changes what objects are selected:
// p is selected object, null for all deletions (we must throw away
// old panel in that case, as the object may no longer exist)
void
selection_changed(svBrowserObject* p)
{
  // store all changes to the current selected objects:
  if(p && the_panel && the_panel->visible()) {
    set_cb(0,0);
    // if there was an error, we try to leave the selected set unchanged:
    if(haderror) {
      svBrowserObject* q = 0;
      for(svBrowserObject* o = svBrowserObject::first; o; o = o->next) {
	o->new_selected = o->selected;
	if(!q && o->selected) q = o;
      }
      if(!p || !p->selected) p = q;
      svBrowserObject::current = p;
      redraw_browser();
      return;
    }
  }
  // update the selected flags to new set:
  svBrowserObject* q = 0;
  for(svBrowserObject* o = svBrowserObject::first; o; o = o->next) {
    o->selected = o->new_selected;
    if(!q && o->selected) q = o;
  }
  if(!p || !p->selected) p = q;
  svBrowserObject::current = p;
  redraw_overlays();

  // load the panel with the new settings:
  load_panel();
}


//
// "$Id: svType.cxx,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $"
//
// Fl_Group object code for the Fast Light Tool Kit (FLTK).
//
// Object describing an Fl_Group and links to Fl_Window_Type.C and
// the Fl_Tabs widget, with special stuff to select tab items and
// insure that only one is visible.
//

#include <FL/Fl.H>
#include <FL/Fl_Group.H>
#include <FL/fl_message.H>


svBrowserObject*
svGroupType::make() {return svWidgetType::make();}

void
fix_group_size(svBrowserObject* tt)
{
  if(!tt || !tt->is_group()) return;
  svGroupType* t = reinterpret_cast<svGroupType*>(tt);
  int X = t->o->x();
  int Y = t->o->y();
  int R = X + t->o->w();
  int B = Y + t->o->h();
  for(svBrowserObject* nn = t->next; nn && nn->level > t->level; nn = nn->next) {
    if(!nn->is_widget()) continue;
    svWidgetType* n = reinterpret_cast<svWidgetType*>(nn);
    int x = n->o->x();	if(x < X) X = x;
    int y = n->o->y();	if(y < Y) Y = y;
    int r = x+n->o->w();if(r > R) R = r;
    int b = y+n->o->h();if(b > B) B = b;
  }
  t->o->resize(X,Y,R-X,B-Y);
}

extern int force_parent;

void
group_cb(Fl_Widget* , void* )
{
  // Find the current widget:
  svBrowserObject* qq = svBrowserObject::current;
  while(qq && !qq->is_widget()) qq = qq->parent;
  if(!qq || qq->level <= 1) {
    fl_message("Please select widgets to group");
    return;
  }
  svWidgetType* q = (svWidgetType*)qq;
  force_parent = 1;
  svGroupType* n = (svGroupType*)(Fl_Group_type.make());
  n->move_before(q);
  n->o->resize(q->o->x(),q->o->y(),q->o->w(),q->o->h());
  for(svBrowserObject* t = svBrowserObject::first; t;) {
    if(t->level != n->level || t == n || !t->selected) {
      t = t->next; continue;}
    svBrowserObject* nxt = t->remove();
    t->add(n);
    t = nxt;
  }
  fix_group_size(n);
}

void
ungroup_cb(Fl_Widget* , void* )
{
  // Find the group:
  svBrowserObject* q = svBrowserObject::current;
  while(q && !q->is_widget()) q = q->parent;
  if(q) q = q->parent;
  if(!q || q->level <= 1) {
    fl_message("Please select widgets in a group");
    return;
  }
  svBrowserObject* n;
  for(n = q->next; n && n->level > q->level; n = n->next) {
    if(n->level == q->level+1 && !n->selected) {
      fl_message("Please select all widgets in group");
      return;
    }
  }
  for(n = q->next; n && n->level > q->level;) {
    svBrowserObject* nxt = n->remove();
    n->insert(q);
    n = nxt;
  }
  delete q;
}

////////////////////////////////////////////////////////////////

// This is called when o is created.  If it is in the tab group make
// sure it is visible:

void
svGroupType::add_child(svBrowserObject* cc, svBrowserObject* before)
{
  svWidgetType* c = reinterpret_cast<svWidgetType*>(cc);
  Fl_Widget* b = (before ? (reinterpret_cast<svWidgetType*>(before))->o : 0);
  (reinterpret_cast<Fl_Group*>(o))->insert(*(c->o), b);
  o->redraw();
}

// This is called when o is deleted.  If it is in the tab group make
// sure it is not visible:

void
svGroupType::remove_child(svBrowserObject* cc)
{
  svWidgetType* c = (svWidgetType*)cc;
  ((Fl_Group*)o)->remove(c->o);
  o->redraw();
}

// move, don't change selected value:

void
svGroupType::move_child(svBrowserObject* cc, svBrowserObject* before)
{
  svWidgetType* c = (svWidgetType*)cc;
  Fl_Widget* b = (before ? ((svWidgetType*)before)->o : 0);
  ((Fl_Group*)o)->remove(c->o);
  ((Fl_Group*)o)->insert(*(c->o), b);
  o->redraw();
}

////////////////////////////////////////////////////////////////

svBrowserObject* in_this_only; // set if menu popped-up in window

void
select_all_cb(Fl_Widget*, void* )
{
  svBrowserObject* p = (svBrowserObject::current ? svBrowserObject::current->parent : 0);

  if(in_this_only) {
    svBrowserObject* t = p;
    for(; t && t != in_this_only; t = t->parent);
    if(t != in_this_only) p = in_this_only;
  }

  for(;;) {
    if(p) {
      int foundany = 0;
      for(svBrowserObject* t = p->next; t && t->level>p->level; t = t->next) {
	if(!t->new_selected) {
          widget_browser->select(t,1,0);
          foundany = 1;
        }
      }
      if(foundany) break;
      p = p->parent;
    } else {
      for(svBrowserObject* t = svBrowserObject::first; t; t = t->next)
	widget_browser->select(t,1,0);
      break;
    }
  }

  selection_changed(p);
}

static void
delete_children(svBrowserObject* p)
{
  svBrowserObject* f;

  for(f = p; f && f->next && f->next->level > p->level; f = f->next);

  for( ; f != p; ) {
    svBrowserObject* g = f->prev;
    delete f;
    f = g;
  }
}

void
delete_all(int selected_only)
{
  for(svBrowserObject* f = svBrowserObject::first; f; ) {
    if(f->selected || !selected_only) {
      delete_children(f);
      svBrowserObject* g = f->next;
      delete f;
      f = g;
    }
    else {
      f = f->next;
    }
  }

  selection_changed(0);
}

//
// End of "$Id: svType.cxx,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $".
//
