/* 
 * $Id: CreateGlomContour.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $
 * 
 */
#include <iostream>
// VTK Common
#include "vtkIdTypeArray.h"
#include "vtkSource.h"
#include "vtkDataSet.h"
#include "vtkPolyData.h"
#include "vtkStructuredPoints.h"
#include "vtkImageData.h"
// VTK Filtering
#include "vtkDataSetSource.h"
#include "vtkPolyDataSource.h"
#include "vtkStructuredPointsSource.h"
#include "vtkImageSource.h"
// VTK Graphics
#include "vtkContourFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataConnectivityFilter.h"
#include "vtkTriangleFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkMassProperties.h"
#include "vtkTriangleFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkGaussianSplatter.h"
// VTK Hybrid
#include "vtkImplicitModeller.h"
// VTK Parallel
#include "vtkPProbeFilter.h"
#include "vtkPPolyDataNormals.h"
// vtkExtensions
#include "vtkUpstream.h"


static void PrintSourceInfo(std::ostream&, vtkSource*, const char*);

vtkPolyDataSource*
CreateSplatter(vtkSource*, float D, float pad, float scale);


// ----------------------------------------------------------------------------
static void
InflationStart (void*)
{ std::cerr << "\nStarting surface inflation resampling." << std::endl; }

static void
InflationEnd (void*)
{ std::cerr << "\nEnding surface inflation resampling.\n" << std::endl; }

static void
InflationProgress (void* aPtr)
{
  vtkImplicitModeller* self = reinterpret_cast<vtkImplicitModeller*>(aPtr);

  std::cerr << "\tInflation Progress: ";
  std::cerr.width (10);
  std::cerr << (100 * self->GetProgress());
  std::cerr << "%\r";
}

// ----------------------------------------------------------------------------
static void
DeflationStart (void*)
{ std::cerr << "\nStarting surface deflation resampling." << std::endl; }

static void
DeflationEnd (void*)
{ std::cerr << "\nEnding surface deflation resampling.\n" << std::endl; }

static void
DeflationProgress (void* aPtr)
{
  vtkImplicitModeller* self = reinterpret_cast<vtkImplicitModeller*>(aPtr);

  std::cerr << "\tDeflation Progress: ";
  std::cerr.width (10);
  std::cerr << (100 * self->GetProgress());
  std::cerr << "%\r";
}

// ----------------------------------------------------------------------------
vtkPolyDataSource*
CreateGlomContour(vtkPolyData* aInputPolyData,
                  float aDistance,
                  float aPadding,
                  float aScale,
                  int   aIterations,
                  float aPassBand,
                  bool  aAbsoluteFlag)
{
  float origLength, origBounds[6], origCenter[3], closest[3];
  vtkIdType ctrPt = -1;

  // Bring the input up to date.
  aInputPolyData->Update();

  // Get the extents of the input data set.
  origLength = aInputPolyData->GetLength();
  aInputPolyData->GetBounds(origBounds);
  origCenter[0] = (origBounds[0] + origBounds[1]) / 2.f;
  origCenter[1] = (origBounds[2] + origBounds[3]) / 2.f;
  origCenter[2] = (origBounds[4] + origBounds[5]) / 2.f;
  // Get the point closest to the center.
  if((ctrPt = aInputPolyData->FindPoint(origCenter))==-1) ctrPt = 0;
  aInputPolyData->GetPoint(ctrPt, closest);

  PrintSourceInfo(std::cout, aInputPolyData->GetSource(), "original");

  float value;
  float dmax;

  if (aAbsoluteFlag)
    {
    value = aDistance;
    dmax  = aPadding + value;
    }
  else
    {
    value = aDistance * origLength;
    dmax  = aPadding  * origLength + value;
    }

  std::cout << "\nvalue: " << value << ", dmax: " << dmax << std::endl;

  float inflationBounds[6];
  int   inflationRange[3];

  // Get inflation extents.
  for(int i=0; i<3; i++)
    {
    inflationBounds[i*2+0] = origBounds[i*2+0] - dmax;
    inflationBounds[i*2+1] = origBounds[i*2+1] + dmax;
    inflationRange[i] =
      int(ceil((inflationBounds[i*2+1] - inflationBounds[i*2+0])*aScale));
    }

  // create implicit model of the input (original) PolyData
  vtkImplicitModeller* inflationModel = vtkImplicitModeller::New();
    {
    /* inflationModel->DebugOn(); */
    inflationModel->SetStartMethod (InflationStart,(void*)inflationModel);
    inflationModel->SetEndMethod (InflationEnd,(void*)inflationModel);
    inflationModel->SetProgressMethod (InflationProgress,(void*)inflationModel);
    inflationModel->SetInput(aInputPolyData);
    inflationModel->SetSampleDimensions(inflationRange);
    inflationModel->SetMaximumDistance(dmax);
    inflationModel->SetModelBounds(inflationBounds);
    inflationModel->SetAdjustBounds(1/*Off*/);  // default: On
    inflationModel->SetCapping(1/*On*/);        // default: On
    inflationModel->SetProcessModeToPerVoxel(); // default: per cell
    //inflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(std::cout,inflationModel,"inflation model");
    /* inflationModel->Print(std::cout); */
    }

  // Create the isocontour through the sample data at the chosen value.
  vtkContourFilter* inflatedContour = vtkContourFilter::New();
    {
    inflatedContour->SetInput(inflationModel->GetOutput());
    inflatedContour->ComputeScalarsOn();
    inflatedContour->UseScalarTreeOn();
    inflatedContour->SetValue(1, value);

    PrintSourceInfo(std::cout,inflatedContour,"inflated contour");
    /* inflatedContour->Print(std::cout); */
    }

  float deflationBounds[6];
  int   deflationRange[3];

  inflatedContour->GetOutput()->GetBounds(deflationBounds);
  if( deflationBounds[0] == VTK_LARGE_FLOAT &&
      deflationBounds[1] == -VTK_LARGE_FLOAT &&
      deflationBounds[2] == VTK_LARGE_FLOAT &&
      deflationBounds[3] == -VTK_LARGE_FLOAT &&
      deflationBounds[4] == VTK_LARGE_FLOAT &&
      deflationBounds[5] == -VTK_LARGE_FLOAT )
    {
    std::cerr << "Something is terribly wrong! Inflated Contour Empty!"
              << std::endl;
#if 1
    inflationModel->Delete();
#endif /* 0 */
    inflatedContour->Delete();

    vtkPolyDataSource* source;
    if(source = vtkPolyDataSource::SafeDownCast(aInputPolyData->GetSource()))
      {
      source->Register(NULL);
      return source;
      }
    else
      {
      return 0;
      }
    }

  for(int i=0; i<3; i++)
    {
    deflationRange[i] =
      int(ceil((deflationBounds[i*2+1] - deflationBounds[i*2+0])*aScale));
    }

  std::cout << "\ndeflation sample range: { " << deflationRange[0] << ", "
            << deflationRange[1] << ", " << deflationRange[2] << " }"
            << std::endl;

  // create implicit model of the inflated contour PolyData
  vtkImplicitModeller* deflationModel = vtkImplicitModeller::New();
    {
    /* deflationModel->DebugOn(); */
    deflationModel->SetStartMethod (DeflationStart,(void*)deflationModel);
    deflationModel->SetEndMethod (DeflationEnd,(void*)deflationModel);
    deflationModel->SetProgressMethod (DeflationProgress,(void*)deflationModel);
    deflationModel->SetInput(inflatedContour->GetOutput());
    deflationModel->SetSampleDimensions(deflationRange);
    deflationModel->SetMaximumDistance(dmax);
    deflationModel->SetModelBounds(deflationBounds);
    deflationModel->SetAdjustBounds(0/*Off*/);  // default: On
    deflationModel->SetCapping(0/*Off*/);       // default: On
    deflationModel->SetProcessModeToPerVoxel();
    //deflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(std::cout,deflationModel,"deflation model");
    /* deflationModel->Print(std::cout); */
    }

  vtkContourFilter* deflatedContour = vtkContourFilter::New();
    {
    deflatedContour->SetInput(deflationModel->GetOutput());
    deflatedContour->ComputeScalarsOn();
    deflatedContour->UseScalarTreeOn();
    deflatedContour->SetValue(0, value);

    PrintSourceInfo(std::cout,deflatedContour,"deflated contour");
    }

  vtkIdType numIds = 0;
  vtkIdTypeArray* pValidPointIds;
  
  vtkPProbeFilter* probeFilter = vtkPProbeFilter::New();
    {
    probeFilter->SetInput(deflatedContour->GetOutput());
    probeFilter->SetSource(aInputPolyData);
    probeFilter->Update();
    pValidPointIds = probeFilter->GetValidPoints();

    PrintSourceInfo(std::cout,probeFilter,"deflated probe");
    
    if(pValidPointIds)
      {
      numIds = pValidPointIds->GetNumberOfTuples();
      std::cout << "\t" << numIds << " valid probe points." << std::endl;
      }
    }

  vtkPPolyDataNormals* normals = vtkPPolyDataNormals::New();
    {
    vtkWindowedSincPolyDataFilter* smoother =
      vtkWindowedSincPolyDataFilter::New();
      {
      vtkPolyDataConnectivityFilter* connectivityFilter =
	vtkPolyDataConnectivityFilter::New();
        {
	vtkTriangleFilter* triangleFilter = vtkTriangleFilter::New();
          {
	  vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
            {
	    cleanPolyData->SetInput(deflatedContour->GetOutput());
	    cleanPolyData->ConvertLinesToPointsOn();
	    cleanPolyData->ConvertPolysToLinesOn();
	    cleanPolyData->ConvertStripsToPolysOn();
	    cleanPolyData->PointMergingOn();
	    cleanPolyData->SetTolerance(0.0);
            }
	  triangleFilter->SetInput(cleanPolyData->GetOutput());
	  triangleFilter->PassVertsOff();
	  triangleFilter->PassLinesOff();
	  cleanPolyData->Delete();
          }
	connectivityFilter->SetInput(triangleFilter->GetOutput());
	triangleFilter->Delete();
	// get the interior surface
	if(numIds>0)
          {
	  connectivityFilter->InitializeSeedList();
	  for(vtkIdType id=0; id<numIds; id++)
            {
	    connectivityFilter->AddSeed(pValidPointIds->GetValue(id));
            }
	  connectivityFilter->SetExtractionModeToPointSeededRegions();
          }
	else
          {
	  connectivityFilter->SetExtractionModeToClosestPointRegion();
	  connectivityFilter->SetClosestPoint(closest);
          }
      
	PrintSourceInfo(std::cout,connectivityFilter,"deflated filtered");
        }
      smoother->SetInput(connectivityFilter->GetOutput());
      smoother->SetNumberOfIterations (aIterations);
      smoother->SetPassBand (aPassBand);
      //smoother->FeatureEdgeSmoothingOn();
      connectivityFilter->Delete();
    
      PrintSourceInfo(std::cout,smoother,"deflated smoothed");
      }
    normals->SetInput(smoother->GetOutput());
    normals->ConsistencyOn();
    normals->ComputePointNormalsOn();
    normals->ComputeCellNormalsOff();
    smoother->Delete();

    PrintSourceInfo(std::cout,normals,"deflated normals");
    }

  probeFilter->Delete();
#if 1
  inflationModel->Delete();
#endif /* 0 */
  deflationModel->Delete();
  inflatedContour->Delete();
  deflatedContour->Delete();

  return normals;
}

// ----------------------------------------------------------------------------
static void
PrintSourceInfo(std::ostream& os, vtkSource* aSource, const char* name)
{
#if 1
  vtkDataSet* dataSet;

  aSource->Update();

  if(vtkPolyDataSource::SafeDownCast(aSource))
    {
    dataSet = (vtkPolyDataSource::SafeDownCast(aSource))->GetOutput();
    }
  else if(vtkStructuredPointsSource::SafeDownCast(aSource))
    {
    dataSet = (vtkStructuredPointsSource::SafeDownCast(aSource))->GetOutput();
    }
  else if(vtkDataSetSource::SafeDownCast(aSource))
    {
    dataSet = (vtkDataSetSource::SafeDownCast(aSource))->GetOutput();
    }
  else if(vtkImageSource::SafeDownCast(aSource))
    {
    dataSet = (vtkImageSource::SafeDownCast(aSource))->GetOutput();
    }
  else
    {
    return;
    }

  float bounds[6];
  dataSet->GetBounds(bounds);

  os << name << " bounds = {" << bounds[0] << "," << bounds[1]
     << "," << bounds[2] << "," << bounds[3]
     << "," << bounds[4] << "," << bounds[5] << "}, "
     << "length = " << dataSet->GetLength() << std::endl;

  os << "\t" << dataSet->GetNumberOfPoints() << " points, "
     << dataSet->GetNumberOfCells() << " cells" << std::endl;

  if(dataSet->GetDataObjectType() == VTK_POLY_DATA)
    {
    vtkPolyData* polyData;

    if(polyData = vtkPolyData::SafeDownCast(dataSet))
      {
      polyData->Update();

      vtkMassProperties* massProperties = vtkMassProperties::New();
        {
	vtkTriangleFilter* triangleFilter = vtkTriangleFilter::New();
          {
	  vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
            {
	    cleanPolyData->SetInput(polyData);
	    cleanPolyData->ConvertLinesToPointsOn();
	    cleanPolyData->ConvertPolysToLinesOn();
	    cleanPolyData->ConvertStripsToPolysOn();
	    cleanPolyData->PointMergingOn();
	    cleanPolyData->SetTolerance(0.0);
            }
	  triangleFilter->SetInput(cleanPolyData->GetOutput());
	  triangleFilter->PassVertsOff();
	  triangleFilter->PassLinesOff();
	  cleanPolyData->Delete();
          }
	massProperties->SetInput(triangleFilter->GetOutput());
	triangleFilter->Delete();
	
	os << "\tvolume: " << massProperties->GetVolume() << ", surface area: "
	   << massProperties->GetSurfaceArea() << ", normalized shape index: "
	   << massProperties->GetNormalizedShapeIndex() << std::endl;
        }
      massProperties->Delete();
      }
    }
#endif /* 0 */
}

// ----------------------------------------------------------------------------
vtkPolyDataSource*
CreateSplatter(vtkSource* aSource,
               float aDistance,
               float aPadding,
               float aScale)
{
  float origLength, origBounds[6], origCenter[3], closest[3];
  vtkIdType ctrPt = -1;

  vtkUpstream* upstream = vtkUpstream::New();
  upstream->SetSource(aSource);

  // Get the input DataSet.
  vtkDataSet* dataSet = upstream->GetOutputDataSet();

  // Get the extents of the input data set.
  origLength = dataSet->GetLength();
  dataSet->GetBounds(origBounds);
  origCenter[0] = (origBounds[0] + origBounds[1]) / 2.f;
  origCenter[1] = (origBounds[2] + origBounds[3]) / 2.f;
  origCenter[2] = (origBounds[4] + origBounds[5]) / 2.f;
  // Get the point closest to the center.
  if((ctrPt = dataSet->FindPoint(origCenter))==-1) ctrPt = 0;
  dataSet->GetPoint(ctrPt, closest);

  float value = aDistance *   origLength;
  float dmax  = aPadding * origLength + value;

  float splatterBounds[6];
  int   splatterRange[3];

  // Get splatter extents.
  for(int i=0; i<3; i++) {
    splatterBounds[i*2+0] = origBounds[i*2+0] - dmax;
    splatterBounds[i*2+1] = origBounds[i*2+1] + dmax;
    splatterRange[i] =
      int(ceil((splatterBounds[i*2+1] - splatterBounds[i*2+0])*aScale));
  }

  vtkContourFilter* splatterContour = vtkContourFilter::New();
  {
    vtkGaussianSplatter* gaussianSplatter = vtkGaussianSplatter::New();
    {
      gaussianSplatter->SetInput(dataSet);
      gaussianSplatter->SetSampleDimensions(splatterRange);
      gaussianSplatter->SetRadius(aDistance);
      gaussianSplatter->SetEccentricity(0.333);
      gaussianSplatter->NormalWarpingOn();
      gaussianSplatter->ScalarWarpingOff();
      gaussianSplatter->CappingOn();
    }
    splatterContour->SetInput(gaussianSplatter->GetOutput());
    splatterContour->SetValue(0,value);
    gaussianSplatter->Delete();
  }

  // clean up
  upstream->Delete();

  return splatterContour;
}


/* 
 * End of: $Id: CreateGlomContour.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $.
 * 
 */
