/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: CreateGlomContour.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <iostream>
// VTK Common
#include "vtkIdTypeArray.h"
#include "vtkSource.h"
#include "vtkDataSet.h"
#include "vtkPolyData.h"
#include "vtkStructuredPoints.h"
#include "vtkImageData.h"
#include "vtkDataSetAttributes.h"
// VTK Filtering
#include "vtkDataSetSource.h"
#include "vtkPolyDataSource.h"
#include "vtkStructuredPointsSource.h"
#include "vtkImageSource.h"
// VTK Graphics
#include "vtkContourFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataConnectivityFilter.h"
#include "vtkTriangleFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkMassProperties.h"
#include "vtkProbeFilter.h"
#include "vtkPolyDataNormals.h"
// VTK Hybrid
#include "vtkImplicitModeller.h"


static void PrintSourceInfo(std::ostream&, vtkSource*, const char*);


// ----------------------------------------------------------------------------
vtkPolyDataSource*
CreateGlomContour (vtkPolyData* aPolyData, float D, float pad, float scale)
{
  float origLength, origBounds[6], origCenter[3], closest[3];
  vtkIdType ctrPt = -1;

  // Bring the input up to date.
  aPolyData->Update();
  // Get the extents of the input data set.
  origLength = aPolyData->GetLength();
  aPolyData->GetBounds(origBounds);
  origCenter[0] = (origBounds[0] + origBounds[1]) / 2.f;
  origCenter[1] = (origBounds[2] + origBounds[3]) / 2.f;
  origCenter[2] = (origBounds[4] + origBounds[5]) / 2.f;
  // Get the point closest to the center.
  if ((ctrPt = aPolyData->FindPoint(origCenter))==-1) ctrPt = 0;
  aPolyData->GetPoint(ctrPt, closest);

  PrintSourceInfo(std::cout, aPolyData->GetSource(), "original");

  float value = D *   origLength;
  float dmax  = pad * origLength + value;

  float inflationBounds[6];
  int   inflationRange[3];

  // Get inflation extents. 
  for (int i=0; i<3; i++)
    {
    inflationBounds[i*2+0] = origBounds[i*2+0] - dmax;
    inflationBounds[i*2+1] = origBounds[i*2+1] + dmax;
    inflationRange[i] =
      int(ceil((inflationBounds[i*2+1] - inflationBounds[i*2+0])*scale));
    }

  // create implicit model of the input (original) PolyData
  vtkImplicitModeller* inflationModel = vtkImplicitModeller::New();
    {
    inflationModel->DebugOn();
    inflationModel->SetInput(aPolyData);
    inflationModel->SetSampleDimensions(inflationRange);
    inflationModel->SetMaximumDistance(dmax);
    inflationModel->SetModelBounds(inflationBounds);
    inflationModel->SetAdjustBounds(1/*Off*/);  // default: On
    inflationModel->SetCapping(1/*On*/);        // default: On
    inflationModel->SetProcessModeToPerVoxel(); // default: per cell
    //inflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(std::cout,inflationModel,"inflation model");
    inflationModel->Print(std::cout);
    }

  // Create the isocontour through the sample data at the chosen value.
  vtkContourFilter* inflatedContour = vtkContourFilter::New();
    {
    inflatedContour->SetInput(inflationModel->GetOutput());
    inflatedContour->ComputeScalarsOn();
    inflatedContour->UseScalarTreeOn();
    inflatedContour->SetValue(1, value);
    
    PrintSourceInfo(std::cout,inflatedContour,"inflated contour");
    inflatedContour->Print(std::cout);
    }

  float deflationBounds[6];
  int   deflationRange[3];

  inflatedContour->GetOutput()->GetBounds(deflationBounds);

  if ( deflationBounds[0] == VTK_LARGE_FLOAT &&
       deflationBounds[1] == -VTK_LARGE_FLOAT &&
       deflationBounds[2] == VTK_LARGE_FLOAT &&
       deflationBounds[3] == -VTK_LARGE_FLOAT &&
       deflationBounds[4] == VTK_LARGE_FLOAT &&
       deflationBounds[5] == -VTK_LARGE_FLOAT )
    {
    std::cerr << "Something is terribly wrong! Inflated Contour Empty!"
              << std::endl;
#if 1
    inflationModel->Delete();
#endif /* 0 */
    inflatedContour->Delete();

    vtkPolyDataSource* source;

    if ((source=vtkPolyDataSource::SafeDownCast(aPolyData->GetSource()))!=NULL)
      {
      source->Register(NULL);
      return source;
      }
    else
      {
      return NULL;
      }
    }

  for (int i=0; i<3; i++)
    {
    deflationRange[i] =
      int(ceil((deflationBounds[i*2+1] - deflationBounds[i*2+0])*scale));
    }

  std::cout << "deflation sample range: { " << deflationRange[0] << ", "
            << deflationRange[1] << ", " << deflationRange[2] << " }"
            << std::endl;

  // create implicit model of the inflated contour PolyData
  vtkImplicitModeller* deflationModel = vtkImplicitModeller::New();
    {
    deflationModel->DebugOn();
    deflationModel->SetInput(inflatedContour->GetOutput());
    deflationModel->SetSampleDimensions(deflationRange);
    deflationModel->SetMaximumDistance(dmax);
    deflationModel->SetModelBounds(deflationBounds);
    deflationModel->SetAdjustBounds(0/*Off*/);  // default: On
    deflationModel->SetCapping(0/*Off*/);       // default: On
    deflationModel->SetProcessModeToPerVoxel();
    //deflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(std::cout,deflationModel,"deflation model");
    deflationModel->Print(std::cout);
    }

  vtkContourFilter* deflatedContour = vtkContourFilter::New();
    {
    deflatedContour->SetInput(deflationModel->GetOutput());
    deflatedContour->ComputeScalarsOn();
    deflatedContour->UseScalarTreeOn();
    deflatedContour->SetValue(0, value);

    PrintSourceInfo(std::cout,deflatedContour,"deflated contour");
    }

  vtkIdType numIds = 0;
  vtkIdTypeArray* validPointIds;
  
  vtkProbeFilter* probeFilter = vtkProbeFilter::New();
    {
    probeFilter->SetInput(deflatedContour->GetOutput());
    probeFilter->SetSource(aPolyData);
    probeFilter->Update();
    validPointIds = probeFilter->GetValidPoints();

    PrintSourceInfo(std::cout,probeFilter,"deflated probe");

    if (validPointIds != NULL)
      {
      numIds = validPointIds->GetNumberOfTuples();
      std::cout << "\t" << numIds << " valid probe points." << std::endl;
      }
    }

  vtkPolyDataNormals* normalsFilter = vtkPolyDataNormals::New();
    {
    vtkWindowedSincPolyDataFilter* smootherFilter =
      vtkWindowedSincPolyDataFilter::New();
      {
      vtkPolyDataConnectivityFilter* connectivityFilter =
        vtkPolyDataConnectivityFilter::New();
        {
        vtkTriangleFilter* triangleFilter = vtkTriangleFilter::New();
          {
          vtkCleanPolyData* cleanerFilter = vtkCleanPolyData::New();
            {
            cleanerFilter->SetInput(deflatedContour->GetOutput());
            cleanerFilter->ConvertLinesToPointsOn();
            cleanerFilter->ConvertPolysToLinesOn();
            cleanerFilter->ConvertStripsToPolysOn();
            cleanerFilter->PointMergingOn();
            //cleanerFilter->SetTolerance(0.0);
            }
          triangleFilter->SetInput(cleanerFilter->GetOutput());
          triangleFilter->PassVertsOff();
          triangleFilter->PassLinesOff();
          cleanerFilter->Delete();
          }
        connectivityFilter->SetInput(triangleFilter->GetOutput());
        triangleFilter->Delete();
        // get the interior surface
        if (numIds>0)
          {
          connectivityFilter->InitializeSeedList();
          for (vtkIdType id=0; id<numIds; id++)
            {
            connectivityFilter->AddSeed(validPointIds->GetValue(id));
            }
          connectivityFilter->SetExtractionModeToPointSeededRegions();
          }
        else
          {
          connectivityFilter->SetExtractionModeToClosestPointRegion();
          connectivityFilter->SetClosestPoint(closest);
          }

        PrintSourceInfo(std::cout,connectivityFilter,"deflated filtered");
        }
      smootherFilter->SetInput(connectivityFilter->GetOutput());
      //smootherFilter->FeatureEdgeSmoothingOn();
      connectivityFilter->Delete();
    
      PrintSourceInfo(std::cout,smootherFilter,"deflated smoothed");
      }
    normalsFilter->SetInput(smootherFilter->GetOutput());
    normalsFilter->ConsistencyOn();
    normalsFilter->ComputePointNormalsOn();
    normalsFilter->ComputeCellNormalsOff();
    smootherFilter->Delete();

    PrintSourceInfo(std::cout,normalsFilter,"deflated normals");
    }

  probeFilter->Delete();
#if 1
  inflationModel->Delete();
#endif /* 0 */
  deflationModel->Delete();
  inflatedContour->Delete();
  deflatedContour->Delete();

  return normalsFilter;
}

// ----------------------------------------------------------------------------
static void
PrintSourceInfo(std::ostream& aTarget, vtkSource* aSource, const char* aName)
{
  if (aSource == NULL || aName == NULL)
    return;

#if 1
  vtkDataSet* dataSet;

  aSource->Update();

  if      (aSource->IsA("vtkPolyDataSource"))
    {
    dataSet = (static_cast<vtkPolyDataSource*>(aSource))->GetOutput();
    }
  else if (aSource->IsA("vtkStructuredPointsSource"))
    {
    dataSet = (static_cast<vtkStructuredPointsSource*>(aSource))->GetOutput();
    }
  else if (aSource->IsA("vtkDataSetSource"))
    {
    dataSet = (static_cast<vtkDataSetSource*>(aSource))->GetOutput();
    }
  else if (aSource->IsA("vtkImageSource"))
    {
    dataSet = (static_cast<vtkImageSource*>(aSource))->GetOutput();
    }
  else
    {
    return;
    }

  float bounds[6];

  dataSet->GetBounds(bounds);

  aTarget << aName
          << " bounds = {" << bounds[0] << "," << bounds[1]
          << "," << bounds[2] << "," << bounds[3]
          << "," << bounds[4] << "," << bounds[5] << "}, "
          << "length = " << dataSet->GetLength() << std::endl;

  aTarget << "\t" << dataSet->GetNumberOfPoints() << " points, "
          << dataSet->GetNumberOfCells() << " cells" << std::endl;

  if (dataSet->GetDataObjectType() == VTK_POLY_DATA)
    {
    vtkPolyData* polyData;

    if ((polyData = vtkPolyData::SafeDownCast(dataSet)) != NULL)
      {
      polyData->Update();

      vtkMassProperties* massProperties = vtkMassProperties::New();
        {
	vtkTriangleFilter* triangleFilter = vtkTriangleFilter::New();
          {
	  vtkCleanPolyData* cleanerFilter = vtkCleanPolyData::New();
            {
	    cleanerFilter->SetInput(polyData);
	    cleanerFilter->ConvertLinesToPointsOn();
	    cleanerFilter->ConvertPolysToLinesOn();
	    cleanerFilter->ConvertStripsToPolysOn();
	    cleanerFilter->PointMergingOn();
	    cleanerFilter->SetTolerance(0.0);
            }
          triangleFilter->SetInput(cleanerFilter->GetOutput());
          triangleFilter->PassVertsOff();
          triangleFilter->PassLinesOff();
          cleanerFilter->Delete();
          }
        massProperties->SetInput(triangleFilter->GetOutput());
        triangleFilter->Delete();
	
        aTarget << "\tvolume: " << massProperties->GetVolume()
                << ", surface area: " << massProperties->GetSurfaceArea()
                << ", normalized shape index: "
                << massProperties->GetNormalizedShapeIndex() << std::endl;
        }
      massProperties->Delete();
      }
    }
#endif /* 0 */
}

/* 
 * End of: $Id: CreateGlomContour.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $.
 * 
 */
