/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Warper0.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
// VTK Common
#include "vtkMath.h"
#include "vtkTransform.h"
#include "vtkPolyData.h"
// VTK Graphics
#include "vtkPolyDataNormals.h"
#include "vtkGlyphSource2D.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkGlyph3D.h"
// VTK Parallel
#include "vtkInputPort.h"
// VTK Hybrid
#include "vtkCubeAxesActor2D.h"
// VTK Rendering
#include "vtkProperty.h"
#include "vtkAssembly.h"
#include "vtkActorCollection.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkPolyDataMapper.h"
// VTK IO
#include "vtkPolyDataWriter.h"
// SurfaceWarp
#include "SurfaceWarp.H"
// SVV
#include "svvController.h"

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);

// ----------------------------------------------------------------------------
static void
CreateAxes (vtkRenderer*);

static vtkPolyDataSource*
CreateArrowGlyphSource (vtkPolyData*, float, float = 0.5);

// ----------------------------------------------------------------------------
void
Warper0 (vtkMultiProcessController* aController, void* userData)
{
  if (aController == NULL || userData == NULL)
    return;

  StringListRef    inputArgs    = *(reinterpret_cast<StringListPtr>(userData));
  vtkCommunicator* communicator = aController->GetCommunicator();

  // 
  // Create the RenderWindow, Renderer and both Actors
  // 
  vtkRenderer*     renderer     = vtkRenderer::New();
  vtkRenderWindow* renderWindow = vtkRenderWindow::New();

  renderWindow->AddRenderer(renderer);
  renderWindow->LineSmoothingOn();
  renderWindow->SetWindowName("SurfaceWarp");
  renderWindow->SetSize(WINDOW_WIDTH, WINDOW_HEIGHT);
  //renderWindow->LineSmoothingOn();
  renderer->SetBackground(0.20, 0.15, 0.1);

  vtkRenderWindowInteractor* interactor = vtkRenderWindowInteractor::New();

  interactor->SetRenderWindow(renderWindow);
  svv.SetInteractor(interactor);

  // 
  // Original data input port.
  // 
  vtkInputPort* originalInputPort = vtkInputPort::New();
    {
    originalInputPort->SetController(aController);
    originalInputPort->SetRemoteProcessId(originalPortId);
    originalInputPort->SetTag(originalPortTag);
    }

  // 
  // Use OpenGL polygon offsetting ...
  // 
  vtkMapper::SetResolveCoincidentTopologyToPolygonOffset();

  // 
  // The original data is behind door number 1 ...
  // 
  vtkActor* originalActor = vtkActor::New();
    {
    vtkPolyDataMapper* originalMapper = vtkPolyDataMapper::New();
      {
      originalMapper->SetInput(originalInputPort->GetPolyDataOutput());
      originalMapper->ScalarVisibilityOff();
      }
    vtkProperty* originalProperty = svv.MakeProperty("gloss dark green");

    originalActor->SetProperty(originalProperty);
    originalActor->SetMapper(originalMapper);
    originalProperty->Delete();
    originalMapper->Delete();
    }
  renderer->AddActor(originalActor);
  originalActor->Delete();

  renderer->ResetCamera();
  renderWindow->Render();

  float scalingFactor;

  // Blocks until receiving the scaling factor (input data has been read).
  if(!communicator->Receive(&scalingFactor,1,deflatedPortId,scalingFactorTag))
    {
    vtkGenericWarningMacro(<< "Server: Error receiving scaling factor.");
    return;
    }

  // 
  // Inflate/Deflate input ports.
  // 
  vtkInputPort* inflatedInputPort = vtkInputPort::New();
    {
    inflatedInputPort->SetController(aController);
    inflatedInputPort->SetRemoteProcessId(inflatedPortId);
    inflatedInputPort->SetTag(inflatedPortTag);
    }
  vtkInputPort* deflatedInputPort = vtkInputPort::New();
    {
    deflatedInputPort->SetController(aController);
    deflatedInputPort->SetRemoteProcessId(deflatedPortId);
    deflatedInputPort->SetTag(deflatedPortTag);
    }

  // 
  // The inflated surface is behind door number 2 ...
  // ... the inflation vectors should also be behind door number 2 ...
  // 
  vtkActor* inflatedActor = vtkActor::New();
    {
    vtkPolyDataMapper* inflatedMapper = vtkPolyDataMapper::New();
      {
      inflatedMapper->SetInput(inflatedInputPort->GetPolyDataOutput());
      inflatedMapper->ScalarVisibilityOn();
      }
    inflatedMapper->ScalarVisibilityOn();
    //inflatedActor->GetProperty()->SetRepresentationToWireframe();
    inflatedActor->GetProperty()->SetOpacity(0.5);
    inflatedActor->SetMapper(inflatedMapper);
    inflatedMapper->Delete();
    }
  renderer->AddActor(inflatedActor);
  inflatedActor->Delete();

  vtkActor* inflatedVectorsActor = vtkActor::New();
    {
    vtkPolyDataMapper* inflatedVectorsMapper = vtkPolyDataMapper::New();
      {
      vtkPolyDataSource* inflatedVectorsGlyph =
        CreateArrowGlyphSource( inflatedInputPort->GetPolyDataOutput(),
                                scalingFactor );
      inflatedVectorsMapper->SetInput(inflatedVectorsGlyph->GetOutput());
      inflatedVectorsGlyph->Delete();
      inflatedVectorsMapper->ScalarVisibilityOff();
      }
    inflatedVectorsActor->SetMapper(inflatedVectorsMapper);
    //inflatedVectorsActor->GetProperty()->SetLineWidth(2.0);
    inflatedVectorsMapper->Delete();
    }
  renderer->AddActor(inflatedVectorsActor);
  inflatedVectorsActor->Delete();

  // 
  // The deflated surface is behind door number 3 ...
  // ... the deflation vectors should also be behind door number 3 ...
  // 
  vtkActor* deflatedActor = vtkActor::New();
    {
    vtkPolyDataMapper* deflatedMapper = vtkPolyDataMapper::New();
      {
      deflatedMapper->SetInput(deflatedInputPort->GetPolyDataOutput());
      deflatedMapper->ScalarVisibilityOff();
      }
    vtkProperty* deflatedProperty = svv.MakeProperty("gloss yellow");

    deflatedProperty->SetRepresentationToWireframe();
    //deflatedProperty->SetOpacity(0.5);
    deflatedActor->SetProperty(deflatedProperty);
    deflatedActor->SetMapper(deflatedMapper);
    deflatedProperty->Delete();
    deflatedMapper->Delete();
    }
  renderer->AddActor(deflatedActor);
  deflatedActor->Delete();

  vtkActor* deflatedVectorsActor = vtkActor::New();
    {
    vtkPolyDataMapper* deflatedVectorsMapper = vtkPolyDataMapper::New();
      {
      vtkPolyDataSource* deflatedVectorsGlyph =
        CreateArrowGlyphSource( deflatedInputPort->GetPolyDataOutput(),
                                scalingFactor );
      deflatedVectorsMapper->SetInput(deflatedVectorsGlyph->GetOutput());
      deflatedVectorsGlyph->Delete();
      deflatedVectorsMapper->ScalarVisibilityOn();
      }
    deflatedVectorsActor->SetMapper(deflatedVectorsMapper);
    //deflatedVectorsActor->GetProperty()->SetLineWidth(2.0);
    deflatedVectorsMapper->Delete();
    }
  renderer->AddActor(deflatedVectorsActor);
  deflatedVectorsActor->Delete();

  renderer->ResetCamera();
  renderWindow->Render();

  // // Now update the Axes ...
  CreateAxes(renderer);

  // Let's go
  renderWindow->Render();
  interactor->Initialize();
  interactor->Start();

  // Maybe write out the glommed data.
  StringListConstIter s;
  for (s=inputArgs.begin(); s!=inputArgs.end(); s++)
    {
    if ((*s) == "-o")
      {
      vtkPolyDataWriter* writer = vtkPolyDataWriter::New();
        {
	writer->SetFileName((*(++s)).c_str());
	writer->SetFileTypeToASCII();
	//writer->SetFileTypeToBinary();
	writer->SetInput(deflatedInputPort->GetPolyDataOutput());
	writer->Write();
      }
      writer->Delete();
      }
    }

  // Tell the other process we are done
  originalInputPort->GetController()->
    TriggerRMI(originalPortId, vtkMultiProcessController::BREAK_RMI_TAG);
  inflatedInputPort->GetController()->
    TriggerRMI(inflatedPortId, vtkMultiProcessController::BREAK_RMI_TAG); 
  deflatedInputPort->GetController()->
    TriggerRMI(deflatedPortId, vtkMultiProcessController::BREAK_RMI_TAG); 

  // Cleanup
  originalInputPort->Delete();
  inflatedInputPort->Delete();
  deflatedInputPort->Delete();
  interactor->Delete();
  renderWindow->Delete();
  renderer->Delete();
}

// ----------------------------------------------------------------------------
static void
CreateAxes (vtkRenderer* aRenderer)
{
  if (aRenderer == NULL)
    return;

  vtkActor* actor;
  vtkActorCollection* actorCollection = aRenderer->GetActors();

  float tmp[6], bounds[6] = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

  int j, j0, j1;

  
  for ( actorCollection->InitTraversal();
        (actor = actorCollection->GetNextActor()) != NULL; )
    {
    actor->GetBounds(tmp);

    for (j0=0,j1=1; j0<6; j0+=2,j1+=2)
      {
      if(tmp[j0] < bounds[j0]) bounds[j0] = tmp[j0];
      if(tmp[j1] > bounds[j1]) bounds[j1] = tmp[j1];
      }
    }

  float center[3], range[3];

  for (j=0,j0=0,j1=1; j<3; j++,j0+=2,j1+=2)
    {
    center[j] = (bounds[j0] + bounds[j1]) / 2.f;
    range[j]  = (bounds[j1] - bounds[j0]);
    }

  float length = vtkMath::Norm(range);

  vtkCubeAxesActor2D* cubeAxesActor2D = vtkCubeAxesActor2D::New();
    {
    cubeAxesActor2D->SetBounds(bounds);
    cubeAxesActor2D->SetCamera(aRenderer->GetActiveCamera());
    cubeAxesActor2D->SetFlyModeToOuterEdges();
    cubeAxesActor2D->SetScaling(0/*Off*/);
    cubeAxesActor2D->SetShadow(1/*On*/);
    cubeAxesActor2D->SetInertia(2);
    }
  aRenderer->AddProp(cubeAxesActor2D);
  cubeAxesActor2D->Delete();

  aRenderer->GetRenderWindow()->Render();
}

// ----------------------------------------------------------------------------
static vtkPolyDataSource*
CreateArrowGlyphSource (vtkPolyData* aPolyData, float aFactor, float aScale)
{
  vtkGlyph3D* glyph3D = vtkGlyph3D::New();
    {
    vtkTransformPolyDataFilter* transformFilter =
      vtkTransformPolyDataFilter::New();
      {
      vtkGlyphSource2D* arrowSource = vtkGlyphSource2D::New();
        {
        //arrowSource->SetGlyphTypeToArrow(); // becomes "thick" when filled
        arrowSource->SetGlyphTypeToThickArrow();
        //arrowSource->SetGlyphTypeToHookedArrow();
        arrowSource->SetScale(aScale);
        arrowSource->FilledOff();
        arrowSource->SetCenter(-0.5*aScale, 0.0, 0.0);
        }
      vtkTransform* transform = vtkTransform::New();
        {
        transform->Scale(1.0, 0.25, 1.0);
        }
      transformFilter->SetInput(arrowSource->GetOutput());
      transformFilter->SetTransform(transform);
      arrowSource->Delete();
      transform->Delete();
      }
    glyph3D->SetInput(aPolyData);
    // Source to use for he glyph (see documentation re: multiple glyphs)
    glyph3D->SetSource(transformFilter->GetOutput());
    // Toggle scaling of source geometry. 
    glyph3D->ScalingOn();
    // Scale by scalar or by vector/normal magnitude. 
    //glyph3D->SetScaleModeToScaleByScalar(); // default
    glyph3D->SetScaleModeToScaleByVector();
    //glyph3D->SetScaleModeToScaleByVectorComponents();
    //glyph3D->SetScaleModeToDataScalingOff();
    // Color by scale, scalar or by vector/normal magnitude. 
    //glyph3D->SetColorModeToColorByScale();
    glyph3D->SetColorModeToColorByScalar(); // default
    //glyph3D->SetColorModeToColorByVector();
    // Scale factor to scale object by.
    glyph3D->SetScaleFactor(aFactor);
    // Toggle orienting of input geometry along vector/normal. 
    glyph3D->OrientOn();
    // Use vector or normal to perform vector operations. 
    glyph3D->SetVectorModeToUseVector(); // default
    //glyph3D->SetVectorModeToUseNormal();
    //glyph3D->SetVectorModeToVectorRotationOff();
    // Clean up ...
    transformFilter->Delete();
    }

  return glyph3D;
}

/* 
 * End of: $Id: Warper0.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $.
 * 
 */
