/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Warper1.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <cstdlib>
// VTK Common
#include "vtkMath.h"
#include "vtkPolyData.h"
// VTK Filtering
#include "vtkPolyDataSource.h"
// VTK Graphics
#include "vtkAppendPolyData.h"
// VTK Parallel
#include "vtkOutputPort.h"
// Warper
#include "SurfaceWarp.H"
// vtkExtensions
#include "vtkInputSource.h"
// SVV
#include "svvController.h"

VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);
SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);

static vtkPolyDataSource* ParseArgs1(vtkMultiProcessController*, void*);


// ----------------------------------------------------------------------------
void
Warper1(vtkMultiProcessController* pController, void* userData)
{
  // Create the AppendFilter and add the inputs to it.
  vtkPolyDataSource* pPolyDataSource = ParseArgs1(pController,userData);

  // Output port
  vtkOutputPort* originalOutputPort = vtkOutputPort::New();
  {
    // Output port.
    originalOutputPort->SetInput(pPolyDataSource->GetOutput());
    // Access to the global controller.
    originalOutputPort->SetController(pController);
    // Identify this output port of this process.
    originalOutputPort->SetTag(originalPortTag);
    // Turn this on for pipeline parallelism.
    originalOutputPort->PipelineFlagOn();
    // Process requests ...
    originalOutputPort->WaitForUpdate();
  }

  // Clean up ... will happen only after BREAK_RMI_TAG.
  originalOutputPort->Delete();
  pPolyDataSource->Delete();
}

// ----------------------------------------------------------------------------
static vtkPolyDataSource*
ParseArgs1(vtkMultiProcessController* pController, void* userData)
{
  StringListRef    inputArgs = *(reinterpret_cast<StringListPtr>(userData));
  vtkCommunicator* pComm     = pController->GetCommunicator();

  float scalingFraction = 0.0625;

  // Create the AppendFilter and add the inputs to it.
  vtkAppendPolyData* pAppender = vtkAppendPolyData::New();
    {
    if (inputArgs.empty())
      {
      vtkGenericWarningMacro(<< "ParseArgs1(): empty input arg list!");
      }
    else
      {
      vtkInputSource* inputSource;
      vtkPolyData* polyData;
      StringListConstIter s;

      for (s=inputArgs.begin(); s!=inputArgs.end(); s++)
        {
        // Input data files.
        if ((*s) == "-i")
          {
          for (s++; s!=inputArgs.end() && (*s).substr(0,1)!="-"; s++)
            {
            if (inputSource = svv.MakeSurfaceDataSource((*s).c_str()))
              {
              if (polyData = inputSource->GetPolyDataOutput())
                {
                pAppender->AddInput(polyData);
                }
              }
            else
              {
              vtkGenericWarningMacro(<< "ParseArgs1(): failed to make"
                                     << " surface data source from \""
                                     << (*s).c_str() << "\"!");
              }
            }
          }

        // Scaling fraction (scaling factor is fraction x bounding box length).
        else if ((*s) == "-D")
          {
          //s++;
          if ((*(++s)).substr(0,1)!="-")
            {
            scalingFraction = float(atof((*s).c_str()));
            }
          else
            {
            vtkGenericWarningMacro(<< "ParseArgs1(): -D flag requires"
                                   << " an argument.");
            }
          }

        } // for each argument ...

      // Make sure that there is at least one valid input.
      if (pAppender->GetNumberOfInputs() < 1)
        {
	vtkGenericWarningMacro(<< "ParseArgs1(): received no valid inputs!");
        }
      }
    }

  float originalBounds[6], originalRange[3], originalCenter[3];
  float originalLength, scalingFactor;
  {
    int j, j0, j1;
    
    pAppender->UpdateWholeExtent();
    pAppender->GetOutput()->GetBounds(originalBounds);
    
    for (j=0,j0=0,j1=1; j<3; j++,j0+=2,j1+=2)
      {
      originalRange[j]  = originalBounds[j1] - originalBounds[j0];
      originalCenter[j] = originalBounds[j0] + (originalRange[j] / 2.f);
      }
    
    originalLength = vtkMath::Norm(originalRange);
    scalingFactor  = scalingFraction * originalLength;
  }
  
  // Send the scaling factor to the inflater ...
  if (!pComm->Send(&scalingFactor,1,inflatedPortId,scalingFactorTag))
    {
    vtkGenericWarningMacro(<< "Server: Error sending scaling factor.");
    }

  return pAppender;
}

/* 
 * End of: $Id: Warper1.cxx,v 1.1.1.1 2006/12/19 22:59:40 christianh Exp $.
 * 
 */
