/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: VoxelToContourCb.cxx,v 1.1.1.1 2006/12/19 22:59:42 christianh Exp $
 * 
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <sys/types.h>
#include <sys/stat.h>
// IOStreams
#include <iostream>
#include <fstream>
// STL
#include <string>
// FLTK
#include <FL/Fl_File_Chooser.H> /* for fl_file_chooser() */
// VTK Common
#include "vtkInstantiator.h"
#include "vtkMath.h"
#include "vtkMatrix4x4.h"
#include "vtkMatrixToLinearTransform.h"
#include "vtkPolyData.h"
#include "vtkImageData.h"
// VTK Imaging
#include "vtkImageThreshold.h"
#include "vtkImageResample.h"
#include "vtkImageDilateErode3D.h"
// VTK Graphics
#include "vtkContourFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkTriangleFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataNormals.h"
#include "vtkCurvatures.h"
#include "vtkMassProperties.h"
// VTK Rendering
#include "vtkPolyDataMapper.h"
// VTK IO
#include "vtkImageReader.h"
#include "vtkDataSetWriter.h"
#include "vtkXMLDataSetWriter.h"
#include "vtkZLibDataCompressor.h"
// VTK Rendering
#include "vtkActor.h"
// SVV
#include "VoxelToContourUI.h"

// ----------------------------------------------------------------------------
#if 0
void
ImageResampleToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkImageResample* imageResample;
  vtkImageThreshold* imageThreshold;
  vtkContourFilter* contourFilter;

  if (!(imageResample = GUI->GetImageResample()))
    {
    Fl::error("ImageResampleToggleCb failed to get ImageResample.\n");
    return;
    }
  if (!(imageThreshold = GUI->GetImageThreshold()))
    {
    Fl::error("ImageResampleToggleCb failed to get ImageThreshold.\n");
    return;
    }
  if (!(contourFilter = GUI->GetContourFilter()))
    {
    Fl::error("ImageResampleToggleCb failed to get ContourFilter.\n");
    return;
    }

  if (aButton->value())
    {
    imageResample->SetInput(imageThreshold->GetOutput());
    contourFilter->SetInput(imageResample->GetOutput());
    GUI->mImageResampleGroup->activate();
    }
  else
    {
    contourFilter->SetInput(imageThreshold->GetOutput());
    imageResample->SetInput(NULL);
    GUI->mImageResampleGroup->deactivate();
    }
}
#else
void
ImageResampleToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkImageResample* imageResample;
  vtkImageSource* prev;
  vtkImageToImageFilter* next;
  vtkContourFilter* contourFilter;

  if (!(imageResample = GUI->GetImageResample()))
    {
    Fl::error("ImageResampleToggleCb failed to get ImageResample.\n");
    return;
    }

  if (!(prev = GUI->GetPrevImageSource(imageResample)))
    {
    Fl::error("ImageResampleToggleCb failed to get preceding ImageData source.\n");
    return;
    }

  if (aButton->value())
    {
    imageResample->SetInput(prev->GetOutput());

    if (next = GUI->GetNextImageToImageFilter(imageResample))
      next->SetInput(imageResample->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(imageResample->GetOutput());
    else
      {
      Fl::error("ImageResampleToggleCb failed to get subsequent ImageData filter.\n");
      return;
      }
    GUI->mImageResampleGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextImageToImageFilter(imageResample))
      next->SetInput(prev->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(prev->GetOutput());
    else
      {
      Fl::error("ImageResampleToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    imageResample->SetInput(NULL);
    GUI->mImageResampleGroup->deactivate();
    }
}
#endif /* 0 */

// ----------------------------------------------------------------------------
void
ImageResliceTransformToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkMatrixToLinearTransform* resliceTransform;
  vtkImageResample* imageResample;

  if (resliceTransform = GUI->GetResliceTransform())
    {
    resliceTransform->Identity();
    }

  if (!(imageResample = GUI->GetImageResample()))
    {
    Fl::error("ImageResampleToggleCb failed to get ImageResample.\n");
    return;
    }

  if (aButton->value())
    {
    imageResample->SetResliceTransform(resliceTransform);
    GUI->mResliceTransformGroup->activate();
    }
  else
    {
    imageResample->SetResliceTransform(NULL);
    GUI->mResliceTransformGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
PreThresholdImageToggleCb(Fl_Check_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkImageThreshold* imagePreThreshold;
  vtkImageSource* prev;
  vtkImageToImageFilter* next;
  vtkContourFilter* contourFilter;

  if (!(imagePreThreshold = GUI->GetImagePreThreshold()))
    {
    Fl::error("PreThresholdImageToggleCb failed to get ImagePreThreshold.\n");
    return;
    }

  if (!(prev = GUI->GetPrevImageSource(imagePreThreshold)))
    {
    Fl::error("PreThresholdImageToggleCb failed to get preceding ImageData source.\n");
    return;
    }

  if (aButton->value() && GUI->mImageDilateErode3DToggle->value())
    {
    imagePreThreshold->SetInput(prev->GetOutput());

    if (next = GUI->GetNextImageToImageFilter(imagePreThreshold))
      next->SetInput(imagePreThreshold->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(imagePreThreshold->GetOutput());
    else
      {
      Fl::error("PreThresholdImageToggleCb failed to get subsequent ImageData filter.\n");
      return;
      }
    }
  else
    {
    if (next = GUI->GetNextImageToImageFilter(imagePreThreshold))
      next->SetInput(prev->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(prev->GetOutput());
    else
      {
      Fl::error("PreThresholdImageToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    imagePreThreshold->SetInput(NULL);
    }
}

// ----------------------------------------------------------------------------
void
ImageDilateErode3DToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  // Invoke the pre-thresholder's callback first.
  GUI->mPreThresholdImageToggle->do_callback();

  vtkImageDilateErode3D* imageDilateErode3D;
  vtkImageSource* prev;
  vtkImageToImageFilter* next;
  vtkContourFilter* contourFilter;

  if (!(imageDilateErode3D = GUI->GetImageDilateErode3D()))
    {
    Fl::error("ImageDilateErode3DToggleCb failed to get ImageDilateErode3D.\n");
    return;
    }

  if (!(prev = GUI->GetPrevImageSource(imageDilateErode3D)))
    {
    Fl::error("ImageDilateErode3DToggleCb failed to get preceding ImageData source.\n");
    return;
    }

  if (aButton->value())
    {
    imageDilateErode3D->SetInput(prev->GetOutput());

    if (next = GUI->GetNextImageToImageFilter(imageDilateErode3D))
      next->SetInput(imageDilateErode3D->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(imageDilateErode3D->GetOutput());
    else
      {
      Fl::error("ImageDilateErode3DToggleCb failed to get subsequent ImageData filter.\n");
      return;
      }
    GUI->mImageDilateErode3DGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextImageToImageFilter(imageDilateErode3D))
      next->SetInput(prev->GetOutput());
    else if (contourFilter = GUI->GetContourFilter())
      contourFilter->SetInput(prev->GetOutput());
    else
      {
      Fl::error("ImageDilateErode3DToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    imageDilateErode3D->SetInput(NULL);
    GUI->mImageDilateErode3DGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
CleanPolyDataToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkCleanPolyData* cleanPolyData;
  vtkPolyDataSource* prev;
  vtkPolyDataToPolyDataFilter* next;
  vtkMassProperties* massProperties;

  if (!(cleanPolyData = GUI->GetCleanPolyData()))
    {
    Fl::error("CleanPolyDataToggleCb failed to get CleanPolyData.\n");
    return;
    }

  if (!(prev = GUI->GetPrevPolyDataSource(cleanPolyData)))
    {
    Fl::error("CleanPolyDataToggleCb failed to get preceding PolyData source.\n");
    return;
    }

  if (aButton->value())
    {
    cleanPolyData->SetInput(prev->GetOutput());

    if (next = GUI->GetNextPolyDataToPolyDataFilter(cleanPolyData))
      next->SetInput(cleanPolyData->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(cleanPolyData->GetOutput());
    else
      {
      Fl::error("CleanPolyDataToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }
    GUI->mCleanPolyDataGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextPolyDataToPolyDataFilter(cleanPolyData))
      next->SetInput(prev->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(prev->GetOutput());
    else
      {
      Fl::error("CleanPolyDataToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    cleanPolyData->SetInput(NULL);
    GUI->mCleanPolyDataGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
TriangleFilterToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkTriangleFilter* triangleFilter;
  vtkPolyDataSource* prev;
  vtkPolyDataToPolyDataFilter* next;
  vtkMassProperties* massProperties;

  if (!(triangleFilter = GUI->GetTriangleFilter()))
    {
    Fl::error("TriangleFilterToggleCb failed to get TriangleFilter.\n");
    return;
    }

  if (!(prev = GUI->GetPrevPolyDataSource(triangleFilter)))
    {
    Fl::error("TriangleFilterToggleCb failed to get preceding PolyData source.\n");
    return;
    }

  if (aButton->value())
    {
    triangleFilter->SetInput(prev->GetOutput());

    if (next = GUI->GetNextPolyDataToPolyDataFilter(triangleFilter))
      next->SetInput(triangleFilter->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(triangleFilter->GetOutput());
    else
      {
      Fl::error("TriangleFilterToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }
    GUI->mTriangleFilterGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextPolyDataToPolyDataFilter(triangleFilter))
      next->SetInput(prev->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(prev->GetOutput());
    else
      {
      Fl::error("TriangleFilterToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    triangleFilter->SetInput(NULL);
    GUI->mTriangleFilterGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
WSincPolyDataFilterToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkWindowedSincPolyDataFilter* wsincPolyDataFilter;
  vtkPolyDataSource* prev;
  vtkPolyDataToPolyDataFilter* next;
  vtkMassProperties* massProperties;

  if (!(wsincPolyDataFilter = GUI->GetWSincPolyDataFilter()))
    {
    Fl::error("WSincPolyDataFilterToggleCb failed to get TriangleFilter.\n");
    return;
    }

  if (!(prev = GUI->GetPrevPolyDataSource(wsincPolyDataFilter)))
    {
    Fl::error("WSincPolyDataFilterToggleCb failed to get preceding PolyData source.\n");
    return;
    }

  if (aButton->value())
    {
    wsincPolyDataFilter->SetInput(prev->GetOutput());

    if (next = GUI->GetNextPolyDataToPolyDataFilter(wsincPolyDataFilter))
      next->SetInput(wsincPolyDataFilter->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(wsincPolyDataFilter->GetOutput());
    else
      {
      Fl::error("WSincPolyDataFilterToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }
    GUI->mWSincPolyDataFilterGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextPolyDataToPolyDataFilter(wsincPolyDataFilter))
      next->SetInput(prev->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(prev->GetOutput());
    else
      {
      Fl::error("WSincPolyDataFilterToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    wsincPolyDataFilter->SetInput(NULL);
    GUI->mWSincPolyDataFilterGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
PolyDataNormalsToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkPolyDataNormals* polyDataNormals;
  vtkPolyDataSource* prev;
  vtkPolyDataToPolyDataFilter* next;
  vtkMassProperties* massProperties;

  if (!(polyDataNormals = GUI->GetPolyDataNormals()))
    {
    Fl::error("PolyDataNormalsToggleCb failed to get TriangleFilter.\n");
    return;
    }

  if (!(prev = GUI->GetPrevPolyDataSource(polyDataNormals)))
    {
    Fl::error("PolyDataNormalsToggleCb failed to get preceding PolyData source.\n");
    return;
    }

  if (aButton->value())
    {
    polyDataNormals->SetInput(prev->GetOutput());

    if (next = GUI->GetNextPolyDataToPolyDataFilter(polyDataNormals))
      next->SetInput(polyDataNormals->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(polyDataNormals->GetOutput());
    else
      {
      Fl::error("PolyDataNormalsToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }
    GUI->mPolyDataNormalsGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextPolyDataToPolyDataFilter(polyDataNormals))
      next->SetInput(prev->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(prev->GetOutput());
    else
      {
      Fl::error("PolyDataNormalsToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    polyDataNormals->SetInput(NULL);
    GUI->mPolyDataNormalsGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
CurvaturesToggleCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkCurvatures* curvatures;
  vtkPolyDataSource* prev;
  vtkPolyDataToPolyDataFilter* next;
  vtkMassProperties* massProperties;

  if (!(curvatures = GUI->GetCurvatures()))
    {
    Fl::error("CurvaturesToggleCb failed to get TriangleFilter.\n");
    return;
    }

  if (!(prev = GUI->GetPrevPolyDataSource(curvatures)))
    {
    Fl::error("CurvaturesToggleCb failed to get preceding PolyData source.\n");
    return;
    }

  if (aButton->value())
    {
    curvatures->SetInput(prev->GetOutput());

    if (next = GUI->GetNextPolyDataToPolyDataFilter(curvatures))
      next->SetInput(curvatures->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(curvatures->GetOutput());
    else
      {
      Fl::error("CurvaturesToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }
    GUI->mCurvaturesGroup->activate();
    }
  else
    {
    if (next = GUI->GetNextPolyDataToPolyDataFilter(curvatures))
      next->SetInput(prev->GetOutput());
    else if (massProperties = GUI->GetMassProperties())
      massProperties->SetInput(prev->GetOutput());
    else
      {
      Fl::error("CurvaturesToggleCb failed to get subsequent PolyData filter.\n");
      return;
      }

    curvatures->SetInput(NULL);
    GUI->mCurvaturesGroup->deactivate();
    }
}

// ----------------------------------------------------------------------------
void
LoadImageDataCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;
  vtkImageReader* imageReader;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;
  if(!(imageReader = GUI->GetImageReader())) return;

  char* fileName = NULL;

  if ( !( fileName = fl_file_chooser(
            "Read Voxel Data?",
            "ImageData Files (*.{bchar,buchar,bshort,bfloat,info})",
            0/*NULL ... defaults to last filename*/ ) ) )
    {
    return; // pressed cancel
    }

  std::string inName(fileName);
  std::string::size_type idx = inName.rfind('.');
  std::string suffix = (idx==std::string::npos ? "" : inName.substr(idx+1));
  const std::string hdr(".hdr");
  std::string hdrName = inName.substr(0,idx);
  int type = -1;
  
  if (idx != std::string::npos) hdrName.append(hdr);
  
  if      (suffix == "buchar")        type = VTK_UNSIGNED_CHAR;
  else if (suffix == "bushort")       type = VTK_UNSIGNED_SHORT;
  else if (suffix == "bshort")        type = VTK_SHORT;
  else if (suffix == "bfloat")        type = VTK_FLOAT;
  else
    {
    Fl::error("Failed to determine scalar type from suffix!\n");
    return;
    }

  if (hdrName.empty())
    {
    Fl::error("Failed to determine header name!\n");
    return;
    }

  struct stat fs;
  
  // Open the header file containing the dimensions.
  if(stat(hdrName.c_str(), &fs) != 0)
    {
    Fl::error("Unable to open file: '%s'",hdrName.c_str());
    return;
    }

  std::ifstream hdrStream;

  // Opens the file and positions the stream pointer at EOF ...
  hdrStream.open(hdrName.c_str(), std::ios::in | std::ios::ate);
  
  if (!hdrStream.is_open())
    {
    Fl::error("Failed to open header '%s'\n",hdrName.c_str());
    return;
    }
  
  int dimensions[3];
  
  // Rewind the stream ... and start reading the items.
  hdrStream.seekg(0, std::ios::beg);
  // Read the file dimensions from the header (rows, cols, slices, 0).
  hdrStream >> dimensions[1] >> dimensions[0] >> dimensions[2];
  hdrStream.close();
  
  imageReader->SetFileName(fileName);
  imageReader->SetDataScalarType(type);
  imageReader->SetDataByteOrderToBigEndian();
  imageReader->SetFileNameSliceOffset(0);
  imageReader->SetFileNameSliceSpacing(1);
  imageReader->FileLowerLeftOff();

  GUI->SetInputDimensions(dimensions);
}


// ----------------------------------------------------------------------------
void
SaveContourCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  char* fileName = 0;

  if ( fileName = fl_file_chooser(
         "Write Contour Polygonal Data?",
         "PolyData Files (*.{vtk,vtp})",
         0/*NULL ... defaults to last filename*/) )
    {
    std::string outName(fileName);
    std::string::size_type idx = outName.rfind('.');
    std::string suffix;

    if (idx==std::string::npos)
      {
      suffix = "vtp";
      outName.append(".vtp");
      }
    else
      {
      suffix = outName.substr(idx+1);
      if (suffix!="vtp" && suffix!="vtk")
        {
        suffix = "vtp";
        outName.replace(idx,outName.size(),".vtp");
        }
      }

    if (!outName.empty())
      {
      if (suffix == "vtp")
        {
        vtkXMLDataSetWriter* writer = vtkXMLDataSetWriter::New();
          {
          writer->SetInput(GUI->GetContourMapper()->GetInput());
          writer->SetByteOrderToBigEndian();
          writer->SetFileName(outName.c_str());
          vtkZLibDataCompressor* compressor = vtkZLibDataCompressor::New();
          writer->SetCompressor(compressor);
          compressor->Delete();
          //writer->SetDataModeToAscii();
          //writer->SetDataModeToBinary();
          writer->SetDataModeToAppended();
          writer->EncodeAppendedDataOn();
          writer->Write();
          }
        writer->Delete();
        }
      else if (suffix == "vtk")
        {
        vtkDataSetWriter* writer = vtkDataSetWriter::New();
          {
          writer->SetInput(GUI->GetContourMapper()->GetInput());
          writer->SetFileName(outName.c_str());
          writer->Write();
          }
        writer->Delete();          
        }
      }
    }
}

// ----------------------------------------------------------------------------
void
UpdateButtonCb(Fl_Return_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

#if 0
  // --------------------------------------------------------------------------
  if (gReferenceName && !gReferenceActor)
    {
    if (gReferenceActor = CreateReferenceActor(gReferenceName,GUI))
      {
      GUI->GetView()->GetDefaultRenderer()->AddProp(gReferenceActor);
      gReferenceActor->Delete();
      GUI->GetView()->GetDefaultRenderer()->ResetCamera();
      GUI->GetView()->GetRenderWindow()->Render();
      }
    }
#endif /* 0 */

  // --------------------------------------------------------------------------
  GUI->UpdateImageReader();
  GUI->UpdateImageThreshold();
  GUI->UpdateImageResample();
  GUI->UpdateImageDilateErode3D();
  GUI->UpdateContourFilter();
  GUI->UpdateCleanPolyData();
  GUI->UpdateTriangleFilter();
  GUI->UpdateWSincPolyDataFilter();
  GUI->UpdatePolyDataNormals();
  GUI->UpdateCurvatures();
  GUI->UpdateMassProperties();
  GUI->UpdateContourMapper();
  GUI->UpdateContourActor();

#if 0
  // --------------------------------------------------------------------------
  CheckImagePlaneWidgets( GUI->GetView()->GetInteractor(),
                          GUI->GetImageResample()->GetOutput() );
#endif /* 0 */

  GUI->UpdateView();
}

// ----------------------------------------------------------------------------
void
ContourVisibilityCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkActor* contour;

  if(contour = GUI->GetContourActor())
    {
    Fl::check();
    contour->SetVisibility(aButton->value());
    GUI->UpdateView();
    }
}

// ----------------------------------------------------------------------------
// Internal function to read in a line up to 'size' characters.
// Returns zero if there was an error, else number of char read.
static std::size_t
ReadLine(ifstream& ifs, char* buffer, std::size_t bufsize)
{
  ifs.getline(buffer,bufsize);

  std::size_t length = ifs.gcount();

  if(ifs.fail())
    {
    if(ifs.eof())
      {
      return 0;
      }
    if(length == (bufsize-1))
      {
      // Read 'bufsize' chars; ignoring the rest of the line.
      ifs.clear();
      ifs.ignore(0xffffffff, '\n');
      }
    }

  return length;
}

// ----------------------------------------------------------------------------
static bool
OrientWithRegister(double aMatrix[16], VoxelToContourUI* aGUI)
{
  int           toDimensions[3] = { 256, 256, 256 };
  double        toSpacing[3] = { 1.0, 1.0, 1.0 };
  int           toRASGoodFlag = 1;
  double        toResliceAxesDirectionCosines[9] = { -1,0,0, 0,0,-1, 0,1,0 };
  double        toResliceAxesOrigin[3] = { 0.0, 0.0, 0.0 };

  vtkMatrix4x4* r = vtkMatrix4x4::New();
  r->SetElement(0,0, aMatrix[0]);
  r->SetElement(0,1, aMatrix[1]);
  r->SetElement(0,2, aMatrix[2]);
  r->SetElement(0,3, aMatrix[3]);
  r->SetElement(1,0, aMatrix[4]);
  r->SetElement(1,1, aMatrix[5]);
  r->SetElement(1,2, aMatrix[6]);
  r->SetElement(1,3, aMatrix[7]);
  r->SetElement(2,0, aMatrix[8]);
  r->SetElement(2,1, aMatrix[9]);
  r->SetElement(2,2, aMatrix[10]);
  r->SetElement(2,3, aMatrix[11]);
  r->SetElement(3,0, aMatrix[12]);
  r->SetElement(3,1, aMatrix[13]);
  r->SetElement(3,2, aMatrix[14]);
  r->SetElement(3,3, aMatrix[15]);

  double determinant;

#if 1
  determinant = r->Determinant();

  if(determinant == 0.0)
    {
    std::cerr << "orient_with_register():"
              << " registration matrix has zero determinant" << std::endl;
    r->Print(std::cerr);
    return false;
    }
#endif /* 0 */

  vtkMatrix4x4* rinv = vtkMatrix4x4::New();
  rinv->DeepCopy(r);
  rinv->Invert();
  r->Delete();

  double m00 = toSpacing[0] * toResliceAxesDirectionCosines[0];
  double m01 = toSpacing[1] * toResliceAxesDirectionCosines[3];
  double m02 = toSpacing[2] * toResliceAxesDirectionCosines[6];
  double m10 = toSpacing[0] * toResliceAxesDirectionCosines[1];
  double m11 = toSpacing[1] * toResliceAxesDirectionCosines[4];
  double m12 = toSpacing[2] * toResliceAxesDirectionCosines[7];
  double m20 = toSpacing[0] * toResliceAxesDirectionCosines[2];
  double m21 = toSpacing[1] * toResliceAxesDirectionCosines[5];
  double m22 = toSpacing[2] * toResliceAxesDirectionCosines[8];

  double ci = (toDimensions[0] - 1.0) / 2.0;
  double cj = (toDimensions[1] - 1.0) / 2.0;
  double ck = (toDimensions[2] - 1.0) / 2.0;

  double m03 = toResliceAxesOrigin[0] - (m00 * ci + m01 * cj + m02 * ck);
  double m13 = toResliceAxesOrigin[1] - (m10 * ci + m11 * cj + m12 * ck);
  double m23 = toResliceAxesOrigin[2] - (m20 * ci + m21 * cj + m22 * ck);

  vtkMatrix4x4* sr = vtkMatrix4x4::New();

  sr->SetElement(0,0,m00);
  sr->SetElement(0,1,m01);
  sr->SetElement(0,2,m02);
  sr->SetElement(0,3,m03);
  
  sr->SetElement(1,0,m10);
  sr->SetElement(1,1,m11);
  sr->SetElement(1,2,m12);
  sr->SetElement(1,3,m13);
  
  sr->SetElement(2,0,m20);
  sr->SetElement(2,1,m21);
  sr->SetElement(2,2,m22);
  sr->SetElement(2,3,m23);

  sr->SetElement(3,0, 0.0);
  sr->SetElement(3,1, 0.0);
  sr->SetElement(3,2, 0.0);
  sr->SetElement(3,3, 1.0);

  vtkMatrix4x4* sa = vtkMatrix4x4::New();

  sa->SetElement(0,0, -toSpacing[0]);
  sa->SetElement(0,1,  0.0);
  sa->SetElement(0,2,  0.0);
  sa->SetElement(0,3,  toDimensions[0] * toSpacing[0] / 2.0);

  sa->SetElement(1,0,  0.0);
  sa->SetElement(1,1,  0.0);
  sa->SetElement(1,2,  toSpacing[2]);
  sa->SetElement(1,3, -toDimensions[2] * toSpacing[2] / 2.0);

  sa->SetElement(2,0,  0.0);
  sa->SetElement(2,1, -toSpacing[1]);
  sa->SetElement(2,2,  0.0);
  sa->SetElement(2,3,  toDimensions[1] * toSpacing[1] / 2.0);

  sa->SetElement(3,0,  0.0);
  sa->SetElement(3,1,  0.0);
  sa->SetElement(3,2,  0.0);
  sa->SetElement(3,3,  1.0);

#if 1
  determinant = sa->Determinant();

  if(determinant == 0.0)
    {
    std::cerr << "orient_with_register(): destination (ijk) ->"
              << " r space matrix has zero determinant" << std::endl;
    sa->Print(std::cerr);
    sr->Delete();
    sa->Delete();
    rinv->Delete();
    return false;
    }
#endif /* 0 */

  vtkMatrix4x4* sainv = vtkMatrix4x4::New();
  sainv->DeepCopy(sa);
  sainv->Invert();
  sa->Delete();

  int   fromDimensions[3];
  float fromSpacing[3];

  aGUI->GetInputDimensions(fromDimensions);
  aGUI->GetInputSpacingValues(fromSpacing);

  vtkMatrix4x4* fa = vtkMatrix4x4::New();

  fa->SetElement(0,0, -fromSpacing[0]);
  fa->SetElement(0,1,  0.0);
  fa->SetElement(0,2,  0.0);
  fa->SetElement(0,3,  fromDimensions[0] * fromSpacing[0] / 2.0);

  fa->SetElement(1,0,  0.0);
  fa->SetElement(1,1,  0.0);
  fa->SetElement(1,2,  fromSpacing[2]);
  fa->SetElement(1,3, -fromDimensions[2] * fromSpacing[2] / 2.0);

  fa->SetElement(2,0,  0.0);
  fa->SetElement(2,1, -fromSpacing[1]);
  fa->SetElement(2,2,  0.0);
  fa->SetElement(2,3,  fromDimensions[1] * fromSpacing[1] / 2.0);

  fa->SetElement(3,0,  0.0);
  fa->SetElement(3,1,  0.0);
  fa->SetElement(3,2,  0.0);
  fa->SetElement(3,3,  1.0);

  vtkMatrix4x4* r1 = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(rinv,fa,r1);

  rinv->Delete();
  fa->Delete();

  vtkMatrix4x4* r2 = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(sainv,r1,r2);

  r1->Delete();
  sainv->Delete();

  vtkMatrix4x4* fr = vtkMatrix4x4::New();

  vtkMatrix4x4::Multiply4x4(sr,r2,fr);

  sr->Delete();
  r2->Delete();

  double        outSpacing[3];
  double        outResliceAxesDirectionCosines[9];
  double        outResliceAxesOrigin[3];

  double ras[3];
  double magnitude;
  
  ras[0] = fr->GetElement(0,0);
  ras[1] = fr->GetElement(1,0);
  ras[2] = fr->GetElement(2,0);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[0] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[1] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[2] = ras[2] / magnitude;
  outSpacing[0]                     = magnitude;
  
  ras[0] = fr->GetElement(0,1);
  ras[1] = fr->GetElement(1,1);
  ras[2] = fr->GetElement(2,1);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[3] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[4] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[5] = ras[2] / magnitude;
  outSpacing[1]                     = magnitude;
  
  ras[0] = fr->GetElement(0,2);
  ras[1] = fr->GetElement(1,2);
  ras[2] = fr->GetElement(2,2);
  magnitude = vtkMath::Norm(ras);
  
  outResliceAxesDirectionCosines[6] = ras[0] / magnitude;
  outResliceAxesDirectionCosines[7] = ras[1] / magnitude;
  outResliceAxesDirectionCosines[8] = ras[2] / magnitude;
  outSpacing[2]                     = magnitude;
  
  double origin[4], c[4];
  
  origin[0] = (fromDimensions[0] - 1.0) / 2.0;
  origin[1] = (fromDimensions[1] - 1.0) / 2.0;
  origin[2] = (fromDimensions[2] - 1.0) / 2.0;
  origin[3] = 1.0;
  
  fr->MultiplyPoint(origin,c);
  
  outResliceAxesOrigin[0] = c[0];
  outResliceAxesOrigin[1] = c[1];
  outResliceAxesOrigin[2] = c[2];
  
  aGUI->SetResliceAxesDirectionCosinesValues(outResliceAxesDirectionCosines);
  aGUI->SetResliceAxesOriginValues(outResliceAxesOrigin);
  aGUI->SetResliceSpacingValues(outSpacing);
  
  fr->Delete();

  return true;
} /* end orient_with_register() */

// ----------------------------------------------------------------------------
void
LoadRegistrationCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  char* fileName = 0;

  if ( !(fileName = fl_file_chooser(
           "Read Registration Data?",
           "Registration (register.{dat,txt})",
           0/*NULL ... defaults to last filename*/)) )
    {
    return;
    }

  struct stat fs;
  
  // Open the file containing the resolutions and rotation matrix.
  if (stat(fileName, &fs) != 0)
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return;
    }

  // Opens the file and positions the stream pointer at EOF ...
  ifstream ifs(fileName, ios::in|ios::ate);

  if(!ifs.is_open())
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return;
    }

  // The read position at EOF is the file size.
  std::streampos size = ifs.tellg();
  // Rewind the stream.
  ifs.seekg(0, ios::beg);

  char pathName[1024];
    
  if(!ReadLine(ifs, pathName, sizeof(pathName)))
    {
    return;
    }

  double ipr, tpr, scaling, m[16];

  ifs >> ipr >> tpr >> scaling
      >> m[0] >> m[1] >> m[2] >> m[3]
      >> m[4] >> m[5] >> m[6] >> m[7]
      >> m[8] >> m[9] >> m[10] >> m[11]
      >> m[12] >> m[13] >> m[14] >> m[15];

  if (ifs.fail())
    {
    return;
    }

  float inputSpacing[3];

  GUI->GetInputSpacingValues(inputSpacing);

  if ( (inputSpacing[0] != ipr) ||
       (inputSpacing[1] != ipr) ||
       (inputSpacing[2] != tpr) )
    {
    Fl::warning("Registration spacing does not match input spacing.");
    }

  vtkMatrix4x4* resliceMatrix;

  if (resliceMatrix = GUI->GetResliceMatrix())
    {
    resliceMatrix->DeepCopy(m);
    }

  if (!OrientWithRegister(m, GUI))
    {
    return;
    }
}

void
InvertRegistrationCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

#if 0
  vtkMatrix4x4* resliceMatrix;

  if (resliceMatrix = GUI->GetResliceMatrix())
    {
    resliceMatrix->Invert();
    }
#else
  vtkMatrixToLinearTransform* resliceTransform;

  if (resliceTransform = GUI->GetResliceTransform())
    {
    resliceTransform->Inverse();
    }
#endif /* 0 */
}

void
ClearRegistrationCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  VoxelToContourUI* GUI;

  if(!(GUI = reinterpret_cast<VoxelToContourUI*>(aPtr))) return;

  vtkMatrixToLinearTransform* resliceTransform;

  if (resliceTransform = GUI->GetResliceTransform())
    {
    resliceTransform->Identity();
    }
}

/* 
 * End of: $Id: VoxelToContourCb.cxx,v 1.1.1.1 2006/12/19 22:59:42 christianh Exp $.
 * 
 */
