/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Fl_Toggle_Tree.cxx,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * Adapted from Flek (The Fast Light Environment Kit).
 * 
 */
// Flek-derived
#include "Fl_Toggle_Tree.H"
#include "Fl_Toggle_Node.H"
// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Pixmap.H>
#include <FL/Fl_Input.H>
#include <FL/Fl_Group.H>
#include <FL/fl_draw.H>
// C++ forwarding ANSI C
#include <cstddef>
// pixmaps
#include "pixmaps/tt_opened_icon.xpm"
#include "pixmaps/tt_closed_icon.xpm"
#include "pixmaps/tt_group_icon.xpm"
#include "pixmaps/tt_node_icon.xpm"


// static initialization 
ImageIconMap* Fl_Toggle_Tree::s_icon_map_;
long Fl_Toggle_Tree::Init::TranslationUnits;


// ----------------------------------------------------------------------------
Fl_Toggle_Tree::Init::Init (void)
{
  if (Fl_Toggle_Tree::Init::TranslationUnits++ == 0)
    {
    Fl_Toggle_Tree::ClassInitialize();
    }
}

Fl_Toggle_Tree::Init::~Init()
{
  if (--Fl_Toggle_Tree::Init::TranslationUnits == 0)
    {
    Fl_Toggle_Tree::ClassFinalize();
    }
}

void
Fl_Toggle_Tree::ClassInitialize (void)
{
  Fl_Toggle_Tree::s_icon_map_ = new ImageIconMap();

  (*(Fl_Toggle_Tree::s_icon_map_))["closed"] =
    static_cast<Fl_Image*>(new Fl_Pixmap(tt_closed_icon_xpm));
  (*(Fl_Toggle_Tree::s_icon_map_))["opened"] =
    static_cast<Fl_Image*>(new Fl_Pixmap(tt_opened_icon_xpm));
  (*(Fl_Toggle_Tree::s_icon_map_))["group"] =
    static_cast<Fl_Image*>(new Fl_Pixmap(tt_group_icon_xpm));
  (*(Fl_Toggle_Tree::s_icon_map_))["node"] =
    static_cast<Fl_Image*>(new Fl_Pixmap(tt_node_icon_xpm));
}

void
Fl_Toggle_Tree::ClassFinalize (void)
{
  ImageIconMap::iterator p;
  for ( p  = Fl_Toggle_Tree::s_icon_map_->begin();
        p != Fl_Toggle_Tree::s_icon_map_->end();
        p++ )
    {
    delete p->second;
    }

  delete Fl_Toggle_Tree::s_icon_map_;
}

// ----------------------------------------------------------------------------
// Find an icon in the static map by name. -----------------------------------
Fl_Image*
Fl_Toggle_Tree::find_icon (std::string aName)
{
  return (*(Fl_Toggle_Tree::s_icon_map_))[aName];
}

// ----------------------------------------------------------------------------
// Add a named icon to the static map. ---------------------------------------
void
Fl_Toggle_Tree::add_icon (std::string aName, Fl_Image* aImage)
{
  Fl_Image* oldImage = (*(Fl_Toggle_Tree::s_icon_map_))[aName];

  if (oldImage && oldImage!=aImage)
    {
    Fl_Toggle_Tree::s_icon_map_->erase(aName);
    delete oldImage;
    }

  if (aImage)
    {
    (*(Fl_Toggle_Tree::s_icon_map_))[aName] = aImage;
    }
}

// ----------------------------------------------------------------------------
// Add a named icon (created as pixmap from data) to the static map. ----------
Fl_Image*
Fl_Toggle_Tree::make_icon (std::string aName, const char* const * aData)
{
  Fl_Toggle_Tree::add_icon(aName, static_cast<Fl_Image*>(new Fl_Pixmap(aData)));
  return Fl_Toggle_Tree::find_icon(aName);
}

// ----------------------------------------------------------------------------
// Remove a named icon from the static map. -----------------------------------
void
Fl_Toggle_Tree::remove_icon (std::string aName, bool aDestroyFlag)
{
  Fl_Image* image = (*(Fl_Toggle_Tree::s_icon_map_))[aName];

  if (image)
    {
    Fl_Toggle_Tree::s_icon_map_->erase(aName);
    if (aDestroyFlag) delete image;
    }
}


// ---------------------------------------------------------------------------
void
Fl_Toggle_Tree::select_range (Fl_Toggle_Node* aStartNode,
                              Fl_Toggle_Node* aEndNode,
                              int             aAddFlag)
{
  Fl_Toggle_Node* tnode     = this->first_node();
  Fl_Toggle_Node* selecting = (Fl_Toggle_Node *) 0;

  this->selection_node_  = (Fl_Toggle_Node *) 0;
  this->selection_count_ = 0;

  this->traverse_start(tnode);

  while (tnode != (Fl_Toggle_Node *) 0)
    {
    if (selecting == (Fl_Toggle_Node *) 0)
      {
      if      (tnode == aStartNode)
        {
	selecting = aEndNode;
        }
      else if (tnode == aEndNode)
        {
	selecting = aStartNode;
        }
      }
    
    // aAddFlag == 0 - pick one only (unpick rest)
    // aAddFlag == 1 - add picked (never unpick)
    // aAddFlag  > 1 - toggle picked
      
    int tmp = tnode->selected_;

    if      ((selecting != (Fl_Toggle_Node *) 0) && (aAddFlag > 1))
      {
      tnode->selected_ = !tnode->selected_;
      }
    else if (selecting != (Fl_Toggle_Node *) 0)
      {
      tnode->selected_ = true;
      }
    else if (aAddFlag == 0)
      {
      tnode->selected_ = false;
      }
    
    tnode->changed_ = (tmp != tnode->selected_);
    
    if (tnode == selecting)
      {
      selecting = (Fl_Toggle_Node *) 0;
      }
    
    tnode = this->traverse_forward();
    }

  this->selected_node_ = (Fl_Toggle_Node_Base *) 0;

  if (this->selection_count() == 1)
    {
    this->selected_node_ = this->selection_next();
    }
}

// ----------------------------------------------------------------------------
static const int no_columns[1] = {0};
static const int s_icon_size   = 24;

// ----------------------------------------------------------------------------
Fl_Toggle_Tree::Fl_Toggle_Tree (int x, int y, int w, int h)
  : Fl_Toggle_Tree_Base(x, y, w, h),
    icon_offset_(s_icon_size),
    label_offset_(icon_offset_ * 5 / 2),
    indent_toggles_(true),
    edit_on_reselect_(true),
    line_visibility_(true),
    selection_label_color_(FL_YELLOW),
    alternate_color_(FL_LIGHT2),
    trim_color_(FL_LIGHT1),
    column_widths_(no_columns),
    column_delimiter_('\t'),
    textfont_(FL_HELVETICA),
    textsize_(12),
    textcolor_(FL_BLACK),
    state_(FL_TOGGLE_NONE),
    opened_icon_((Fl_Image *) 0),
    closed_icon_((Fl_Image *) 0),
    selection_node_((Fl_Toggle_Node *) 0),
    selection_index_(0),
    selection_count_(0),
    edit_input_((Fl_Input *) 0)
{
  this->closed_icon(Fl_Toggle_Tree::find_icon("closed"));
  this->opened_icon(Fl_Toggle_Tree::find_icon("opened"));

  try
    {
    this->edit_input_ = new Fl_Input(x, y, 0, 0);
    this->edit_input_->box(FL_FLAT_BOX);
    this->edit_input_->color(FL_WHITE);
#if FL_MAJOR_VERSION == 1
    this->edit_input_->textcolor(FL_BLACK);
    this->edit_input_->textfont(this->textfont_);
    this->edit_input_->textsize(this->textsize_);
#endif
    this->edit_callback( (Fl_Callback *) Fl_Toggle_Tree::edit_default_callback,
                         (void *) this );
    this->edit_input_->when( FL_WHEN_RELEASE |
                             FL_WHEN_ENTER_KEY |
                             FL_WHEN_NOT_CHANGED );
    }
  catch (...)
    {
    delete this->edit_input_;
    this->edit_input_ = (Fl_Input *) 0;
    Fl::error("Fl_Toggle_Tree constructor"
              " failed to create EditInput widget.");
    }

  this->edit_input_->hide();

  this->color(FL_BACKGROUND_COLOR);             // originally FL_WHITE
  this->selection_color(FL_SELECTION_COLOR);    // originally FL_BLACK
  this->selection_label_color(fl_contrast(FL_YELLOW, this->selection_color()));
}

Fl_Toggle_Tree::~Fl_Toggle_Tree()
{
  delete this->edit_input_;
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree::node_height (Fl_Toggle_Node_Base*)
{
  return s_icon_size + 1; // could be 'this->textsize() + 5'
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::draw_node (int                  aDepth,
                           int                  aCurrentY,
                           Fl_Toggle_Node_Base* aNode)
{
  Fl_Toggle_Node* tnode = static_cast<Fl_Toggle_Node*>(aNode);

  if ( this->damage() == FL_DAMAGE_CHILD &&
       tnode->changed_ == false &&
       this->damaged_node_ == (Fl_Toggle_Node_Base *) 0 )
    {
    return;
    }

  tnode->changed_ = false;

  if (tnode->selected_)
    {
    fl_color(this->selection_color());
    fl_rectf(this->x(), aCurrentY + 1, this->w(), this->height_(tnode) - 1);
    }
  else
    {
    fl_color((aCurrentY-this->y())&1 ? this->color() : this->alternate_color());
    fl_rectf(this->x(), aCurrentY + 1, this->w(), this->height_(tnode) - 1);
    }

  fl_color(this->trim_color());
  fl_line(this->x(), aCurrentY, this->x() + this->w(), aCurrentY);
  fl_color(FL_BLACK);

  if (this->line_visibility_)
    {
    fl_xyline( this->x() + aDepth*s_icon_size + (s_icon_size/2),
               aCurrentY + (s_icon_size/2),
               this->x() + (aDepth+1)*s_icon_size,
               aCurrentY + (s_icon_size/2) ); 

    if (tnode->next_sibling_ != NULL)
      {
      fl_xyline( this->x() + aDepth*s_icon_size + (s_icon_size/2),
                 aCurrentY,
                 this->x() + aDepth*s_icon_size + (s_icon_size/2),
                 aCurrentY + s_icon_size );
      }
    else
      {
      fl_xyline( this->x() + aDepth*s_icon_size + (s_icon_size/2),
                 aCurrentY,
                 this->x() + aDepth*s_icon_size + (s_icon_size/2),
                 aCurrentY + (s_icon_size/2) );
      }

    Fl_Toggle_Node* node = static_cast<Fl_Toggle_Node*>(tnode->parent_node_);

    for (int i=(aDepth-1); node; i--)
      {
      if (node->next_sibling_ != NULL)
        {
        fl_xyline( this->x() + i*s_icon_size + (s_icon_size/2),
                   aCurrentY,
                   this->x() + i*s_icon_size + (s_icon_size/2),
                   aCurrentY + s_icon_size );
        }
      node = static_cast<Fl_Toggle_Node*>(node->parent_node_);
      }
    }

  if (tnode->openable_)
    {
    if (tnode->opened_)
      {
      this->opened_icon_->draw(this->x() + aDepth*s_icon_size, aCurrentY);
      }
    else
      {
      this->closed_icon_->draw(this->x() + aDepth*s_icon_size, aCurrentY);
      }
    }

#if FL_MAJOR_VERSION == 1
  if (tnode->selected_)
    {
    this->textcolor(this->selection_label_color());
    }
  else
    {
    this->textcolor(this->labelcolor());
    }
#endif

  if (tnode->label_ != (char *) 0)
    {
    this->draw_label( tnode->label_, aDepth * s_icon_size + this->label_offset_,
                      this->x(), aCurrentY, this->w(), s_icon_size );
    }

  if (tnode->icon_ != (Fl_Image *) 0)
    {
    tnode->icon_->draw( this->x() + aDepth*s_icon_size + this->icon_offset_,
                        aCurrentY + 1 );
    }
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::draw_label (char* aLabel, int aIndent,
                            int x, int y, int w, int h)
{
  const int* i = this->column_widths();

  while (w > 6)
    {
    // do each delimiter-seperated field
    int   fieldWidth = w;          // width for this field
    char* end        = (char *) 0; // pointer to end of field or null if none

    if (*i)
      {
      // find end of field and temporarily replace with 0
      for (end=aLabel; *end != '\0' && *end != this->column_delimiter(); end++);

      if (*end != '\0')
        {
        *end = '\0';
        fieldWidth = *i++;
        }
      else
        {
        end = (char *) 0;
        }
      }

    int      size  = this->textsize();
    Fl_Font  font  = this->textfont();
    Fl_Color lcol  = this->textcolor();
    Fl_Align align = Fl_Align(FL_ALIGN_LEFT | FL_ALIGN_CLIP);

#if 0
    // check for all the @-lines recognized by XForms:
    while (*aLabel == format_char() && *++aLabel && *aLabel != format_char())
      {
      switch (*aLabel++)
        {
        case 'l': case 'L': size = 24; break;
        case 'm': case 'M': size = 18; break;
        case 's': size = 11; break;
        case 'b': font = (Fl_Font)(font|FL_BOLD); break;
        case 'i': font = (Fl_Font)(font|FL_ITALIC); break;
        case 'f': case 't': font = FL_COURIER; break;
        case 'c': align = FL_ALIGN_CENTER; break;
        case 'r': align = FL_ALIGN_RIGHT; break;
        case 'B': 
          fl_color((Fl_Color)strtol(aLabel, &aLabel, 10));
          fl_rectf(x, y, fieldWidth, h);
          break;
        case 'C':
          lcol = (Fl_Color)strtol(aLabel, &aLabel, 10);
          break;
        case 'F':
          font = (Fl_Font)strtol(aLabel, &aLabel, 10);
          break;
        case 'N':
          lcol = FL_INACTIVE_COLOR;
          break;
        case 'S':
          size = strtol(aLabel, &aLabel, 10);
          break;
        case '-':
          fl_color(FL_DARK3);
          fl_line(x+3, y+h/2, x+fieldWidth-3, y+h/2);
          fl_color(FL_LIGHT3);
          fl_line(x+3, y+h/2+1, x+fieldWidth-3, y+h/2+1);
          break;
        case 'u':
        case '_':
          fl_color(lcol);
          fl_line(x+3, y+h-1, x+fieldWidth-3, y+h-1);
          break;
        case '.':
          goto BREAK;
        case '@':
          aLabel--; goto BREAK;
        }
      }
      BREAK:
#endif /* 0 */

    fl_font(font, size);

#if FL_MAJOR_VERSION == 1
#  if FL_MINOR_VERSION == 0     
    if (!this->active_r()) lcol = inactive(lcol);
#  else
    if (!this->active_r()) lcol = fl_inactive(lcol);
#  endif
#endif

    // if (((FL_BLINE*)v)->flags & SELECTED)
    //   {
    //   lcol = contrast(lcol, this->selection_color());
    //   }

    fl_color(lcol);
    fl_draw(aLabel, x+aIndent, y+1, fieldWidth-aIndent, h+1, align);

    if (end == (char *) 0)
      {
      break; // no more fields...
      }

    *end = this->column_delimiter(); // put the seperator back

    x += fieldWidth;
    w -= fieldWidth;

    aLabel  = end + 1;
    aIndent = 0;
    }
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::open (Fl_Toggle_Node* aNode)
{
  if (aNode->opened_)
    {
    return;
    }

  int th = this->Fl_Toggle_Tree_Base::open(aNode);

  this->damaged_node_ = aNode;

  if (th != 0)
    {
    this->damage(FL_DAMAGE_TREE);
    this->resize(this->x(), this->y(), this->w(), this->h() + th);

    if (this->parent() != (Fl_Group *) 0)
      this->parent()->damage(FL_DAMAGE_CHILD);
    }
  else
    {
    this->damage(FL_DAMAGE_CHILD);
    }
}

void
Fl_Toggle_Tree::close (Fl_Toggle_Node* aNode)
{
  if (!aNode->opened_)
    {
    return;
    }

  int th = this->Fl_Toggle_Tree_Base::close(aNode);

  this->damaged_node_ = aNode;

  if (th != 0)
    {
    this->damage(FL_DAMAGE_TREE);
    this->resize(this->x(), this->y(), this->w(), this->h() - th);

    if (this->parent() != (Fl_Group *) 0)
      this->parent()->damage(FL_DAMAGE_SCROLL);
    }
  else
    {
    this->damage(FL_DAMAGE_CHILD);
    }
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::edit (Fl_Toggle_Node* aNode, int aX, int aY)
{
  if (!this->edit_input_->visible() && aNode != (Fl_Toggle_Node *) 0)
    {
    this->edit_input_->resize( aX - 3,
                               aY + 1,
                               this->w() - (aX - 3 - this->x()),
                               this->height_(aNode) - 1 );
    this->edit_input_->value(aNode->label());
    this->edit_input_->show();
    this->edit_input_->take_focus();
    }
}

void
Fl_Toggle_Tree::edit_default_callback (Fl_Input*, void* aPtr)
{
  Fl_Toggle_Tree* tree = reinterpret_cast<Fl_Toggle_Tree*>(aPtr);
  if (tree != (Fl_Toggle_Tree *) 0) tree->end_edit();
}

void
Fl_Toggle_Tree::end_edit (void)
{
  if (this->selected_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    (static_cast<Fl_Toggle_Node*>(this->selected_node_))->
      label(this->edit_input_->value());
    }

  this->edit_input_->hide();

  this->damaged_node_ = this->selected_node_;
  this->damage(FL_DAMAGE_CHILD);
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree::handle (int aEvent)
{
  static Fl_Toggle_Node* prev = (Fl_Toggle_Node *) 0;

  switch (aEvent)
    {
      
    case FL_ENTER:
    case FL_PUSH: 
      break;

    case FL_RELEASE:
      {
      if (this->edit_input_->visible())
        {
        this->end_edit();
        }

      int depth;
      int cy;

      Fl_Toggle_Node* tnode = static_cast<Fl_Toggle_Node*>
        (this->Fl_Toggle_Tree_Base::find(Fl::event_y(), depth, cy));

      if (Fl::event_x() < this->x() + depth*s_icon_size + s_icon_size)
        {
        if (tnode->opened_)
          {
          this->selected_node_ = tnode;
          this->state_         = FL_TOGGLE_OPENED;
          this->do_callback();
          this->close(tnode);
          }
        else
          {
          this->selected_node_ = tnode;
          this->state_         = FL_TOGGLE_CLOSED;
          this->do_callback();
          this->open(tnode);
          }
        }
      else
        {
        if      (Fl::event_state(FL_SHIFT))
          {
          if (prev == (Fl_Toggle_Node *) 0)
            {
            prev = tnode;
            }

          this->select_range(prev, tnode, 1);
          // this->selected_node_ = (Fl_Toggle_Node_Base *) 0;
          this->state_ = FL_TOGGLE_SELECT;
          this->do_callback();
          }
        else if (Fl::event_state(FL_CTRL))
          {
          /*
           * if (!tnode->selected_)
           *   {
           */
          this->select_range(tnode, tnode, Fl::event_state(FL_CTRL));
          /*
           *   }
           * else
           *   {
           *   this->selection_node_ = (Fl_Toggle_Node *) 0;
           *   tnode->selected_      = false;
           *   tnode->changed_       = true;
           *   tnode                 = (Fl_Toggle_Node *) 0;
           *   }
           * this->selected_node_ = (Fl_Toggle_Node_Base *) 0;
           */
          this->state_ = FL_TOGGLE_SELECT;
          this->do_callback();
          }
        else
          {
          this->select_range(tnode, tnode, 0);
          this->state_=(Fl::event_clicks() ? FL_TOGGLE_HIT : FL_TOGGLE_SELECT);
          if (tnode == this->selected_node_ && this->state_ == FL_TOGGLE_SELECT)
            {
            this->state_ = FL_TOGGLE_RESELECT;
            }
          // this->selected_node_ = tnode;
          if (this->state_==FL_TOGGLE_RESELECT && this->edit_on_reselect_)
            {
            this->edit( tnode,
                        this->x() + depth*s_icon_size + this->label_offset_,
                        cy );
            }
          this->do_callback();
          }

        this->damaged_node_ = (Fl_Toggle_Node_Base *) 0;
        this->damage(FL_DAMAGE_CHILD);

        prev = tnode;
        }
      } break; // FL_RELEASE

    case FL_KEYBOARD:
    case FL_SHORTCUT:
    default:
      {
      return Fl_Toggle_Tree_Base::handle(aEvent);
      }

    } // switch (aEvent)

  return 1;
}

// ----------------------------------------------------------------------------
Fl_Toggle_Node*
Fl_Toggle_Tree::selection_next (void)
{
  if (this->selected_node_ == (Fl_Toggle_Node_Base *) 0)
    {
    this->selected_node_ = this->traverse_start();
    }
  else
    {
    this->traverse_start(this->selected_node_);
    this->selected_node_ = this->traverse_forward();
    }

  while (this->selected_node_ != (Fl_Toggle_Node_Base *) 0)
    {
    if ((static_cast<Fl_Toggle_Node*>(this->selected_node_))->selected_)
      {
      return static_cast<Fl_Toggle_Node*>(this->selected_node_);
      }
    this->selected_node_ = this->traverse_forward();
    }

  return (Fl_Toggle_Node *) 0;
}

int
Fl_Toggle_Tree::selection_count (void)
{
  if (this->selection_node_ == (Fl_Toggle_Node *) 0)
    {
    this->selection_count_ = 0;
    }

  if (this->selection_count_ != 0)
    {
    return this->selection_count_;
    }

  Fl_Toggle_Node* t;
  t = this->traverse_start();

  while (t != (Fl_Toggle_Node *) 0)
    {
    if (t->selected_)
      {
      this->selection_count_++;
      }
    t = this->traverse_forward();
    }

  return this->selection_count_;
}

Fl_Toggle_Node*
Fl_Toggle_Tree::selection_node (int aIndex)
{
  int backwards = 0;

  if (this->selection_node_ == (Fl_Toggle_Node *) 0)
    {
    this->selection_node_ = this->traverse_start();
    this->selection_index_ = 0;
    }
  else
    {
    this->traverse_start(this->selection_node_);
    if (aIndex > this->selection_index_)
      {
      this->selection_node_ = this->traverse_forward();
      this->selection_index_++;
      }
    else if (aIndex < this->selection_index_)
      {
      this->selection_node_ = this->traverse_backward();
      this->selection_index_--;
      backwards = 1;
      }
    }

  if (backwards)
    {
    while (this->selection_node_ != (Fl_Toggle_Node *) 0)
      {
      if (this->selection_node_->selected_)
        {
        if (aIndex == this->selection_index_)
          {
          return this->selection_node_;
          }
        this->selection_index_--;
        }
      this->selection_node_ = this->traverse_backward();
      }
    }
  else
    {
    while (this->selection_node_ != (Fl_Toggle_Node *) 0)
      {
      if (this->selection_node_->selected_)
        {
        if (aIndex == this->selection_index_)
          {
          return this->selection_node_;
          }
        this->selection_index_++;
        }
      this->selection_node_ = this->traverse_forward();
      }
    }

  return (Fl_Toggle_Node *) 0;
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::opened_icon (Fl_Image* aImage)
{
  if ( this->opened_icon_ != (Fl_Image *) 0 &&
       this->opened_icon_ != (*(Fl_Toggle_Tree::s_icon_map_))["opened"] )
    {
    delete this->opened_icon_;
    }
  this->opened_icon_ = aImage;
}

void
Fl_Toggle_Tree::closed_icon (Fl_Image* aImage)
{
  if ( this->closed_icon_ != (Fl_Image *) 0 &&
       this->closed_icon_ != (*(Fl_Toggle_Tree::s_icon_map_))["closed"] )
    {
    delete this->closed_icon_;
    }
  this->closed_icon_ = aImage;
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree::remove (const char* aLabel)
{
  Fl_Toggle_Node* node;

  if ((node = this->find(aLabel)) != (Fl_Toggle_Node *) 0)
    {
    return this->remove(node);
    }

  return 0;
}

int
Fl_Toggle_Tree::remove (void* aPtr)
{
  Fl_Toggle_Node* node;

  if ((node = this->find(aPtr)) != (Fl_Toggle_Node *) 0)
    {
    return this->remove(node);
    }

  return 0;
}

// ----------------------------------------------------------------------------
Fl_Toggle_Node*
Fl_Toggle_Tree::find (const char* a)
{
  const char* label;

  for ( Fl_Toggle_Node* node = this->traverse_start();
        node != (Fl_Toggle_Node *) 0;
        node = this->traverse_forward() )
    if ((label = node->label()) != (char *) 0 && strcmp(label, a) == 0)
      return node;

  return (Fl_Toggle_Node *) 0;
}

Fl_Toggle_Node*
Fl_Toggle_Tree::find (void* a)
{
  void* data;

  for ( Fl_Toggle_Node* node = this->traverse_start();
        node != (Fl_Toggle_Node *) 0;
        node = this->traverse_forward() )
    if ((data = node->user_data()) != (void *) 0 && data == a)
      return node;

  return (Fl_Toggle_Node *) 0;
}

// ----------------------------------------------------------------------------
int
Fl_Toggle_Tree::sort_by_label( Fl_Toggle_Node_Base* aNodeA,
                               Fl_Toggle_Node_Base* aNodeB )
{
  return strcmp( (static_cast<Fl_Toggle_Node*>(aNodeA))->label(),
                 (static_cast<Fl_Toggle_Node*>(aNodeB))->label() );
}

// ----------------------------------------------------------------------------
void
Fl_Toggle_Tree::edit_callback(Fl_Callback* aCallback, void* aPtr)
{
  this->edit_input_->callback((Fl_Callback *) aCallback, aPtr);
}

void
Fl_Toggle_Tree::edit_callback(Fl_Callback* aCallback)
{
  this->edit_input_->callback((Fl_Callback *) aCallback);
}

void
Fl_Toggle_Tree::edit_callback(Fl_Callback0* aCallback)
{
  this->edit_input_->callback(aCallback);
}

void
Fl_Toggle_Tree::edit_callback(Fl_Callback1* aCallback, long aArg)
{
  this->edit_input_->callback(aCallback, aArg);
}

/* 
 * End of: $Id: Fl_Toggle_Tree.cxx,v 1.1.1.1 2006/12/19 22:59:53 christianh Exp $.
 * 
 */
