/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Fl_VTK_View.H,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef _FL_VTK_VIEW_H_
#  define _FL_VTK_VIEW_H_
#  include "Fl_VTK_Window.H"
// vtkExtensions
#  include "vtkRenderingContextFwd.h"

/** \internal
 * hack alert!!! ... for VTK event handler callback (not currently used).
 */
#  define FL_VTK_VIEW (FL_VTK_WINDOW+1) // all subclasses have type() >= this
#  define FL_VTK_VIEW_XCLASS "vtkfltk, Vtkfltk" 


class vtkObject;
class vtkProp;
class vtkAssembly;
class vtkCamera;
class vtkRenderer;
class vtkRendererCollection;
class vtkLightCollection;


/** \class   Fl_VTK_View
 *  \brief   Extended FLTK event-driven interface to a RenderWindow.
 * 
 * User interface window peer class integrating a FLTK event-driven interface 
 * to a VTK pipeline. Fl_VTK_View provides methods for saving the contents of 
 * its RenderWindow in addition to its ancestor methods.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:43 $
 * 
 * \sa
 * Fl_VTK_Window vtkFileType
 */

class Fl_VTK_View : public Fl_VTK_Window
{
  /** \internal
   * constructor internals
   */
  void _Fl_VTK_View (void);

protected:
  // Fl_VTK_Window overrides
  int  handle (int event);

public:
  /** Get the Rendering Context. */
  VTK_EXTENSIONS_NAMESPACE_QUALIFIER vtkRenderingContext* GetRenderingContext(void);

  /** Return the Collection of Renderers in the RenderWindow. */
  vtkRendererCollection* GetRenderers (void);

  /** Get the first Renderer held in the RenderWindow's Collection. */
  vtkRenderer* GetDefaultRenderer (void);

  /** Get the active Camera held by the default Renderer. */
  vtkCamera* GetDefaultCamera (void);

  /*@{*/
  /** Add/Remove an Object to/from the RenderWindow. */
  void  Add (vtkObject* object);
  void  Remove (vtkObject* object);
  /*@}*/

  /*@{*/
  /** Add/Remove a Prop to the current Renderer. */
  void  AddProp (vtkProp* prop);
  void  RemoveProp (vtkProp* prop);
  /*@}*/

  /*@{*/
  /** Add/Remove a Renderer to/from the RenderWindow. */
  void  AddRenderer (vtkRenderer* renderer);
  void  RemoveRenderer (vtkRenderer* renderer);
  /*@}*/

  /*@{*/
  /** Get/Set the current Renderer in the RendererCollection. */
  vtkRenderer*          GetCurrentRenderer (void);
  void                  SetCurrentRenderer (vtkRenderer* renderer);
  /*@}*/

  /** Get the active Camera held by the CurrentRenderer. */
  vtkCamera*            GetCurrentCamera (void);

  /** Get the current Renderer's LightCollection. */
  vtkLightCollection*   GetCurrentLights (void);

  /** Reset the active camera of the current Renderer. */
  void                  ResetCurrentCamera (void);

  /*@{*/
  /** Get/Set the current Assembly.
   * Must be in the Renderers' PropCollection or in an Assembly therein.
   */
  vtkAssembly*          GetCurrentAssembly (void);
  void                  SetCurrentAssembly (vtkAssembly* assembly);
  /*@}*/

  /*@{*/
  /** Get/Set the current Prop.
   * Must be in the Renderers' PropCollection or in an Assembly therein.
   */
  vtkProp*              GetCurrentProp (void);
  void                  SetCurrentProp (vtkProp* prop);
  /*@}*/

  /*@{*/
  /** Get/Set file output scaling (may be rounded to integral value).
   * Currently applies only to writing of 2D images and RIB export.
   */
  float                 GetOutputScaling (void);
  void                  SetOutputScaling (float factor);
  /*@}*/

  /** Writes the composite geometric and topological data to a file. */
  virtual void  Write (const char* pathname, int type = 0);

  /** Save the contents of the RenderWindow in a 2D image file.
   * Optionally scales the image by an integral factor. The file format 
   * is parsed from the filename suffix if the optional file type 
   * parameter is omitted. Availible file formats include: Windows BMP, 
   * JPEG, PNG, PNM, Postscript, and TIFF.
   * 
   * Returns 1 on success, 0 on failure.
   */
  virtual int   WriteImage (const char*, int type = 0, int destroy = 0);

  /** Imports a scene from a file.
   * Reads not only geometric and topological data for an object, but may 
   * also specify material properties, lighting, camera parameters, etc. 
   * Not all valid 3D file types offer all of this information.
   * 
   * The file format is parsed from the filename suffix if the optional 
   * file type parameter is omitted. The importer itself is destroyed if the
   * optional destroy flag is non-zero, otherwise, a reference is retained
   * by this view's RenderingContext. Availible file formats include: 
   * 3D Studio 3DS and VRML 2.0.
   * 
   * Returns the Importer's Renderer on success, 0 on failure.
   */
  virtual vtkRenderer* Import (const char*, int type = 0, int destroy = 0);

  /** Exports a scene to a file.
   * Similar to Write(), except that Write() only writes out the 
   * geometric and topological data for an object, while Export() can 
   * write out material properties, lighting, camera parameters, etc. 
   * Not all valid 3D file types write out all of this information. For 
   * example, Wavefront obj files which do not include support for 
   * camera parameters.
   * 
   * The file format is parsed from the filename suffix if the optional 
   * file type parameter is omitted. The exporter itself is destroyed if the
   * optional destroy flag is non-zero, otherwise, a reference is retained
   * by this view's RenderingContext. Availible file formats include: 
   * Wavefront OBJ, RenderMan RIB, OpenInventor 2.0, VRML 2.0, and 
   * Geomview OOGL.
   * 
   * Returns 1 on success, 0 on failure.
   */
  virtual int Export (const char*, int type = 0, int destroy = 0);

  inline int    push_enabled (void) const
    { return this->push_enabled_; }
  inline void   push_enabled (int a)
    { this->push_enabled_ = a; }

  inline void   reset_push_enabled (void)
    { this->push_enabled_ = FL_BUTTONS | FL_SHIFT | FL_CTRL; }
  inline void   clear_push_enabled (int a)
    { this->push_enabled_ &= ~a; }
  inline void   append_push_enabled (int a)
    { this->push_enabled_ |= a; }

public:
  Fl_VTK_View (int w = 256, int h = 256, const char* l = "");
  Fl_VTK_View (int x, int y, int w, int h, const char* l = "");

  ~Fl_VTK_View();

  virtual const char* GetClassName (void) {return "Fl_VTK_View";}

private:
  VTK_EXTENSIONS_NAMESPACE_QUALIFIER vtkRenderingContext* RenderingContext;

  int   push_enabled_;

private:
  Fl_VTK_View (const Fl_VTK_View&);  // Not implemented.
  void operator= (const Fl_VTK_View&);  // Not implemented.
};


#endif /* _FL_VTK_VIEW_H_ */
/* 
 * End of: $Id: Fl_VTK_View.H,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $.
 * 
 */
