/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Fl_VTK_View.cxx,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */

// Fl_VTK_View - extended FLTK event-driven interface to a RenderWindow
// User interface window peer class integrating a FLTK event-driven interface 
// to a VTK pipeline. Fl_VTK_View provides methods for saving the contents 
// of its RenderWindow in addition to its ancestor methods.
#include "Fl_VTK_View.H"
// SV
#include "svvController.h"
// IOStreams
#include <iostream>
// FLTK
#include <FL/Fl.H>
// VTK for SaveAsImage()
#include "vtkImageWriter.h"
// VTK Rendering
#include "vtkProp.h"
#include "vtkProp3D.h"
#include "vtkPropAssembly.h"
#include "vtkImageActor.h"
#include "vtkActor2D.h"
#include "vtkActor.h"
#include "vtkVolume.h"
#include "vtkAssembly.h"
#include "vtkLODProp3D.h"
#include "vtkCamera.h"
#include "vtkRenderer.h"
#include "vtkRendererCollection.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkImporter.h"
#include "vtkExporter.h"

#include "vtkRenderWindowInteractor.h"
#include "vtkAbstractPicker.h"
#include "vtkAssemblyPath.h"
#include "vtkAssemblyNode.h"
#include "vtkAbstractPropPicker.h"
#include "vtkInteractorStyle.h"

// VTK Rendering (in Write())
#include "vtkAxisActor2D.h"
#include "vtkCaptionActor2D.h"
#include "vtkCubeAxesActor2D.h"
#include "vtkLegendBoxActor.h"
#include "vtkParallelCoordinatesActor.h"
#include "vtkScalarBarActor.h"
#include "vtkTextActor.h"
#include "vtkTextProperty.h"
#include "vtkXYPlotActor.h"
#include "vtkFollower.h"
#include "vtkLODActor.h"
// vtkExtensions
#include "vtkRenderingContext.h"

#ifdef SVV_SUPPORT_MRML
// vtkExtensions
#  include "vtkFileTypeEnum.h"
// vtkMrml
#  include "vtkMrml.h"
#  include "vtkMrmlRegistry.h"
#  include "vtkMrmlImporter.h"
#  include "vtkMrmlExporter.h"
#endif /* SVV_SUPPORT_MRML */

VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);
SVV_NAMESPACE_USING(svv);

#ifdef SVV_SUPPORT_MRML
VTK_MRML_NAMESPACE_USING(vtkMrml);
VTK_MRML_NAMESPACE_USING(vtkMrmlRegistry);
VTK_MRML_NAMESPACE_USING(vtkMrmlImporter);
VTK_MRML_NAMESPACE_USING(vtkMrmlExporter);
#endif /* SVV_SUPPORT_MRML */

//---------------------------------------------------------------------------
void
Fl_VTK_View::_Fl_VTK_View (void)
{
  this->type(FL_VTK_VIEW);
  this->xclass(FL_VTK_VIEW_XCLASS);

  this->RenderingContext = vtkRenderingContext::New();
  //this->RenderingContext->DebugOn();
  this->RenderingContext->Initialize(this->Interactor,this->RenderWindow);
  this->RenderingContext->SetQueuedRendererPointer(&(this->QueuedRenderer));

#ifdef SVV_SUPPORT_MRML
  this->RenderingContext->
    AddImporterCreator( int(VTK_EXTENSIONS_NAMESPACE_QUALIFIER eMRMLFileType),
                        vtkMrmlImporter::Create );
  this->RenderingContext->
    AddExporterCreator( int(VTK_EXTENSIONS_NAMESPACE_QUALIFIER eMRMLFileType),
                        vtkMrmlExporter::Create );
#endif /* SVV_SUPPORT_MRML */
}

//---------------------------------------------------------------------------
Fl_VTK_View::Fl_VTK_View (int w, int h, const char* l)
  : Fl_VTK_Window(w,h,l),
    push_enabled_(FL_BUTTONS | FL_SHIFT | FL_CTRL)
{
  this->_Fl_VTK_View();
}

Fl_VTK_View::Fl_VTK_View (int x, int y, int w, int h, const char* l)
  : Fl_VTK_Window(x,y,w,h,l),
    push_enabled_(FL_BUTTONS | FL_SHIFT | FL_CTRL)
{
  this->_Fl_VTK_View();
}

//---------------------------------------------------------------------------
Fl_VTK_View::~Fl_VTK_View()
{
  // according to the fltk docs, destroying a widget does NOT remove it from
  // its parent, so we have to do that explicitly at destruction
  if (this->parent() != NULL)
    {
    (static_cast<Fl_Group*>(this->parent()))->
      remove(*(static_cast<Fl_Widget*>(this)));
    }

  if (this->RenderingContext != NULL)
    {
    this->RenderingContext->Delete();
    }
}

// ----------------------------------------------------------------------------
vtkRenderingContext*
Fl_VTK_View::GetRenderingContext (void)
{
  return this->RenderingContext;
}

// ----------------------------------------------------------------------------
vtkRendererCollection*
Fl_VTK_View::GetRenderers (void)
{
  return this->RenderingContext->GetRenderers();
}

// ----------------------------------------------------------------------------
vtkRenderer*
Fl_VTK_View::GetDefaultRenderer (void)
{
  return this->RenderingContext->GetDefaultRenderer();
}

vtkCamera*
Fl_VTK_View::GetDefaultCamera (void)
{
  return this->RenderingContext->GetDefaultCamera();
}

// ----------------------------------------------------------------------------
void
Fl_VTK_View::Add (vtkObject* aObject)
{
  vtkProp*     prop;
  vtkRenderer* renderer;

  if      ((prop = vtkProp::SafeDownCast(aObject)) != NULL)
    {
    this->RenderingContext->AddProp(prop);
    }
  else if ((renderer = vtkRenderer::SafeDownCast(aObject)) != NULL)
    {
    this->RenderingContext->AddRenderer(renderer);
    }
  else
    {
    vtkGenericWarningMacro(<< "Add(" << aObject << ") failed to add unexpected"
                    << " object of type '" << aObject->GetClassName() << "'.");
    
    }
}

void
Fl_VTK_View::Remove (vtkObject* aObject)
{
  vtkProp*     prop;
  vtkRenderer* renderer;

  if      ((prop = vtkProp::SafeDownCast(aObject)) != NULL)
    {
    this->RenderingContext->RemoveProp(prop);
    }
  else if ((renderer = vtkRenderer::SafeDownCast(aObject)) != NULL)
    {
    vtkRenderWindow*       renderWindow;
    vtkRendererCollection* rendererCollection;

    if ( (renderWindow = this->RenderingContext->GetRenderWindow()) &&
         (rendererCollection = renderWindow->GetRenderers()) )
      {
      int pos;

      if ((pos = rendererCollection->IsItemPresent(renderer)) > 0)
        {
        renderer->RemoveAllProps();
        this->RenderingContext->RemoveRenderer(renderer);
        }
      else
        {
        vtkGenericWarningMacro(<< "Remove(" << aObject << ") failed to find"
                               << " Renderer in current RenderWindow.");
        }
      }
    else
      {
      vtkGenericWarningMacro(<< "Remove(" << aObject << ") failed to find"
                             " Renderer collection for current RenderWindow.");
      }
    }
  else
    {
    vtkGenericWarningMacro(<< "Remove(" << aObject << ") failed to remove"
                           << " unexpected object of type '"
                           << aObject->GetClassName() << "'.");
    
    }
}

// ----------------------------------------------------------------------------
void
Fl_VTK_View::AddRenderer (vtkRenderer* aRenderer)
{
  this->RenderingContext->AddRenderer(aRenderer);
}

void
Fl_VTK_View::RemoveRenderer (vtkRenderer* aRenderer)
{
  this->RenderingContext->RemoveRenderer(aRenderer);
}

vtkRenderer*
Fl_VTK_View::GetCurrentRenderer (void)
{
  return this->RenderingContext->GetCurrentRenderer();
}

void
Fl_VTK_View::SetCurrentRenderer (vtkRenderer* aRenderer)
{
  this->RenderingContext->SetCurrentRenderer(aRenderer);
}

// ----------------------------------------------------------------------------
vtkCamera*
Fl_VTK_View::GetCurrentCamera (void)
{
  return this->RenderingContext->GetCurrentCamera();
}

vtkLightCollection*
Fl_VTK_View::GetCurrentLights (void)
{
  return this->RenderingContext->GetCurrentLights();
}

void
Fl_VTK_View::ResetCurrentCamera (void)
{
  this->RenderingContext->ResetCurrentCamera();
}

// ----------------------------------------------------------------------------
void
Fl_VTK_View::AddProp (vtkProp* aProp)
{
  this->RenderingContext->AddProp(aProp);
}

void
Fl_VTK_View::RemoveProp (vtkProp* aProp)
{
  this->RenderingContext->RemoveProp(aProp);
}

// ----------------------------------------------------------------------------
vtkAssembly*
Fl_VTK_View::GetCurrentAssembly (void)
{
  return this->RenderingContext->GetCurrentAssembly();
}

void
Fl_VTK_View::SetCurrentAssembly (vtkAssembly* pAssembly)
{
  this->RenderingContext->SetCurrentAssembly(pAssembly);
}

// ----------------------------------------------------------------------------
vtkProp*
Fl_VTK_View::GetCurrentProp (void)
{
  return this->RenderingContext->GetCurrentProp();
}

void
Fl_VTK_View::SetCurrentProp (vtkProp* aProp)
{
  this->RenderingContext->SetCurrentProp(aProp);
}

// ----------------------------------------------------------------------------
float
Fl_VTK_View::GetOutputScaling (void)
{
  return this->RenderingContext->GetOutputScaling();
}

void
Fl_VTK_View::SetOutputScaling (float scale)
{
  this->RenderingContext->SetOutputScaling(scale);
}

// ----------------------------------------------------------------------------
void
Fl_VTK_View::Write (const char* aFileName, int aType)
{
  vtkDebugMacro(<<"Write(\"" << aFileName << "\", " << aType << ")");

  vtkProp* prop;

  if ((prop = this->RenderingContext->GetCurrentProp()) == NULL)
    {
    vtkProp3D*            prop3D;
    vtkPropAssembly*      propAssembly;
    vtkImageActor*        imageActor;
    vtkActor2D*           actor2D;

    if      ((prop3D = vtkProp3D::SafeDownCast(prop)) != NULL)
      {
      vtkActor*           actor;
      vtkVolume*          volume;
      vtkAssembly*        assembly;
      vtkLODProp3D*       LODProp3D;

      if      ((actor = vtkActor::SafeDownCast(prop3D)) != NULL)
        {
        vtkFollower* follower;
        vtkLODActor* LODActor;
        
        if      ((follower = vtkFollower::SafeDownCast(actor)) != NULL)
          {
          // ...
          }
        else if ((LODActor = vtkLODActor::SafeDownCast(actor)) != NULL)
          {
          // ...
          }
        else // vtkMesaActor or vtkOpenGLActor
          {
          // ...
          }
        }
      else if ((volume = vtkVolume::SafeDownCast(prop3D)) != NULL)
        {
        // ...
        }
      else if ((assembly = vtkAssembly::SafeDownCast(prop3D)) != NULL)
        {
        // ...
        }
      else if ((LODProp3D = vtkLODProp3D::SafeDownCast(prop3D)) != NULL)
        {
        // ...
        }
      }
    else if ((propAssembly = vtkPropAssembly::SafeDownCast(prop)) != NULL)
      {
      // ...
      }
    else if ((imageActor = vtkImageActor::SafeDownCast(prop)) != NULL)
      {
      // ...
      }
    else if ((actor2D = vtkActor2D::SafeDownCast(prop)) != NULL)
      {
      vtkAxisActor2D*              axisActor2D;
      vtkCaptionActor2D*           captionActor2D;
      vtkCubeAxesActor2D*          cubeAxesActor2D;
      vtkLegendBoxActor*           legendBoxActor;
      vtkParallelCoordinatesActor* parallelCoordinatesActor;
      vtkScalarBarActor*           scalarBarActor;
      vtkTextActor*                textActor;
      vtkXYPlotActor*              xyPlotActor;

      if      ( ( axisActor2D =
                  vtkAxisActor2D::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( captionActor2D =
                  vtkCaptionActor2D::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( cubeAxesActor2D =
                  vtkCubeAxesActor2D::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( legendBoxActor =
                  vtkLegendBoxActor::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( parallelCoordinatesActor =
                  vtkParallelCoordinatesActor::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( scalarBarActor =
                  vtkScalarBarActor::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( textActor =
                  vtkTextActor::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      else if ( ( xyPlotActor =
                  vtkXYPlotActor::SafeDownCast(actor2D) ) != NULL )
        {
        // ...
        }
      }
    }
}

// ----------------------------------------------------------------------------
int
Fl_VTK_View::WriteImage (const char* aFileName, int aType, int aDestroyFlag)
{
  vtkDebugMacro(<< "WriteImage(\"" << aFileName << "\", " << aType << ")");

  vtkImageWriter* imageWriter;

  if ((imageWriter = this->RenderingContext->WriteImage(aFileName,aType))==NULL)
    {
    vtkGenericWarningMacro(<< "Failed to make ImageWriter.");
    return 0; /* failure */
    }

  if (this->window()) this->window()->take_focus();
  else                this->take_focus();

  int interactive;
  vtkRenderer* renderer;

  vtkCubeAxesActor2D*   cubeAxes = NULL;
  vtkProp*              axesProp = NULL;
  int                   axesInertia = 4;
  int                   axesTitleFontSize = 12;
  int                   axesLabelFontSize = 12;
  int                   axesTitleAA = 1;
  int                   axesLabelAA = 1;

  if ((renderer = this->GetCurrentRenderer()) != NULL)
    {
    if ((axesProp = svv.GetAxes(renderer)) != NULL)
      {
      if ((cubeAxes = vtkCubeAxesActor2D::SafeDownCast(axesProp)) != NULL)
        {
        axesInertia = cubeAxes->GetInertia();
        axesTitleFontSize = cubeAxes->GetAxisTitleTextProperty()->GetFontSize();
        axesLabelFontSize = cubeAxes->GetAxisLabelTextProperty()->GetFontSize();
        axesTitleAA = cubeAxes->GetAxisTitleTextProperty()->GetAntiAliasing();
        axesLabelAA = cubeAxes->GetAxisLabelTextProperty()->GetAntiAliasing();
      
        cubeAxes->SetInertia(0x7FFFFFFF);
        cubeAxes->GetAxisTitleTextProperty()->SetFontSize(32);
        cubeAxes->GetAxisLabelTextProperty()->SetFontSize(32);
        cubeAxes->GetAxisTitleTextProperty()->SetAntiAliasing(1);
        cubeAxes->GetAxisLabelTextProperty()->SetAntiAliasing(1);
        }
      }

    // save Interactive state to be restored after the save.
    interactive = renderer->GetInteractive();
    renderer->InteractiveOff();
    }

  int aaFrames = this->RenderWindow->GetAAFrames();

  this->RenderWindow->SetAAFrames(7);

  // Write scalar data to output.
  imageWriter->Write();

  // get up-to-date
  this->RenderWindow->SetAAFrames(aaFrames);
  this->RenderWindow->Render();

  // Image writer still held by the RenderingContext unless we destroy it here.
  if (aDestroyFlag != 0)
    {
    vtkDebugMacro(<< "Destroying ImageWriter.");
    // destroy the image writer
    this->RenderingContext->WriteImage(NULL);
    imageWriter = NULL;
    }

  if (renderer != NULL)
    {
    // Restore saved Interactive state.
    renderer->SetInteractive(interactive);

    if (cubeAxes != NULL)
      {
      cubeAxes->SetInertia(axesInertia);
      cubeAxes->GetAxisTitleTextProperty()->SetFontSize(axesTitleFontSize);
      cubeAxes->GetAxisLabelTextProperty()->SetFontSize(axesLabelFontSize);
      cubeAxes->GetAxisTitleTextProperty()->SetAntiAliasing(axesTitleAA);
      cubeAxes->GetAxisLabelTextProperty()->SetAntiAliasing(axesLabelAA);
      }
    }

  return 1; /* success */
} // WriteImage()

// ----------------------------------------------------------------------------
vtkRenderer*
Fl_VTK_View::Import (const char* aFileName, int aType, int aDestroyFlag)
{
  vtkDebugMacro(<< "Import(\"" << aFileName << "\", " << aType
                << ", " << aDestroyFlag << ")");

  vtkImporter* importer;

  int debug = this->RenderingContext->GetDebug();
  this->RenderingContext->DebugOn();

  if ((importer = this->RenderingContext->Import(aFileName,aType)) == NULL)
    {
    vtkGenericWarningMacro(<< "Import(): failed to create Importer.");
    return NULL; /* failure */
    }

  importer->DebugOn();

  // Read data from input.
  importer->Read();

  // importer->Print(vtkstd::clog);

  vtkRenderer* renderer;

  // Get the renderer that contains the imported actors, cameras and lights.
  if ((renderer = importer->GetRenderer()) == NULL)
    {
    vtkGenericWarningMacro(<< "Import(): failed to get Renderer!");
    // Forces destruction of the exporter.
    this->RenderingContext->Import(NULL);
    importer = NULL;
    }

  // Importer still held by the RenderingContext unless we destroy it here.
  if (aDestroyFlag != 0)
    {
    vtkDebugMacro(<< "Import(): destroying Importer.");
    // Forces destruction of the exporter.
    this->RenderingContext->Import(NULL);
    importer = NULL;
    }

  this->Update();
  this->ResetCurrentCamera();

  vtkDebugMacro(<< "Returning imported Renderer (" << renderer << ").");

  this->RenderingContext->SetDebug(debug);

  return renderer;
}

// ----------------------------------------------------------------------------
int
Fl_VTK_View::Export (const char* aFileName, int aType, int aDestroyFlag)
{
  vtkDebugMacro(<< "Export(\"" << aFileName << "\", " << aType
                << ", " << aDestroyFlag << ")");

  vtkExporter* exporter;

  int debug = this->RenderingContext->GetDebug();
  this->RenderingContext->DebugOn();

  if ((exporter = this->RenderingContext->Export(aFileName,aType)) == NULL)
    {
    vtkGenericWarningMacro(<< "Export(): Failed to make Exporter.");
    return 0; /* failure */
    }

  exporter->DebugOn();

#ifdef SVV_SUPPORT_MRML
  vtkMrmlExporter* mrmlExporter;
  if ((mrmlExporter = vtkMrmlExporter::SafeDownCast(exporter)) != NULL)
    {
    const char* mrmlKey = (char *) 0;
    vtkMrml*    impl;
    
    mrmlExporter->SaveColorsOn();
    mrmlExporter->Build();
    mrmlKey = mrmlExporter->GetMrmlKey();

    if ((impl = vtkMrmlRegistry::GetImplementation(mrmlKey)) != (vtkMrml *) 0)
      {
      
      }
    }
#endif /* SVV_SUPPORT_MRML */

  vtkRenderer* renderer;
  int interactive;
  
  if ((renderer = this->RenderingContext->GetCurrentRenderer()) == NULL)
    {
    vtkGenericWarningMacro(<< "Export(): Could not get Renderer!");
    }
  else
    {
    // save Interactive state to be restored after the export.
    interactive = renderer->GetInteractive();
    renderer->InteractiveOff();
    }
  
  // Write data to output.
  exporter->Update();
  
  if (aDestroyFlag != 0)
    {
    vtkDebugMacro(<< "Export(): Destroying Exporter.");
    // Forces destruction of the exporter.
    this->RenderingContext->Export(NULL);
    exporter = NULL;
    }
  
  if (renderer != NULL)
    {
    // restore saved Interactive state
    renderer->SetInteractive(interactive);
    }

  this->RenderingContext->SetDebug(debug);
  
  return 1; /* success */
}

//---------------------------------------------------------------------------
// main FLTK event handler
int
Fl_VTK_View::handle (int aEvent)
{
  if (aEvent!=FL_MOVE && aEvent!=FL_DRAG)
    {
    // // if printing type of event
    // vtkDebugMacro(<< "Fl_VTK_View::handle( " << aEvent << " = "
    //               << Fl_VTK_Window::GetEventAsString(aEvent) << " )");
    }

  vtkRenderWindowInteractor* rwi = this->Interactor;
  vtkInteractorObserver*     obs = rwi->GetInteractorStyle();
  vtkAbstractPicker*         ap  = rwi->GetPicker();

  int enabled = rwi->GetEnabled();

  switch (aEvent)
    {

    case FL_PUSH:
      {
      if (!enabled || (this->push_enabled() & Fl::event_state()))
        break;

      vtkInteractorStyle*       style;

      if ((style = vtkInteractorStyle::SafeDownCast(obs)) != NULL)
        {
        rwi->SetEventInformationFlipY( Fl::event_x(),
                                       Fl::event_y(),
                                       Fl::event_state(FL_CTRL),
                                       Fl::event_state(FL_SHIFT) );

        style->FindPokedRenderer( rwi->GetEventPosition()[0],
                                  rwi->GetEventPosition()[1] );

        rwi->StartPickCallback();

        ap->Pick( rwi->GetEventPosition()[0],
                  rwi->GetEventPosition()[1],
                  0.0,
                  style->GetCurrentRenderer() );

        vtkDebugMacro(<< "Fl_VTK_View(" << this << ")::handle(): "
                      << ap->GetClassName() << "(" << ap
                      << ")->Pick(" << rwi->GetEventPosition()[0] << ", "
                      << rwi->GetEventPosition()[1] << ", 0.0, "
                      << style->GetCurrentRenderer() << ")");

        vtkAssemblyPath*       path = NULL;
        vtkAbstractPropPicker* propPicker;

        if ( (propPicker = vtkAbstractPropPicker::SafeDownCast(ap)) != NULL &&
             (path = propPicker->GetPath()) != NULL )
          {
          style->HighlightProp(path->GetFirstNode()->GetProp());

          vtkDebugMacro(<< "Fl_VTK_View(" << this << ")::handle(): "
                        << style->GetClassName() << "("
                        << style << ")->HighlightProp("
                        << path->GetFirstNode()->GetProp() << ")");
          }
        else
          {
          style->HighlightProp(NULL);
          }

        rwi->EndPickCallback();

        return 1;
        }
      
      return Fl_Gl_Window::handle(aEvent);
      } // FL_KEYDOWN

    case FL_RELEASE:
      {
      if (!enabled) break;
      
      return Fl_VTK_Window::handle(aEvent);
      } // FL_KEYDOWN

    case FL_KEYDOWN: // KeyPress
      {
      if (!enabled) break;
      
      return Fl_VTK_Window::handle(aEvent);
      } // FL_KEYDOWN
    
    case FL_KEYUP: // KeyRelease
      {
      if (!enabled) break;
      
      return Fl_VTK_Window::handle(aEvent);
      } // FL_KEYUP

    } // switch (aEvent)

      // let the base class handle everything else 
  return Fl_VTK_Window::handle(aEvent);
} // handle()

/* 
 * End of: $Id: Fl_VTK_View.cxx,v 1.1.1.1 2006/12/19 22:59:43 christianh Exp $.
 * 
 */
