/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMessagesUI.h,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_MESSAGES_UI_H_
#  define SVV_MESSAGES_UI_H_
#  include "svvConfiguration.h"
// FLTK
#  include <FL/Fl.H>
#  include <FL/Fl_Window.H>
#  include <FL/Fl_Browser.H>
#  include <FL/Fl_Box.H>
#  include <FL/Fl_Button.H>
#  include <FL/Fl_Menu_Button.H>
// STL
#  include <sstream>
#  include <string>

#  define svvLogMacro(_level, _msg)     { std::stringstream ss;         \
                                          ss _msg;                      \
                                          SvvMsg.log((_level), ss.str()); }

#  define svvMessage(_msg)      svvLogMacro(svvMessagesUI::DIRECT, _msg)
#  define svvFatal(_msg)        svvLogMacro(svvMessagesUI::FATAL, _msg)
#  define svvError(_msg)        svvLogMacro(svvMessagesUI::ERROR, _msg)
#  define svvWarning(_msg)      svvLogMacro(svvMessagesUI::WARNING, _msg)
#  define svvInfo(_msg)         svvLogMacro(svvMessagesUI::INFO, _msg)
#  define svvStatus(_msg)       svvLogMacro(svvMessagesUI::STATUS, _msg)
#  define svvLogFile(_msg)      svvLogMacro(svvMessagesUI::LOGFILE, _msg)

#  ifdef SVV_DEBUG
#    define svvDebug(_msg)      svvLogMacro(svvMessagesUI::DEBUG, _msg)
#  else
#    define svvDebug(_msg)      ((void *) 0)
#  endif /* SVV_DEBUG */

/** \class   svvMessagesBrowser
 *  \brief   Output-end of the widget-stream.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:44 $
 * 
 * \sa Fl_Browser
 */
class svvMessagesBrowser : public Fl_Browser 
{
  char  CBuffer[1024];
  char  SnBuffer[1024];

public:
  void PutC (int c);
  void PutSn (const char* s, int n);

  svvMessagesBrowser (int x, int y, int w, int h);
};

/** \class   svvMessagesUI
 *  \brief   Messages composite widget.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:59:44 $
 * 
 * \sa svvMessagesBrowser
 */
class svvMessagesUI : public Fl_Window
{
private:
  svvMessagesBrowser    mBrowser;
  Fl_Button             mHelpButton;
  Fl_Box                mStatusBox;
  Fl_Menu_Button        mLevelButton;
  Fl_Button             mSaveButton;
  Fl_Button             mClearButton;
  Fl_Button             mCloseButton;
  Fl_Menu_Button        mLevelPopup;
  Fl_End                mEnd;

public:
  enum
  {
    NONE        = 0x00, /*!< Null Level. */
    DIRECT      = 0x01, /*!< Direct message (no special formatting). */
    FATAL       = 0x02, /*!< Fatal error (causes SVV to exit). */
    ERROR       = 0x04, /*!< Error (but SVV can live with it). */
    WARNING     = 0x08, /*!< Warning. */
    INFO        = 0x10, /*!< Long information. */
    DEBUG       = 0x20, /*!< Long debug information. */
    STATUS      = 0x40, /*!< Small information in status bar. */
    LOGFILE     = 0x80  /*!< Info put only in the logfile. */
  };

private:
  uchar                 threshold_;
  long                  next_save_start_;
  char                  filename_[1024];
  char                  status_[1024];

  static Fl_Menu_Item   LevelMenuItems[];

protected:
  svvMessagesUI (void);

  static char   ErrorFileName[1024];

  static void   ClassInitialize (void);
  static void   ClassFinalize (void);

  static void   BrowserCb (Fl_Browser*, void*);
  static void   HelpCb (Fl_Button*, void*);
  static void   LevelCb (Fl_Menu_*, void*);
  static void   SaveCb (Fl_Button*, void*);
  static void   ClearCb (Fl_Button*, void*);
  static void   CloseCb (Fl_Button*, void*);

public:
  /**  */
  void  threshold (uchar level);
  uchar threshold (void) const;

  /** Add a message at the specified severity level. */
  void  log (uchar level, const std::string& message);

  static void error (const char* format, ...);
  static void fatal (const char* format, ...);
  static void warning (const char* format, ...);

  /** Save the messages, possibly appending, to a file. */
  void  save (void);
  void  save (char fileName[1024]);
  void  save (const std::string& fileName);

  /** Clear the currently stored messages. */
  void  clear (void);

  svvMessagesBrowser& browser (void) { return this->mBrowser; }

public:
  /** \internal Nested initializer class. */
  class Init
  {
  public:
    Init (void);
    ~Init();
  private:
    static long TranslationUnits;
    friend class svvMessagesUI;
  };

  friend class Init;

private:
  svvMessagesUI (const svvMessagesUI&); // Not implemented.;
  void operator= (const svvMessagesUI&); // Not implemented.;
};

#  ifndef SVV_MESSAGES_UI_INITIALIZED_
#    define SVV_MESSAGES_UI_INITIALIZED_
/** \internal Global initializer; ensure construction of static objects. */
static svvMessagesUI::Init svvMessagesUISingletonInitInstance;
#  endif /* SVV_MESSAGES_UI_INITIALIZED_ */

/** Singleton instance reference. */
extern svvMessagesUI& SvvMsg;

#endif /* SVV_MESSAGES_UI_H_ */
/*
 * End of: $Id: svvMessagesUI.h,v 1.1.1.1 2006/12/19 22:59:44 christianh Exp $.
 *
 */
