/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkAbstractSurfacePropertiesUI.cxx,v 1.1.1.1 2006/12/19 22:59:46 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "vtkSurfacePropertiesUI.h"
// C++ forwarding ANSI C
#include <cmath>        /* rint(3M) */
// SVV
#include "Fl_VTK_View.H"
#include "Fl_Color_Chooser_A.H"
#include "svvModelSource.h"
#include "svvMaterialNameMap.h"
// FLTK
#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Tabs.H>
#include <FL/Fl_Widget.H>
#include <FL/Fl_Group.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Menu_.H>
#include <FL/Fl_Menu_Item.H>
// vtkExtensions
#include "vtkRenderingContext.h"
// VTK Rendering
#include "vtkActor.h"
#include "vtkProperty.h"
#include "vtkPolyDataMapper.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
// VTK Hybrid
#include "vtkDepthSortPolyData.h"
// VTK Common
#include "vtkDebugLeaks.h"
#include "vtkObjectFactory.h"

#include "vtkDataSetToDataSetFilter.h"
#include "vtkTransformTextureCoords.h"

VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);
SVV_NAMESPACE_USING(svvMaterialNameMap);
SVV_NAMESPACE_USING(svvModelSource);

// ----------------------------------------------------------------------------
//      v t k A b s t r a c t S u r f a c e P r o p e r t i e s U I
// ----------------------------------------------------------------------------
vtkCxxRevisionMacro (vtkAbstractSurfacePropertiesUI, "$Revision: 1.1.1.1 $");
vtkInstantiatorNewMacro (vtkAbstractSurfacePropertiesUI);

// ----------------------------------------------------------------------------
vtkAbstractSurfacePropertiesUI*
vtkAbstractSurfacePropertiesUI::New (void)
{
  vtkObject* result =
    vtkObjectFactory::CreateInstance("vtkAbstractSurfacePropertiesUI");

  if (result != NULL)
    {
    return static_cast<vtkAbstractSurfacePropertiesUI*>(result);
    }

  vtkDebugLeaks::DestructClass("vtkAbstractSurfacePropertiesUI");

  return vtkSurfacePropertiesUI::New();
}

//----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::ClassInitialize (void)
{
  vtkAbstractSurfacePropertiesUI::NamedMaterials = svvMaterialNameMap::New();
  vtkAbstractSurfacePropertiesUI::NamedMaterials->InsertDefaults();
}

void
vtkAbstractSurfacePropertiesUI::ClassFinalize (void)
{
  vtkAbstractSurfacePropertiesUI::NamedMaterials->Delete();
  vtkAbstractSurfacePropertiesUI::NamedMaterials = NULL;
}

vtkAbstractSurfacePropertiesUI::Init::Init (void)
{
  if(vtkAbstractSurfacePropertiesUI::Init::TranslationUnits++ == 0)
    {
    vtkAbstractSurfacePropertiesUI::ClassInitialize();
    }
}

vtkAbstractSurfacePropertiesUI::Init::~Init()
{
  if(--vtkAbstractSurfacePropertiesUI::Init::TranslationUnits == 0)
    {
    vtkAbstractSurfacePropertiesUI::ClassFinalize();
    }
}

// Purposely not initialized.  ClassInitialize will handle it.
long                    vtkAbstractSurfacePropertiesUI::Init::TranslationUnits;
svvMaterialNameMap*     vtkAbstractSurfacePropertiesUI::NamedMaterials;

// ----------------------------------------------------------------------------
vtkAbstractSurfacePropertiesUI::vtkAbstractSurfacePropertiesUI (void)
  : Initialized (0),
    ModelSource (svvModelSource::New()),
    DemoActor (vtkActor::New()),
    StashedProperty (0),
    OriginalFrontProperty (0),
    OriginalBackProperty (0)
{
  vtkPolyDataMapper* polyDataMapper = vtkPolyDataMapper::New();
  this->DemoActor->SetMapper(polyDataMapper);
  polyDataMapper->Delete();
}

// ----------------------------------------------------------------------------
vtkAbstractSurfacePropertiesUI::~vtkAbstractSurfacePropertiesUI()
{
  if (this->ModelSource != NULL)
    this->ModelSource->UnRegister(this);

  if (this->DemoActor != NULL)
    this->DemoActor->UnRegister(this);

  if (this->StashedProperty != NULL)
    this->StashedProperty->UnRegister(this);

  if (this->OriginalFrontProperty != NULL)
    this->OriginalFrontProperty->UnRegister(this);

  if (this->OriginalBackProperty != NULL)
    this->OriginalBackProperty->UnRegister(this);
}

// ----------------------------------------------------------------------------
vtkCxxSetObjectMacro( vtkAbstractSurfacePropertiesUI, StashedProperty,
                      vtkProperty );

vtkCxxSetObjectMacro( vtkAbstractSurfacePropertiesUI, OriginalFrontProperty,
                      vtkProperty );

vtkCxxSetObjectMacro( vtkAbstractSurfacePropertiesUI, OriginalBackProperty,
                      vtkProperty );

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::SetDemoFrontProperty (vtkProperty* aProperty)
{
  if (this->DemoActor != (vtkActor *) 0)
    this->DemoActor->SetProperty(aProperty);
}

vtkProperty*
vtkAbstractSurfacePropertiesUI::GetDemoFrontProperty (void)
{
  if (this->DemoActor != (vtkActor *) 0)
    return this->DemoActor->GetProperty();

  return (vtkProperty *) 0;
}

void
vtkAbstractSurfacePropertiesUI::SetDemoBackProperty (vtkProperty* aProperty)
{
  if (this->DemoActor != (vtkActor *) 0)
    this->DemoActor->SetBackfaceProperty(aProperty);
}

vtkProperty*
vtkAbstractSurfacePropertiesUI::GetDemoBackProperty (void)
{
  if (this->DemoActor != (vtkActor *) 0)
    return this->DemoActor->GetBackfaceProperty();

  return (vtkProperty *) 0;
}

void
vtkAbstractSurfacePropertiesUI::SetDemoTexture (vtkTexture* aTexture)
{
  if (this->DemoActor != (vtkActor *) 0)
    this->DemoActor->SetTexture(aTexture);
}

vtkTexture*
vtkAbstractSurfacePropertiesUI::GetDemoTexture (void)
{
  if (this->DemoActor != (vtkActor *) 0)
    return this->DemoActor->GetTexture();

  return (vtkTexture *) 0;
}

// ----------------------------------------------------------------------------
vtkActor*
vtkAbstractSurfacePropertiesUI::GetCurrentClientActor (void)
{
  return vtkActor::SafeDownCast(this->GetCurrentClientProp());
}

//-----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::InitializeDemo (void)
{
  vtkRenderWindow*     demoRenderWindow;
  vtkRenderer*         demoRenderer;
  vtkPolyDataMapper*   demoMapper;

  if ((demoRenderWindow = this->GetDemoRenderWindow()) == NULL)
    return;
  if ((demoRenderer     = this->GetCurrentDemoRenderer()) == NULL)
    return;
  if ( ( demoMapper = vtkPolyDataMapper::SafeDownCast( this->DemoActor->
                                                       GetMapper() ) ) == NULL )
    return;

  vtkProperty*         frontProperty = this->DemoActor->GetProperty();
  vtkProperty*         backProperty  = this->DemoActor->GetBackfaceProperty();

  if ( (frontProperty != NULL && (frontProperty->GetOpacity() < 1.f)) ||
       (backProperty  != NULL && (backProperty->GetOpacity()  < 1.f)) )
    {
    vtkDepthSortPolyData* depthSorter = vtkDepthSortPolyData::New();
      {
      depthSorter->SetInput(this->ModelSource->GetOutput());
      depthSorter->SetProp3D(this->DemoActor);
      depthSorter->SetDepthSortModeToParametricCenter();
      depthSorter->SetDirectionToBackToFront();
      depthSorter->SetCamera(demoRenderer->GetActiveCamera());
      }
    demoMapper->SetInput(depthSorter->GetOutput());
    depthSorter->Delete();
    // NOTE: The Mapper "owns" the DepthSorter; i.e. the DepthSorter
    //       will be destroyed should the Mapper input be reset.
    }
  else
    {
    demoMapper->SetInput(this->ModelSource->GetOutput());
    }

  vtkRenderWindow*     clientRenderWindow;
  vtkRenderer*         clientRenderer;

  if (clientRenderWindow = this->GetClientRenderWindow())
    {
    demoRenderWindow->
      SetPointSmoothing(clientRenderWindow->GetPointSmoothing());
    demoRenderWindow->
      SetLineSmoothing(clientRenderWindow->GetLineSmoothing());

    if (clientRenderer = this->GetCurrentClientRenderer())
      {
      demoRenderer->
	SetTwoSidedLighting(clientRenderer->GetTwoSidedLighting());
      }
    }

  if (this->Initialized == 0)
    {
    demoRenderer->AddProp(this->DemoActor);
    demoRenderer->ResetCamera();
    this->InitializedOn();
    }

  // Sync properties with those held by the interface.
  this->UpdateDemoGroups();
}

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::Update (void)
{
  this->Receive();
  this->UpdateDemoRenderingContext();
}

//-----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::Receive (vtkObject* vtkNotUsed(a))
{
  vtkActor*     clientActor = this->GetCurrentClientActor();

  vtkProperty* frontProperty;
  vtkProperty* backProperty = NULL;
  vtkProperty* originalFrontProperty;
  vtkProperty* originalBackProperty = NULL;

  if ((frontProperty = this->GetDemoFrontProperty()) == NULL)
    {
    vtkErrorMacro(<< "Failed to get Front Property.");
    return;
    }

  // clone the property(ies)
  if (clientActor != (vtkActor *) 0)
    {
    if ((originalFrontProperty = this->GetOriginalFrontProperty()) == NULL)
      {
      originalFrontProperty = vtkProperty::New();
      this->SetOriginalFrontProperty(originalFrontProperty);
      originalFrontProperty->Delete();
      }

    // Copy the front face property.
    frontProperty->DeepCopy(clientActor->GetProperty());
    originalFrontProperty->DeepCopy(clientActor->GetProperty());

    // Get the back face property (or lack thereof).
    if (clientActor->GetBackfaceProperty() != NULL)
      {
      // Make sure we have a back face property if the client actor has one.
      if ((backProperty = this->GetDemoBackProperty()) == NULL)
        {
        // Maybe it's been stashed ... otherwise, create it.
	if ((backProperty = this->GetStashedProperty()) == NULL)
          {
	  backProperty = vtkProperty::New();
	  this->SetDemoBackProperty(backProperty);
	  backProperty->Delete();
          }
	else
          {
	  this->SetDemoBackProperty(backProperty);
          }
        }

      if ((originalBackProperty = this->GetOriginalBackProperty()) == NULL)
        {
        originalBackProperty = vtkProperty::New();
        this->SetOriginalBackProperty(originalBackProperty);
        originalBackProperty->Delete();
        }

      // Copy the back face property.
      backProperty->DeepCopy(clientActor->GetBackfaceProperty());
      originalBackProperty->DeepCopy(clientActor->GetBackfaceProperty());
      }
    else if ((backProperty = this->GetDemoBackProperty()) != NULL)
      {
      this->SetStashedProperty(backProperty);
      this->SetDemoBackProperty(NULL);
      this->SetOriginalBackProperty(NULL);
      }
    }

  // Make sure our pointers are up to date.
  frontProperty = this->GetDemoFrontProperty();
  backProperty  = this->GetDemoBackProperty();

  // Update front composite color as well as reflectance and features.
  this->ReceiveFront();

  // update the back face inputs if the client uses a backface property.
  if ((backProperty != NULL) && (backProperty != frontProperty))
    {
    // Update front composite color as well as reflectance and features.
    this->ReceiveBack();
    }

  this->ReceiveTexture();

  // Sync Demo properties with those held by the interface.
  this->UpdateDemoGroups();
}

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::Show (int aC, char* aV[])
{
  if (this->FltkWindow != (Fl_Window *) 0)
    this->FltkWindow->show(aC,aV);

  if (this->Initialized == 0)
    this->InitializeDemo();

  this->Update();
}

void
vtkAbstractSurfacePropertiesUI::Show (void)
{
  if (this->FltkWindow != (Fl_Window *) 0)
    this->FltkWindow->show();

  if (this->Initialized == 0)
    this->InitializeDemo();

  this->Update();
}

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::UpdateDemoGroups (void)
{
  if (this->DemoActor->GetBackfaceProperty() != (vtkProperty *) 0)
    this->ActivateBackGroups();
  else
    this->DeactivateBackGroups();
}

// ----------------------------------------------------------------------------
static Fl_Color
PropertyToFlColor (vtkProperty*& aProperty)
{
  if (aProperty == NULL)
    return FL_BLACK;

  float rgb[3];

  aProperty->GetColor(rgb);
    
  return fl_rgb_color( int(rint(rgb[0] * 255.f)),
		       int(rint(rgb[1] * 255.f)),
		       int(rint(rgb[2] * 255.f)) );
}

// ----------------------------------------------------------------------------
#define MAX2(_a,_b) ((_a)>(_b) ? (_a) : (_b))
#define MAX3(_v)    MAX2(MAX2(_v[0],_v[1]), MAX2(_v[0],_v[2]))

void
vtkAbstractSurfacePropertiesUI::MaterialToProperty (vtkProperty*& aProperty,
                                                    SvvMaterial&  aMaterial)
{
  if (aProperty == NULL)
    return;

  float magnitude;

  magnitude = (MAX3(aMaterial.Ka)<0.2 ? 0.2 : MAX3(aMaterial.Ka));
  aProperty->SetAmbient(magnitude);
  aProperty->SetAmbientColor( aMaterial.Ka[0] / magnitude,
			      aMaterial.Ka[1] / magnitude,
			      aMaterial.Ka[2] / magnitude );

  magnitude = (MAX3(aMaterial.Kd)<0.8 ? 0.8 : MAX3(aMaterial.Kd));
  aProperty->SetDiffuse(magnitude);
  aProperty->SetDiffuseColor( aMaterial.Kd[0] / magnitude,
			      aMaterial.Kd[1] / magnitude,
			      aMaterial.Kd[2] / magnitude );

  magnitude = (MAX3(aMaterial.Ks)<=0.f ? 0.f : MAX3(aMaterial.Ks));
  aProperty->SetSpecular(magnitude);
  aProperty->SetSpecularColor( aMaterial.Ks[0] / magnitude,
			       aMaterial.Ks[1] / magnitude,
			       aMaterial.Ks[2] / magnitude );

  aProperty->SetSpecularPower(aMaterial.Se);
  aProperty->SetOpacity(aMaterial.Kd[3]);
}

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::BuildNamedMaterialsMenu (Fl_Menu_* aMenu)
{
  if (aMenu == NULL)
    return;

  svvMaterialNameMap* materials =
    vtkAbstractSurfacePropertiesUI::NamedMaterials;

  Fl_Menu_Item* item;
  vtkProperty*  property = vtkProperty::New();
  int           i;

  const char* name = NULL;
  SvvMaterial material;

  for (materials->InitTraversal(); materials->TraverseForward(name, material); )
    {
    // Add a defined material name to the menu with a pointer to its data.
    i    = aMenu->add(name, 0/*shortcut*/, 0/*Cb*/);
    // Color the item label with the composite material color.
    item = const_cast<Fl_Menu_Item*>(&(aMenu->menu())[i]);
    MaterialToProperty(property, material);
    item->labelcolor(PropertyToFlColor(property));
    }

  // Clean up references.
  property->Delete();
}

// ----------------------------------------------------------------------------
bool
vtkAbstractSurfacePropertiesUI::FindNamedMaterial (const char*  aName,
                                                   SvvMaterial& aMaterial)
{
  return
    vtkAbstractSurfacePropertiesUI::NamedMaterials->FindMaterial( aName,
                                                                  aMaterial );
}

// ----------------------------------------------------------------------------
vtkDataSetToDataSetFilter*
vtkAbstractSurfacePropertiesUI::
FindTextureCoordsGenerator (vtkActor*                   aActor,
                            vtkTransformTextureCoords** aTexTxPtr)
{
#if 0
  vtkUpstream* upstream = vtkUpstream::New();

  upstream->SetSource(aActor);

  vtkSource*                 source;
  vtkDataSetToDataSetFilter* filter;
  vtkTransformTextureCoords* texTx;

  while ((source = upstream->GetSource()) != NULL)
    {
    if ((filter = vtkDataSetToDataSetFilter::SafeDownCast(source)) != NULL)
      {
      if      ( (filter->IsA("vtkTextureMapToPlane")) ||
                (filter->IsA("vtkTextureMapToCylinder")) ||
                (filter->IsA("vtkTextureMapToSphere")) )
        {
	return filter;
        }
      else if ( (aTexTxPtr != NULL) &&
                ((texTx = vtkTransformTextureCoords::SafeDownCast(filter))
                 != NULL ) )
        {
	*aTexTxPtr = texTx;
        }
      }

    source = upstream->GetInputSource();
    upstream->SetSource(source);
    }
#endif /* 0 */

  return NULL;
}

// ----------------------------------------------------------------------------
void
vtkAbstractSurfacePropertiesUI::PrintSelf (ostream& aTarget, vtkIndent aIndent)
{
  this->Superclass::PrintSelf(aTarget, aIndent);

  aTarget << aIndent << "Initialized:           "
          << this->Initialized << endl;

  aTarget << aIndent << "ModelSource:           "
          << this->ModelSource << endl;
  if (this->ModelSource != 0)
    this->ModelSource->PrintSelf(aTarget, aIndent.GetNextIndent());

  aTarget << aIndent << "DemoActor:             "
          << this->DemoActor << endl;
  if (this->DemoActor != 0)
    this->DemoActor->PrintSelf(aTarget, aIndent.GetNextIndent());

  aTarget << aIndent << "StashedProperty:       "
          << this->StashedProperty << endl;
  if (this->StashedProperty != 0)
    this->StashedProperty->PrintSelf(aTarget, aIndent.GetNextIndent());

  aTarget << aIndent << "OriginalFrontProperty: "
          << this->OriginalFrontProperty << endl;
  if (this->OriginalFrontProperty != 0)
    this->OriginalFrontProperty->PrintSelf(aTarget, aIndent.GetNextIndent());

  aTarget << aIndent << "OriginalBackProperty:  "
          << this->OriginalBackProperty << endl;
  if (this->OriginalBackProperty != 0)
    this->OriginalBackProperty->PrintSelf(aTarget, aIndent.GetNextIndent());
}

/*
 * End of: $Id: vtkAbstractSurfacePropertiesUI.cxx,v 1.1.1.1 2006/12/19 22:59:46 christianh Exp $.
 *
 */
