/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkLoadDialogUIImpl.cxx,v 1.1.1.1 2006/12/19 22:59:48 christianh Exp $
 * 
 * Copyright (c) 2003, 2004 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <string>
#include "vtkDirectory.h"

void
vtkLoadDialogUI::Check (void)
{
  Fl::check();
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::ShowSingleGroup (void)
{
  if (!this->mSingleRadio->value()) this->mSingleRadio->value(1);

  this->mMultipleGroup->hide();
  this->mSeriesGroup->hide();
  this->mDirectoryGroup->hide();
  this->mFilePatternGroup->hide();

  this->mSingleGroup->show();
}

void
vtkLoadDialogUI::ShowMultipleGroup (void)
{
  if (!this->mMultipleRadio->value()) this->mMultipleRadio->value(1);

  this->mSingleGroup->hide();
  this->mSeriesGroup->hide();
  this->mDirectoryGroup->hide();
  this->mFilePatternGroup->hide();

  this->mMultipleGroup->show();
}

void
vtkLoadDialogUI::ShowSeriesGroup (void)
{
  if (!this->mSeriesRadio->value()) this->mSeriesRadio->value(1);

  this->mSingleGroup->hide();
  this->mMultipleGroup->hide();
  this->mDirectoryGroup->hide();

  this->mFilePatternGroup->show();
  this->mSeriesGroup->show();
}

void
vtkLoadDialogUI::ShowDirectoryGroup (void)
{
  if (!this->mDirectoryRadio->value()) this->mDirectoryRadio->value(1);

  this->mSingleGroup->hide();
  this->mMultipleGroup->hide();
  this->mSeriesGroup->hide();

  this->mFilePatternGroup->show();
  this->mDirectoryGroup->show();
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::SetSingleGroupToDefault (void)
{
  this->mSingleFileInput->value(NULL);
}

void
vtkLoadDialogUI::SetMultipleGroupToDefault (void)
{
  this->mMultipleFileBrowser->clear();
}

void
vtkLoadDialogUI::SetSeriesGroupToDefault (void)
{
  this->mFirstFileInput->value(NULL);
  this->mLastFileInput->value(NULL);
}

void
vtkLoadDialogUI::SetDirectoryGroupToDefault (void)
{
  this->mDirectoryNameInput->value(NULL);
  this->mFileOffsetInput->value(0);
  this->mFileCountInput->value(1);
}

void
vtkLoadDialogUI::SetFilePatternGroupToDefault (void)
{
  this->mFilePrefixInput->value(NULL);
  this->mFileDigitsInput->value(1);
  this->mFileInfixInput->value(NULL);
  this->mFileSuffixInput->value(NULL);
}

// ----------------------------------------------------------------------------
void
sParsePathname (const char* const& aPathName,
                vtkstd::string&    aDirname,
                vtkstd::string&    aPrefix,
                int&               aIndex,
                vtkstd::size_t&    aDigits,
                vtkstd::string&    aInfix,
                vtkstd::string&    aSuffix)
{
  if (aPathName == NULL || *aPathName == '\0')
    {
    aDirname.clear();
    aPrefix.clear();
    aIndex = -1;
    aDigits = 0;
    aInfix.clear();
    aSuffix.clear();
    return;
    }

  vtkstd::string pathName(aPathName);

  vtkstd::string::size_type tailBeg = vtkstd::string::npos;
  vtkstd::string::size_type sfxBeg  = vtkstd::string::npos;
  vtkstd::string::size_type ifxBeg  = vtkstd::string::npos;

  if ((tailBeg = pathName.find_last_of('/')) == vtkstd::string::npos)
    tailBeg = 0;
  else
    ++tailBeg;

  if ( (sfxBeg = pathName.find_last_of('.')) != vtkstd::string::npos &&
       sfxBeg > tailBeg )
    {
    aSuffix.assign(pathName.substr(sfxBeg + 1));

    ifxBeg = pathName.find_first_of('.', tailBeg - 1);

    aInfix.assign(pathName.substr(ifxBeg, 1 + sfxBeg - ifxBeg));
    }
  else
    {
    aInfix.clear();
    aSuffix.clear();
    }

  vtkstd::string::size_type digitsEnd =
    (ifxBeg!=vtkstd::string::npos ? ifxBeg-1 : pathName.length()-1);

  vtkstd::string::size_type digitsBeg = digitsEnd;

  while (digitsBeg >= tailBeg && isdigit(pathName[digitsBeg])) --digitsBeg;
  ++digitsBeg;

  if (tailBeg == 0)
    aDirname.clear();
  else
    aDirname.assign(pathName.substr(0, tailBeg));

  aPrefix.assign(pathName.substr(tailBeg, digitsBeg - tailBeg));

  if ((aDigits = 1 + digitsEnd - digitsBeg) == 0)
    {
    vtkstd::string::iterator pos;

    for (pos=aSuffix.begin(); pos!=aSuffix.end(); ++pos)
      if (!isdigit(*pos))
        break;

    if (pos == aSuffix.end())
      {
      aDigits = aSuffix.length();
      aIndex = atoi(aSuffix.c_str());
      aPrefix.append(aInfix);
      aInfix.clear();
      aSuffix.clear();
      }
    else
      {
      aIndex = -1;
      }
    }
  else
    {
    aIndex = atoi(pathName.substr(digitsBeg, aDigits).c_str());
    }
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::ReceiveFromSeriesGroup (void)
{
  const char* first = this->mFirstFileInput->value();
  const char* last  = this->mLastFileInput->value();

  vtkstd::string dirname1, prefix1, infix1, suffix1;
  vtkstd::size_t ndigits1;
  int index1;

  vtkstd::string dirname2, prefix2, infix2, suffix2;
  vtkstd::size_t ndigits2;
  int index2;

  char buf[1024] = {0};

  if      (first != NULL && *first != '\0')
    {
    sParsePathname( first,
                    dirname1, prefix1, index1, ndigits1, infix1, suffix1 );

    if (last != NULL && *last != '\0')
      {
      sParsePathname( last,
                      dirname2, prefix2, index2, ndigits2, infix2, suffix2 );

      if (dirname1 == dirname2)
        {
        if (dirname1.empty())
          {
          dirname1.assign("./");
          dirname2.assign("./");
          }
        else if (*(dirname1.rbegin()) != '/')
          {
          dirname1.push_back('/');
          dirname2.push_back('/');
          }
        }

      if ( dirname1 != dirname2 ||
           prefix1  != prefix2  ||
           infix1   != infix2   ||
           suffix1  != suffix2  ||
           index1   >  index2   ||
           ndigits1 >  ndigits2 )
        {
        this->SetFilePatternGroupToDefault();
        this->SetDirectoryGroupToDefault();
        }
      else
        {
        this->mFilePrefixInput->value((dirname1 + prefix1).c_str());
        this->mFileDigitsInput->value(ndigits1);
        this->mFileInfixInput->value(infix1.c_str());
        this->mFileSuffixInput->value(suffix1.c_str());

        this->mDirectoryNameInput->value(dirname1.c_str());
        this->mFileOffsetInput->value(index1);
        this->mFileCountInput->value(1 + index2 - index1);
        }
      }
    else
      {
      if (dirname1.empty())
        {
        dirname1.assign("./");
        }
      else if (*(dirname1.rbegin()) != '/')
        {
        dirname1.push_back('/');
        }

      this->mFilePrefixInput->value((dirname1 + prefix1).c_str());
      this->mFileDigitsInput->value(ndigits1);
      this->mFileInfixInput->value(infix1.c_str());
      this->mFileSuffixInput->value(suffix1.c_str());

      this->mDirectoryNameInput->value(dirname1.c_str());
      this->mFileOffsetInput->value(index1);

      (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                       this->mFilePrefixInput->value(),
                       this->mFileDigitsInput->value(),
                       this->mFileOffsetInput->value() +
                       this->mFileCountInput->value() - 1,
                       this->mFileInfixInput->value(),
                       this->mFileSuffixInput->value() );

      this->mLastFileInput->value(buf);
      }
    }
  else if (last != NULL && *last != '\0')
    {
    sParsePathname( last,
                    dirname2, prefix2, index2, ndigits2, infix2, suffix2 );

    if (dirname2.empty())
      {
      dirname2.assign("./");
      }
    else if (*(dirname2.rbegin()) != '/')
      {
      dirname2.push_back('/');
      }

    this->mFilePrefixInput->value((dirname2 + prefix2).c_str());
    this->mFileDigitsInput->value(ndigits2);
    this->mFileInfixInput->value(infix2.c_str());
    this->mFileSuffixInput->value(suffix2.c_str());

    this->mDirectoryNameInput->value(dirname2.c_str());
    this->mFileOffsetInput->value(1 + index2 - this->mFileCountInput->value());

    (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                     this->mFilePrefixInput->value(),
                     this->mFileDigitsInput->value(),
                     this->mFileOffsetInput->value(),
                     this->mFileInfixInput->value(),
                     this->mFileSuffixInput->value() );

    this->mFirstFileInput->value(buf);
    }
  else
    {
    this->SetFilePatternGroupToDefault();
    this->SetDirectoryGroupToDefault();
    }
}

void
vtkLoadDialogUI::ReceiveFromDirectoryGroup (void)
{
  vtkstd::string dirname(this->mDirectoryNameInput->value());

  if (dirname.empty())
    {
    dirname.assign("./");
    }
  else if (*(dirname.rbegin()) != '/')
    {
    dirname.push_back('/');
    }

  vtkstd::string prefix(this->mFilePrefixInput->value());
  vtkstd::string::size_type tailBeg = prefix.find_last_of('/');

  if (tailBeg == vtkstd::string::npos) tailBeg = 0;
  else ++tailBeg;

  if (!dirname.compare(prefix.substr(0, tailBeg)))
    {
    prefix.replace(0, tailBeg, dirname);
    this->mFilePrefixInput->value(prefix.c_str());
    }

  char buf[1024] = {0};

  (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                   this->mFilePrefixInput->value(),
                   this->mFileDigitsInput->value(),
                   this->mFileOffsetInput->value(),
                   this->mFileInfixInput->value(),
                   this->mFileSuffixInput->value() );

  this->mFirstFileInput->value(buf);

  (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                   this->mFilePrefixInput->value(),
                   this->mFileDigitsInput->value(),
                   this->mFileOffsetInput->value() +
                   this->mFileCountInput->value() - 1,
                   this->mFileInfixInput->value(),
                   this->mFileSuffixInput->value() );

  this->mLastFileInput->value(buf);
}

void
vtkLoadDialogUI::ReceiveFromPatternGroup (void)
{
  char        buf[1024] = {0};
  const char* prefix    = this->mFilePrefixInput->value();
  char*       ptr       = strrchr(prefix, '/');

  if (ptr == NULL)
    {
    strcpy(buf, "./");
    strcat(buf, prefix);
    this->mFilePrefixInput->value(buf);
    prefix = this->mFilePrefixInput->value();
    ptr    = strrchr(prefix, '/');
    }

  (void) memcpy(buf, prefix, ptr - prefix);
  this->mDirectoryNameInput->value(buf);

  (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                   prefix,
                   this->mFileDigitsInput->value(),
                   this->mFileOffsetInput->value(),
                   this->mFileInfixInput->value(),
                   this->mFileSuffixInput->value() );

  this->mFirstFileInput->value(buf);

  (void) snprintf( buf, sizeof buf, "%s%0*d%s%s",
                   prefix,
                   this->mFileDigitsInput->value(),
                   this->mFileOffsetInput->value() +
                   this->mFileCountInput->value() - 1,
                   this->mFileInfixInput->value(),
                   this->mFileSuffixInput->value() );

  this->mLastFileInput->value(buf);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::SyncInputType (void)
{
  this->InputType =
    vtkLoadDialogUI::InputTypeEnum(this->mInputType->mvalue()->argument());
}

void
vtkLoadDialogUI::SetInputTypeToDefault (void)
{
  this->mInputType->value(0);
  this->SyncInputType();
}

void
vtkLoadDialogUI::GetInputTypeValue (long & aValue) const
{
  aValue = this->mInputType->mvalue()->argument();
}

void
vtkLoadDialogUI::GetInputTypeText (const char* & aValue) const
{
  aValue = this->mInputType->mvalue()->label();
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::SetSelectionTypeToDefault (void)
{
  this->mSingleRadio->value(1);
  this->SelectionType = vtkLoadDialogUI::VTK_SELECTION_TYPE_SINGLE;
  this->mMultipleRadio->value(0);
  this->mSeriesRadio->value(0);
  this->mDirectoryRadio->value(0);
  this->ShowSingleGroup();
}

void
vtkLoadDialogUI::SetSelectionTypeToSingle (void)
{
  this->mSingleRadio->value(1);
  this->SelectionType = vtkLoadDialogUI::VTK_SELECTION_TYPE_SINGLE;
  this->mMultipleRadio->value(0);
  this->mSeriesRadio->value(0);
  this->mDirectoryRadio->value(0);
  this->ShowSingleGroup();
}

void
vtkLoadDialogUI::SetSelectionTypeToMultiple (void)
{
  this->mMultipleRadio->value(1);
  this->SelectionType = vtkLoadDialogUI::VTK_SELECTION_TYPE_MULTIPLE;
  this->mSingleRadio->value(0);
  this->mSeriesRadio->value(0);
  this->mDirectoryRadio->value(0);
  this->ShowMultipleGroup();
}

void
vtkLoadDialogUI::SetSelectionTypeToSeries (void)
{
  this->mSeriesRadio->value(1);
  this->SelectionType = vtkLoadDialogUI::VTK_SELECTION_TYPE_SERIES;
  this->mSingleRadio->value(0);
  this->mMultipleRadio->value(0);
  this->mDirectoryRadio->value(0);
  this->ShowSeriesGroup();
}

void
vtkLoadDialogUI::SetSelectionTypeToDirectory (void)
{
  this->mDirectoryRadio->value(1);
  this->SelectionType = vtkLoadDialogUI::VTK_SELECTION_TYPE_DIRECTORY;
  this->mSingleRadio->value(0);
  this->mMultipleRadio->value(0);
  this->mSeriesRadio->value(0);
  this->ShowDirectoryGroup();
}

void
vtkLoadDialogUI::GetSelectionTypeValue (long & aValue) const
{
  aValue = this->SelectionType;
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetSingleFileInput (const char* & aValue) const
{
  aValue = this->mSingleFileInput->value();
}

void
vtkLoadDialogUI::SetSingleFileInput (const char* const& aValue)
{
  this->mSingleFileInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetMultipleFileBrowserSize (int & aValue) const
{
  aValue = int(this->mMultipleFileBrowser->size());
}

void
vtkLoadDialogUI::GetMultipleFileFilter (const char* & aValue) const
{
  aValue = this->mMultipleFileBrowser->filter();
}

void
vtkLoadDialogUI::SetMultipleFileFilter (const char* const& aValue)
{
  this->mMultipleFileBrowser->filter(aValue);
}

void
vtkLoadDialogUI::GetMultipleFileSelection (int         const& aLine,
                                           const char*      & aValue) const
{
  aValue = this->mMultipleFileBrowser->text(aLine);
}

void
vtkLoadDialogUI::SetMultipleFileSelection (int         const& aLine,
                                           const char* const& aValue)
{
  this->mMultipleFileBrowser->text(aLine, aValue);
}

void
vtkLoadDialogUI::AddToMultipleFileBrowser (const char* const& aValue)
{
  this->mMultipleFileBrowser->add(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFirstFileInput (const char* & aValue) const
{
  aValue = this->mFirstFileInput->value();
}

void
vtkLoadDialogUI::SetFirstFileInput (const char* const& aValue)
{
  this->mFirstFileInput->value(aValue);
}

void
vtkLoadDialogUI::GetLastFileInput (const char* & aValue) const
{
  aValue = this->mLastFileInput->value();
}

void
vtkLoadDialogUI::SetLastFileInput (const char* const& aValue)
{
  this->mLastFileInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetDirectoryNameInput (const char* & aValue) const
{
  aValue = this->mDirectoryNameInput->value();
}

void
vtkLoadDialogUI::SetDirectoryNameInput (const char* const& aValue)
{
  this->mDirectoryNameInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::ClearSingleFileInput (void)
{
  this->mSingleFileInput->value(NULL);
}

void
vtkLoadDialogUI::ClearMultipleFileBrowser (void)
{
  this->mMultipleFileBrowser->clear();
}

void
vtkLoadDialogUI::ClearFirstFileInput (void)
{
  this->mFirstFileInput->value(NULL);
}

void
vtkLoadDialogUI::ClearLastFileInput (void)
{
  this->mLastFileInput->value(NULL);
}

void
vtkLoadDialogUI::ClearDirectoryNameInput (void)
{
  this->mDirectoryNameInput->value(NULL);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternPrefix (const char* & aValue) const
{
  aValue = this->mFilePrefixInput->value();
}

void
vtkLoadDialogUI::SetFilePatternPrefix (const char* const& aValue)
{
  this->mFilePrefixInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternDigits (int & aValue) const
{
  aValue = int(this->mFileDigitsInput->value());
  if (aValue < 0) aValue = 0;
}

void
vtkLoadDialogUI::SetFilePatternDigits (int const& aValue)
{
  this->mFileDigitsInput->value(aValue<0 ? 0 : aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternInfix (const char* & aValue) const
{
  aValue = this->mFileInfixInput->value();
}

void
vtkLoadDialogUI::SetFilePatternInfix (const char* const& aValue)
{
  this->mFileInfixInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternSuffix (const char* & aValue) const
{
  aValue = this->mFileSuffixInput->value();
}

void
vtkLoadDialogUI::SetFilePatternSuffix (const char* const& aValue)
{
  this->mFileSuffixInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternOffset (int & aValue) const
{
  aValue = int(this->mFileOffsetInput->value());
}

void
vtkLoadDialogUI::SetFilePatternOffset (int const& aValue)
{
  this->mFileOffsetInput->value(aValue);
}

// ----------------------------------------------------------------------------
void
vtkLoadDialogUI::GetFilePatternCount (int & aValue) const
{
  aValue = int(this->mFileCountInput->value());
}

void
vtkLoadDialogUI::SetFilePatternCount (int const& aValue)
{
  this->mFileCountInput->value(aValue);
}

/*
 * End of: $Id: vtkLoadDialogUIImpl.cxx,v 1.1.1.1 2006/12/19 22:59:48 christianh Exp $.
 *
 */
