/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkSessionExporter.h,v 1.1.1.1 2006/12/19 23:00:08 christianh Exp $
 * 
 */
#ifndef _VTK_SESSION_EXPORTER_H_
#  define _VTK_SESSION_EXPORTER_H_
// vtkExtensions
#  include "vtkExtensionsSystem.h"
// VTK Common
#  include "vtkObject.h"

// VTK IO
class vtkDataCompressor;
class vtkOutputStream;
// vtkExtensions
class vtkRenderingContext;

/** \class   vtkSessionExporter
 *  \brief   XML visualization session exporter.
 * 
 * vtkSessionExporter writes an XML file containing the state 
 * of a visualization session.
 * 
 * \author  Sean McInerney
 * \version $Version$
 * \date    $Date: 2006/12/19 23:00:08 $
 * 
 * \sa
 * vtkObject vtkExporter
 */

class VTK_EXPORT vtkSessionExporter : public vtkObject
{
public:
  static vtkSessionExporter* New(void);
  vtkTypeRevisionMacro(vtkSessionExporter,vtkObject);
  void PrintSelf(std::ostream&, vtkIndent);

  /** Write data to output.
   * Method executes subclasses WriteData() method, as 
   * well as StartWrite() and EndWrite() methods.
   */
  virtual void Write(void);

  /** Convenient alias for Write() method. */
  void Update(void);
  
  // Description:
  // Get the number of elements nested in this one.
  int GetNumberOfRenderingContexts(void);
  
  // Description:
  // Get the element nested in this one at the given index.
  RenderingContext* GetRenderingContext(int);

  // Description:
  // Set/Get the rendering window that contains the scene to be written.
  virtual void SetRenderingContext(vtkRenderingContext*);
  vtkGetObjectMacro(RenderingContext,vtkRenderingContext);

  /** Specify a function to be called before data is written.
   * Function will be called with argument provided.
   */
  void SetStartWrite(void (*)(void*), void*);

  /** Specify a function to be called after data is written.
   * Function will be called with argument provided.
   */
  void SetEndWrite(void (*)(void*), void*);

  /** Set the arg delete method. This is used to free user memory. */
  void SetStartWriteArgDelete(void (*)(void*));

  /** Set the arg delete method. This is used to free user memory. */
  void SetEndWriteArgDelete(void (*)(void*));

  /** Returns the MTime also considering the RenderingContext. */
  unsigned long GetMTime(void);

  /*@{*/
  /** Specify the name of the file to write. */
  vtkSetStringMacro(FileName);
  vtkGetStringMacro(FileName);
  /*@}*/

  /*@{*/
  /** Specify the prefix of the file to write. */
  vtkSetStringMacro(FilePrefix);
  vtkGetStringMacro(FilePrefix);
  /*@}*/

  /*@{*/
  /** Specify the infix of the file to write (defaults to "."). */
  vtkSetStringMacro(FileInfix);
  vtkGetStringMacro(FileInfix);
  /*@}*/

  /*@{*/
  /** Specify the suffix of the file to write (defaults to "svs"). */
  vtkSetStringMacro(FileSuffix);
  vtkGetStringMacro(FileSuffix);
  /*@}*/

  /*@{*/
  /** Specify the base URI of the file to write. */
  vtkSetStringMacro(BaseName);
  vtkGetStringMacro(BaseName);
  /*@}*/

  /*@{*/
  /** Set the file stream to write to, overriding any URI, if specified. */
  vtkSetMacro(Stream, std::ostream*);
  vtkGetMacro(Stream, std::ostream*);
  /*@}*/

  /*@{*/
  /** Get/Set the byte order of data written to the file.
   * The default is the machine's hardware byte order.
   */
  vtkSetMacro(BigEndian, int);
  vtkGetMacro(BigEndian, int);
  vtkBooleanMacro(BigEndian, int);
  void SetByteOrderToBigEndian(void);
  void SetByteOrderToLittleEndian(void);
  /*@}*/

  /*@{*/
  /** Get/Set compressor used for binary and appended data before stream write.
   * Default is no compression.
   */
  virtual void SetCompressor(vtkDataCompressor*);
  vtkGetObjectMacro(Compressor, vtkDataCompressor);
  /*@}*/
  
  /*@{*/
  /** Get/Set block size used in compression.
   * When reading, this controls the granularity of how much extra 
   * information must be read when only part of the data are requested.
   */
  vtkSetMacro(BlockSize, unsigned int);  
  vtkGetMacro(BlockSize, unsigned int);  
  /*@}*/

protected:
  vtkSessionExporter(void);
  ~vtkSessionExporter(void);

  // The set of nested elements.
  int NumberOfRenderingContexts;
  int RenderingContextsSize;
  vtkRenderingContext** RenderingContexts;

  virtual void WriteData(void) = 0;

  void  (*StartWrite)(void*);
  void  (*StartWriteArgDelete)(void*);
  void* StartWriteArg;
  void  (*EndWrite)(void*);
  void  (*EndWriteArgDelete)(void*);
  void* EndWriteArg;

  char* FileName;
  char* FilePrefix;
  char* FileInfix;
  char* FileSuffix;
  char* BaseName;

  // Output stream pointer to which the session is written.
  std::ostream*  Stream;
  std::ofstream* FileStream;

  // Is the output byte order big endian.
  int BigEndian;
    
  // Compression information.
  vtkDataCompressor* Compressor;
  unsigned int   BlockSize;  
  unsigned long  CompressionBlockNumber;
  unsigned int*  CompressionHeader;
  unsigned int   CompressionHeaderLength;
  unsigned long  CompressionHeaderPosition;

  /** Output stream for binary and appended data (may do transparent encode). */
  vtkOutputStream* DataStream;
  
  // Allow subclasses to set the data stream.
  virtual void SetDataStream(vtkOutputStream*);
  vtkGetObjectMacro(DataStream, vtkOutputStream);  

  virtual int IsSafeToWrite(void);
  virtual int CheckStream(void);

  /** The current indentation level. */
  indent_stream Indent;

private:
  vtkSessionExporter(const vtkSessionExporter&);  // Not implemented.
  void operator=(const vtkSessionExporter&);  // Not implemented.
};

#endif /* _VTK_SESSION_EXPORTER_H_ */
/* 
 * End of: $Id: vtkSessionExporter.h,v 1.1.1.1 2006/12/19 23:00:08 christianh Exp $.
 * 
 */
