/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: vtkSessionImporter.cxx,v 1.1.1.1 2006/12/19 23:00:09 christianh Exp $
 * 
 */
#include <sys/param.h>
#include <cstdlib>
#include <cstring>

#include "vtkProp.h"
#include "vtkPropAssembly.h"
#include "vtkAssembly.h"
#include "vtkProperty.h"
#include "vtkActor.h"
#include "vtkFollower.h"
#include "vtkVolume.h"
#include "vtkCamera.h"
#include "vtkRenderer.h"
#include "vtkRenderingContext.h"

#include "vtkXMLChar.h"
#include "vtkXMLString.h"
#include "vtkSessionImporter.h"

VTK_EXTENSIONS_NAMESPACE_USE

// ----------------------------------------------------------------------------
vtkSessionImporter::vtkSessionImporter(void) :
  TagMap(SessionTagMap::GetInstance())
{
  this->CurrentSection = eNoneSection;
  this->Header         = new HeaderRec;

  this->pCurrentRenderingContext = 0;
  this->pCurrentRenderer         = 0;
  this->pCurrentCamera           = 0;
  this->pCurrentProp             = 0;
  this->pCurrentActor            = 0;
  this->pCurrentVolume           = 0;
  this->pCurrentProperty         = 0;
}

vtkSessionImporter::~vtkSessionImporter(void)
{
}

// ----------------------------------------------------------------------------
HeaderRec*
vtkSessionImporter::GetHeaderCopy(void)
{
  HeaderRec* hdr = new HeaderRec;
  *hdr = *(this->Header);
  return hdr;
}

// ----------------------------------------------------------------------------
static void
PrintAttrs(std::ostream& aOS, indent_stream& aIndent, const XML_Char** aAttrs)
{
  aOS << aIndent;
  while(*aAttrs)
    {
    aOS << " " << aAttrs[0] << "=\"" << aAttrs[1] << "\"";
    aAttrs += 2;
    }
  aOS << std::endl;
}

// ----------------------------------------------------------------------------
static const XML_Char*
GetStringAttrVal(const XML_Char* aTarget, const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if(vtkXMLString::Equals(aTarget,aAttrs[0]))
      {
      return aAttrs[1];
      }
    aAttrs += 2;
    }

  return NULL; // Not found.
}

static int
GetIntAttrVal(const XML_Char* aTarget, const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if(vtkXMLString::Equals(aTarget,aAttrs[0]))
      {
      return int(vtkXMLString::ParseInt(aAttrs[1]));
      }
    aAttrs += 2;
    }

  return -VTK_LARGE_INTEGER; // Not found.
}

static float
GetFloatAttrVal(const XML_Char* aTarget, const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if(vtkXMLString::Equals(aTarget,aAttrs[0]))
      {
      return float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
  
  return -VTK_LARGE_FLOAT; // Not found.
}

static void
GetRGBAttrVals(float a[3], const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if ((*aAttrs)[1]==0x0000/*NUL*/)
      {
      if      ((*aAttrs)[0]==0x0072/*'r'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0067/*'g'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0062/*'b'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0052/*'R'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0047/*'G'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0042/*'B'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
}

static void
GetRGBAAttrVals(float a[4], const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if ((*aAttrs)[1]==0x0000/*NUL*/)
      {
      if      ((*aAttrs)[0]==0x0072/*'r'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0067/*'g'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0062/*'b'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0061/*'a'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0052/*'R'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0047/*'G'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0042/*'B'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0041/*'A'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
}

static void
GetHSVAttrVals(float a[3], const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if ((*aAttrs)[1]==0x0000/*NUL*/)
      {
      if      ((*aAttrs)[0]==0x0068/*'h'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0073/*'s'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0076/*'v'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0048/*'H'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0053/*'S'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0056/*'V'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
}

static void
GetHSVAAttrVals(float a[4], const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if ((*aAttrs)[1]==0x0000/*NUL*/)
      {
      if      ((*aAttrs)[0]==0x0068/*'h'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0073/*'s'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0076/*'v'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0061/*'a'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0048/*'H'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0053/*'S'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0056/*'V'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0041/*'A'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
}

static void
GetSpecularAttrVals(float a[4], const XML_Char** aAttrs)
{
  while(*aAttrs)
    {
    if ((*aAttrs)[1]==0x0000/*NUL*/)
      {
      if      ((*aAttrs)[0]==0x0072/*'r'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0067/*'g'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0062/*'b'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0065/*'e'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0052/*'R'*/)
        a[0] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0047/*'G'*/)
        a[1] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0042/*'B'*/)
        a[2] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      else if ((*aAttrs)[0]==0x0045/*'E'*/)
        a[3] = float(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    aAttrs += 2;
    }
}


/* 
 * Element      Content
 * =======      =======
 * Property     EdgeColor Material 
 * 
 * Element      Attribute
 * =======      =========
 * Property     culling face id interpolation linewidth name pointsize 
 *              representation stipplepattern stipplerepeat 
 */
static void
GetPropertyAttrVals(vtkProperty* pProperty, const XML_Char** aAttrs)
{
  const XML_Char* id   = 0;
  const XML_Char* name = 0;
  bool isFront = true;

  while(*aAttrs)
    { // Polygon face culling.
    if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("culling")))
      {
      if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("front")))
        {
        pProperty->FrontfaceCullingOn();
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("back")))
        {
        pProperty->BackfaceCullingOn();
        }
      else
        {
        pProperty->FrontfaceCullingOff();
        pProperty->BackfaceCullingOff();
        }
      }
    // face.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("face")))
      {
      if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("front")))
        {
        isFront = true;
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("back")))
        {
        isFront = false;
        }
      }
    // Id.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("id")))
      {
      id = aAttrs[1];
      }
    // Shading.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("interpolation")))
      {
      if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("flat")))
        {
        pProperty->SetInterpolation(VTK_FLAT);
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("gouraud")))
        {
        pProperty->SetInterpolation(VTK_GOURAUD);
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("phong")))
        {
        pProperty->SetInterpolation(VTK_PHONG);
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("smooth")))
        {
        pProperty->SetInterpolation(VTK_GOURAUD);
        }
      }
    // Line width.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("linewidth")))
      {
      pProperty->SetLineWidth(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("name")))
      {
      name = aAttrs[1];
      }
    // Point size.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("pointsize")))
      {
      pProperty->SetPointSize(vtkXMLString::ParseDouble(aAttrs[1]));
      }
    // Model representation.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("representation")))
      {
      if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("points")))
        {
        pProperty->SetRepresentation(VTK_POINTS);
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("wireframe")))
        {
        pProperty->SetRepresentation(VTK_WIREFRAME);
        }
      else if (vtkXMLString::Equals(aAttrs[1],LATIN1_TO_UTF16("surface")))
        {
        pProperty->SetRepresentation(VTK_SURFACE);
        }
      }
    // Model representation.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("stipplepattern")))
      {
      pProperty->SetLineStipplePattern(vtkXMLString::ParseInt(aAttrs[1]));
      }
    // Model representation.
    else if (vtkXMLString::Equals(aAttrs[0],LATIN1_TO_UTF16("stipplerepeat")))
      {
      pProperty->SetLineStippleRepeatFactor(vtkXMLString::ParseInt(aAttrs[1]));
      }
    aAttrs += 2;
    }
}

// ----------------------------------------------------------------------------
void
vtkSessionImporter::StartElement(const XML_Char* aName, const XML_Char** aAttrs)
{
  // Increment the indentation level.
  this->Indent += 2;
  // Push the new element onto the stack.
  this->OpenElements.push(this->TagMap[UTF16_TO_LATIN1(aName)]);

  switch (this->CurrentSection)
    {

    case eHeaderSection:
      this->HeaderHandler(aName,aAttrs);
      break;

    case eSourceCollectionSection:
      this->SourceCollectionHandler(aName,aAttrs);
      break;

    case eRenderingContextSection:
      this->RenderingContextHandler(aName,aAttrs);
      break;

    case eNoneSection:
    default:

      switch (this->OpenElements.top())
        {

        case eHeaderTag:
          // NOTE: Should handle 'dir', 'lang', and 'xml:lang' attributes here.
          this->CurrentSection = eHeaderSection;
          break;

        case eSourceCollectionTag:
          this->CurrentSection = eSourceCollectionSection;
          break;

        case eRenderingContextTag:
          this->CurrentSection = eRenderingContextSection;
          break;

        default:
          vtkErrorMacro(<< "Illegal element \""
                        << vtkXMLString::ToLatin1(aName) << "\" at top level.");
          return;

        } // switch (OpenElements)

    } // switch (CurrentSection)
}

// ----------------------------------------------------------------------------
void
vtkSessionImporter::EndElement(const XML_Char* aName)
{
  switch (this->OpenElements.top())
    {
    case eHeaderTag:
    case eSourceCollectionTag:
    case eRenderingContextTag:
      this->CurrentSection = eNoneSection;
      break;
    } // switch (OpenElements)
  
  // ... handle anything else ...
  
  // Pop the ending element off the stack.
  this->OpenElements.pop();
  // Decrement the indentation level.
  this->Indent -= 2;
}

// ----------------------------------------------------------------------------
void
vtkSessionImporter::CharacterData(const XML_Char* s, int len)
{
  switch (this->OpenElements.top())
    {

    case eSignatureTag:
      this->Header->Authors.back().Signature.assign(UTF16_TO_LATIN1(s),len);
      break;

    case eAuthorTag:
      this->Header->Authors.back().Author.assign(UTF16_TO_LATIN1(s),len);
      break;

    case eTitleTag:
      this->Header->Title.assign(UTF16_TO_LATIN1(s),len);
      break;

    case eDescriptionTag:
      this->Header->Description.assign(UTF16_TO_LATIN1(s),len);
      break;

    case eMatrix4x4Tag:
      // Handle 4 cols x 4 rows matrix.
      break;

    default:
      vtkErrorMacro(<< "Unexpected top element in CharacterData() handler.");

    } // switch (OpenElements)
}

// ----------------------------------------------------------------------------
// Specialized Handlers below ... 
// ----------------------------------------------------------------------------

/* 
 * Mostly meta data (i.e. no effect on rendering context).
 * 
 * Element      Content
 * =======      =======
 * Header       Author Base Description Link Meta Title 
 * 
 * Element      Attribute       (NOT handling these attributes yet!)
 * =======      =========
 * Header       dir lang xml:lang 
 */
void
vtkSessionImporter::HeaderHandler(const XML_Char*  aName,
                                  const XML_Char** aAttrs)
{
  const vtkUTF16ChType hrefAttr[] =
    { ASCII_latin_h, ASCII_latin_r, ASCII_latin_e, ASCII_latin_f, ASCII_NUL };
  const vtkUTF16ChType typeAttr[] =
    { ASCII_latin_t, ASCII_latin_y, ASCII_latin_p, ASCII_latin_e, ASCII_NUL };
  const vtkUTF16ChType rsaType[] =
    { ASCII_latin_r, ASCII_latin_s, ASCII_latin_a, ASCII_NUL };
  const vtkUTF16ChType dsaType[] =
    { ASCII_latin_d, ASCII_latin_s, ASCII_latin_a, ASCII_NUL };


  switch (this->OpenElements.top())
    {

    case eBaseTag:
      // Attributes: 'href'.
      this->Header->Base = UTF16_TO_LATIN1(GetStringAttrVal(hrefAttr,aAttrs));
      break;

    case eSignatureTag: {
    // <Signature> enclosed data is handled by CharacterData().
    // Attributes: 'type'.
    const XML_Char* type = GetStringAttrVal(typeAttr,aAttrs);
      
    if(!strcmp(UTF16_TO_LATIN1(type),"rsa"))
      this->Header->Authors.back().SignatureType = AuthorRec::eRSASignature;
    else if(!strcmp(UTF16_TO_LATIN1(type),"dsa"))
      this->Header->Authors.back().SignatureType = AuthorRec::eDSASignature;
    } break;

    case eAuthorTag:
      // <Author> enclosed data is handled by CharacterData().
      // Attributes: 'uid'.
      this->Header->
        Authors.push_back(AuthorRec("",GetIntAttrVal( LATIN1_TO_UTF16("uid"),
                                                      aAttrs ))); 
      break;

    case eTitleTag:
      // <Title> enclosed data is handled by CharacterData().
      // Attributes: 'dir', 'lang', and 'xml:lang'.
      break;

    case eDescriptionTag:
      // <Description> enclosed data is handled by CharacterData().
      // Attributes: 'dir', 'lang', and 'xml:lang'.
      break;

    case eLinkTag:
      // Attributes: 'charset', 'href', 'id', 'media', 'rel', 'rev', and 'type'.
      break;

    case eMetaTag:
      // Attributes: 'dir', 'lang', 'name', 'version', and 'xml:lang'.
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in Header.");

    } // switch (OpenElements)
}

/* 
 * Element              Content
 * =======              =======
 * SourceCollection     PolyDataSource VolumeSource 
 * 
 */
void
vtkSessionImporter::SourceCollectionHandler(const XML_Char*  aName,
                                            const XML_Char** aAttrs)
{
  const XML_Char* fileName;

  switch (this->OpenElements.top())
    {

    case ePolyDataSourceTag:
      fileName = GetStringAttrVal(LATIN1_TO_UTF16("src"),aAttrs);
      // Now read in the polygonal data ...
      break;

    case eVolumeSourceTag:
      fileName = GetStringAttrVal(LATIN1_TO_UTF16("src"),aAttrs);
      // Now read in the volume data ...
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in SourceCollection.");

    } // switch (OpenElements)
}

/* 
 * Element              Content
 * =======              =======
 * RenderingContext     Renderer 
 */
void
vtkSessionImporter::RenderingContextHandler(const XML_Char*  aName,
                                            const XML_Char** aAttrs)
{
  switch (this->OpenElements.top())
    {

    case eRendererTag:
      this->pCurrentRenderer = vtkRenderer::New();
      this->GetRenderWindow()->AddRenderer(this->pCurrentRenderer);
      this->pCurrentRenderer->Delete(); // Reference remains with RenderWindow.
      break;

    default:
      this->RendererHandler(aName,aAttrs);

    } // switch (OpenElements)
}

/* 
 * Element              Content
 * =======              =======
 * Renderer             Ambient Background Camera DisplayPoint LightCollection 
 *                      PixelAspect PropCollection SurfaceCollection ViewPoint 
 *                      VolumeCollection WorldPoint 
 */
void
vtkSessionImporter::RendererHandler(const XML_Char*  aName,
                                    const XML_Char** aAttrs)
{
  switch (this->OpenElements.top())
    {

    case eAmbientTag:
      // Attributes: 'a', 'b', 'g', and 'r'.
      break;

    case eBackgroundTag:
      // Attributes: 'b', 'g', and 'r'.
      break;

    case eCameraTag:
      // Attributes: 'ParallelProjection', 'ParallelScale', 'id', and 'name'.
      // Content:
      //  'ClippingRange', 'DirectionOfProjection', 'Distance', 'EyeAngle', 
      //  'FocalDisk', 'FocalPoint', 'ObliqueAngles', 'Position', 'Thickness', 
      //  'ViewAngle', 'ViewPlaneNormal', 'ViewShear', 'ViewUp', and 
      //  'WindowCenter'. 
      break;

    case eDisplayPointTag:
      // Attributes: 'x', 'y', and 'z'.
      break;

    case eLightCollectionTag:
      // Attributes: 'id' and 'name'.
      // Content: 'Light'.
      break;

    case ePixelAspectTag:
      // Attributes: 'x' and 'y'.
      break;

    case ePropCollectionTag:
      // Attributes: 'id' and 'name'.
      // Content: 'Assembly', 'CubeAxesActor2D', 'Follower', 'PropAssembly', 
      //        'Surface', and 'Volume 
      break;

    case eSurfaceCollectionTag:
      // Attributes: 'id' and 'name'.
      // Content: 'Surface'.
      break;

    case eViewPointTag:
      // Attributes: 'x', 'y', and 'z'.
      break;

    case eVolumeCollectionTag:
      // Attributes: 'id' and 'name'.
      // Content: 'Volume'.
      break;

    case eWorldPointTag:
      // Attributes: 'w', 'x', 'y', and 'z'.
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in Renderer.");

    } // switch (OpenElements)
}

// ----------------------------------------------------------------------------
void
vtkSessionImporter::PropHandler(const XML_Char*  aName,
                                const XML_Char** aAttrs)
{
  vtkAssembly*     pCurrentAssembly     = this->OpenAssemblies.top();
  vtkPropAssembly* pCurrentPropAssembly = this->OpenPropAssemblies.top();

  float rgba[4];

  switch (this->OpenElements.top())
    {

    case eFollowerTag:
      this->pCurrentProp = this->pCurrentActor = vtkFollower::New();
      break;

      // 
      // Surfaces
      // 

    case eSurfaceTag:
      this->pCurrentProp = this->pCurrentActor = vtkActor::New();
      break;

    case eMaterialTag:
      break;

    case eAmbientTag:
      GetRGBAAttrVals(rgba, aAttrs);
      vtkDebugMacro(<< this->Indent << "Ka = {" << rgba[0] << "," << rgba[1]
                    << "," << rgba[2] << "," << rgba[3] << "}");
      break;

    case eDiffuseTag:
      GetRGBAAttrVals(rgba, aAttrs);
      vtkDebugMacro(<< this->Indent << "Kd = {" << rgba[0] << "," << rgba[1]
                    << "," << rgba[2] << "," << rgba[3] << "}");
      break;

    case eSpecularTag:
      GetRGBAAttrVals(rgba, aAttrs);
      vtkDebugMacro(<< this->Indent << "Ks = {" << rgba[0] << "," << rgba[1]
                    << "," << rgba[2] << "," << rgba[3] << "}");
      break;

    case eEmissiveTag:
      GetRGBAAttrVals(rgba, aAttrs);
      vtkDebugMacro(<< this->Indent << "Ke = {" << rgba[0] << "," << rgba[1]
                    << "," << rgba[2] << "," << rgba[3] << "}");
      break;

    case ePropertyTag:
      GetPropertyAttrVals(this->pCurrentProperty, aAttrs);
      break;

      // 
      // Volumes
      // 

    case eVolumeTag:
      this->pCurrentProp = this->pCurrentVolume = vtkVolume::New();
      break;

    case eSegmentTag:
      break;

    case eRGBSegmentTag:
      break;

    case eHSVSegmentTag:
      break;

    case eOpacityTFuncTag:
      break;

    case eColorTFuncTag:
      break;

    case eGradientTFuncTag:
      break;

    case eVolumePropertyTag:
      break;

      // 
      // Groups / Assemblies
      // 

    case eAssemblyTag:
      this->pCurrentProp = vtkAssembly::New();
      break;

    case ePropAssemblyTag:
      this->pCurrentProp = vtkPropAssembly::New();
      break;

    case ePropCollectionTag:
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in Prop.");

    } // switch (OpenElements)
}

/* 
 * Element      Content
 * =======      =======
 * Camera       ClippingRange DirectionOfProjection Distance EyeAngle 
 *              FocalDisk FocalPoint ObliqueAngles Position Thickness 
 *              ViewAngle ViewPlaneNormal ViewShear ViewUp WindowCenter 
 */
void
vtkSessionImporter::CameraHandler(const XML_Char*  aName,
                                  const XML_Char** aAttrs)
{
  switch (this->OpenElements.top())
    {

    case eClippingRangeTag:
      break;

    case eDirectionOfProjectionTag:
      break;

    case eDistanceTag:
      break;

    case eEyeAngleTag:
      break;

    case eFocalDiskTag:
      break;

    case eFocalPointTag:
      break;

    case eObliqueAnglesTag:
      break;

    case ePositionTag:
      break;

    case eThicknessTag:
      break;

    case eViewAngleTag:
      break;

    case eViewPlaneNormalTag:
      break;

    case eViewShearTag:
      break;

    case eViewUpTag:
      break;

    case eWindowCenterTag:
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in CameraHandler().");

    } // switch (OpenElements)
}

/* 
 * Element           Content
 * =======           =======
 * Property          EdgeColor Material 
 * 
 * EdgeColor         EMPTY
 * Material          Ambient Composite Diffuse Emissive Specular 
 * 
 */
void
PropertyHandler(const XML_Char* aName, const XML_Char** aAttrs)
{
}

// ----------------------------------------------------------------------------
void
ActorHandler(const XML_Char* aName, const XML_Char** aAttrs)
{
}

// ----------------------------------------------------------------------------
void
vtkSessionImporter::FilterHandler(const XML_Char*  aName,
                                  const XML_Char** aAttrs)
{
  switch (this->OpenElements.top())
    {

    case eInputTag:
      break;

    case eOutputTag:
      break;

    case eAppendFilterTag:
      break;

    case eTriangleFilterTag:
      break;

    case eNormalsFilterTag:
      break;

    case eWindowedSincFilterTag:
      break;

    case eCleanFilterTag:
      break;

    case eDecimateProFilterTag:
      break;

    case eHullFilterTag:
      break;

    case eFeatureEdgesFilterTag:
      break;

    case eDepthSortFilterTag:
      break;

    case eSurfaceFilterCollectionTag:
      break;

    default:
      vtkErrorMacro(<< "Unexpected element \"" << vtkXMLString::ToLatin1(aName)
                    << "\" in CameraHandler().");

    } // switch (OpenElements)
}


/* 
 * End of: $Id: vtkSessionImporter.cxx,v 1.1.1.1 2006/12/19 23:00:09 christianh Exp $.
 * 
 */
