package caslayout.codegen;

import java.io.IOException;
import java.io.Writer;
import java.util.Map;
import java.util.StringTokenizer;

import caslayout.output.AbstractGenerator;
import caslayout.ui.ButtonDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CAGridLayout;
import caslayout.ui.CellConstraint;
import caslayout.ui.CheckBoxDisplayComponent;
import caslayout.ui.DropdownDisplayComponent;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.MultipleCellSpanConstraint;
import caslayout.ui.PercentCellConstraint;
import caslayout.ui.RadioButtonDisplayComponent;
import caslayout.ui.TextAreaDisplayComponent;
import caslayout.ui.TextDisplayComponent;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.util.CodegenUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: HTMLGenerator.java,v 1.11 2008/10/14 23:21:16 bozyurt Exp $
 */
public class HTMLGenerator extends AbstractGenerator {
   CAContainer root;

   public HTMLGenerator(CAContainer root) {
      this.root = root;
   }

   public void generate(Writer out) throws IOException {
      out.write("<html>\n<body>\n");
      doIndent(out, 1);
      out.write("<form method=\"get\" action=\"\">\n");

      generateTable(out, root, 2);
      doIndent(out, 1);
      out.write("</form>\n");
      out.write("</body>\n</html>\n");
   }

   protected int getIndexForCell(int row, int col, CAGridLayout layout) {
      int idx = 0;
      for (int i = 0; i < row - 1; i++) {
         idx += layout.getEffectiveColumnCount(i);
      }
      idx += col;
      return idx;
   }

   protected void generateTable(Writer out, CAContainer container, int level)
         throws IOException {
      doIndent(out, level);
      if (container == root) {
         tableStart(out, 1, "100%");
      } else {
         tableStart(out);
      }
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();
      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         doIndent(out, level + 1);
         tableRowStart(out);
         int cols = layout.getEffectiveColumnCount(i);

         TDColumn[] tdCols = CodegenUtils.prepareTableColumns(i, layout, cols,
               container, index);

         // boolean allColWidthsEqual =
         // CodegenUtils.areAllColumnWidthsEqual(layout, i, cols);
         // int totColSpan = 0;
         int offset = index;
         for (int j = 0; j < tdCols.length; j++) {

            doIndent(out, level + 2);
            // boolean startedTableColumn = false;
            out.write(tdCols[j].tagStart());

            if (index < container.getComponents().size()) {
               // int dcIdx = getIndexForCell(i,tdCols[ j ].col, layout );
               // System.out.println("dcIdx=" + dcIdx + " row=" + i + " col=" +
               // tdCols[ j ].col);
               IDisplayComponent ic = CodegenUtils.getDisplayComponent(
                     container, offset + tdCols[j].col);
               if (ic != null) {
                  if (ic instanceof CAContainer) {
                     out.write("\n");
                     generateTable(out, (CAContainer) ic, level + 3);
                  } else {
                     generateFormField(out, ic);
                  }
               } else {
                  out.write(" &nbsp; ");
               }
            }

            if ((offset + tdCols[j].col) >= container.getComponents().size()) {
               out.write(" &nbsp; ");
            }

            out.write(tdCols[j].tagEnd() + "\n");

         }// j

         index = offset + cols;

         doIndent(out, level + 1);
         tableRowEnd(out);
      }// i
      doIndent(out, level);
      tableEnd(out);
   }

   class ColumnPosInfo {
      int totColSpan = 0;
      double totPercent = 0;
      boolean first = true;
      boolean usePercent = false;

      public void updatePosition(CellConstraint cc) {
         first = false;
         if (cc instanceof MultipleCellSpanConstraint) {
            MultipleCellSpanConstraint mc = (MultipleCellSpanConstraint) cc;
            totColSpan += mc.getColSpan();
         } else {
            usePercent = true;
            PercentCellConstraint pcc = (PercentCellConstraint) cc;
            totPercent += pcc.getColPercent();
         }
      }

      public boolean atEnd(int colsInRow) {
         if (first) {
            return false;
         }
         if (usePercent)
            return totPercent >= 100;
         return colsInRow >= totColSpan;
      }
   }

   protected void generateFormField(Writer out, IDisplayComponent ic)
         throws IOException {
      if (ic instanceof TextDisplayComponent) {
         createText(out, (TextDisplayComponent) ic);
      } else if (ic instanceof TextFieldDisplayComponent) {
         createInput(out, (TextFieldDisplayComponent) ic);
      } else if (ic instanceof TextAreaDisplayComponent) {
         createTextarea(out, (TextAreaDisplayComponent) ic);
      } else if (ic instanceof CheckBoxDisplayComponent) {
         createCheckBox(out, (CheckBoxDisplayComponent) ic);
      } else if (ic instanceof RadioButtonDisplayComponent) {
         createRadioButton(out, (RadioButtonDisplayComponent) ic);
      } else if (ic instanceof DropdownDisplayComponent) {
         createDropdown(out, (DropdownDisplayComponent) ic);
      } else if (ic instanceof ButtonDisplayComponent) {
         createButton(out, (ButtonDisplayComponent) ic);
      }
   }

   protected String wrapSpan(String s, String cssClass) {
      if (cssClass == null || cssClass.trim().length() == 0)
         return s;

      StringBuilder sb = new StringBuilder(s.length() + 20);
      sb.append("<span class=\"").append(cssClass).append("\"> ");
      sb.append(s).append(" </span>");
      return sb.toString();
   }

   protected void createText(Writer out, TextDisplayComponent tdc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      StringTokenizer stok = new StringTokenizer(tdc.getLabel().getText(), "\n");
      while (stok.hasMoreTokens()) {
         buf.append(stok.nextToken());
         if (stok.hasMoreTokens())
            buf.append("<br>\n");
      }

      out.write(wrapSpan(buf.toString(), tdc.getCSSClass()));
   }

   protected void createInput(Writer out, TextFieldDisplayComponent tfdc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<input ");
      buf.append("size=").append(tfdc.getFieldLength()).append(" maxlength=");
      buf.append(tfdc.getMaxFieldLength());
      buf.append(" name=\"\" value=\"\">");
      out.write(wrapSpan(buf.toString(), tfdc.getCSSClass()));
   }

   protected void createTextarea(Writer out, TextAreaDisplayComponent tadc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<textarea ");
      buf.append("cols=").append(tadc.getNumCols()).append(" rows=");
      buf.append(tadc.getNumRows());
      buf.append(" name=\"\" value=\"\">");
      out.write(wrapSpan(buf.toString(), tadc.getCSSClass()));
   }

   protected void createCheckBox(Writer out, CheckBoxDisplayComponent cdc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<input type=\"checkbox\" ");
      buf.append(" name=\"\" value=\"\">");
      buf.append("&nbsp;").append(cdc.getText());
      buf.append("</input>");
      out.write(wrapSpan(buf.toString(), cdc.getCSSClass()));
   }

   protected void createRadioButton(Writer out, RadioButtonDisplayComponent rdc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<input type=\"radio\" ");
      buf.append(" name=\"\" value=\"\">");
      buf.append("&nbsp;").append(rdc.getText());
      buf.append("</input>");
      out.write(wrapSpan(buf.toString(), rdc.getCSSClass()));
   }

   protected void createDropdown(Writer out, DropdownDisplayComponent ddc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<select name=\"dummy\">\n");
      for(Map.Entry<String, String> entry : ddc.getLabelValueMap().entrySet()) {
         String label = entry.getKey();
         String value = entry.getValue();
         buf.append("\t<option value=\"").append(value).append("\">").append(
               label).append("</option>\n");
      }
      buf.append("</select>\n");
      out.write(wrapSpan(buf.toString(), ddc.getCSSClass()));
   }

   protected void createButton(Writer out, ButtonDisplayComponent bdc)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<input type=\"submit\" ");
      buf.append(" name=\"\" value=\"");
      buf.append(bdc.getAction().getValue()).append("\">");
      buf.append("</input>");
      out.write(wrapSpan(buf.toString(), bdc.getCSSClass()));
   }

   protected void tableStart(Writer out) throws IOException {
      tableStart(out, 0, "100%");
   }

   protected void tableStart(Writer out, int border) throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<table cellpadding=0 cellspacing=0 border=").append(border)
            .append("\">\n");
      out.write(buf.toString());
   }

   protected void tableStart(Writer out, int border, String width)
         throws IOException {
      StringBuilder buf = new StringBuilder();
      buf.append("<table cellpadding=0 cellspacing=0 border=").append(border)
            .append(" width=\"");
      buf.append(width).append("\">\n");
      out.write(buf.toString());
   }

   protected void tableEnd(Writer out) throws IOException {
      out.write("</table>\n");
   }

   protected void tableRowStart(Writer out) throws IOException {
      out.write("<tr>\n");
   }

   protected void tableRowEnd(Writer out) throws IOException {
      out.write("</tr>\n");
   }

   protected void columnEnd(Writer out) throws IOException {
      out.write("</td>\n");
   }
}
