package caslayout.codegen;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import caslayout.util.GenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: SourceFile.java,v 1.2 2008/10/13 23:58:06 bozyurt Exp $
 */
public class SourceFile {
	protected File file;
	protected List<String> lines = new ArrayList<String>();
	public final static int CASE_INSENSITIVE = 1;
	public final static int REGEX = 2;
	public final static int EXACT = 4;

	public SourceFile(File file) {
		this.file = file;
	}

	public void load() throws IOException {
		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(file));
			String line = null;
			while ((line = in.readLine()) != null) {
				lines.add(line);
			}
		} finally {
			GenUtils.close(in);
		}
	}
   
	public void save(boolean doBackup) throws IOException {
	   File backupFile = new File( file.getParent(), file.getName() + ".bak");
	   BufferedWriter out = null;
	   try {
	      if ( doBackup) {
	         GenUtils.copyFile(file.getAbsolutePath(), backupFile.getAbsolutePath());
	      }
	      out = new BufferedWriter( new FileWriter(file));
	      for (Iterator<String> iter = lines.iterator(); iter.hasNext();) {
	         String line = iter.next();
	         out.write(line);
	         out.newLine();
	      }
	   } finally {
	      GenUtils.close(out);
	   }
	}

   public int indexOf(String pattern) {
      return indexOf(pattern, EXACT, 0, lines.size());
   }
   
   public int indexOf(String pattern, int startIdx) {
      return indexOf(pattern, EXACT, startIdx, lines.size());
   }
   
   public int indexOf(String pattern, int startIdx, int endIdx) {
      return indexOf(pattern, EXACT, startIdx, endIdx);
   }
   
   
   public int indexOfRegex(String regex) {
      return indexOf(regex, REGEX, 0, lines.size());
   }
   
   public int indexOfRegex(String regex, int startIdx) {
      return indexOf(regex, REGEX, startIdx, lines.size());
   }
   
	public int indexOf(String pattern, int flags, int startIdx, int endIdx) {
      Pattern p = null;
		if ((flags & CASE_INSENSITIVE) > 0) {
			pattern = pattern.toLowerCase();
		} else if ( (flags & REGEX) > 0) {
          p = Pattern.compile(pattern);    
      }
      
		// int size = lines.size();
		for (int i = startIdx; i < endIdx; i++) {
			String line = lines.get(i);
			if ((flags & EXACT) > 0) {
				if ((flags & CASE_INSENSITIVE) > 0) {
					if (line.toLowerCase().indexOf(pattern) != -1) {
						return i;
					}
				} else {
					if (line.indexOf(pattern) != -1) {
						return i;
					}
				}
			} else if ( (flags & REGEX) > 0) {
             Matcher m = p.matcher(line);
             if ( m.find()) {
                return i;
             }
         }
		}
		return -1;
	}
   
   public void insertLineBefore(String line, int lineIdx) {
      lines.add(lineIdx, line);
   }
   
   public void insertLineAfter(String line, int lineIdx) {
      lines.add(lineIdx + 1, line);
   }
     
   public String getLineAt(int lineIdx) {
      return lines.get(lineIdx);
   }
   
   public int getLineIndentAt(int lineIdx) {
      String line = lines.get(lineIdx);
      if ( line.length() == 0 || line.trim().length() == 0) {
         return 0;
      }
      char[] carr = line.toCharArray();
      int idx = 0;
      while (idx < carr.length &&  Character.isWhitespace(carr[idx]) ) {
         ++idx;
      }
      return idx;
   }

   public int indexOfFirstNonEmptyLineBefore(int startIdx) {
      if ( startIdx == 0) {
         return -1;
      }
      int idx = startIdx - 1;
      do {
        String line = lines.get(idx);
        if ( line.trim().length() > 0) {
           return idx;
        }
        idx--;
      } while (idx >= 0);
      return -1;  
   }
}
