package caslayout.codegen;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.StringTokenizer;

import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import caslayout.exception.CodeGenException;
import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.ButtonDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CAGridLayout;
import caslayout.ui.CheckBoxDisplayComponent;
import caslayout.ui.RadioButtonDisplayComponent;
import caslayout.ui.TextAreaDisplayComponent;
import caslayout.ui.TextDisplayComponent;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.ui.model.AssessmentAssociation;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.ScoreAssociation;
import caslayout.util.CodegenUtils;
import caslayout.util.XMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: XFormsGenerator.java,v 1.8 2008/10/14 23:21:16 bozyurt Exp $
 */
public class XFormsGenerator {
   protected CAContainer root;
   protected int indent = 2;
   final static int RADIO_BUTTON = 1;
   final static int CHECK_BOX = 2;

   final static int SINGLE = 1;
   final static int LOGICAL_GROUP = 2;
   final static int CONTAINER = 3;

   public XFormsGenerator(CAContainer root) {
      this.root = root;
   }

   public void generate(Writer out) throws IOException, CodeGenException {
      out.write("<caslayout");
      out.write("\txmlns:ev=\"http://www.w3.org/2001/xml-events\"\n");
      out.write("\txmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n");
      out.write("\txmlns:xs=\"http://www.w3.org/2001/XMLSchema\"\n");
      out.write("\txmlns:calm=\"http://www.nbirn.net\"\n");

      out.write("\txmlns:xforms=\"http://www.w3.org/2002/xforms\">\n\n");

      generateModelPortion("calm", out, 1);
      generateTag(out, root, 1);

      out.write("</caslayout>\n");
   }

   protected void generateModelPortion(String modelID, Writer out, int level)
         throws IOException {
      AssociationHelper ah = AssociationHelper.getInstance();
      AssessmentAssociation aa = ah.getAsAssoc();
      if (aa == null)
         return;

      doIndent(out, level);
      out.write("<xforms:model id=\"");
      out.write(modelID);
      out.write("\">\n");

      AssessmentInfo ai = aa.getLeft();
      Namespace ns = Namespace.getNamespace("calm", "http://www.nbirn.net");
      Element e = ai.toXFormsXML(ns);
      XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat()); // "
                                                                        // ",true);
      // xout.setLineSeparator( System.getProperty("line.separator") );

      StringWriter sw = new StringWriter(2000);
      xout.output(e, sw);
      doIndent(out, level + 1);
      out.write(sw.toString());
      out.write("\n\n");

      // now write the bindings also
      for (Object element : ah.getAssocMap().values()) {
         ScoreAssociation sa = (ScoreAssociation) element;
         generateBind(out, level + 1, sa);
      }

      for (Object element : ah.getMandatoryFieldsMap().values()) {
         MandatoryFieldAssociation mfa = (MandatoryFieldAssociation) element;
         generateBind(out, level + 1, mfa);
      }

      // write the submission tag
      doIndent(out, level + 1);
      out
            .write("<xforms:submission id=\"asform\" method=\"post\" action=\"/assessmentman\"/>\n");

      doIndent(out, level);
      out.write("</xforms:model>\n");
   }

   protected void generateBind(Writer out, int level, ScoreAssociation sa)
         throws IOException {
      doIndent(out, level);
      String id = "";
      out.write("<xforms:bind nodeset=\"calm:assessment/calm:score[@id='");
      if (sa.getRight() != null) {
         id = sa.getRHSID();
      }
      out.write(id);
      out.write("']/..\" id=\"");
      out.write(id);
      out.write("\"/>\n");
   }

   protected void generateBind(Writer out, int level,
         MandatoryFieldAssociation mfa) throws IOException {
      doIndent(out, level);
      String id = "";
      out.write("<xforms:bind nodeset=\"calm:assessment/calm:field[@id='");
      if (mfa.getRight() != null) {
         id = mfa.getRight().getId();
      }
      out.write(id);
      out.write("']/..\" id=\"");
      out.write(id);
      out.write("\"/>\n");
   }

   protected void generateTag(Writer out, CAContainer container, int level)
         throws IOException {

      List<DisplayComponentInfo> dcInfos = prepareDisplayComponentInfos(container);

      for (Iterator<DisplayComponentInfo> iter = dcInfos.iterator(); iter.hasNext();) {
         DisplayComponentInfo dci = iter.next();
         if (dci.getType() == SINGLE) {
            prepareSingleComponent(out, dci.getComponent(), level);
         } else if (dci.getType() == LOGICAL_GROUP) {
            prepareGroup(out, dci, level);
         } else if (dci.getType() == CONTAINER) {
            generateTag(out, (CAContainer) dci.getComponent(), level + 1);
         }
      }
   }

   protected void prepareSingleComponent(Writer out,
         AbstractDisplayComponent adc, int level) throws IOException {
      if (adc instanceof TextFieldDisplayComponent) {
         createInputTag(out, null, (TextFieldDisplayComponent) adc, level + 1);
      } else if (adc instanceof TextAreaDisplayComponent) {
         createTextareaTag(out, null, (TextAreaDisplayComponent) adc, level + 1);
      } else if (adc instanceof TextDisplayComponent) {
         createLabelTag(out, (TextDisplayComponent) adc, level + 1);
      } else if (adc instanceof ButtonDisplayComponent) {
         createSubmitTag(out, (ButtonDisplayComponent) adc, level + 1);
      }
   }

   protected void prepareGroup(Writer out, DisplayComponentInfo dci, int level)
         throws IOException {
      String groupID = dci.getComponent().getGroupID();
      writeGroupStart(out, level + 1, groupID);

      if (dci.areAllSameComponentType(RADIO_BUTTON)) {
         generateSelect1Element(out, dci, "", level + 1);
      } else if (dci.areAllSameComponentType(CHECK_BOX)) {
         generateSelectElement(out, dci, "", level + 1);
      } else {
         for (Iterator<AbstractDisplayComponent> iter = dci.getComponents().iterator(); iter.hasNext();) {
            AbstractDisplayComponent adc = iter
                  .next();
            prepareSingleComponent(out, adc, level + 1);
         }
      }
      writeGroupEnd(out, level + 1);
   }

   protected void writeGroupStart(Writer out, int level, CAContainer c)
         throws IOException {
      doIndent(out, level);
      out.write("<xforms:group ref=\"");
      out.write(c.getId());
      out.write("\">\n");
   }

   protected void writeGroupStart(Writer out, int level, String groupID)
         throws IOException {
      doIndent(out, level);
      out.write("<xforms:group ref=\"");
      out.write(groupID);
      out.write("\">\n");
   }

   protected void writeGroupEnd(Writer out, int level) throws IOException {
      doIndent(out, level);
      out.write("</xforms:group>\n");
   }

   protected void generateSelect1Element(Writer out, DisplayComponentInfo dci,
         String label, int level) throws IOException {
      String groupID = dci.getComponent().getGroupID();

      doIndent(out, level);

      out.write("<xforms:select1 ref=\"");
      out.write(groupID);
      out.write("\">\n");

      doIndent(out, level + 1);
      out.write("<calm:id value=\"");
      out.write(groupID);
      out.write("\"/>\n");

      if (label != null) {
         doIndent(out, level + 1);
         out.write("<xforms:label>");
         out.write(XMLUtils.toXML(label));
         out.write("</xforms:label>\n");
      }

      for (Iterator<AbstractDisplayComponent> iter = dci.getComponents().iterator(); iter.hasNext();) {
         AbstractDisplayComponent adc = iter.next();

         doIndent(out, level + 1);
         out.write("<xforms:item>\n");
         if (adc instanceof RadioButtonDisplayComponent) {
            RadioButtonDisplayComponent rdc = (RadioButtonDisplayComponent) adc;
            out.write("<calm:id value=\"");
            out.write(rdc.getId());
            out.write("\"/>\n");
            CodegenUtils.prepareCALMTagForCSS(out, rdc, level + 1, indent);

            doIndent(out, level + 2);
            out.write("<xforms:label>");
            out.write(XMLUtils.toXML(rdc.getText()));
            out.write("</xforms:label>\n");
            doIndent(out, level + 2);
            out.write("<xforms:value>");
            if (rdc.getValue() != null)
               out.write(XMLUtils.toXML(rdc.getValue()));
            else
               out.write(XMLUtils.toXML(rdc.getText()));
            out.write("</xforms:value>\n");
         } else if (adc instanceof CheckBoxDisplayComponent) {
            CheckBoxDisplayComponent cdc = (CheckBoxDisplayComponent) adc;
            out.write("<calm:id value=\"");
            out.write(cdc.getId());
            out.write("\"/>\n");
            CodegenUtils.prepareCALMTagForCSS(out, cdc, level + 1, indent);

            doIndent(out, level + 2);
            out.write("<xforms:label>");
            out.write(XMLUtils.toXML(cdc.getText()));
            out.write("</xforms:label>\n");
            doIndent(out, level + 1);
            out.write("<xforms:value>");
            if (cdc.getValue() != null)
               out.write(XMLUtils.toXML(cdc.getValue()));
            else
               out.write(XMLUtils.toXML(cdc.getText()));
            out.write("</xforms:value>\n");
         }
         doIndent(out, level + 1);
         out.write("</xforms:item>\n");

      }
      doIndent(out, level);
      out.write("</xforms:select1>\n");
   }

   protected void generateSelectElement(Writer out, DisplayComponentInfo dci,
         String label, int level) throws IOException {
      String groupID = dci.getComponent().getGroupID();

      doIndent(out, level);
      doIndent(out, level + 1);
      out.write("<calm:id value=\"");
      out.write(groupID);
      out.write("\"/>\n");

      out.write("<xforms:select ref=\"");
      out.write(groupID);
      out.write("\" appearance=\"full\">\n");

      if (label != null) {
         doIndent(out, level + 1);
         out.write("<xforms:label>");
         out.write(label);
         out.write("</xforms:label>\n");
      }

      doIndent(out, level + 1);
      out.write("<xforms:choices>\n");

      for (Iterator<AbstractDisplayComponent> iter = dci.getComponents().iterator(); iter.hasNext();) {
         AbstractDisplayComponent adc = iter.next();

         doIndent(out, level + 2);
         out.write("<xforms:item>\n");
         if (adc instanceof RadioButtonDisplayComponent) {
            RadioButtonDisplayComponent rdc = (RadioButtonDisplayComponent) adc;
            out.write("<calm:id value=\"");
            out.write(rdc.getId());
            out.write("\"/>\n");
            CodegenUtils.prepareCALMTagForCSS(out, rdc, level + 1, indent);

            doIndent(out, level + 3);
            out.write("<xforms:label>");
            out.write(XMLUtils.toXML(rdc.getText()));
            out.write("</xforms:label>\n");
            doIndent(out, level + 3);
            out.write("<xforms:value>");
            out.write(XMLUtils.toXML(rdc.getText()));
            out.write("</xforms:value>\n");
         } else if (adc instanceof CheckBoxDisplayComponent) {
            CheckBoxDisplayComponent cdc = (CheckBoxDisplayComponent) adc;
            out.write("<calm:id value=\"");
            out.write(cdc.getId());
            out.write("\"/>\n");
            CodegenUtils.prepareCALMTagForCSS(out, cdc, level + 1, indent);

            doIndent(out, level + 3);
            out.write("<xforms:label>");
            out.write(XMLUtils.toXML(cdc.getText()));
            out.write("</xforms:label>\n");
            doIndent(out, level + 3);
            out.write("<xforms:value>");
            out.write(XMLUtils.toXML(cdc.getText()));
            out.write("</xforms:value>");
         }
         doIndent(out, level + 2);
         out.write("</xforms:item>\n");
      }
      doIndent(out, level + 1);
      out.write("</xforms:choices>\n");
      doIndent(out, level);
      out.write("</xforms:select>\n");
   }

   protected void createOutputTag(Writer out, TextDisplayComponent tdc)
         throws IOException {
      StringBuffer buf = new StringBuffer();
      buf.append("<xforms:output ref=\"");
      out.write(tdc.getId());
      buf.append("\">");

      out.write("<calm:id value=\"");
      out.write(tdc.getId());
      out.write("\"/>\n");
      StringTokenizer stok = new StringTokenizer(tdc.getLabel().getText(), "\n");
      while (stok.hasMoreTokens()) {
         buf.append(stok.nextToken());
         if (stok.hasMoreTokens()) {
            buf.append("&lt;br&gt;\n");

         }
      }
      buf.append("</xforms:output>");
      out.write(buf.toString());
   }

   protected void createLabelTag(Writer out, TextDisplayComponent tdc, int level)
         throws IOException {
      doIndent(out, level);
      out.write("<xforms:label ref=\"");
      out.write(tdc.getId());
      out.write("\">");
      out.write("<calm:id value=\"");
      out.write(tdc.getId());
      out.write("\"/>\n");
      CodegenUtils.prepareCALMTagForCSS(out, tdc, level + 1, indent);

      String xmlStr = XMLUtils.toXML(tdc.getLabel().getText());
      StringTokenizer stok = new StringTokenizer(xmlStr, "\n");
      StringBuffer buf = new StringBuffer(xmlStr.length() + 20);
      while (stok.hasMoreTokens()) {
         buf.append(stok.nextToken());
         if (stok.hasMoreTokens()) {
            buf.append(" &lt;br&gt;\n");
         }
      }

      out.write(buf.toString());
      out.write("</xforms:label>\n");
   }

   protected void createInputTag(Writer out, String label,
         TextFieldDisplayComponent tfdc, int level) throws IOException {
      doIndent(out, level);
      out.write("<xforms:input ref=\"");
      out.write(tfdc.getId());
      out.write("\">\n");
      out.write("<calm:id value=\"");
      out.write(tfdc.getId());
      out.write("\"/>\n");
      CodegenUtils.prepareCALMTagForCSS(out, tfdc, level + 1, indent);
      if (label != null) {
         doIndent(out, level + 1);
         out.write("<xforms:label>");
         out.write(XMLUtils.toXML(label));
         out.write("</label>\n");
      }
      doIndent(out, level);
      out.write("</xforms:input>\n");
   }

   protected void createTextareaTag(Writer out, String label,
         TextAreaDisplayComponent tadc, int level) throws IOException {
      doIndent(out, level);
      out.write("<xforms:textarea ref=\"");
      out.write(tadc.getId());
      out.write("\">\n");
      doIndent(out, level);
      out.write("<calm:id value=\"");
      out.write(tadc.getId());
      out.write("\"/>\n");
      doIndent(out, level);
      out.write("<calm:rows value=\"");
      out.write(String.valueOf(tadc.getNumRows()));
      out.write("\"/>\n");
      doIndent(out, level);
      out.write("<calm:cols value=\"");
      out.write(String.valueOf(tadc.getNumCols()));
      out.write("\"/>\n");

      CodegenUtils.prepareCALMTagForCSS(out, tadc, level + 1, indent);
      if (label != null) {
         doIndent(out, level + 1);
         out.write("<xforms:label>");
         out.write(XMLUtils.toXML(label));
         out.write("</label>\n");
      }
      doIndent(out, level);
      out.write("</xforms:textarea>\n");
   }

   protected void createSubmitTag(Writer out, ButtonDisplayComponent bdc,
         int level) throws IOException {
      doIndent(out, level);
      out.write("<xforms:submit submission=\"asform\">\n");
      doIndent(out, level + 1);
      out.write("<calm:id value=\"");
      out.write(bdc.getId());
      out.write("\"/>\n");
      CodegenUtils.prepareCALMTagForCSS(out, bdc, level + 1, indent);
      doIndent(out, level + 1);
      out.write("<xforms:label>");
      out.write(XMLUtils.toXML(bdc.getLabel()));
      out.write("</xforms:label>\n");
      doIndent(out, level);
      out.write("</xforms:submit>\n");
   }

   protected void doIndent(Writer out, int level) throws IOException {
      for (int i = 0; i < level * indent; i++) {
         out.write(' ');
      }
   }

   /**
    * Given a container, returns the <code>DisplayComponentInfo</code> object
    * list containing all logically grouped components as a single
    * <code>DisplayComponentInfo</code> object and each encountered container
    * as a separate <code>DisplayComponentInfo</code> object.
    *
    * @param container
    *           a Layout container containing display components or other layout
    *           containers
    *
    * @return
    */
   protected List<DisplayComponentInfo> prepareDisplayComponentInfos(CAContainer container) {
      List<DisplayComponentInfo> dcInfos = new LinkedList<DisplayComponentInfo>();
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();

      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         int cols = layout.getEffectiveColumnCount(i);

         for (int j = 0; j < cols; j++) {

            if (index < container.getComponents().size()) {
               AbstractDisplayComponent adc = (AbstractDisplayComponent) container
                     .getComponents().get(index);
               if (adc != null) {
                  if (adc instanceof CAContainer) {
                     DisplayComponentInfo dci = new DisplayComponentInfo(
                           CONTAINER);
                     dci.addComponent(adc);
                     dcInfos.add(dci);
                  } else {
                     if (adc.getGroupID() != null) {
                        DisplayComponentInfo dci = findLogicalGroup(dcInfos,
                              adc.getGroupID());
                        if (dci == null) {
                           dci = new DisplayComponentInfo(LOGICAL_GROUP);
                           dcInfos.add(dci);
                        }
                        dci.addComponent(adc);
                     } else {
                        // individual display components
                        DisplayComponentInfo dci = new DisplayComponentInfo(
                              SINGLE);
                        dci.addComponent(adc);
                        dcInfos.add(dci);
                     }
                  }
               } // adc != null

            }
            index++;
         }// j
      }
      return dcInfos;
   }

   /**
    * within the given list of <code>DisplayComponentInfo</code> objects,
    * finds and returns the <code>LogicalGroup</code> with the given group id,
    * otherwise returns null
    *
    * @param dcInfos
    *           list of <code>DisplayComponentInfo</code> objects
    * @param groupID
    * @return the <code>DisplayComponentInfo</code> object with the given
    *         groupID or null
    */
   protected DisplayComponentInfo findLogicalGroup(List<DisplayComponentInfo> dcInfos, String groupID) {
      for (Iterator<DisplayComponentInfo> iter = dcInfos.iterator(); iter.hasNext();) {
         DisplayComponentInfo dci = iter.next();
         if (dci.getType() == LOGICAL_GROUP) {
            AbstractDisplayComponent adc = dci.getComponent();
            if (adc.getGroupID().equals(groupID))
               return dci;
         }
      }
      return null;
   }

   public static class DisplayComponentInfo {
      int type;
      List<AbstractDisplayComponent> components = new ArrayList<AbstractDisplayComponent>();

      public DisplayComponentInfo(int type) {
         this.type = type;
      }

      public AbstractDisplayComponent getComponent() {
         if (components.isEmpty())
            return null;
         return components.get(0);
      }

      public List<AbstractDisplayComponent> getComponents() {
         return components;
      }

      public int getType() {
         return type;
      }

      public void addComponent(AbstractDisplayComponent adc) {
         components.add(adc);
      }

      public boolean areAllSameComponentType(int compType) {
         for (Iterator<AbstractDisplayComponent> iter = components.iterator(); iter.hasNext();) {
            AbstractDisplayComponent adc = iter
                  .next();
            switch (compType) {
            case RADIO_BUTTON:
               if (!(adc instanceof RadioButtonDisplayComponent))
                  return false;
               break;
            case CHECK_BOX:
               if (!(adc instanceof CheckBoxDisplayComponent))
                  return false;
               break;
            default:
               break;
            }
         }
         return true;
      }

      public String toString() {
         StringBuffer sb = new StringBuffer(128);
         sb.append("DisplayComponentInfo::[");
         sb.append("num components=").append(components.size());
         for (Iterator<AbstractDisplayComponent> iter = components.iterator(); iter.hasNext();) {
            AbstractDisplayComponent adc = iter
                  .next();
            sb.append("\n").append(adc.toString());
         }
         sb.append(']');
         return sb.toString();
      }
   }
}
