package caslayout.codegen.struts;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;

import org.apache.log4j.Logger;

import caslayout.output.AbstractGenerator;
import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.CheckBoxDisplayComponent;
import caslayout.ui.Document;
import caslayout.ui.DropdownDisplayComponent;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.LogicalGroup;
import caslayout.ui.QuestionGroup;
import caslayout.ui.QuestionGroupRepository;
import caslayout.ui.model.AssessmentAssociation;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.FieldInfo;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.ScoreAssociation;
import caslayout.ui.model.ScoreInfo;
import caslayout.util.CodegenUtils;
import caslayout.util.GenUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: FormBeanGenerator.java,v 1.28 2008/10/14 23:21:16 bozyurt Exp $
 */

public class FormBeanGenerator extends AbstractGenerator {
   protected AssociationHelper assocHelper;
   protected StrutsCodegenConfig config;
   protected Document doc;
   protected QuestionGroupRepository qgRep;
   protected Properties props;
   protected static Logger log = Logger.getLogger(FormBeanGenerator.class);
   public final static String TIMESTAMP = "timestamp";
   public final static String INTEGER = "integer";
   public final static String FLOAT = "float";
   public final static String VARCHAR = "varchar";
   public final static String BOOLEAN = "boolean";

   public FormBeanGenerator(AssociationHelper assocHelper,
         StrutsCodegenConfig config, Document doc) throws IOException {
      this.assocHelper = assocHelper;
      this.config = config;
      this.doc = doc;
      qgRep = QuestionGroupRepository.getInstance();
      props = GenUtils.loadProperties("caslayout.properties");
   }

   public void generate(Writer out) throws IOException {
      writeHeader(out, 1);
      out.write("\n");
      writeInstanceVariables(out, 2);
      writeStandardInstanceVariables(out, 2);
      writeStaticVariables(out, 2);
      out.write("\n");
      writeConstructor(out, 2);
      out.write("\n");
      writeAccessors(out, 2, "get");
      out.write("\n");
      writeAccessors(out, 2, "set");
      out.write("\n");
      writeAccessors(out, 2, "add");
      out.write("\n");
      writeAccessors(out, 2, "removeLast");
      out.write("\n");
      writeAccessors(out, 2, "size");
      out.write("\n");
      writeQuestionCounterMethods(out, 2);
      out.write("\n");
      writeStandardAcessors(out, 2);

      // writeValidateMethod(out, 2);
      createInnerClassAndStaticVariables(out, 2);
      createGetVariableMapMethod(out, 2);
      createGetMandatoryFieldMetaDataMethod(out, 2);

      createPageQuestionInfoClass(out, 2);
      createGetPageQuestionsMapMethod(out, 2);

      createHiddenVariablesAndAccessors(out, 2);

      createResetMethod(out, 2);
      writeFooter(out);
   }

   protected void writeHeader(Writer out, int level) throws IOException {
      out.write("package ");
      out.write(config.getPackageName());
      out.write(";\n\n");
      out.write("import java.util.*;\n");
      out.write("import javax.servlet.http.*;\n");
      out.write("import java.text.*;\n\n");

      out.write("import org.apache.commons.logging.*;\n");
      out.write("import org.apache.struts.action.*;\n");
      out.write("import org.apache.struts.util.*;\n");

      out.write("/**\n  *\n");
      out.write("  * @version $Id" + "$\n");
      out
            .write("  * @author Generated by Clinical Assessment Layout Manager (CALM)\n");
      out.write("  */\n\n");

      out.write("public class ");
      out.write(config.getFormBeanName());
      out.write(" extends ActionForm {\n");
   }

   protected void writeFooter(Writer out) throws IOException {
      out.write("}\n");
   }

   protected void writeInstanceVariables(Writer out, int level)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;

         doIndent(out, level);
         String varName = GenUtils.convertToJavaVariableName(si.getScoreName());

         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.warn("Score " + si.getScoreName()
                  + " is not associated with any component!");
            continue;
         }
         QuestionGroup qg = qgRep.findMatching(sa.getRHSID(), sa);
         if (qg != null) {
            if (qg.isMultiAnswer()) {
               out.write("private List " + varName + " = new ArrayList();\n");
               continue;
            }
         }

         if (si.getScoreType().equalsIgnoreCase("integer")) {
            if (sa.getRight() != null
                  && sa.getRight() instanceof CheckBoxDisplayComponent) {
               out.write("private boolean ");
            } else {
               // for missing values representation and since form beans are
               // essentially data for the view (JSP), use Strings (IBO)
               // out.write("private int ");
               out.write("private String ");
            }
         } else if (si.getScoreType().equalsIgnoreCase(VARCHAR)) {
            out.write("private String ");
         } else if (si.getScoreType().equalsIgnoreCase(FLOAT)) {
            // out.write("private float ");
            out.write("private String ");
         } else if (si.getScoreType().equalsIgnoreCase(BOOLEAN)) {
            out.write("private boolean ");
         } else if (si.getScoreType().equalsIgnoreCase(TIMESTAMP)) {
            out.write("private String ");
         } else {
            throw new RuntimeException("Unsupported data type:"
                  + si.getScoreType());
         }
         out.write(varName);

         if (si.getScoreType().equalsIgnoreCase(VARCHAR)) {
            out.write(" = \"\";\n");
         } else if (si.getScoreType().equalsIgnoreCase(INTEGER)) {
            if (sa.getRight() != null
                  && sa.getRight() instanceof CheckBoxDisplayComponent) {
               out.write(" = false;\n");
            } else {
               // out.write(" = -10000;\n");
               out.write("= \"\";\n");
            }
         } else if (si.getScoreType().equalsIgnoreCase(FLOAT)) {
            // out.write(" = -10000f;\n");
            out.write("= \"\";\n");
         } else if (si.getScoreType().equalsIgnoreCase(BOOLEAN)) {
            out.write(" = false;\n");
         } else if (si.getScoreType().equalsIgnoreCase(TIMESTAMP)) {
            out.write(" = \"\";\n");
         } else {
            out.write(";\n");
         }
      }

      // write mandatory fields also
      for (Object element : asi.getMandatoryFields()) {
         FieldInfo fi = (FieldInfo) element;
         MandatoryFieldAssociation mfa = assocHelper
               .findMandatoryFieldAssociation(fi.getName());
         doIndent(out, level);
         // dynamic dropdowns are handled differently
         if (mfa.getRight() instanceof DropdownDisplayComponent
               && ((DropdownDisplayComponent) mfa.getRight())
                     .getPopulatedFromDatabase()) {
            out.write("private DynamicDropDownSelector ");
            out.write(fi.getName());
            out.write(";\n");
         } else {
            if (fi.getType().equalsIgnoreCase("string")) {
               out.write("private String ");
               out.write(fi.getName());
               out.write(" = \"\";\n");
            } else {
               throw new RuntimeException("Unsupported data type:"
                     + fi.getType());
            }
         }
      }
   }

   protected void writeStandardInstanceVariables(Writer out, int level)
         throws IOException {
      out.write("\n");
      print(out, level,
            "// variables standard for each generated Assessment Struts Form Bean\n");
      print(out, level, "private String assessmentID;\n");
      print(out, level, "private String subjectID;\n");
      print(out, level, "private String experimentID;\n");
      print(out, level, "private String visitID;\n");
      print(out, level, "private String segmentID;\n");
      print(out, level, "private int currentPageIdx = 1;\n");
      print(out, level, "private String selectedQuestion = \"\";\n");
      print(out, level, "private String skippedQuestions = \"\";\n");
      print(out, level, "private String dataClassificationForSkipped = \"\";\n");
   }

   protected void writeStaticVariables(Writer out, int level)
         throws IOException {
      out.write("\n");
      print(out, level, "private static Map pageQuestionsMap;\n");
      print(out, level, "private static Map variableMap = null;\n");
      print(out, level,
            "private static Map mandatoryFieldMetaDataMap = null;\n");

   }

   protected void writeStandardAcessors(Writer out, int level)
         throws IOException {
      out.write("\n");
      doIndent(out, level);
      writeGetter(out, "String", "getAssessmentID", "assessmentID");
      doIndent(out, level);
      writeGetter(out, "String", "getSubjectID", "subjectID");
      doIndent(out, level);
      writeGetter(out, "String", "getExperimentID", "experimentID");
      doIndent(out, level);
      writeGetter(out, "String", "getVisitID", "visitID");
      doIndent(out, level);
      writeGetter(out, "String", "getSegmentID", "segmentID");
      doIndent(out, level);
      writeGetter(out, "int", "getCurrentPageIdx", "currentPageIdx");
      doIndent(out, level);
      writeGetter(out, "String", "getSelectedQuestion", "selectedQuestion");
      doIndent(out, level);
      writeGetter(out, "String", "getSkippedQuestions", "skippedQuestions");
      doIndent(out, level);
      writeGetter(out, "String", "getDataClassificationForSkipped",
            "dataClassificationForSkipped");

      out.write("\n");
      doIndent(out, level);
      writeSetter(out, "String", "setAssessmentID", "assessmentID", "");
      doIndent(out, level);
      writeSetter(out, "String", "setSubjectID", "subjectID", "");
      doIndent(out, level);
      writeSetter(out, "String", "setExperimentID", "experimentID", "");
      doIndent(out, level);
      writeSetter(out, "String", "setVisitID", "visitID", "");
      doIndent(out, level);
      writeSetter(out, "String", "setSegmentID", "segmentID", "");
      doIndent(out, level);
      writeSetter(out, "int", "setCurrentPageIdx", "currentPageIdx", "");
      doIndent(out, level);
      writeSetter(out, "String", "setSelectedQuestion", "selectedQuestion", "");
      doIndent(out, level);
      writeSetter(out, "String", "setSkippedQuestions", "skippedQuestions", "");
      doIndent(out, level);
      writeSetter(out, "String", "setDataClassificationForSkipped",
            "dataClassificationForSkipped", "");

      out.write("\n");
   }

   protected void writeConstructor(Writer out, int level) throws IOException {
      print(out, level, "public " + config.getFormBeanName() + "() {\n");
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;
         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.warn("Score " + si.getScoreName()
                  + " is not associated with any component!");
            continue;
         }

         QuestionGroup qg = qgRep.findMatching(sa.getRHSID(), sa);

         if (qg != null && qg.isMultiAnswer()) {
            String varName = GenUtils.convertToJavaVariableName(si
                  .getScoreName());
            if (si.getScoreType().equals(INTEGER)) {
               // print(out, level + 1, varName + ".add( new Integer(-10000)
               // );\n");
               print(out, level + 1, varName + ".add( \"\" );\n");

            } else if (si.getScoreType().equals(FLOAT)) {
               // print(out, level + 1, varName + ".add( new Float(-10000f)
               // );\n");
               print(out, level + 1, varName + ".add( \"\" );\n");
            } else if (si.getScoreType().equals(VARCHAR)) {
               print(out, level + 1, varName + ".add(\"\");\n");
            } else if (si.getScoreType().equals(TIMESTAMP)) {
               print(out, level + 1, varName + ".add(\"\");\n");
            } else {
               throw new IOException(
                     "Unsupported score type for multiple answer questions:"
                           + si.getScoreType());
            }
         }
      }

      print(out, level, "}\n");
   }

   protected void writeQuestionCounterMethods(Writer out, int level)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      List<QuestionGroup> alreadyDoneQGs = new ArrayList<QuestionGroup>();
      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;
         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.warn("Score " + si.getScoreName()
                  + " is not associated with any component!");
            continue;
         }

         QuestionGroup qg = qgRep.findMatching(sa.getRHSID(), sa);

         if (qg != null && !alreadyDoneQGs.contains(qg)) {
            CodegenUtils.createScoreIDMap(qg);
            alreadyDoneQGs.add(qg);
         } else {
            continue;
         }

         if (qg != null && qg.isMultiAnswer()) {
            String varName = CodegenUtils
                  .createMultiAnswerIteratorVariableName(qg);
            String methodName = GenUtils.prepareAccessorName(varName, "get");
            varName = GenUtils.convertToJavaVariableName(si.getScoreName());

            print(out, level, "public Collection " + methodName + "() {\n");
            print(out, level + 1, "List list = new ArrayList( " + varName
                  + ".size() -1 );\n");
            print(out, level + 1, "for(int i = 0; i < " + varName
                  + ".size() -1; i++) {\n");
            print(out, level + 2, "list.add( new Integer(i + 1) );\n");
            print(out, level + 1, "}\n");
            print(out, level + 1, "return list;\n");
            print(out, level, "}\n");
            out.write("\n");
         }
      }
   }

   protected void writeAccessors(Writer out, int level, String prefix)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;
         // doIndent(out, level);
         String varName = GenUtils.convertToJavaVariableName(si.getScoreName());
         String methodName = GenUtils.prepareAccessorName(varName, prefix);
         String format = si.getScoreTypeFormat();
         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.warn("Score " + si.getScoreName()
                  + " is not associated with any component!");
            continue;
         }

         QuestionGroup qg = null;
         if (sa.getRight() instanceof LogicalGroup) {
            LogicalGroup lg = (LogicalGroup) sa.getRight();
            for (Object element2 : lg.getElements()) {
               AbstractDisplayComponent adc = (AbstractDisplayComponent) element2;
               qg = qgRep.findMatching(adc.getId(), sa);
               if (qg != null) {
                  break;
               }
            }
         } else {
            qg = qgRep.findMatching(sa.getRHSID(), sa);
         }
         boolean indexed = false;
         if (qg != null) {
            if (qg.isMultiAnswer()) {
               indexed = true;
            }
         }

         String type = null;
         if (si.getScoreType().equalsIgnoreCase("integer")) {
            if (sa.getRight() != null
                  && sa.getRight() instanceof CheckBoxDisplayComponent) {
               type = "boolean";
            } else {
               // type = "int";
               type = "String";
            }
         } else if (si.getScoreType().equalsIgnoreCase(VARCHAR)) {
            type = "String";
         } else if (si.getScoreType().equalsIgnoreCase(FLOAT)) {
            // type = "float";
            type = "String";
         } else if (si.getScoreType().equalsIgnoreCase(BOOLEAN)) {
            type = "boolean";
         } else if (si.getScoreType().equalsIgnoreCase(TIMESTAMP)) {
            type = "String";
         } else {
            throw new RuntimeException("Unsupported data type:"
                  + si.getScoreType());
         }

         if (prefix.equals("get")) {
            doIndent(out, level);
            writeGetter(out, type, methodName, varName, indexed);
         } else if (prefix.equals("set")) {
            doIndent(out, level);
            writeSetter(out, type, methodName, varName, indexed, format);
         } else if (indexed && prefix.equals("add")) {
            doIndent(out, level);
            writeAdder(out, type, varName);
         } else if (indexed && prefix.equals("removeLast")) {
            writeRemoveLastMethod(out, varName);
         } else if (indexed && prefix.equals("size")) {
            doIndent(out, level);
            writeSizeMethod(out, varName);
         }
      }

      for (Object element : asi.getMandatoryFields()) {
         FieldInfo fi = (FieldInfo) element;
         doIndent(out, level);
         String varName = GenUtils.convertToJavaVariableName(fi.getName());
         String methodName = GenUtils.prepareAccessorName(varName, prefix);
         MandatoryFieldAssociation mfa = assocHelper
               .findMandatoryFieldAssociation(fi.getName());

         // dynamic dropdowns are handled differently
         if (mfa.getRight() instanceof DropdownDisplayComponent
               && ((DropdownDisplayComponent) mfa.getRight())
                     .getPopulatedFromDatabase()) {
            if (prefix.equals("get")) {
               writeGetter(out, "DynamicDropDownSelector", methodName, varName);
            } else if (prefix.equals("set")) {
               writeSetter(out, "DynamicDropDownSelector", methodName, varName,
                     "");
            }
         } else {
            if (fi.getType().equalsIgnoreCase("string")) {
               if (prefix.equals("get")) {
                  writeGetter(out, "String", methodName, varName);
               } else if (prefix.equals("set")) {
                  writeSetter(out, "String", methodName, varName, "");
               }
            } else {
               throw new RuntimeException("Unsupported data type:"
                     + fi.getType());
            }
         }
      }
   }

   protected void createInnerClassAndStaticVariables(Writer out, int level)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      print(out, level, "\npublic static class PageVariableInfo {\n");
      doIndent(out, level + 1);
      out.write("int pageNumber;\n");
      doIndent(out, level + 1);
      out.write("String formVarName;\n");
      doIndent(out, level + 1);
      out.write("String dbVarName;\n");
      doIndent(out, level + 1);
      out.write("Map metaDataMap;\n");
      doIndent(out, level + 1);
      out
            .write("public PageVariableInfo(int pageNumber, String formVarName, String dbVarName) {\n");
      doIndent(out, level + 2);
      out.write("this.pageNumber = pageNumber;\n");
      doIndent(out, level + 2);
      out.write("this.formVarName = formVarName;\n");
      doIndent(out, level + 2);
      out.write("this.dbVarName = dbVarName;\n");
      doIndent(out, level + 1);
      out.write("}\n\n");
      doIndent(out, level + 1);
      out.write("public int getPageNumber() { return this.pageNumber; }\n");
      doIndent(out, level + 1);
      out
            .write("public String getFormVarName() { return this.formVarName; }\n");
      doIndent(out, level + 1);
      out.write("public String getDbVarName() { return this.dbVarName; }\n");

      doIndent(out, level + 1);
      out.write("public void addMetaData(String name, String value) {\n");
      doIndent(out, level + 2);
      out.write("if (metaDataMap == null) {\n");
      doIndent(out, level + 3);
      out.write("metaDataMap = new HashMap(3);\n");
      doIndent(out, level + 2);
      out.write("}\n");
      doIndent(out, level + 2);
      out.write("metaDataMap.put(name, value);\n");
      doIndent(out, level + 1);
      out.write("}\n");
      doIndent(out, level + 1);
      out.write("public Map getMetaDataMap() { return metaDataMap; }\n");

      doIndent(out, level);
      out.write("}\n\n");

      // as in the database
      print(out, level, "private final static String assessmentName = \"");
      out.write(asi.getName());
      out.write("\";\n\n");
      print(out, level,
            "public static String getAssessmentName() { return assessmentName; }\n\n");
   }

   protected void createPageQuestionInfoClass(Writer out, int level)
         throws IOException {
      String dir = props.getProperty("code.snippet.dir");
      String filename = dir + File.separator + "java_snippets.txt";
      StringBuilder buf = CodegenUtils.loadCodeSnippet(filename,
            "pagequestioninfo");
      out.write(buf.toString());
      out.write("\n");
   }

   static List<ScoreAssociation> findScoreAssociation(QuestionGroup qg) {
      List<ScoreAssociation> ret = new ArrayList<ScoreAssociation>();
      Set<Association> uniqueAsocs = new HashSet<Association>(11);
      for (Object element : qg.getElements()) {
         IDisplayComponent ic = (IDisplayComponent) element;
         Association asoc = AssociationHelper.getInstance()
               .findScoreAssociationForDisplayComponent(ic);
         if (asoc == null) {
            continue;
         }
         if (uniqueAsocs.contains(asoc)) {
            continue;
         } else {
            uniqueAsocs.add(asoc);
         }
         if (asoc instanceof ScoreAssociation) {
            ret.add((ScoreAssociation) asoc);
         }
      }
      return ret;
   }

   List<String> findScoreNames(QuestionGroup qg) {
      List<String> scoreNames = new LinkedList<String>();
      Map<String, String> scoreNameMap = new HashMap<String, String>(7);
      for (Object element : qg.getElements()) {
         IDisplayComponent ic = (IDisplayComponent) element;
         Association asoc = assocHelper
               .findScoreAssociationForDisplayComponent(ic);
         if (asoc != null) {
            if (asoc instanceof ScoreAssociation) {
               ScoreAssociation sa = (ScoreAssociation) asoc;
               if (scoreNameMap.get(sa.getLeft().getScoreName()) == null) {
                  scoreNameMap.put(sa.getLeft().getScoreName(), sa.getLeft()
                        .getScoreName());
                  scoreNames.add(sa.getLeft().getScoreName());
               }
            } else {
               MandatoryFieldAssociation mfa = (MandatoryFieldAssociation) asoc;
               if (scoreNameMap.get(mfa.getLeft()) == null) {
                  scoreNameMap.put(mfa.getLeft(), mfa.getLeft());
                  scoreNames.add((String) mfa.getLeft());
               }
            }
         }
      }
      return scoreNames;
   }

   private int findPageNumber(QuestionGroup qg) {
      for (Object element : qg.getElements()) {
         IDisplayComponent ic = (IDisplayComponent) element;
         int pageNumber = doc.findPageForDisplayComponent(ic);
         if (pageNumber >= 0) {
            return pageNumber;
         }
      }
      // should not happen
      return -1;
   }

   public static class QuestionScorePageInfo {
      QuestionGroup qg;
      // Association asoc;
      int pageNumber;
      List<String> scores = new LinkedList<String>();
      List<Association> asocList = new LinkedList<Association>();

      public QuestionScorePageInfo(QuestionGroup qg, Association asoc,
            int pageNumber) {
         this.qg = qg;
         asocList.add(asoc);
         this.pageNumber = pageNumber;
      }

      public QuestionScorePageInfo(QuestionGroup qg, int pageNumber) {
         this.qg = qg;
         this.pageNumber = pageNumber;
      }

      public QuestionGroup getQg() {
         return this.qg;
      }

      public void addAsoc(Association asoc) {
         asocList.add(asoc);
      }

      public List<Association> getAsocList() {
         return this.asocList;
      }

      public int getPageNumber() {
         return this.pageNumber;
      }

      public void addScore(String scoreName) {
         scores.add(scoreName);
      }

      public List<String> getScores() {
         return scores;
      }
   }

   protected void createGetPageQuestionsMapMethod(Writer out, int level)
         throws IOException {
      // AssessmentAssociation aa = assocHelper.getAsAssoc();

      print(out, level, "public static Map getPageQuestionsMap() {\n");
      doIndent(out, level + 1);
      out.write("synchronized(");
      out.write(config.getFormBeanName());
      out.write(".class) {\n");
      print(out, level + 2, "if (pageQuestionsMap == null) {\n");
      print(out, level + 3, "pageQuestionsMap = new LinkedHashMap();\n\n");

      Map<Integer, List<QuestionScorePageInfo>> qspiMap = new LinkedHashMap<Integer, List<QuestionScorePageInfo>>(
            17);
      for (Object element : qgRep.getQuestionIDs()) {
         String groupID = (String) element;
         QuestionGroup qg = qgRep.get(groupID);

         List<ScoreAssociation> sal = findScoreAssociation(qg);
         if ((sal == null) || (sal.size() == 0)) {
            log.info("No score Assoc for question " + qg.getId());
            continue;
         }

         int pageNumber = findPageNumber(qg);
         List<QuestionScorePageInfo> qspiList = qspiMap.get(new Integer(
               pageNumber));
         if (qspiList == null) {
            qspiList = new ArrayList<QuestionScorePageInfo>();
            qspiMap.put(new Integer(pageNumber), qspiList);
         }

         QuestionScorePageInfo qspi = new QuestionScorePageInfo(qg, pageNumber);

         for (int i = 0; i < sal.size(); i++) {
            ScoreAssociation sa = sal.get(i);
            qspi.addAsoc(sa);
         }

         List<String> scoreNames = findScoreNames(qg);
         for (Iterator<String> it = scoreNames.iterator(); it.hasNext();) {
            String sn = it.next();
            qspi.addScore(sn);
         }

         qspiList.add(qspi);
      }

      print(out, level + 3, "List list = null;\n");
      print(out, level + 3, "PageQuestionInfo pqi = null;\n");

      for (Iterator<List<QuestionScorePageInfo>> iter = qspiMap.values()
            .iterator(); iter.hasNext();) {
         List<QuestionScorePageInfo> qList = iter.next();
         if (qList.isEmpty()) {
            continue;
         }

         print(out, level + 3, "list = new LinkedList();\n");
         int pageNumber = ((QuestionScorePageInfo) qList.get(0))
               .getPageNumber();

         print(out, level + 3, "pageQuestionsMap.put( new Integer("
               + pageNumber + "), list);\n");

         for (Object element : qList) {
            QuestionScorePageInfo qspi = (QuestionScorePageInfo) element;
            StringBuilder buf = new StringBuilder();
            if (qspi.getQg().isMultiAnswer()) {
               buf.append("list.add( pqi = new PageQuestionInfo(");
               buf.append(qspi.getPageNumber()).append(',');
               buf.append(qspi.getQg().getId()).append(',');
               buf.append("\"multiple-answer\"").append(',');
               buf.append(qspi.getQg().getMinAnswer()).append(',');
               buf.append(qspi.getQg().getMaxAnswer()).append(") );\n");

            } else {
               buf.append("list.add( pqi = new PageQuestionInfo(");
               buf.append(qspi.getPageNumber()).append(',');
               buf.append(qspi.getQg().getId()).append(',');
               buf.append("\"regular\"").append(") );\n");
            }

            print(out, level + 3, buf.toString());

            for (Iterator<String> it2 = qspi.getScores().iterator(); it2
                  .hasNext();) {
               String sn = it2.next();
               print(out, level + 3, "pqi.addScoreName(\"" + sn + "\");\n");
            }
            for (Iterator<String> it2 = qspi.getScores().iterator(); it2
                  .hasNext();) {
               String sn = it2.next();
               System.out.println("score name=" + sn);
               int snID = qspi.getQg().getScoreID(sn);
               print(out, level + 3, "pqi.addScoreNameIDAssoc(\"" + sn + "\","
                     + snID + ");\n");

            }
         }
      }
      print(out, level + 2, "}\n");
      print(out, level + 1, "}\n");
      print(out, level + 1, "return pageQuestionsMap;\n");
      print(out, level, "}\n");
   }

   protected void createHiddenVariablesAndAccessors(Writer out, int level)
         throws IOException {
      out.write("\n");
      for (Object element : qgRep.getQuestionIDs()) {
         String groupID = (String) element;
         QuestionGroup qg = qgRep.get(groupID);
         if (qg.isMultiAnswer()) {
            int maxAnswer = qg.getMaxAnswer();
            List<ScoreAssociation> sal = findScoreAssociation(qg);
            for (int i = 0; i < maxAnswer; i++) {
               for (int j = 0; j < sal.size(); j++) {
                  ScoreAssociation sa = sal.get(j);
                  String varRealName = (String) sa.getLeft().getScoreName();
                  int varCode = qg.getScoreID(varRealName);

                  String varName = CodegenUtils.createHiddenNotesVariable(qg,
                        i + 1, varCode);
                  print(out, level, "private String " + varName + " = \"\";\n");
               }
            }

         } else {
            String varName = CodegenUtils.createHiddenNotesVariable(qg, -1, -1);
            print(out, level, "private String " + varName + " = \"\";\n");
         }
      }
      out.write("\n");

      for (Object element : qgRep.getQuestionIDs()) {
         String groupID = (String) element;
         QuestionGroup qg = qgRep.get(groupID);
         if (qg.isMultiAnswer()) {
            int maxAnswer = qg.getMaxAnswer();
            List<ScoreAssociation> sal = findScoreAssociation(qg);
            for (int i = 0; i < maxAnswer; i++) {
               for (int j = 0; j < sal.size(); j++) {
                  ScoreAssociation sa = sal.get(j);
                  String varRealName = (String) sa.getLeft().getScoreName();
                  int varCode = qg.getScoreID(varRealName);
                  String varName = CodegenUtils.createHiddenNotesVariable(qg,
                        i + 1, varCode);

                  String methodName = CodegenUtils.createAccessorName(varName,
                        "get");
                  doIndent(out, level);
                  writeGetter(out, "String", methodName, varName);
                  methodName = CodegenUtils.createAccessorName(varName, "set");
                  doIndent(out, level);
                  writeSetter(out, "String", methodName, varName, "");
               }
            }
         } else {
            String varName = CodegenUtils.createHiddenNotesVariable(qg, -1, -1);
            String methodName = CodegenUtils.createAccessorName(varName, "get");
            doIndent(out, level);
            writeGetter(out, "String", methodName, varName);
            methodName = CodegenUtils.createAccessorName(varName, "set");
            doIndent(out, level);
            writeSetter(out, "String", methodName, varName, "");
         }
      }
   }

   /**
    * creates getMandatoryFieldMetaDataMap() static method
    *
    * @param out
    * @param level
    * @throws IOException
    */
   protected void createGetMandatoryFieldMetaDataMethod(Writer out, int level)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();

      print(out, level, "public static Map getMandatoryFieldMetaDataMap() {\n");
      print(out, level + 1, "if (mandatoryFieldMetaDataMap == null) {\n");
      print(out, level + 2, "mandatoryFieldMetaDataMap = new HashMap(3);\n");

      boolean first = true;
      for (Object element : asi.getMandatoryFields()) {

         FieldInfo fi = (FieldInfo) element;
         if (fi.hasMetaData()) {

            print(out, level + 2, (first) ? "Map mdMap = new HashMap(3);\n"
                  : "" + "mdMap = new HashMap(3);\n");
            print(out, level + 2, "mandatoryFieldMetaDataMap.put(\""
                  + fi.getName() + "\",mdMap);\n");
            first = false;
            for (String name : fi.getMetaDataVarNames()) {
               print(out, level + 2, "mdMap.put(\"" + name + "\",\""
                     + fi.getMetaDataValue(name) + "\");\n");
            }
         }
      }

      print(out, level + 1, "}\n");
      print(out, level + 1, "return mandatoryFieldMetaDataMap;\n");

      print(out, level, "}\n\n");
   }

   protected void createResetMethod(Writer out, int level) throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();
      print(out, level, "\n");
      print(out, level,
            "public void reset(ActionMapping mapping, HttpServletRequest request) {\n");
      Map<Integer, List<String>> cbFormVarNamePerPageMap = new TreeMap<Integer, List<String>>();
      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;
         String formVarName = GenUtils.convertToJavaVariableName(si
               .getScoreName());

         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.info("No score Assoc for " + si.getScoreName());
            continue;
         }

         AbstractDisplayComponent adc = null;
         if (sa != null) {
            if (sa.getRight() instanceof AbstractDisplayComponent) {
               adc = (AbstractDisplayComponent) sa.getRight();
            } else {
               LogicalGroup lg = (LogicalGroup) sa.getRight();
               if (lg.getElements().size() >= 1) {
                  adc = (AbstractDisplayComponent) lg.getElements().get(0);
               } else {
                  continue;
               }
            }
            if (adc instanceof CheckBoxDisplayComponent) {

               int pageNumber = doc.findPageForDisplayComponent(adc);
               List<String> fvNames = cbFormVarNamePerPageMap.get(new Integer(
                     pageNumber));
               if (fvNames == null) {
                  fvNames = new LinkedList<String>();
                  cbFormVarNamePerPageMap.put(new Integer(pageNumber), fvNames);
               }
               fvNames.add(formVarName);
            }
         }
      }
      if (!cbFormVarNamePerPageMap.isEmpty()) {
         print(out, level + 1, "switch(this.currentPageIdx) {\n");
      }

      for (Map.Entry<Integer, List<String>> entry : cbFormVarNamePerPageMap
            .entrySet()) {
         int pageNum = ((Integer) entry.getKey()).intValue();
         List<String> fvNames = entry.getValue();
         print(out, level + 2, "case " + pageNum + ":\n");
         for (Object element : fvNames) {
            String fvName = (String) element;
            print(out, level + 3, fvName + " = false;\n");
         }
         print(out, level + 3, "break;\n");
      }
      if (!cbFormVarNamePerPageMap.isEmpty()) {
         print(out, level + 1, "}\n");
      }
      print(out, level, "}\n");
   }

   protected void createGetVariableMapMethod(Writer out, int level)
         throws IOException {
      AssessmentAssociation aa = assocHelper.getAsAssoc();
      AssessmentInfo asi = aa.getLeft();

      print(out, level, "public static Map getVariableMap() {\n");
      doIndent(out, level + 1);
      out.write("synchronized(");
      out.write(config.getFormBeanName());
      out.write(".class) {\n");
      print(out, level + 2, "if (variableMap == null) {\n");
      print(out, level + 1, "variableMap = new LinkedHashMap();\n");
      print(out, level + 1, "PageVariableInfo pvi = null;\n\n");

      for (Object element : asi.getScores()) {
         ScoreInfo si = (ScoreInfo) element;
         String dbVarName = si.getScoreName();
         String formVarName = GenUtils.convertToJavaVariableName(si
               .getScoreName());

         ScoreAssociation sa = assocHelper.findScoreAssociation(si);
         if (sa == null) {
            log.info("No score Assoc for " + si.getScoreName());
         }

         AbstractDisplayComponent adc = null;
         if (sa != null) {
            if (sa.getRight() instanceof AbstractDisplayComponent) {
               adc = (AbstractDisplayComponent) sa.getRight();
            } else {
               LogicalGroup lg = (LogicalGroup) sa.getRight();
               if (lg.getElements().size() >= 1) {
                  adc = (AbstractDisplayComponent) lg.getElements().get(0);
               } else {
                  continue;
               }
            }

            int pageNumber = doc.findPageForDisplayComponent(adc);
            doIndent(out, level + 3);
            String format = si.getScoreTypeFormat();
            writeVariableMapEntry(out, pageNumber, formVarName, dbVarName,
                  format, si.getScoreType());

            if (sa.hasMetaData()) {
               for (String name : sa.getMetaDataVarNames()) {
                  print(out, level + 3, " pvi.addMetaData(\"" + name + "\",\""
                        + sa.getMetaDataValue(name) + "\");\n");
               }

            }
         }
      }

      print(out, level + 2, "}\n");
      print(out, level + 1, "}\n");
      print(out, level + 1, "return variableMap;\n");
      print(out, level, "}\n");
   }

   protected void writeVariableMapEntry(Writer out, int pageNumber,
         String formVarName, String dbVarName, String format, String scoreType)
         throws IOException {
      out.write("variableMap.put(\"");
      out.write(dbVarName);
      out.write("\", pvi = new PageVariableInfo(");
      out.write(String.valueOf(pageNumber));
      out.write(",\"");
      out.write(formVarName);
      out.write("\",\"");
      out.write(dbVarName);
      out.write("\") );\n");
      if (scoreType.equals(TIMESTAMP)) {
         out.write("pvi.addMetaData(\"format\", \"" + format + "\");\n");
      }
   }

   protected void writeSetter(Writer out, String type, String methodName,
         String varName, String format) throws IOException {
      writeSetter(out, type, methodName, varName, false, format);
   }

   protected void writeSetter(Writer out, String type, String methodName,
         String varName, boolean indexed, String format) throws IOException {
      out.write("public void ");
      out.write(methodName);

      if (indexed) {
         out.write("(int idx, " + type + " newValue) { ");
         if (type.equals("int")) {
            // out.write( varName + ".set(idx, new Integer(newValue)); }\n");
            out.write(varName + ".set(idx, newValue); }\n");
         } else if (type.equals("float")) {
            // out.write( varName + ".set(idx, new Float(newValue)); }\n");
            out.write(varName + ".set(idx, newValue); }\n");
         } else if (type.equals("String")) {
            out.write(varName + ".set(idx, newValue); }\n");
         } else {
            throw new RuntimeException(
                  "Not a supported type for indexed setters:" + type);
         }
      } else {
         out.write("(");
         out.write(type);
         out.write(" newValue) {  this.");
         out.write(varName);
         out.write(" = newValue; }\n");
      }
   }

   protected void writeAdder(Writer out, String type, String varName)
         throws IOException {
      String methodName = GenUtils.prepareAccessorName(varName, "add");
      out.write("public void ");
      out.write(methodName);
      if (type.equals("int")) {
         // out.write("(" + type + " value) { " + varName + ".add( new
         // Integer(value)); }\n");
         out.write("(" + type + " value) { " + varName + ".add( value ); }\n");
      } else if (type.equals("float")) {
         // out.write("(" + type + " value) { " + varName + ".add( new
         // Float(value)); }\n");
         out.write("(" + type + " value) { " + varName + ".add( value ); }\n");
      } else if (type.equals("String")) {
         out.write("(" + type + " value) { " + varName + ".add(value); }\n");
      } else {
         throw new RuntimeException("Not a supported type for adder method:"
               + type);
      }
   }

   protected void writeRemoveLastMethod(Writer out, String varName)
         throws IOException {
      String methodName = GenUtils.prepareAccessorName(varName, "removeLast");
      out.write("public void ");
      out.write(methodName);
      out.write("() { " + varName + ".remove(");
      out.write(varName);
      out.write(".size() -1); }\n");
   }

   protected void writeSizeMethod(Writer out, String varName)
         throws IOException {
      String methodName = GenUtils.prepareAccessorName(varName, "get") + "Size";
      out.write("public int " + methodName + "() { return " + varName
            + ".size(); }\n");
   }

   protected void writeGetter(Writer out, String type, String methodName,
         String varName) throws IOException {
      writeGetter(out, type, methodName, varName, false);
   }

   protected void writeGetter(Writer out, String type, String methodName,
         String varName, boolean indexed) throws IOException {
      out.write("public ");
      out.write(type);
      out.write(" ");
      out.write(methodName);
      if (indexed) {
         if (type.equals("int")) {
            out.write("(int idx) { return (String) " + varName
                  + ".get(idx); } \n");

         } else if (type.equals("float")) {
            out.write("(int idx) { return (String) " + varName
                  + ".get(idx); } \n");
         } else if (type.equals("String")) {
            out.write("(int idx) { return (" + type + " ) " + varName
                  + ".get(idx); } \n");
         } else {
            throw new RuntimeException(
                  "Not a supported type for indexed getters:" + type);
         }
      } else {
         out.write("() {  return this.");
         out.write(varName);
         out.write("; }\n");
      }
   }

   protected void print(Writer out, int level, String msg) throws IOException {
      if (level > 0) {
         doIndent(out, level);
      }
      out.write(msg);
   }
}
