package caslayout.codegen.struts;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import caslayout.codegen.QuestionInfo;
import caslayout.codegen.TDColumn;
import caslayout.output.AbstractGenerator;
import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.ButtonDisplayComponent;
import caslayout.ui.ButtonType;
import caslayout.ui.CAContainer;
import caslayout.ui.CAGridLayout;
import caslayout.ui.CheckBoxDisplayComponent;
import caslayout.ui.DropdownDisplayComponent;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.LogicalGroup;
import caslayout.ui.QuestionGroup;
import caslayout.ui.QuestionGroupRepository;
import caslayout.ui.RadioButtonDisplayComponent;
import caslayout.ui.TextAreaDisplayComponent;
import caslayout.ui.TextDisplayComponent;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.ScoreAssociation;
import caslayout.ui.model.ScoreCodeInfo;
import caslayout.ui.model.ScoreInfo;
import caslayout.util.CodegenUtils;
import caslayout.util.GenUtils;
import caslayout.util.HTMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: StrutsJSPGenerator.java,v 1.20.2.3 2007/06/13 23:23:10 bozyurt
 *          Exp $
 */
public class StrutsJSPGenerator extends AbstractGenerator {
   protected CAContainer root;
   protected StrutsCodegenConfig config;
   protected int pageIdx;
   protected Properties props;
   protected AssociationHelper ah;
   protected Map<CAContainer, QuestionGroup> contMAQMap;
   /** Map containing calculated fields if any */
   protected Map<TextFieldDisplayComponent, Association> cfMap;
   protected int skipID = 0;

   final static int RADIO_BUTTON = 1;
   final static int CHECK_BOX = 2;

   final static int SINGLE = 1;
   final static int LOGICAL_GROUP = 2;
   final static int CONTAINER = 3;
   final static int QUESTION = 4;

   public StrutsJSPGenerator() {
      super();
   }

   public StrutsJSPGenerator(CAContainer root, StrutsCodegenConfig config,
         int pageIdx) throws IOException {
      super();
      this.root = root;
      this.config = config;
      this.pageIdx = pageIdx;
      props = GenUtils.loadProperties("caslayout.properties");
      ah = AssociationHelper.getInstance();
   }

   public void initialize(CAContainer root) {
      this.root = root;
   }

   public void generate(Writer out) throws IOException {
      Map<QuestionGroup, QuestionInfo> questionInfoMap = new LinkedHashMap<QuestionGroup, QuestionInfo>();
      createHeader(out, 1);

      this.cfMap = CodegenUtils.prepareCalculatedFieldMap(root, ah);
      if (!cfMap.isEmpty()) {
         out.write("<script language=\"javascript\">\n");

         List<ScoreInfo> siList = ah.getAssociatedScores();
         Map<String, ScoreInfo> jv2ScoreInfoMap = CodegenUtils
               .prepJavaVar2ScoreInfoMap(siList);
         // for (Iterator iter = cfMap.entrySet().iterator(); iter.hasNext();) {
         for (Map.Entry<TextFieldDisplayComponent, Association> entry : cfMap
               .entrySet()) {
            TextFieldDisplayComponent tdc = entry.getKey();
            ScoreAssociation sa = (ScoreAssociation) entry.getValue();
            String methodStr = CodegenUtils.prepareCalcFieldJSMethod(sa, tdc,
                  jv2ScoreInfoMap);
            System.out.println("function: \n" + methodStr);
            out.write(methodStr);
            out.write("\n\n");
         }
         createSupportJSMethods(out);
         out.write("</script>\n");
      }

      doIndent(out, 2);
      out.write("<html:errors/>\n");
      doIndent(out, 2);
      out.write("<center>");
      doIndent(out, 2);
      String suffix = (config.getJspPageNames().length == 1) ? "" : "_Page"
            + pageIdx;
      String actionPath = config.getStrutsActionRoot() + suffix;

      out.write("<html:form action=\"");
      out.write(actionPath);
      out.write("\">\n\n");
      createCommonHiddenFields(out, 2);

      this.contMAQMap = CodegenUtils.prepareQuestionGroupContainerMap(root);

      generateTables(out, root, 3, questionInfoMap);

      doIndent(out, 2);
      out.write("</html:form>\n");
      createFooter(out, 1);
      out.write("</center>\n");
   }

   protected void createCommonHiddenFields(Writer out, int level)
         throws IOException {
      doIndent(out, level);
      out
            .write("<input type=\"hidden\" name=\"selectedQuestion\" value=\"\">\n");
      doIndent(out, level);
      out.write("<input type=\"hidden\" name=\"action\" value=\"\">\n");

      doIndent(out, level);
      out
            .write("<input type=\"hidden\" name=\"skippedQuestions\" value=\"\">\n");

      doIndent(out, level);
      out
            .write("<input type=\"hidden\" name=\"dataClassificationForSkipped\" value=\"\">\n");
   }

   protected QuestionGroup findMultiAnswerQuestionGroup(CAContainer container,
         int offset, TDColumn[] tdCols,
         Map<QuestionGroup, QuestionInfo> questionInfoMap) {
      int qgCount = 0;
      QuestionGroup multiAnswer = null;
      for (int j = 0; j < tdCols.length; j++) {
         IDisplayComponent ic = CodegenUtils.getDisplayComponent(container,
               offset + tdCols[j].getCol());
         if (ic != null) {
            QuestionGroup qg = QuestionGroupRepository.getInstance()
                  .findMatching(ic);
            QuestionInfo qi = null;
            if (qg != null) {
               qi = questionInfoMap.get(qg);
               if (qi == null) {
                  qi = new QuestionInfo(qg, (AbstractDisplayComponent) ic);
                  questionInfoMap.put(qg, qi);
               }

               if (qg.isMultiAnswer()) {
                  if (multiAnswer != null && multiAnswer != qg) {
                     // multiple multianswer questions in the same
                     // container row
                     throw new RuntimeException(
                           "You cannot have multiple multianswer questions in the same container row");
                  }
                  multiAnswer = qg;
               } else {
                  qgCount++;
               }
            }
         }
      }
      if (multiAnswer != null && qgCount > 0) {
         throw new RuntimeException(
               "You cannot have a multianswer question intermixed with other question types in the same container row");
      }
      return multiAnswer;
   }

   class QuestionGroupHolder {
      QuestionGroup qg;

      public QuestionGroupHolder(QuestionGroup qg) {
         this.qg = qg;
      }
   }

   protected void containsMultiAnswerQuestionGroup(CAContainer container,
         Map<QuestionGroup, QuestionInfo> questionInfoMap,
         QuestionGroupHolder qgHolder) {
      int qgCount = 0;
      QuestionGroup multiAnswer = null;
      for (IDisplayComponent ic : container.getComponents()) {
         if (ic == null) {
            continue;
         }
         if (ic instanceof CAContainer) {
            containsMultiAnswerQuestionGroup((CAContainer) ic, questionInfoMap,
                  qgHolder);
         }
         QuestionGroup qg = QuestionGroupRepository.getInstance().findMatching(
               ic);
         QuestionInfo qi = null;
         if (qi != null) {
            qi = questionInfoMap.get(qg);
            if (qi == null) {
               qi = new QuestionInfo(qg, (AbstractDisplayComponent) ic);
               questionInfoMap.put(qg, qi);
            }

            if (qg.isMultiAnswer()) {
               if (multiAnswer != null && multiAnswer != qg) {
                  // multiple multianswer questions in the same container
                  throw new RuntimeException(
                        "You cannot have multiple multianswer questions in the same container");
               }
               multiAnswer = qg;
            } else {
               qgCount++;
            }
         }
      }
   }

   protected void generateTables(Writer out, CAContainer container, int level,
         Map<QuestionGroup, QuestionInfo> questionInfoMap) throws IOException {
      QuestionGroup maQG = (QuestionGroup) contMAQMap.get(container);
      if (maQG != null) {
         QuestionInfo qi = CodegenUtils.prepareQuestionInfo(maQG);
         generateMultiAnswerQuestion(out, level, container, qi);
         return;
      }

      doIndent(out, level);

      if (container == root) {
         HTMLUtils.tableStart(out, 1, "80%");
      } else {
         HTMLUtils.tableStart(out);
      }
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();
      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         doIndent(out, level + 1);
         HTMLUtils.tableRowStart(out);
         int cols = layout.getEffectiveColumnCount(i);

         TDColumn[] tdCols = CodegenUtils.prepareTableColumns(i, layout, cols,
               container, index);

         int offset = index;

         for (int j = 0; j < tdCols.length; j++) {
            doIndent(out, level + 2);
            out.write(tdCols[j].tagStart());

            if (index < container.getComponents().size()) {
               IDisplayComponent ic = CodegenUtils.getDisplayComponent(
                     container, offset + tdCols[j].getCol());

               if (ic != null) {
                  QuestionGroup qg = QuestionGroupRepository.getInstance()
                        .findMatching(ic);
                  QuestionInfo qi = null;
                  if (qg != null) {
                     qi = questionInfoMap.get(qg);
                     if (qi == null) {
                        qi = new QuestionInfo(qg, (AbstractDisplayComponent) ic);
                        questionInfoMap.put(qg, qi);
                     }
                  }
                  if (ic instanceof CAContainer) {
                     out.write("\n");
                     generateTables(out, (CAContainer) ic, level,
                           questionInfoMap);
                  } else {
                     renderDisplayComponent(ic, out, level + 3, qi, null);
                  }
               } else {
                  // no component in the current grid cell
                  out.write("&nbsp;");
               }
            }

            if ((offset + tdCols[j].getCol()) >= container.getComponents()
                  .size()) {
               out.write("&nbsp;");
            }
            out.write(tdCols[j].tagEnd() + "\n");
         } // j

         index = offset + cols;

         doIndent(out, level + 1);
         HTMLUtils.tableRowEnd(out);
      } // i
      doIndent(out, level);
      HTMLUtils.tableEnd(out);
   }

   protected void generateMultiAnswerQuestion(Writer out, int level,
         CAContainer container, QuestionInfo qi) throws IOException {
      generateFirstMAQuestion(out, level, container, qi);

      QuestionGroup qg = qi.getQuestionGroup();
      // now create the expanding logic
      createMultiAnswerLoopStart(out, level + 1, qi);
      doIndent(out, level + 2);
      out.write("<%\n");
      doIndent(out, level + 3);
      out.write("int di = ((Integer) idx).intValue() + 1;\n");
      doIndent(out, level + 3);
      out.write("java.util.Map notesVarnameMap = new java.util.HashMap(7);\n");
      doIndent(out, level + 3);
      out.write("String notesVarname = null;\n");
      doIndent(out, level + 3);
      out.write("String noteHandlerVarname = null;\n");
      doIndent(out, level + 3);
      out.write("String noteBorderVarname = null;\n");
      doIndent(out, level + 2);
      out.write("%>\n");

      int idx = 0;
      // for (Iterator iter = qg.getScoreIDMap().entrySet().iterator(); iter
      // .hasNext();) {
      for (Map.Entry<String, Integer> entry : qg.getScoreIDMap().entrySet()) {
         int scoreVarID = entry.getValue();
         String scoreName = entry.getKey();
         String notesVarName = "q" + qg.getId() + "_ans\" + di + \"_"
               + scoreVarID;
         doIndent(out, level + 2);
         out.write("<% notesVarnameMap.put(\"" + scoreName + "\",\""
               + notesVarName + "\");\n");
         doIndent(out, level + 3);
         out.write("notesVarname =\"" + notesVarName + "_notes\";\n");
         doIndent(out, level + 2);
         out.write("%>\n");
         doIndent(out, level + 2);
         out.write("<html:hidden property=\"<%= notesVarname %>\" />\n");
         idx++;
      }

      // layout the parametrized question
      doIndent(out, level);
      HTMLUtils.tableStart(out);
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();
      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         doIndent(out, level + 1);
         HTMLUtils.tableRowStart(out);
         int cols = layout.getEffectiveColumnCount(i);

         TDColumn[] tdCols = CodegenUtils.prepareTableColumns(i, layout, cols,
               container, index);

         int offset = index;
         for (int j = 0; j < tdCols.length; j++) {
            doIndent(out, level + 2);
            out.write(tdCols[j].tagStart());
            IDisplayComponent ic = CodegenUtils.getDisplayComponent(container,
                  offset + tdCols[j].getCol());
            if (ic != null) {
               if (ic instanceof CAContainer) {
                  out.write("\n");
                  generateContainer(out, level, (CAContainer) ic, qi, "idx");
               } else {
                  doIndent(out, level + 3);
                  renderDisplayComponent(ic, out, level, qi, "idx");
               }
            } else {
               // no component in the current grid cell
               out.write("&nbsp;");
            }

            if ((offset + tdCols[j].getCol()) >= container.getComponents()
                  .size()) {
               out.write("&nbsp;");
            }
            out.write(tdCols[j].tagEnd() + "\n");
         } // j

         index = offset + cols;
         doIndent(out, level + 1);
         HTMLUtils.tableRowEnd(out);

      } // i
      doIndent(out, level);
      HTMLUtils.tableEnd(out);

      // close the loop over the parametrized question
      doIndent(out, level + 1);
      out.write("</logic:iterate>\n");
   }

   private void generateFirstMAQuestion(Writer out, int level,
         CAContainer container, QuestionInfo qi) throws IOException {
      doIndent(out, level);

      HTMLUtils.tableStart(out);
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();
      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         doIndent(out, level + 1);
         HTMLUtils.tableRowStart(out);
         int cols = layout.getEffectiveColumnCount(i);

         TDColumn[] tdCols = CodegenUtils.prepareTableColumns(i, layout, cols,
               container, index);

         int offset = index;
         for (int j = 0; j < tdCols.length; j++) {
            doIndent(out, level + 2);
            out.write(tdCols[j].tagStart());
            IDisplayComponent ic = CodegenUtils.getDisplayComponent(container,
                  offset + tdCols[j].getCol());
            if (ic != null) {
               if (ic instanceof CAContainer) {
                  out.write("\n");
                  // System.out.println("creating container" +(
                  // (CAContainer) ic).toString());
                  generateContainer(out, level, (CAContainer) ic, qi, "0");
                  // System.out.println("finished container" + (
                  // (CAContainer) ic).toString());
               } else {
                  doIndent(out, level + 3);
                  renderDisplayComponent(ic, out, level, qi, "0");
               }
            } else {
               // no component in the current grid cell
               out.write("&nbsp;");
            }

            if ((offset + tdCols[j].getCol()) >= container.getComponents()
                  .size()) {
               out.write("&nbsp;");
            }
            out.write(tdCols[j].tagEnd() + "\n");
         }
         index = offset + cols;

         doIndent(out, level + 1);
         HTMLUtils.tableRowEnd(out);
      } // i;
      doIndent(out, level);
      HTMLUtils.tableEnd(out);
   }

   protected void generateContainer(Writer out, int level,
         CAContainer container, QuestionInfo qi, String indexStr)
         throws IOException {
      doIndent(out, level);
      HTMLUtils.tableStart(out);
      CAGridLayout layout = (CAGridLayout) container.getLayoutManager();
      int rows = layout.getEffectiveRowCount();
      int index = 0;
      for (int i = 0; i < rows; i++) {
         doIndent(out, level + 1);
         HTMLUtils.tableRowStart(out);
         int cols = layout.getEffectiveColumnCount(i);

         TDColumn[] tdCols = CodegenUtils.prepareTableColumns(i, layout, cols,
               container, index);
         int offset = index;
         for (int j = 0; j < tdCols.length; j++) {
            doIndent(out, level + 2);
            out.write(tdCols[j].tagStart());

            if (index < container.getComponents().size()) {
               IDisplayComponent ic = CodegenUtils.getDisplayComponent(
                     container, offset + tdCols[j].getCol());

               if (ic != null) {
                  if (ic instanceof CAContainer) {
                     out.write("\n");
                     generateContainer(out, level, (CAContainer) ic, qi,
                           indexStr);
                  } else {
                     doIndent(out, level + 3);
                     renderDisplayComponent(ic, out, level, qi, indexStr);
                  }
               } else {
                  // no component in the current grid cell
                  out.write("&nbsp;");
               }
            }

            if ((offset + tdCols[j].getCol()) >= container.getComponents()
                  .size()) {
               out.write("&nbsp;");
            }

            out.write(tdCols[j].tagEnd() + "\n");
         } // j
         index = offset + cols;
         doIndent(out, level + 1);
         HTMLUtils.tableRowEnd(out);
      } // j
      doIndent(out, level);
      HTMLUtils.tableEnd(out);
   }

   protected void createMultiAnswerLoopStart(Writer out, int level,
         QuestionInfo qi) throws IOException {
      doIndent(out, level);

      String formBeanID = config.getFormBeanID();

      QuestionGroup qg = qi.getQuestionGroup();
      String varName = CodegenUtils.createMultiAnswerIteratorVariableName(qg);
      out.write("<logic:iterate id=\"idx\" name=\"" + formBeanID
            + "\" property=\"" + varName + "\" >\n");
   }

   protected void listSelectAndOptions(Writer out, int level,
         String selectName, boolean activateJavascript) throws IOException {
      doIndent(out, level);
      if (activateJavascript) {
         out
               .write("<select onFocus=\"noteFocusHandler(this);\" onChange=\"noteChangeFocusHandler(this);\" onBlur=\"noteBlurHandler(this);\" STYLE=\"width: 20px;\" NAME=\""
                     + selectName + "\">\n");
      } else {
         out
               .write("<select onFocus=\"noteFocusHandler(this);\" onBlur=\"javascript: if(document.all) { this.style.width = '20px';}\" STYLE=\"width: 20px;\" NAME=\""
                     + selectName + "\">\n");
      }
      level++;
      doIndent(out, level);
      out.write("<option></option>\n");
      doIndent(out, level);
      out.write("<option>No data entered</option>\n");
      doIndent(out, level);
      out.write("<option>Subject declined to answer the question</option>\n");
      doIndent(out, level);
      out.write("<option>Answer is unknown to the informant</option>\n");
      doIndent(out, level);
      out.write("<option>Interviewer forgot to ask the question</option>\n");
      doIndent(out, level);
      out.write("<option>Interviewer deferred the question</option>\n");
      doIndent(out, level);
      out.write("<option>Question is excluded from the protocol</option>\n");
      doIndent(out, level);
      out.write("<option>Question is not applicable to the subject</option>\n");
      doIndent(out, level);
      out.write("<option>Answer is not intelligible</option>\n");
      level--;
      doIndent(out, level);
      if (activateJavascript) {
         out.write("</select>");
      } else {
         out.write("</select>&nbsp;&nbsp;&nbsp;\n");
      }
   }

   protected void createNotesButtonCode(Writer out, int level,
         String notesBasename, boolean first, String scoreName)
         throws IOException {
      if (first) {
         doIndent(out, level);
         out
               .write("<span style=\"border-style: solid; border-width: 5px; border-color: white;\" ID=\""
                     + notesBasename + "_note_border\">\n");
         level++;
         listSelectAndOptions(out, level, notesBasename + "_note_handler", true);
         level--;
         doIndent(out, level);
         out.write("</span>&nbsp;&nbsp;&nbsp;\n");
      } else {
         doIndent(out, level);
         out.write("<% notesVarname = (String) notesVarnameMap.get(\""
               + scoreName + "\") + \"_notes\";\n");
         doIndent(out, level);
         out.write("   noteHandlerVarname = (String) notesVarnameMap.get(\""
               + scoreName + "\") + \"_note_handler\";\n");
         doIndent(out, level);
         out.write("   noteBorderVarname = (String) notesVarnameMap.get(\""
               + scoreName + "\") + \"_note_border\";\n");
         doIndent(out, level);
         out.write("%>\n");
         doIndent(out, level);
         out
               .write("<span style=\"border-style: solid; border-width: 5px; border-color: white;\" ID=\"<%= noteBorderVarname %>\">\n");
         level++;
         listSelectAndOptions(out, level, "<%= noteHandlerVarname %>", true);
         level--;
         doIndent(out, level);
         out.write("</span>&nbsp;&nbsp;&nbsp;\n");
      }
   }

   protected void renderDisplayComponent(IDisplayComponent ic, Writer out,
         int level, QuestionInfo qi, String index) throws IOException {
      String formBeanID = config.getFormBeanID();
      IDisplayComponent leftMost = null;

      String propertyName = "";
      ScoreInfo si = null;
      Association asoc = ah.findScoreAssociationForDisplayComponent(ic);
      if (asoc != null) {
         if (asoc instanceof ScoreAssociation) {
            ScoreAssociation sa = (ScoreAssociation) asoc;
            propertyName = GenUtils.convertToJavaVariableName(sa.getLeft()
                  .getScoreName());
            si = sa.getLeft();
         } else if (asoc instanceof MandatoryFieldAssociation) {
            MandatoryFieldAssociation mfa = (MandatoryFieldAssociation) asoc;
            propertyName = mfa.getLeft();
         }
      }

      boolean multiAnswer = false;
      int maIdx = -1;
      if (index != null && index.equals("0")) {
         maIdx = 0;
      }

      String questionID = null;

      if (qi != null) {
         QuestionGroup qg = qi.getQuestionGroup();
         multiAnswer = qg.isMultiAnswer();

         questionID = qg.getId();

         String notesVarname = null;
         String notesBasename = null;
         if (qg.isMultiAnswer()) {
            // multiple answer question are assumed to have notes variable
            // names encoded
            // ALWAYS as q<question#>_ans<questionIdx>_<scoreVarID>_notes
            //
            // e.g. q3_ans1_0_notes
            Map<String, IDisplayComponent> lmMap = qi.findLeftMostComponents();

            if (maIdx == 0) {
               // for (Iterator iter = qg.getScoreIDMap().entrySet().iterator();
               // iter
               // .hasNext();) {
               for (Map.Entry<String, Integer> entry : qg.getScoreIDMap()
                     .entrySet()) {
                  String scoreName = entry.getKey();
                  leftMost = lmMap.get(scoreName);
                  if (leftMost != null && ic == leftMost) {
                     int scoreVarID = ((Integer) entry.getValue()).intValue();
                     notesVarname = CodegenUtils.createHiddenNotesVariable(qg,
                           1, scoreVarID);
                     notesBasename = CodegenUtils.createHiddenNotesVariable(qg,
                           1, scoreVarID, "");

                     out.write("<html:hidden property=\"" + notesVarname
                           + "\" />\n");

                     createNotesButtonCode(out, level, notesBasename, true,
                           scoreName);
                  }
               }
            } else {
               // in the loop do nothing here, since the hidden var for the
               // notes created in a different place
               // just create the notes button
               // for (Iterator iter = qg.getScoreIDMap().entrySet().iterator();
               // iter
               // .hasNext();) {
               for (Map.Entry<String, Integer> entry : qg.getScoreIDMap()
                     .entrySet()) {
                  String scoreName = (String) entry.getKey();
                  leftMost = lmMap.get(scoreName);
                  if (leftMost != null && ic == leftMost) {
                     int scoreVarID = ((Integer) entry.getValue()).intValue();
                     notesBasename = CodegenUtils.createHiddenNotesVariable(qg,
                           0, scoreVarID, "");
                     notesVarname = "<%= notesVarname %>";

                     createNotesButtonCode(out, level, notesBasename, false,
                           scoreName);
                  }
               }
            }
         } else { // ! qg.isMultiAnswer()

            // find the left most component
            leftMost = qi.findLeftMostComponent();

            System.out.println(">> checking ic with ID: " + ic.getId());
            if (leftMost != null && ic == leftMost) {
               // only for the left most component for the question
               notesVarname = CodegenUtils
                     .createHiddenNotesVariable(qg, -1, -1);
               notesBasename = CodegenUtils.createHiddenNotesVariable(qg, -1,
                     -1, "");
               out.write("<html:hidden property=\"" + notesVarname + "\" />");
               out.write("\n");
               createNotesButtonCode(out, level, notesBasename, true, null);
            }
         }
      } // qi != null

      if (ic instanceof TextDisplayComponent) {
         writeText(out, (TextDisplayComponent) ic, level);
      } else if (ic instanceof RadioButtonDisplayComponent) {
         createRadioButton(out, (RadioButtonDisplayComponent) ic, level,
               formBeanID, propertyName, si, (ScoreAssociation) asoc, index);
      } else if (ic instanceof CheckBoxDisplayComponent) {
         createCheckBox(out, (CheckBoxDisplayComponent) ic, level, formBeanID,
               propertyName, si, (ScoreAssociation) asoc, index);
      } else if (ic instanceof TextFieldDisplayComponent) {
         createInputTag(out, (TextFieldDisplayComponent) ic, level, formBeanID,
               propertyName, index);
      } else if (ic instanceof TextAreaDisplayComponent) {
         createTextareaTag(out, null, (TextAreaDisplayComponent) ic, level,
               formBeanID, propertyName, index);
      } else if (ic instanceof DropdownDisplayComponent) {
         createDropdown(out, (DropdownDisplayComponent) ic, level,
               propertyName, index);
      } else if (ic instanceof ButtonDisplayComponent) {
         ButtonDisplayComponent bdc = (ButtonDisplayComponent) ic;
         if (multiAnswer
               && index != null
               && index.equals("idx")
               && (bdc.getAction() == ButtonType.ADD_BUTTON || bdc.getAction() == ButtonType.REMOVE_LAST_BUTTON)) {
            // do not render the add button multiple times for the
            // multianswer case
            out.write("&nbsp; ");
         } else {
            createSubmitButton(out, bdc, level, questionID);
         }
      }
   }

   protected void writeText(Writer out, TextDisplayComponent tdc, int level)
         throws IOException {
      doIndent(out, level);
      if (tdc.getCSSClass() != null) {
         out.write("<span class=\"" + tdc.getCSSClass() + "\"> ");
      }

      String textStr = tdc.getLabel().getText();
      StringTokenizer stok = new StringTokenizer(textStr, "\n");
      StringBuilder buf = new StringBuilder(textStr.length() + 20);
      while (stok.hasMoreTokens()) {
         buf.append(stok.nextToken());
         if (stok.hasMoreTokens()) {
            buf.append(" <br>\n");
         }
      }

      out.write(buf.toString());
      if (tdc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void createPropertyExpression(Writer out, String propertyName,
         String index) throws IOException {
      createPropertyExpression(out, propertyName, index, null);
   }

   protected void createPropertyExpression(Writer out, String propertyName,
         String index, String suffix) throws IOException {
      if (index != null) {
         try {
            int idx = Integer.parseInt(index);
            out.write("\"" + propertyName + "[" + idx + "]");
            if (suffix != null) {
               out.write(suffix);
            }
            out.write("\"");
         } catch (NumberFormatException nfe) {
            // must be a variable index
            out.write("'<%= \"" + propertyName + "[\" + " + index + " + \"]");
            if (suffix != null) {
               out.write(suffix);
            }
            out.write("\" %>'");
         }
      } else {
         out.write("\"" + propertyName);
         if (suffix != null) {
            out.write(suffix);
         }
         out.write("\"");
      }
   }

   protected void createInputTag(Writer out, TextFieldDisplayComponent tfdc,
         int level, String formName, String propertyName, String index)
         throws IOException {
      doIndent(out, level);
      if (tfdc.getCSSClass() != null) {
         out.write("<span class=\"" + tfdc.getCSSClass() + "\"> ");
      }
      if (propertyName == null || propertyName.length() == 0) {
         // component is not associated, just show the component
         out.write("<input type=\"text\">");
      } else {
         ScoreAssociation sa = (ScoreAssociation) cfMap.get(tfdc);
         String cfId = null;
         if (sa != null) {
            cfId = propertyName + "Field";
         }
         out.write("<html:text name=\"" + formName + "\" property=");
         createPropertyExpression(out, propertyName, index);
         out.write(" size=\"" + tfdc.getFieldLength() + "\"");
         if (sa != null) {
            // a calculated field
            out.write(" styleId=\"" + cfId + "\"");
         }
         out.write(" />");
         if (sa != null) {
            out
                  .write(" &nbsp; \n<input type=\"button\" value=\"Calculate\" onclick=\"");
            String funcName = "calc" + GenUtils.toTitleCase(propertyName);
            out.write(funcName + "(this,'" + cfId + "')\" />\n");
         }
      }
      if (tfdc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void createTextareaTag(Writer out, String label,
         TextAreaDisplayComponent tadc, int level, String formName,
         String propertyName, String index) throws IOException {
      doIndent(out, level);
      if (tadc.getCSSClass() != null) {
         out.write("<span class=\"" + tadc.getCSSClass() + "\"> ");
      }
      if (propertyName == null || propertyName.length() == 0) {
         // component is not associated, just show the component
         out.write("<textarea></textarea>");

      } else {
         out.write("<html:textarea name=\"" + formName + "\" property=");
         createPropertyExpression(out, propertyName, index);

         out.write(" rows=\"" + tadc.getNumRows() + "\" cols=\""
               + tadc.getNumCols() + "\" />");
      }
      if (tadc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void createSubmitButton(Writer out, ButtonDisplayComponent bdc,
         int level, String questionID) throws IOException {
      doIndent(out, level);
      if (bdc.getCSSClass() != null) {
         out.write("<span class=\"" + bdc.getCSSClass() + "\">");
      }

      if (bdc.getAction() == ButtonType.SKIP_BUTTON) {
         out.write("\n");
         handleSkipButton(out, level, bdc);
      } else {

         out.write("<html:submit styleClass=\"submit-button\" onclick=\"");
         if (bdc.getAction() == ButtonType.ADD_BUTTON) {
            out.write("addAnswer(this.form,'" + questionID + "')\"> ");
         } else if (bdc.getAction() == ButtonType.REMOVE_LAST_BUTTON) {
            out.write("deleteLastAnswer(this.form,'" + questionID + "')\"> ");
         } else {
            out.write("return submitAssessment(this.form,'" + bdc.getLabel()
                  + "')\"> ");
         }

         out.write(bdc.getLabel());
         out.write("</html:submit>");
      }
      if (bdc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void handleSkipButton(Writer out, int level,
         ButtonDisplayComponent bdc) throws IOException {
      String selectName = "s" + skipID + "_note_handler";
      skipID++;
      listSelectAndOptions(out, level, selectName, false);

      StringBuilder buf = new StringBuilder();
      for (Iterator<Integer> iter = bdc.getQuestionIDs().iterator(); iter
            .hasNext();) {
         Integer qid = iter.next();
         buf.append(qid);
         if (iter.hasNext()) {
            buf.append(',');
         }
      }

      out.write("<html:submit styleClass=\"submit-button\" value=\""
            + bdc.getLabel() + "\" onclick=\"");
      out.write("return setAnswers(this,");
      if (bdc.getQuestionIDs().isEmpty()) {
         out.write(" null, this.form," + bdc.getJavascriptOnly() + ")\"> ");
      } else {
         out.write("'" + buf.toString() + "', this.form, "
               + bdc.getJavascriptOnly() + ", '" + selectName + "');\"> ");
      }
      out.write(bdc.getAction().getValue());

      out.write("</html:submit>");
   }

   protected void createRadioButton(Writer out,
         RadioButtonDisplayComponent rdc, int level, String formName,
         String propertyName, ScoreInfo si, ScoreAssociation sa, String index)
         throws IOException {
      doIndent(out, level);
      if (rdc.getCSSClass() != null) {
         out.write("<span class=\"" + rdc.getCSSClass() + "\"> ");
      }
      if (si == null || sa == null) {
         // component is not associated, just show the component
         out.write("<input type=\"radio\">" + rdc.getText() + " ");
      } else {
         out.write("<html:radio name=\"" + formName + "\" property=");
         createPropertyExpression(out, propertyName, index);
         String value = prepareScoreValue(rdc.getValue(), rdc.getText(), rdc
               .getId(), si, sa);

         out.write(" title=\"" + rdc.getText() + "\" value=\"" + value
               + "\" />" + rdc.getText() + " ");
      }
      if (rdc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   private String prepareScoreValue(String compValue, String compText,
         String compID, ScoreInfo si, ScoreAssociation sa) {
      String value = compValue;
      boolean foundScoreCode = false;
      if (value != null) {
         ScoreCodeInfo sci = si.findScoreCodeInfo(value);
         foundScoreCode = sci != null;
      }

      if (value == null || !foundScoreCode) {
         ScoreCodeInfo sci = si.findScoreCode(compText);
         if (sci != null) {
            value = sci.getScoreCode();
         } else {
            // check if there is score-code association info
            if (!sa.getScoreCode2CompIDMap().isEmpty()) {
               sci = sa.findScoreCodeInfo(compID);
               value = sci.getScoreCode();
            } else {
               // FIXME don't assume the logical group component order is
               // same as score code order in database
               value = getScoreCode(sa, compID, si);
            }
         }
      }
      return value;
   }

   protected String getScoreCode(ScoreAssociation sa, String compID,
         ScoreInfo si) {
      LogicalGroup lg = (LogicalGroup) sa.getRight();
      int idx = -1;
      for (Iterator<IDisplayComponent> iter = lg.getElements().iterator(); iter
            .hasNext();) {
         AbstractDisplayComponent adc = (AbstractDisplayComponent) iter.next();
         ++idx;
         if (adc.getId().equals(compID)) {
            break;
         }
      }
      List<ScoreCodeInfo> scoreCodes = si.getScoreCodes();
      if (idx >= 0 && scoreCodes.size() > idx) {
         ScoreCodeInfo sci = scoreCodes.get(idx);
         return sci.getScoreCode();
      }
      return null;
   }

   protected void createDropdown(Writer out, DropdownDisplayComponent ddc,
         int level, String propertyName, String index) throws IOException {
      doIndent(out, level);
      if (!ddc.getPopulatedFromDatabase() && ddc.getCSSClass() != null) {
         out.write("<span class=\"" + ddc.getCSSClass() + "\"> ");
      }
      if (propertyName == null || propertyName.length() == 0) {
         out.write("<select></select> ");
         return;
      }
      if (ddc.getPopulatedFromDatabase() == true) {
         createDynamicDropDown(out, ddc, level, propertyName, index);
         return;
      }

      out.write("<html:select property=");
      createPropertyExpression(out, propertyName, index);
      out.write(">\n");
      for (Map.Entry<String, String> entry : ddc.getLabelValueMap().entrySet()) {
         StringBuilder buf = new StringBuilder();

         String label = (String) entry.getKey();
         String value = (String) entry.getValue();
         buf.append("<html:option value=\"").append(value).append("\">")
               .append(label).append("</html:option>\n");
         doIndent(out, level + 1);
         out.write(buf.toString());
      }
      doIndent(out, level);
      out.write("</html:select>\n");

      if (ddc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   private void createDynamicDropDown(Writer out, DropdownDisplayComponent ddc,
         int level, String propertyName, String index) throws IOException {
      if (ddc.getCSSClass() != null) {
         out.write("<span class=\"" + ddc.getCSSClass() + "\"> ");
      }

      String listBeanName = propertyName + "ValueList";
      String formBeanID = config.getFormBeanID();
      out.write("<bean:define id=\"" + listBeanName + "\" name=\"" + formBeanID
            + "\" property=");
      createPropertyExpression(out, propertyName, index, ".possibleSelections");
      out.write(" type=\"java.util.Collection\"/>\n");

      doIndent(out, level);
      out.write("<html:select property=");
      createPropertyExpression(out, propertyName, index, ".selectedItemIdx");
      out.write(">\n");
      doIndent(out, level + 1);
      out.write("<html:options collection=\"" + listBeanName
            + "\" property=\"value\" labelProperty=\"label\" />\n");
      doIndent(out, level);
      out.write("</html:select>");
      if (ddc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void createCheckBox(Writer out, CheckBoxDisplayComponent cdc,
         int level, String formName, String propertyName, ScoreInfo si,
         ScoreAssociation sa, String index) throws IOException {
      doIndent(out, level);
      if (cdc.getCSSClass() != null) {
         out.write("<span class=\"" + cdc.getCSSClass() + "\"> ");
      }
      if (si == null || sa == null) {
         out.write("<input type=\"checkbox\">" + cdc.getText() + " ");
      } else {

         out.write("<html:checkbox name=\"" + formName + "\" property=");
         createPropertyExpression(out, propertyName, index);

         out.write(" title=\"" + cdc.getText() + "\" />" + cdc.getText() + " ");
      }
      if (cdc.getCSSClass() != null) {
         out.write("<span> ");
      }
      out.write("\n");
   }

   protected void createHeader(Writer out, int indent) throws IOException {
      String dir = props.getProperty("code.snippet.dir");
      String filename = dir + File.separator + "jsp_snippets.txt";
      StringBuilder buf = CodegenUtils.loadCodeSnippet(filename, "header");
      String s = buf.toString();

      s = s.replaceAll("\\$\\{CONTEXT\\}", config.getAppContext());
      out.write(s);
      out.write("\n");
   }

   protected void createSupportJSMethods(Writer out) throws IOException {
      String dir = props.getProperty("code.snippet.dir");
      String filename = dir + File.separator + "jsp_snippets.txt";
      StringBuilder buf = CodegenUtils.loadCodeSnippet(filename, "calcFields");
      assert (buf != null);
      out.write(buf.toString());
      out.write("\n");
   }

   protected void createFooter(Writer out, int indent) throws IOException {
      String dir = props.getProperty("code.snippet.dir");
      String filename = dir + File.separator + "jsp_snippets.txt";
      StringBuilder buf = CodegenUtils.loadCodeSnippet(filename, "footer");
      String s = buf.toString();

      s = s.replaceAll("\\$\\{CONTEXT\\}", config.getAppContext());
      out.write(s);
      out.write("\n");
   }

   public static class DisplayComponentInfo {
      int type;
      List<AbstractDisplayComponent> components = new ArrayList<AbstractDisplayComponent>();

      public DisplayComponentInfo(int type) {
         this.type = type;
      }

      public AbstractDisplayComponent getComponent() {
         if (components.isEmpty()) {
            return null;
         }
         return components.get(0);
      }

      public List<AbstractDisplayComponent> getComponents() {
         return components;
      }

      public int getType() {
         return type;
      }

      public void addComponent(AbstractDisplayComponent adc) {
         components.add(adc);
      }

      public boolean areAllSameComponentType(int compType) {
         for (Iterator<AbstractDisplayComponent> iter = components.iterator(); iter
               .hasNext();) {
            AbstractDisplayComponent adc = iter.next();
            switch (compType) {
            case RADIO_BUTTON:
               if (!(adc instanceof RadioButtonDisplayComponent)) {
                  return false;
               }
               break;
            case CHECK_BOX:
               if (!(adc instanceof CheckBoxDisplayComponent)) {
                  return false;
               }
               break;
            default:
               break;
            }
         }
         return true;
      }

      public String toString() {
         StringBuilder sb = new StringBuilder(128);
         sb.append("DisplayComponentInfo::[");
         sb.append("num components=").append(components.size());
         for (Iterator<AbstractDisplayComponent> iter = components.iterator(); iter
               .hasNext();) {
            AbstractDisplayComponent adc = iter.next();
            sb.append("\n").append(adc.toString());
         }
         sb.append(']');
         return sb.toString();
      }
   }

}
